/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.logstreams.impl.log.fs;

import io.zeebe.dispatcher.impl.PositionUtil;
import io.zeebe.logstreams.impl.Loggers;
import io.zeebe.logstreams.impl.log.fs.FsLogSegment;
import io.zeebe.logstreams.impl.log.fs.FsLogSegmentDescriptor;
import io.zeebe.logstreams.impl.log.fs.FsLogSegments;
import io.zeebe.logstreams.impl.log.fs.FsLogStorageConfiguration;
import io.zeebe.logstreams.spi.LogStorage;
import io.zeebe.logstreams.spi.ReadResultProcessor;
import io.zeebe.util.FileUtil;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import org.slf4j.Logger;

public class FsLogStorage
implements LogStorage {
    public static final Logger LOG = Loggers.LOGSTREAMS_LOGGER;
    private static final int STATE_CREATED = 0;
    private static final int STATE_OPENED = 1;
    private static final int STATE_CLOSED = 2;
    private static final String ERROR_MSG_APPEND_BLOCK_SIZE = "Expected to append block with smaller block size then %d, but actual block size was %d.";
    protected final FsLogStorageConfiguration config;
    private final ReadResultProcessor defaultReadResultProcessor = (buffer, readResult) -> readResult;
    protected volatile int state = 0;
    private FsLogSegments logSegments;
    private FsLogSegment currentSegment;
    private int dirtySegmentId = -1;

    public FsLogStorage(FsLogStorageConfiguration cfg) {
        this.config = cfg;
    }

    @Override
    public long append(ByteBuffer buffer) throws IOException {
        int segmentSize;
        this.ensureOpenedStorage();
        if (this.currentSegment == null) {
            throw new IllegalStateException("Current segment is not initialized.");
        }
        int size = this.currentSegment.getSize();
        int capacity = this.currentSegment.getCapacity();
        int remainingCapacity = capacity - size;
        int requiredCapacity = buffer.remaining();
        if (requiredCapacity > (segmentSize = this.config.getSegmentSize())) {
            throw new IllegalArgumentException(String.format(ERROR_MSG_APPEND_BLOCK_SIZE, segmentSize, requiredCapacity));
        }
        if (remainingCapacity < requiredCapacity) {
            this.onSegmentFilled();
        }
        int appendResult = this.currentSegment.append(buffer);
        long opresult = PositionUtil.position((int)this.currentSegment.getSegmentId(), (int)appendResult);
        this.markSegmentAsDirty(this.currentSegment);
        return opresult;
    }

    @Override
    public void delete(long address) {
        this.ensureOpenedStorage();
        int segmentId = PositionUtil.partitionId((long)address);
        int firstSegmentId = this.logSegments.initialSegmentId;
        int lastSegmentId = this.logSegments.getLastSegmentId();
        if (segmentId > firstSegmentId && segmentId <= lastSegmentId) {
            for (int i = this.logSegments.initialSegmentId; i < segmentId; ++i) {
                FsLogSegment segmentToDelete = this.logSegments.getSegment(i);
                if (segmentToDelete == null) continue;
                segmentToDelete.closeSegment();
                segmentToDelete.delete();
            }
            int diff = segmentId - firstSegmentId;
            LOG.info("Deleted {} segments from log storage ({} to {}).", new Object[]{diff, firstSegmentId, segmentId});
            this.dirtySegmentId = Math.max(this.dirtySegmentId, segmentId);
            this.logSegments.removeSegmentsUntil(segmentId);
        }
    }

    @Override
    public long read(ByteBuffer readBuffer, long addr) {
        return this.read(readBuffer, addr, this.defaultReadResultProcessor);
    }

    @Override
    public long read(ByteBuffer readBuffer, long addr, ReadResultProcessor processor) {
        this.ensureOpenedStorage();
        int segmentId = PositionUtil.partitionId((long)addr);
        int segmentOffset = PositionUtil.partitionOffset((long)addr);
        FsLogSegment segment = this.logSegments.getSegment(segmentId);
        long opStatus = -1L;
        if (segment != null) {
            int readResult = segment.readBytes(readBuffer, segmentOffset);
            if (readResult >= 0) {
                int processingResult = processor.process(readBuffer, readResult);
                opStatus = processingResult < 0 ? (long)processingResult : PositionUtil.position((int)segmentId, (int)(segmentOffset + processingResult));
            } else {
                if (readResult == -3) {
                    long nextAddr = PositionUtil.position((int)(segmentId + 1), (int)FsLogSegmentDescriptor.METADATA_LENGTH);
                    return this.read(readBuffer, nextAddr, processor);
                }
                if (readResult == -2) {
                    opStatus = -2L;
                } else if (readResult == -4) {
                    opStatus = 0L;
                }
            }
        }
        return opStatus;
    }

    @Override
    public boolean isByteAddressable() {
        return true;
    }

    @Override
    public void open() throws IOException {
        this.ensureNotOpenedStorage();
        String path = this.config.getPath();
        File logDir = new File(path);
        logDir.mkdirs();
        this.initLogSegments(logDir);
        this.checkConsistency();
        this.state = 1;
    }

    @Override
    public void close() {
        this.ensureOpenedStorage();
        this.logSegments.closeAll();
        if (this.config.isDeleteOnClose()) {
            String logPath = this.config.getPath();
            try {
                FileUtil.deleteFolder((String)logPath);
            }
            catch (Exception e) {
                LOG.error("Failed to delete folder {}: {}", (Object)logPath, (Object)e);
            }
        }
        this.dirtySegmentId = -1;
        this.state = 2;
    }

    @Override
    public boolean isOpen() {
        return this.state == 1;
    }

    @Override
    public boolean isClosed() {
        return this.state == 2;
    }

    @Override
    public long getFirstBlockAddress() {
        this.ensureOpenedStorage();
        FsLogSegment firstSegment = this.logSegments.getFirst();
        if (firstSegment != null && firstSegment.getSizeVolatile() > FsLogSegmentDescriptor.METADATA_LENGTH) {
            return PositionUtil.position((int)firstSegment.getSegmentId(), (int)FsLogSegmentDescriptor.METADATA_LENGTH);
        }
        return -1L;
    }

    @Override
    public void flush() throws Exception {
        this.ensureOpenedStorage();
        if (this.dirtySegmentId >= 0) {
            for (int id = this.dirtySegmentId; id <= this.currentSegment.getSegmentId(); ++id) {
                FsLogSegment segment = this.logSegments.getSegment(id);
                if (segment != null) {
                    segment.flush();
                    continue;
                }
                LOG.warn("Ignoring segment {} on flush as it does not exist", (Object)id);
            }
            this.dirtySegmentId = -1;
        }
    }

    private void onSegmentFilled() throws IOException {
        FsLogSegment filledSegment = this.currentSegment;
        int nextSegmentId = 1 + filledSegment.getSegmentId();
        String nextSegmentName = this.config.fileName(nextSegmentId);
        FsLogSegment newSegment = new FsLogSegment(nextSegmentName);
        newSegment.allocate(nextSegmentId, this.config.getSegmentSize());
        this.logSegments.addSegment(newSegment);
        this.currentSegment = newSegment;
        filledSegment.setFilled();
    }

    private void initLogSegments(File logDir) throws IOException {
        int initialSegmentId;
        ArrayList<FsLogSegment> readableLogSegments = new ArrayList<FsLogSegment>();
        List<File> logFiles = Arrays.asList(logDir.listFiles(this.config::matchesFragmentFileNamePattern));
        logFiles.forEach(file -> {
            FsLogSegment segment = new FsLogSegment(file.getAbsolutePath());
            if (!segment.openSegment(false)) {
                throw new RuntimeException("Cannot init log segment " + file);
            }
            readableLogSegments.add(segment);
        });
        readableLogSegments.sort(Comparator.comparingInt(FsLogSegment::getSegmentId));
        for (int i = 0; i < readableLogSegments.size() - 1; ++i) {
            FsLogSegment segment = (FsLogSegment)readableLogSegments.get(i);
            segment.setFilled();
        }
        int existingSegments = readableLogSegments.size();
        if (existingSegments > 0) {
            this.currentSegment = (FsLogSegment)readableLogSegments.get(existingSegments - 1);
            initialSegmentId = ((FsLogSegment)readableLogSegments.get(0)).getSegmentId();
        } else {
            initialSegmentId = this.config.getInitialSegmentId();
            String initialSegmentName = this.config.fileName(initialSegmentId);
            int segmentSize = this.config.getSegmentSize();
            FsLogSegment initialSegment = new FsLogSegment(initialSegmentName);
            initialSegment.allocate(initialSegmentId, segmentSize);
            this.currentSegment = initialSegment;
            readableLogSegments.add(initialSegment);
        }
        FsLogSegment[] segmentsArray = readableLogSegments.toArray(new FsLogSegment[readableLogSegments.size()]);
        FsLogSegments logSegments = new FsLogSegments();
        logSegments.init(initialSegmentId, segmentsArray);
        this.logSegments = logSegments;
    }

    private void checkConsistency() {
        try {
            if (!this.currentSegment.isConsistent()) {
                this.currentSegment.truncateUncommittedData();
            }
            if (!this.currentSegment.isConsistent()) {
                throw new RuntimeException("Inconsistent log segment: " + this.currentSegment.getFileName());
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Fail to check consistency", e);
        }
    }

    private void markSegmentAsDirty(FsLogSegment segment) {
        if (this.dirtySegmentId < 0) {
            this.dirtySegmentId = segment.getSegmentId();
        }
    }

    public FsLogStorageConfiguration getConfig() {
        return this.config;
    }

    private void ensureOpenedStorage() {
        if (this.state == 0) {
            throw new IllegalStateException("log storage is not open");
        }
        if (this.state == 2) {
            throw new IllegalStateException("log storage is already closed");
        }
    }

    private void ensureNotOpenedStorage() {
        if (this.state == 1) {
            throw new IllegalStateException("log storage is already opened");
        }
    }
}

