/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.netty;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Ticker;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.StatusException;
import io.grpc.internal.ClientTransport;
import io.grpc.internal.GrpcUtil;
import io.grpc.internal.Http2Ping;
import io.grpc.netty.BufferingHttp2ConnectionEncoder;
import io.grpc.netty.CancelClientStreamCommand;
import io.grpc.netty.CreateStreamCommand;
import io.grpc.netty.GoAwayClosedStreamException;
import io.grpc.netty.NettyClientStream;
import io.grpc.netty.RequestMessagesCommand;
import io.grpc.netty.SendGrpcFrameCommand;
import io.grpc.netty.SendPingCommand;
import io.grpc.netty.WriteQueue;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http2.DefaultHttp2ConnectionDecoder;
import io.netty.handler.codec.http2.Http2CodecUtil;
import io.netty.handler.codec.http2.Http2Connection;
import io.netty.handler.codec.http2.Http2ConnectionAdapter;
import io.netty.handler.codec.http2.Http2ConnectionDecoder;
import io.netty.handler.codec.http2.Http2ConnectionEncoder;
import io.netty.handler.codec.http2.Http2ConnectionHandler;
import io.netty.handler.codec.http2.Http2Error;
import io.netty.handler.codec.http2.Http2Exception;
import io.netty.handler.codec.http2.Http2FrameAdapter;
import io.netty.handler.codec.http2.Http2FrameListener;
import io.netty.handler.codec.http2.Http2FrameReader;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2LocalFlowController;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.handler.codec.http2.Http2Stream;
import io.netty.handler.codec.http2.Http2StreamVisitor;
import io.netty.util.CharsetUtil;
import io.netty.util.concurrent.GenericFutureListener;
import java.util.Random;
import java.util.concurrent.Executor;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

class NettyClientHandler
extends Http2ConnectionHandler {
    private static final Logger logger = Logger.getLogger(NettyClientHandler.class.getName());
    static final Object NOOP_MESSAGE = new Object();
    private static final Status EXHAUSTED_STREAMS_STATUS = Status.UNAVAILABLE.withDescription("Stream IDs have been exhausted");
    private final Http2Connection.PropertyKey streamKey;
    private final Ticker ticker;
    private final Random random = new Random();
    private WriteQueue clientWriteQueue;
    private int flowControlWindow;
    private Http2Settings initialSettings = new Http2Settings();
    private Http2Ping ping;
    private Status goAwayStatus;
    private ChannelHandlerContext ctx;
    private int nextStreamId;

    public NettyClientHandler(BufferingHttp2ConnectionEncoder encoder, Http2Connection connection, Http2FrameReader frameReader, int flowControlWindow) {
        this(encoder, connection, frameReader, flowControlWindow, Ticker.systemTicker());
    }

    @VisibleForTesting
    NettyClientHandler(BufferingHttp2ConnectionEncoder encoder, Http2Connection connection, Http2FrameReader frameReader, int flowControlWindow, Ticker ticker) {
        super((Http2ConnectionDecoder)new DefaultHttp2ConnectionDecoder(connection, (Http2ConnectionEncoder)encoder, frameReader, (Http2FrameListener)new LazyFrameListener()), (Http2ConnectionEncoder)encoder);
        this.ticker = ticker;
        Preconditions.checkArgument((flowControlWindow > 0 ? 1 : 0) != 0, (Object)"flowControlWindow must be positive");
        this.flowControlWindow = flowControlWindow;
        this.initListener();
        this.streamKey = connection.newKey();
        this.nextStreamId = connection.local().nextStreamId();
        connection.addListener((Http2Connection.Listener)new Http2ConnectionAdapter(){

            public void onGoAwayReceived(int lastStreamId, long errorCode, ByteBuf debugData) {
                NettyClientHandler.this.goAwayStatus(NettyClientHandler.this.statusFromGoAway(errorCode, debugData));
                NettyClientHandler.this.goingAway();
            }
        });
        this.initialSettings.pushEnabled(false);
        this.initialSettings.initialWindowSize(flowControlWindow);
        this.initialSettings.maxConcurrentStreams(0L);
    }

    @Nullable
    public Status errorStatus() {
        return this.goAwayStatus;
    }

    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
        this.ctx = ctx;
        super.handlerAdded(ctx);
        this.sendInitialSettings();
    }

    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        super.channelActive(ctx);
        this.sendInitialSettings();
    }

    public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) throws Exception {
        if (msg instanceof CreateStreamCommand) {
            this.createStream((CreateStreamCommand)msg, promise);
        } else if (msg instanceof SendGrpcFrameCommand) {
            this.sendGrpcFrame(ctx, (SendGrpcFrameCommand)((Object)msg), promise);
        } else if (msg instanceof CancelClientStreamCommand) {
            this.cancelStream(ctx, (CancelClientStreamCommand)msg, promise);
        } else if (msg instanceof RequestMessagesCommand) {
            ((RequestMessagesCommand)msg).requestMessages();
        } else if (msg instanceof SendPingCommand) {
            this.sendPingFrame(ctx, (SendPingCommand)msg, promise);
        } else if (msg == NOOP_MESSAGE) {
            ctx.write((Object)Unpooled.EMPTY_BUFFER, promise);
        } else {
            throw new AssertionError((Object)("Write called for unexpected type: " + msg.getClass().getName()));
        }
    }

    void startWriteQueue(Channel channel) {
        this.clientWriteQueue = new WriteQueue(channel);
    }

    WriteQueue getWriteQueue() {
        return this.clientWriteQueue;
    }

    void returnProcessedBytes(Http2Stream stream, int bytes) {
        try {
            this.decoder().flowController().consumeBytes(stream, bytes);
        }
        catch (Http2Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void initListener() {
        ((LazyFrameListener)this.decoder().listener()).setHandler(this);
    }

    private void onHeadersRead(int streamId, Http2Headers headers, boolean endStream) throws Http2Exception {
        NettyClientStream stream = this.clientStream(this.requireHttp2Stream(streamId));
        stream.transportHeadersReceived(headers, endStream);
    }

    private void onDataRead(int streamId, ByteBuf data, boolean endOfStream) throws Http2Exception {
        NettyClientStream stream = this.clientStream(this.requireHttp2Stream(streamId));
        stream.transportDataReceived(data, endOfStream);
    }

    private void onRstStreamRead(int streamId, long errorCode) throws Http2Exception {
        NettyClientStream stream = this.clientStream(this.requireHttp2Stream(streamId));
        Status status = GrpcUtil.Http2Error.statusForCode((int)errorCode);
        stream.transportReportStatus(status, false, new Metadata());
    }

    public void close(ChannelHandlerContext ctx, ChannelPromise promise) throws Exception {
        logger.fine("Network channel being closed by the application.");
        super.close(ctx, promise);
    }

    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        try {
            logger.fine("Network channel is closed");
            this.goAwayStatus(this.goAwayStatus().augmentDescription("Network channel closed"));
            this.cancelPing();
            this.connection().forEachActiveStream(new Http2StreamVisitor(){

                public boolean visit(Http2Stream stream) throws Http2Exception {
                    NettyClientHandler.this.clientStream(stream).transportReportStatus(NettyClientHandler.this.goAwayStatus, false, new Metadata());
                    return true;
                }
            });
        }
        finally {
            super.channelInactive(ctx);
        }
    }

    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        if (Http2CodecUtil.getEmbeddedHttp2Exception((Throwable)cause) == null) {
            this.goAwayStatus(Status.fromThrowable(cause));
            cause = new Http2Exception(Http2Error.INTERNAL_ERROR, null, cause);
        }
        super.exceptionCaught(ctx, cause);
    }

    protected void onConnectionError(ChannelHandlerContext ctx, Throwable cause, Http2Exception http2Ex) {
        logger.log(Level.FINE, "Caught a connection error", cause);
        this.goAwayStatus(this.statusFromError(cause));
        super.onConnectionError(ctx, cause, http2Ex);
    }

    protected void onStreamError(ChannelHandlerContext ctx, Throwable cause, Http2Exception.StreamException http2Ex) {
        Http2Stream stream = this.connection().stream(http2Ex.streamId());
        if (stream != null) {
            this.clientStream(stream).transportReportStatus(this.statusFromError(cause), false, new Metadata());
        }
        super.onStreamError(ctx, cause, http2Ex);
    }

    private Status statusFromError(Throwable cause) {
        return cause instanceof Http2Exception ? Status.INTERNAL.withCause(cause) : Status.fromThrowable(cause);
    }

    protected boolean isGracefulShutdownComplete() {
        return super.isGracefulShutdownComplete() && ((BufferingHttp2ConnectionEncoder)this.encoder()).numBufferedStreams() == 0;
    }

    private void createStream(CreateStreamCommand command, ChannelPromise promise) throws Exception {
        int streamId;
        try {
            streamId = this.getAndIncrementNextStreamId();
        }
        catch (StatusException e) {
            promise.setFailure((Throwable)e);
            if (!this.connection().goAwaySent()) {
                logger.fine("Stream IDs have been exhausted for this connection. Initiating graceful shutdown of the connection.");
                super.close(this.ctx, this.ctx.newPromise());
            }
            return;
        }
        final NettyClientStream stream = command.stream();
        Http2Headers headers = command.headers();
        stream.id(streamId);
        this.encoder().writeHeaders(this.ctx, streamId, headers, 0, false, promise).addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    Http2Stream http2Stream = NettyClientHandler.this.connection().stream(streamId);
                    if (http2Stream != null) {
                        http2Stream.setProperty(NettyClientHandler.this.streamKey, (Object)stream);
                    }
                    stream.setHttp2Stream(http2Stream);
                } else if (future.cause() instanceof GoAwayClosedStreamException) {
                    GoAwayClosedStreamException e = (GoAwayClosedStreamException)future.cause();
                    NettyClientHandler.this.goAwayStatus(NettyClientHandler.this.statusFromGoAway(e.errorCode(), e.debugData()));
                    stream.transportReportStatus(NettyClientHandler.this.goAwayStatus, false, new Metadata());
                } else {
                    stream.transportReportStatus(Status.fromThrowable(future.cause()), true, new Metadata());
                }
            }
        });
    }

    private void cancelStream(ChannelHandlerContext ctx, CancelClientStreamCommand cmd, ChannelPromise promise) {
        NettyClientStream stream = cmd.stream();
        stream.transportReportStatus(cmd.reason(), true, new Metadata());
        this.encoder().writeRstStream(ctx, stream.id().intValue(), Http2Error.CANCEL.code(), promise);
    }

    private void sendGrpcFrame(ChannelHandlerContext ctx, SendGrpcFrameCommand cmd, ChannelPromise promise) {
        this.encoder().writeData(ctx, cmd.streamId(), cmd.content(), 0, cmd.endStream(), promise);
    }

    private void sendPingFrame(ChannelHandlerContext ctx, SendPingCommand msg, ChannelPromise promise) {
        ClientTransport.PingCallback callback = msg.callback();
        Executor executor = msg.executor();
        if (!ctx.channel().isOpen()) {
            Http2Ping.notifyFailed(callback, executor, this.goAwayStatus().asException());
            return;
        }
        if (this.ping != null) {
            this.ping.addCallback(callback, executor);
            return;
        }
        long data = this.random.nextLong();
        ByteBuf buffer = ctx.alloc().buffer(8);
        buffer.writeLong(data);
        Stopwatch stopwatch = Stopwatch.createStarted((Ticker)this.ticker);
        this.ping = new Http2Ping(data, stopwatch);
        this.ping.addCallback(callback, executor);
        this.encoder().writePing(ctx, false, buffer, promise);
        ctx.flush();
        final Http2Ping finalPing = this.ping;
        promise.addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (!future.isSuccess()) {
                    finalPing.failed(future.cause());
                    if (NettyClientHandler.this.ping == finalPing) {
                        NettyClientHandler.this.ping = null;
                    }
                }
            }
        });
    }

    private void goingAway() {
        final Status goAwayStatus = this.goAwayStatus();
        final int lastKnownStream = this.connection().local().lastStreamKnownByPeer();
        try {
            this.connection().forEachActiveStream(new Http2StreamVisitor(){

                public boolean visit(Http2Stream stream) throws Http2Exception {
                    if (stream.id() > lastKnownStream) {
                        NettyClientHandler.this.clientStream(stream).transportReportStatus(goAwayStatus, false, new Metadata());
                        stream.close();
                    }
                    return true;
                }
            });
        }
        catch (Http2Exception e) {
            throw new RuntimeException(e);
        }
    }

    private Status goAwayStatus() {
        if (this.goAwayStatus != null) {
            return this.goAwayStatus;
        }
        return Status.UNAVAILABLE.withDescription("Connection going away, but for unknown reason");
    }

    private void goAwayStatus(Status status) {
        this.goAwayStatus = this.goAwayStatus == null ? status : this.goAwayStatus;
    }

    private void cancelPing() {
        if (this.ping != null) {
            this.ping.failed(this.goAwayStatus().asException());
            this.ping = null;
        }
    }

    private Status statusFromGoAway(long errorCode, ByteBuf debugData) {
        Status status = GrpcUtil.Http2Error.statusForCode((int)errorCode);
        if (debugData.isReadable()) {
            String msg = debugData.toString(CharsetUtil.UTF_8);
            status = status.augmentDescription(msg);
        }
        return status;
    }

    private NettyClientStream clientStream(Http2Stream stream) {
        return (NettyClientStream)stream.getProperty(this.streamKey);
    }

    private int getAndIncrementNextStreamId() throws StatusException {
        if (this.nextStreamId < 0) {
            logger.fine("Stream IDs have been exhausted for this connection. Initiating graceful shutdown of the connection.");
            throw EXHAUSTED_STREAMS_STATUS.asException();
        }
        int id = this.nextStreamId;
        this.nextStreamId += 2;
        return id;
    }

    private Http2Stream requireHttp2Stream(int streamId) {
        Http2Stream stream = this.connection().stream(streamId);
        if (stream == null) {
            throw new AssertionError((Object)("Stream does not exist: " + streamId));
        }
        return stream;
    }

    private void sendInitialSettings() throws Http2Exception {
        if (!this.ctx.channel().isActive()) {
            return;
        }
        boolean needToFlush = false;
        if (this.initialSettings != null) {
            needToFlush = true;
            this.encoder().writeSettings(this.ctx, this.initialSettings, this.ctx.newPromise());
            this.initialSettings = null;
        }
        if (this.flowControlWindow > 0) {
            needToFlush = true;
            Http2Stream connectionStream = this.connection().connectionStream();
            int currentSize = ((Http2LocalFlowController)this.connection().local().flowController()).windowSize(connectionStream);
            int delta = this.flowControlWindow - currentSize;
            this.decoder().flowController().incrementWindowSize(connectionStream, delta);
            this.flowControlWindow = -1;
        }
        if (needToFlush) {
            this.ctx.flush();
        }
    }

    private static class LazyFrameListener
    extends Http2FrameAdapter {
        private NettyClientHandler handler;

        private LazyFrameListener() {
        }

        void setHandler(NettyClientHandler handler) {
            this.handler = handler;
        }

        public int onDataRead(ChannelHandlerContext ctx, int streamId, ByteBuf data, int padding, boolean endOfStream) throws Http2Exception {
            this.handler.onDataRead(streamId, data, endOfStream);
            return padding;
        }

        public void onHeadersRead(ChannelHandlerContext ctx, int streamId, Http2Headers headers, int streamDependency, short weight, boolean exclusive, int padding, boolean endStream) throws Http2Exception {
            this.handler.onHeadersRead(streamId, headers, endStream);
        }

        public void onRstStreamRead(ChannelHandlerContext ctx, int streamId, long errorCode) throws Http2Exception {
            this.handler.onRstStreamRead(streamId, errorCode);
        }

        public void onPingAckRead(ChannelHandlerContext ctx, ByteBuf data) throws Http2Exception {
            Http2Ping p = this.handler.ping;
            if (p != null) {
                long ackPayload = data.readLong();
                if (p.payload() == ackPayload) {
                    p.complete();
                    this.handler.ping = null;
                } else {
                    logger.log(Level.WARNING, String.format("Received unexpected ping ack. Expecting %d, got %d", p.payload(), ackPayload));
                }
            } else {
                logger.warning("Received unexpected ping ack. No ping outstanding");
            }
        }
    }
}

