package com.gtis.portal.web.config;

import com.gtis.config.AppConfig;
import com.gtis.portal.model.FjStoreInfo;
import cn.gtmap.estateplat.core.support.mybatis.page.model.Page;
import cn.gtmap.estateplat.core.support.mybatis.page.model.PageImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Pageable;
import org.springframework.data.querydsl.QPageRequest;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ch.ethz.ssh2.Connection;
import ch.ethz.ssh2.Session;
import ch.ethz.ssh2.StreamGobbler;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

/**
 * @author <a href="mailto:liangxichang@gtmap.cn">liangxichang</a>
 * @version 1.0 , 2020/3/24
 * @description 用于fileCenter中附件所存储服务器磁盘容量使用的显示信息
 **/
@Controller
@RequestMapping("/showFileStore")
public class ShowFileStoreController {

    private final Logger log = LoggerFactory.getLogger(ShowFileStoreController.class);

    //获取要查询的服务器信息
    private List<FjStoreInfo> fjStoreInfoList;

    @RequestMapping("")
    public String showFileStore(Model model) {
        getFjStoreInfoList();
        model.addAttribute("fjStoreInfoList",fjStoreInfoList);
        return "config/show-fileCenter-fileStore";
    }

    @RequestMapping("/getShowFileStorePagesJson")
    @ResponseBody
    public Object getShowFileStorePagesJson(Pageable pageable, String name, String host, @RequestParam(value = "number", required = false) String number, @RequestParam(value = "currentPage", required = false) String currentPage){
        int pageNumber = pageable.getPageNumber();
        if (currentPage != null)
            pageNumber = Integer.parseInt(currentPage) - 1;

        QPageRequest pageable1 = new QPageRequest(pageNumber, 10);
        name = StringUtils.trim(name);
        host = StringUtils.trim(host);
        List<FjStoreInfo> fjStoreInfoListTemp = new ArrayList<FjStoreInfo>();
        getFjStoreInfoList();
        if (CollectionUtils.isNotEmpty(fjStoreInfoList)) {
            if (StringUtils.isNotBlank(name) && StringUtils.isNotBlank(host)) {
                for (FjStoreInfo fjStoreInfo : fjStoreInfoList) {
                    if(StringUtils.equals(fjStoreInfo.getName(),name) && StringUtils.equals(fjStoreInfo.getHost(),host)) {
                        fjStoreInfoListTemp.add(fjStoreInfo);
                    }
                }
            } else if (StringUtils.isNotBlank(name)) {
                for (FjStoreInfo fjStoreInfo : fjStoreInfoList) {
                    if(StringUtils.equals(fjStoreInfo.getName(),name)) {
                        fjStoreInfoListTemp.add(fjStoreInfo);
                    }
                }
            } else if (StringUtils.isNotBlank(host)) {
                for (FjStoreInfo fjStoreInfo : fjStoreInfoList) {
                    if(StringUtils.equals(fjStoreInfo.getHost(),host)) {
                        fjStoreInfoListTemp.add(fjStoreInfo);
                    }
                }
            } else {
                fjStoreInfoListTemp = fjStoreInfoList;
            }
        }
        org.springframework.data.domain.Page<FjStoreInfo> FjStoreInfoPage = new org.springframework.data.domain.PageImpl<FjStoreInfo>(fjStoreInfoListTemp,pageable1,fjStoreInfoListTemp.size());


        Page<FjStoreInfo> FjStoreInfoPage1 = null;
        if (FjStoreInfoPage != null) {
            FjStoreInfoPage1 = new PageImpl(FjStoreInfoPage.getContent(), pageNumber + 1, Integer.parseInt(FjStoreInfoPage.getTotalElements() + ""), 10);
        }
        return FjStoreInfoPage1;
    }

    /**
     * 获取所有磁盘列表信息
     */
    private void getFjStoreInfoList() {
        fjStoreInfoList = new ArrayList<FjStoreInfo>();
        String nameVal = AppConfig.getProperty("fileCenter.showStoreInfo.names");
        String hostVal = AppConfig.getProperty("fileCenter.showStoreInfo.values");

        if (StringUtils.isNotBlank(nameVal) && StringUtils.isNotBlank(hostVal)) {
            String[] nameSplit = nameVal.split(",");
            String[] hostSplit = hostVal.split(",");
            if (nameSplit.length == hostSplit.length) {
                for (int i = 0; i<hostSplit.length; i++) {
                    String[] hostSplit1 = hostSplit[i].split("&");
                    //存储单个的详细信息
                    FjStoreInfo fjStoreInfo = new FjStoreInfo();
                    if (hostSplit1.length == 4) {
                        //远程服务器
                        fjStoreInfo.setName(nameSplit[i]);
                        fjStoreInfo.setHost(hostSplit1[0]);
                        fjStoreInfo.setLoginName(hostSplit1[1]);
                        fjStoreInfo.setPassWord(hostSplit1[2]);
                        fjStoreInfo.setPath(hostSplit1[3].substring(0,2));//只保存盘符
                        fjStoreInfoList.add(fjStoreInfo);
                    } else if (hostSplit1.length == 2) {
                        //本地服务器
                        fjStoreInfo.setName(nameSplit[i]);
                        fjStoreInfo.setHost(hostSplit1[0]);
                        fjStoreInfo.setPath(hostSplit1[1].substring(0,2));
                        fjStoreInfoList.add(fjStoreInfo);
                    }else {
                        log.error("portal配置fileCenter的host错误");
                    }
                }
            } else {
                log.error("portal配置fileCenter的storeInfo错误");
            }

        }
        if (CollectionUtils.isNotEmpty(fjStoreInfoList)) {
            //获取存储信息
            fjStoreInfoList = getFjStoreInfoFromList(fjStoreInfoList);
        }

    }
    /**
     * 获取附件存储的容量信息
     * @param fjStoreInfoList fjStoreInfoList
     * @return List<FjStoreInfo>
     */
    private List<FjStoreInfo> getFjStoreInfoFromList(List<FjStoreInfo> fjStoreInfoList) {
        List<FjStoreInfo> fjStoreInfos = new ArrayList<FjStoreInfo>();
        for (FjStoreInfo fjStoreInfo : fjStoreInfoList) {
            FjStoreInfo diskInfo = null;
            if(StringUtils.isNotBlank(fjStoreInfo.getLoginName()) && StringUtils.isNotBlank(fjStoreInfo.getPassWord())) {
                diskInfo = getRemoteDiskInfo(fjStoreInfo);
            } else {
                diskInfo = getLocalDiskInfo(fjStoreInfo);
            }

            if(diskInfo != null){
                fjStoreInfos.add(diskInfo);
            }

        }
        return fjStoreInfos;
    }

    /**
     * 获取本地服务的磁盘信息
     * @param fjStoreInfo fjStoreInfo
     * @return 磁盘信息
     */
    private FjStoreInfo getLocalDiskInfo(FjStoreInfo fjStoreInfo) {
        long totalSpace = 0;//磁盘总容量
        long freeSpace = 0;
        long usedSpace = 0;
        try {
            Process process = Runtime.getRuntime().exec("cmd.exe /c "+"wmic LogicalDisk where \"Caption='"+
                    fjStoreInfo.getPath()+"'\" get FreeSpace,Size /value");
            InputStream stdout = new StreamGobbler(process.getInputStream());
            BufferedReader br = new BufferedReader(new InputStreamReader(stdout, Charset.forName("GBK")));
            String len = null;
            while((len = br.readLine()) != null){
                if(len.startsWith("FreeSpace")) {
                    String[] str = len.split("=");
                    freeSpace = Long.parseLong(str[1])/1024/ 1024;
                }
                if(len.startsWith("Size")){
                    String[] str = len.split("=");
                    totalSpace = Long.parseLong(str[1])/ 1024 / 1024;
                }
            }
            usedSpace = totalSpace - freeSpace;
            log.info("总空间大小 : " + totalSpace + "M");
            log.info("剩余空间大小 : " + freeSpace + "M");
            log.info("已用空间大小 : " + usedSpace + "M");

            fjStoreInfo.setMaxCapacity(totalSpace + "M");
            fjStoreInfo.setUsedCapacity(usedSpace + "M");
            fjStoreInfo.setProportion((int)(usedSpace*100/totalSpace)+ "%");

        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            //清除FjStoreInfo里的用户名及密码信息
            fjStoreInfo.setLoginName("***");
            fjStoreInfo.setPassWord("***");
        }


        return fjStoreInfo;
    }


    /**
     * 获取远程服务的磁盘信息
     * @param fjStoreInfo fjStoreInfo
     * @return 磁盘信息
     */
    private FjStoreInfo getRemoteDiskInfo(FjStoreInfo fjStoreInfo) {
        HashMap<String,Long> map = new HashMap<String,Long>();
        Connection conn = new Connection(fjStoreInfo.getHost(), 22);
        Session ssh = null;

        try {
            conn.connect();
            boolean flag = conn.authenticateWithPassword(fjStoreInfo.getLoginName(), fjStoreInfo.getPassWord());
            if (!flag) {
                System.out.println("用户名或密码错误");
            } else {
                log.info("连接成功");
                ssh = conn.openSession();
                //windows
                map = getWindowsServerDiskInfo(ssh,map,fjStoreInfo.getPath());
                fjStoreInfo.setMaxCapacity(map.get("total") + "M");
                fjStoreInfo.setUsedCapacity(map.get("used") + "M");
                fjStoreInfo.setProportion((int)(map.get("used")*100/map.get("total")) + "%");
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            //清除FjStoreInfo里的用户名及密码信息
            fjStoreInfo.setLoginName("***");
            fjStoreInfo.setPassWord("***");
            try {
                if (ssh != null) {
                    ssh.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return fjStoreInfo;
    }

    /**
     *获取Windows服务器磁盘信息
     * @param ssh ssh
     * @param map map
     * @param diskPath 磁盘
     * @return Map
     * @throws IOException
     */
    private HashMap<String,Long> getWindowsServerDiskInfo(Session ssh, HashMap<String,Long> map, String diskPath) throws IOException {
        long totalSpace = 0;//磁盘总容量
        long freeSpace = 0;
        long usedSpace = 0;
        ssh.execCommand("cmd.exe /c "+"wmic LogicalDisk where \"Caption='"+
                diskPath+"'\" get FreeSpace,Size /value");
        InputStream stdout = new StreamGobbler(ssh.getStdout());
        BufferedReader br = new BufferedReader(new InputStreamReader(stdout, Charset.forName("GBK")));
        String len = null;
        while((len = br.readLine()) != null){
            if(len.startsWith("FreeSpace")) {
                String[] str = len.split("=");
                freeSpace = Long.parseLong(str[1])/1024/ 1024;
            }
            if(len.startsWith("Size")){
                String[] str = len.split("=");
                totalSpace = Long.parseLong(str[1])/ 1024 / 1024;
            }
        }
        usedSpace = totalSpace - freeSpace;
        log.info("总空间大小 : " + totalSpace + "M");
        log.info("剩余空间大小 : " + freeSpace + "M");
        log.info("已用空间大小 : " + usedSpace + "M");
        map.put("total", totalSpace);
        map.put("free", freeSpace);
        map.put("used", usedSpace);

        return map;
    }


}
