package com.gtis.plat.service.impl;

import java.util.ArrayList;
import java.util.List;

import com.gtis.plat.dao.SysUserDAO;
import com.gtis.plat.service.SysUserService;
import com.gtis.plat.vo.PfConfigVo;
import com.gtis.plat.vo.PfOrganVo;
import com.gtis.plat.vo.PfRoleVo;
import com.gtis.plat.vo.PfTaskAgentVo;
import com.gtis.plat.vo.PfUserVo;
import com.gtis.plat.wf.model.PerformerModel;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;

public class SysUserServiceImpl implements SysUserService {
    /** 用户持久服务 */
    private SysUserDAO userDao;

    @Cacheable(value="UserCache",key="#userName+'UserListByName'")
    public List<PfUserVo> getUserVoListByUserName(String userName) {
        return userDao.getUserVoListByUserName(userName);
    }

    @Cacheable(value="UserCache",key="#userId+'UserOrganListById'")
    public List<PfOrganVo> getOrganListByUser(String userId) {
        return userDao.getOrganListByUser(userId);
    }

    @Cacheable(value="UserCache",key="#userId+'UserRoleListById'")
    public List<PfRoleVo> getRoleListByUser(String userId) {
        return userDao.getRoleListByUser(userId);
    }

    @Cacheable(value="UserCache",key="#loginName+'UserByLoginName'")
    public PfUserVo getUserByloginName(String loginName) {
        return userDao.getUserByloginName(loginName);
    }

    @Cacheable(value="UserCache",key="#userId+'RoleListByUserId'")
    public List<String> getRoleListByAgentUser(String userId) {
        return userDao.getRoleListByAgentUser(userId);
    }


    public List<PfUserVo> getUserListByOragn(String oragnId) {
        return userDao.getUserListByOragn(oragnId);
    }

    @Cacheable(value="UserCache",key="#roleId+'UserListByRoleId'")
    public List<PfUserVo> getUserListByRole(String roleId) {
        return userDao.getUserListByRole(roleId);
    }

    @Cacheable(value="UserCache",key="#userId+'UserById'")
    public PfUserVo getUserVo(String userId) {
        return userDao.getUserVo(userId);
    }

    @Cacheable(value="UserCache",key="#roleId+'RoleById'")
    public PfRoleVo getRoleVo(String roleId) {
        return userDao.getRoleVo(roleId);
    }

    @Cacheable(value="UserCache",key="#organId+'OrganById'")
    public PfOrganVo getOrganVo(String organId) {
        return userDao.getOrganVo(organId);
    }

    @Cacheable(value="UserCache",key="#organName+'OrganByName'")
    public PfOrganVo getOrganVoByName(String organName){
        return userDao.getOrganVoByName(organName);
    }

    @Cacheable(value="UserCache",key="#userId+'UserConfig'")
    public PfConfigVo getConfigVo(String userId) {
		PfConfigVo configVo=userDao.getConfigVo(userId);
		if (configVo==null){   //默认信息
			configVo=new PfConfigVo();
			configVo.setUserId(userId);
			configVo.setMessageAutoTime(5);
			configVo.setTaskAutoTime(2);
			configVo.setShowAllNewTask(true);
			configVo.setShowNewTaskHandle(true);
		}
		return configVo;
	}

    @Caching(evict={
            @CacheEvict(value="UserCache",key = "#vo.userId + 'UserConfig'")
    })
	public void saveConfig(PfConfigVo vo) {
		PfConfigVo configVo=userDao.getConfigVo(vo.getUserId());
		if (configVo==null)
			userDao.insertConfig(vo);
		else
			userDao.saveConfig(vo);
	}

    /**
     * 根据地域码获得部门列表
     * @param regionCode
     * @return
     */
    @Cacheable(value="UserCache",key="#regionCode+'OrganByRegionCode'")
    public List<PfOrganVo> getOrganByRegionCode(String regionCode){
        List<PfOrganVo> lstOrgan =this.getOrganList();

        List<PfOrganVo> organList = new ArrayList<PfOrganVo>();
        if(StringUtils.isNotBlank(regionCode)){
            for(PfOrganVo organ : lstOrgan){
                if(StringUtils.isNotBlank(organ.getRegionCode())){
                    if(regionCode.equals(organ.getRegionCode())){
                        organList.add(organ);    
                    }
                }else {
                    if(StringUtils.isNotBlank(organ.getSuperOrganId())){
                        PfOrganVo o = dualRegionCode(lstOrgan,organ,regionCode);
                        if(o != null){
                            organList.add(o);    
                        }
                    }
                }
            }
        }

        return organList;
    }

    /**
     * 根据用户Id获得地狱码
     * @param userId
     * @return
     */
    @Cacheable(value="UserCache",key="#userId+'RegionCodeByUserId'")
    public String getRegionCodeByUserId(String userId){
        List<PfOrganVo> lstOrgan =this.getOrganList();
        
        List<PfOrganVo> organList = this.getOrganListByUser(userId);
        if(organList.size() > 0){
            return dualRegionCode(lstOrgan,organList.get(0));    
        }
        return null;
    }

    /**
	 * 根据参与者获取所有的人员Id
	 * @param lstPerformers
	 * @return
	 */
	public List<String> getAllUserListPerformers(
			List<PerformerModel> lstPerformers) {
		List<String> lstUserIds = new ArrayList<String>();
		for (PerformerModel performer : lstPerformers) {
			if (performer.getUserId() != null && !performer.getUserId().equals("")) {
				if (!lstUserIds.contains(performer.getUserId())) {
					lstUserIds.add(performer.getUserId());
				}
			} else if (performer.getOrganId() != null && !performer.getOrganId().equals("")) {
				List<PfUserVo> lstUsers = getUserListByOragn(performer
						.getOrganId());
				for (PfUserVo userVo : lstUsers) {
					if (!lstUserIds.contains(userVo.getUserId())) {
						lstUserIds.add(userVo.getUserId());
					}
				}
			} else if (performer.getRoleId() != null) {
				List<PfUserVo> lstUsers = getUserListByRole(performer
						.getRoleId());
				for (PfUserVo userVo : lstUsers) {
					if (!lstUserIds.contains(userVo.getUserId())) {
						lstUserIds.add(userVo.getUserId());
					}
				}
			}
		}
		return lstUserIds;
	}

    /**
     * 获取所有和创建者在同一个部门的人员
     * @param lstPerformers
     * @param userId
     * @return
     */
	public List<String> getUserListPerformersByUserOrgan(
			List<PerformerModel> lstPerformers, String userId) {
		List<PfOrganVo> lstOrgans = this.getOrganListByUser(userId);
		List<String> lstOrganIds = new ArrayList<String>();
		for (PfOrganVo organVo : lstOrgans) {
			lstOrganIds.add(organVo.getOrganId());
		}

		List<String> lstUserIds = new ArrayList<String>();
		for (PerformerModel performer : lstPerformers) {
			if (StringUtils.isNotBlank(performer.getUserId())) {
				if (!lstUserIds.contains(performer.getUserId())) {
					List<PfOrganVo> Organs = this.getOrganListByUser(performer.getUserId());
					for (PfOrganVo organVo : lstOrgans) {
						if (lstOrganIds.contains(organVo.getOrganId())){
							lstUserIds.add(performer.getUserId());
							break;
						}
					}
				}
			} else if (StringUtils.isNotBlank(performer.getOrganId())){
				if (lstOrganIds.contains(performer.getOrganId())) {
					List<PfUserVo> lstUsers = getUserListByOragn(performer
							.getOrganId());
					for (PfUserVo userVo : lstUsers) {
						if (!lstUserIds.contains(userVo.getUserId())) {
							lstUserIds.add(userVo.getUserId());
						}
					}
				}
			} else if (StringUtils.isNotBlank(performer.getRoleId())) {
//				lstUserIds = userDao.getUserListByRoleUser(performer.getRoleId(),userId);
                List<String> userIdlist = userDao.getUserListByRoleUser(performer.getRoleId(),userId);
                for (String userid : userIdlist) {
                    if (!lstUserIds.contains(userid)) {
                        lstUserIds.add(userid);
                    }
                }
			}
		}
		return lstUserIds;
	}

    /**
     * 获得同一regionCode的部门
     * @param organList
     * @param regionCode
     * @return
     */
    private PfOrganVo dualRegionCode(List<PfOrganVo> organList,PfOrganVo organ,String regionCode) {
        for(PfOrganVo o : organList){
            if(o.getOrganId().equals(organ.getSuperOrganId())){
                if(StringUtils.isNotBlank(o.getRegionCode())){
                    if(o.getRegionCode().equals(regionCode)){
                        return organ;
                    }
                }else {
                    if(StringUtils.isNotBlank(o.getSuperOrganId())){
                        if(dualRegionCode(organList,o,regionCode) != null){
                            return organ;   
                        }
                    }
                }
            }
        }
        return null;
    }

    /**
     * 获得部门的regionCode，如果没有返回空
     * @param organList
     * @param organ
     * @return
     */
    private String dualRegionCode(List<PfOrganVo> organList,PfOrganVo organ){
        if(StringUtils.isNotBlank(organ.getRegionCode())){
            return organ.getRegionCode();
        }else {
            if(StringUtils.isNotBlank(organ.getSuperOrganId())){
                for(PfOrganVo o : organList){
                    if(o.getOrganId().equals(organ.getSuperOrganId())){
                        if(StringUtils.isNotBlank(o.getRegionCode())){
                            return o.getRegionCode();
                        }else {
                            return dualRegionCode(organList,o);
                        }
                    }
                }
            }
        }

        return null;
    }

    /**
     * 查询一个部门和角色下的人员
     * @param roleId
     * @param organId
     * @return
     */
    @Cacheable(value="UserCache",key="#roleId+#organId+'UserListByRoleAndOrgan'")
	public List<PfUserVo> getUserListByRoleAndOrgan(String roleId,String organId) {
		return userDao.getUserListByRoleAndOrgan(roleId,organId);
	}

    /**
     * 查询多个部门和角色下的人员
     * @param roleId
     * @param organList
     * @return
     */
    public List<PfUserVo> getUserListByRoleAndOrganIds(String roleId, List<String> organList) {
        return userDao.getUserListByRoleAndOrganIds(roleId,organList);
    }

    /**
     * 获取有效用户代理人列表
     * @return
     */
    @Cacheable(value="UserAgentCache",key="#userId+'UserAgentListByUserId'")
	public String getUserAgentList(String userId) {
		List<PfUserVo> lstUsers = userDao.getUserAgentList(userId);
		StringBuffer strBuffer = new StringBuffer("");
		for (PfUserVo userVo : lstUsers) {
			strBuffer.append("'" + userVo.getUserId() + "',");
		}
		strBuffer.append("'" + userId + "'");
		return strBuffer.toString();
	}

    /**
     * 获得有效任务代理信息列表
     * @param userId
     * @return
     */
    @Cacheable(value="UserAgentCache",key="#userId+'AgentUserListByUserId'")
    public List<PfTaskAgentVo> getAgentUserList(String userId) {
    	if(userId.equals("0")){
    		return userDao.getAllAgentUserList();
    	}else{
    		return userDao.getAgentUserList(userId);
    	}
    }

    @Cacheable(value="UserCache",key="#roleName+'RoleVoListByRoleNameFuzzy'")
    public List<PfRoleVo> getRoleVoListByRoleNameFuzzy(String roleName){
        return userDao.getRoleVoListByRoleNameFuzzy(roleName);
    }
    /**
     * 删除任务代理
     */
    @Caching(evict={
            @CacheEvict(value = "UserAgentCache", allEntries = true)
    })
    public void deleteTaskAgent(String taskAgentId) {
		userDao.deleteTaskAgent(taskAgentId);
	}

    public List<PfTaskAgentVo> getTaskAgentListByUser(String userId){
       return userDao.getTaskAgentListByUser(userId);
    }
    /**
     * 终止任务代理
     * 
     */
    @Caching(evict={
            @CacheEvict(value = "UserAgentCache", allEntries = true)
    })
    public void stopTaskAgent(String taskAgentId){
    	userDao.stopTaskAgent(taskAgentId);
    }

    /**
     * 增加任务代理
     */
    @Caching(evict={
            @CacheEvict(value = "UserAgentCache", allEntries = true)
    })
	public void insertTaskAgent(PfTaskAgentVo pfTaskAgentVo) {
		userDao.insertTaskAgent(pfTaskAgentVo);
	}


	/**
	 * 更新任务代理
	 */
    @Caching(evict={
            @CacheEvict(value = "UserAgentCache", allEntries = true)
    })
	public void updateTaskAgent(PfTaskAgentVo pfTaskAgentVo){
		userDao.updateTaskAgent(pfTaskAgentVo);
	}
	/**
	 * 查看任务代理
	 */
    @Cacheable(value="UserAgentCache",key="#taskAgentId+'TaskAgent'")
	public PfTaskAgentVo getTaskAgent(String taskAgentId){
		return userDao.getTaskAgent(taskAgentId);
	}
	/**
	 * 更新用户信息
	 */
    @Caching(evict={
            @CacheEvict(value = "UserCache", allEntries = true)
    })
	public void updateUser(PfUserVo pfUserVo) {
		userDao.updateUser(pfUserVo);
	}

    @Cacheable(value="UserCache",key="'AllUsers'")
    public List<PfUserVo> getAllUsers(){
		return userDao.getAllUsers();
	}

	public List<PfOrganVo> getAllOrganList(){
		return  this.getOrganList();
	}

    @Cacheable(value="UserCache",key="'AllOrganList'")
    public List<PfOrganVo> getOrganList() {
		return userDao.getOrganList();
	}

	public List<PfOrganVo> getOrganListByRemark(String remark) {
		return userDao.getOrganListByRemark(remark);
	}

    @Cacheable(value="UserCache",key="'AllRoleList'")
	public List<PfRoleVo> getRoleList() {
		return userDao.getRoleList();
	}

    @Cacheable(value="UserCache",key="#userId+'UserRegionCode'")
	public String getUserRegionCode(String userId) {
		return userDao.getUserRegionCode(userId);
	}

    @Cacheable(value="UserCache",key="#orginName+'UsersByOrganName'")
	public List<PfUserVo> getUsersByOrganName(String orginName) {
		return userDao.getUsersByOrganName(orginName);
	}

    public boolean savePassWord(String userId, String loginPassWord) {
		return userDao.savePassWord(userId, loginPassWord);
	}

    @Cacheable(value="UserCache",key="#userName+'UserListByUserNameFuzzy'")
    public List<PfUserVo> getUserVoListByUserNameFuzzy(String userName) {
        return userDao.getUserVoListByUserNameFuzzy(userName);
    }

	public SysUserDAO getUserDao() {
		return userDao;
	}
	public void setUserDao(SysUserDAO userDao) {
		this.userDao = userDao;
	}

    @Cacheable(value="UserCache",key="#regionCode+'OrganListByRegionCode'")
    public List<PfOrganVo> getOrganListByRegionCode(String regionCode){
        return this.userDao.getOrganListByRegionCode(regionCode);
    }

}
