/*
 * Author: xyang
 *
 * Project: egov
 *
 * File: AbstractPortalManager.java
 *
 * LastModified: 2009-11-30 07:55:09
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.plat.portal.impl;

import com.gtis.plat.portal.PortalDao;
import com.gtis.plat.portal.PortalManager;
import com.gtis.plat.portal.model.*;
import com.gtis.plat.service.CountProvider;
import com.gtis.plat.service.SysUserService;
import com.gtis.plat.vo.PfRoleVo;
import com.thoughtworks.xstream.XStream;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.util.ResourceUtils;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-11-30
 */
public class PortalManagerImpl implements PortalManager {

    public static final String PORTLETAPPS_CONFIG = "classpath:portlets.xml";

    private Site globalSite;
    private List<PortletApp> portletApps;
    private Map<String, PortletEntity> pes;
    private XStream sm;
    private List<CountProvider> countProviders;
    private SysUserService sysUserService;

    public void setSysUserService(SysUserService sysUserService) {
        this.sysUserService = sysUserService;
    }

    public void setCountProviders(List<CountProvider> countProviders) {
        this.countProviders = countProviders;
    }

    private PortalDao portalDao;

    public void setPortalDao(PortalDao portalDao) {
        this.portalDao = portalDao;
    }

    public PortletApp getPortletApp(String name) {
        for (PortletApp pa : portletApps) {
            if (pa.getName().equals(name))
                return pa;
        }
        return null;
    }

    public List<PortletApp> getPortletApps() {
        return portletApps;
    }

    public PortletEntity getPortletEntity(String id) {
        return pes.get(id);
    }

    public Collection<PortletEntity> getPortletEntities() {
        return pes.values();
    }

    public PortletEntity savePortletEntity(PortletEntity pe) {
        boolean isCreate = StringUtils.isEmpty(pe.getId());
        portalDao.savePortletEntity(pe);
        if (!isCreate) {
            PortletEntity sPe = getPortletEntity(pe.getId());
            BeanUtils.copyProperties(pe, sPe);
            pe = sPe;
        } else
            pes.put(pe.getId(), pe);
        pe.setApp(getPortletApp(pe.getAppName()));
        return pe;
    }

    public void removePortletEntity(String id) {
        portalDao.removePortletEntity(id);
        pes.remove(id);
    }

    public Site getSite(String owner) {
        Site site;
        if (Site.GLOBAL_SITE.equals(owner)) {
            site = globalSite;
        } else if (owner.startsWith(Site.ROLE_SITE_PREFIX)) {
            site = portalDao.getSite(owner);
            if (site == null) {
                try {
                    site = globalSite.clone();
                    site.setOwner(owner);
                } catch (CloneNotSupportedException e) {
                    throw new RuntimeException(e);
                }
            }
        } else {
            site = portalDao.getSite(owner);
            if (site == null) {
                Map<String, Site> map = portalDao.getRoleSitesMap();
                for (PfRoleVo role : sysUserService.getRoleListByUser(owner)) {
                    site = map.get(Site.ROLE_SITE_PREFIX + role.getRoleId());
                    if (site != null)
                        break;
                }
                if (site == null)
                    site = globalSite;
                try {
                    site = site.clone();
                    site.setOwner(owner);
                } catch (CloneNotSupportedException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        initPortlets(site);
        return site;
    }

    public Site saveSite(Site site) {
        initPortlets(site);
        return portalDao.saveSite(site);
    }

    public void removeSite(String owner) {
        portalDao.removeSite(owner);
    }

    public List<CountProvider> getCountProviders() {
        return countProviders;
    }

    public void resetUserSite() {
        portalDao.resetUserSite();
    }

    public synchronized void init() throws Exception {
        sm = new XStream();
        sm.alias("portlet", PortletApp.class);
        loadPortletApps();
        pes = new HashMap<String, PortletEntity>();
        List<PortletEntity> entities = portalDao.getPortletEntities();
        for (PortletEntity pe : entities) {
            pe.setApp(getPortletApp(pe.getAppName()));
            pes.put(pe.getId(), pe);
        }
        globalSite = portalDao.getSite(Site.GLOBAL_SITE);
        if (globalSite == null) {
            globalSite = new Site();
            globalSite.setOwner(Site.GLOBAL_SITE);
            Page page = new Page();
            page.setName("index");
            globalSite.getPages().add(page);
            page.setLayout("2-0");
            for (int i = 0; i < 3; i++) {
                Fragment fragment = new Fragment();
                page.getFragments().add(fragment);
            }
            portalDao.saveSite(globalSite);
        }
    }


    private void initPortlets(Site site) {
        for (Page page : site.getPages()) {
            for (Fragment fm : page.getFragments()) {
                for (Portlet portlet : fm.getPortlets()) {
                    PortletEntity entity = getPortletEntity(portlet.getEntityId());
                    if (portlet.getEntity() == null)
                        portlet.setEntity(entity);
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void loadPortletApps() {
        try {
            File file = ResourceUtils.getFile(PORTLETAPPS_CONFIG);
            portletApps = (List<PortletApp>) sm.fromXML(new FileReader(file));
        } catch (IOException e) {
            portletApps = new ArrayList<PortletApp>();
        }
    }
}
