package cn.gtmap.landtax.util;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Created by zhouzhiwei on 2015-11-16.
 * 从未知类型的Object对象中，指定field取值
 * object对象在前台取值是直接使用属性名(object.field[.field])
 * 该类通过反射方法，能够类似前台方式取值
 */
public class ObjectValueManager implements Serializable {
    /**
     * 空对象异常信息
     */
    private static final String MSG_NULLOBJECT = "对象为空";

    /**
     * 索引超出范围异常信息
     */
    private static  final String MSG_OUTBOUND = "索引超出范围";

    /**
     * object中定义的field若class type位于class_path，则需要便利获取field
     */
    private static  final String CLASS_PATH = "cn.gtmap";

    /**
     * 存放传进来的Object对象
     */
    private List<Object> objectList;

    /**
     * 存放field名称跟对应get方法
     */
    private HashMap<String, Method> methodHashMap;

    /**
     * 构造传一个Object对象，对象不允许为空，否则抛出异常
     * @param o
     */
    public ObjectValueManager(Object o) throws Exception {
        // 检查对象
        checkNullObject(o);

        List<Object> list = new ArrayList<Object>();
        list.add(o);
        this.objectList = list;

        // 初始化对象field的get方法
        initObjectMethodFactory();
    }

    /**
     *
     * @param list
     */
    public ObjectValueManager(List<Object> list) throws Exception {
        // 检查对象
        checkNullObject(list);

        this.objectList = list;

        // 初始化对象field的get方法
        initObjectMethodFactory();
    }

    /**
     * 根据fieldName获取object的field值
     * @param fieldName 字段名称
     * @return
     */
    public String getObjectValue(String fieldName) {
        return getObjectValue(fieldName, 0);
    }

    /**
     * 获取object的field值
     * @param fieldName 字段名称
     * @param index 索引值，获取第index个Object的field值
     * @return
     */
    public String getObjectValue(String fieldName, Integer index) {
        String fieldValue = null;
        try {
            Object o = getObject(index);

            // 分割属性名，一层一层反射获取值
            // 获取field.subField，要先获取field，再获取field中的subField
            String[] fieldNames = fieldName.split("\\.");
            fieldName = "";
            Object fieldValueObject = null;
            for (int i = 0; i < fieldNames.length; i++) {
                if (fieldName != "") {
                    fieldName += ".";
                }
                fieldName += fieldNames[i];

                Method method = methodHashMap.get(fieldName);
                // 若未找到方法或对象为空，直接返回空
                if (method == null || o == null) {
                    return null;
                }
                fieldValueObject = method.invoke(o, null);
                o = fieldValueObject;
            }
            fieldValue = objectToString(fieldValueObject);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return fieldValue;
    }

    /**
     * 判断对象是否为空，为空则抛出异常
     * @param o
     * @throws Exception
     */
    private  void checkNullObject(Object o) throws Exception {
        if (o == null) {
            throw new Exception(MSG_NULLOBJECT);
        }
    }

    /**
     * 判断集合跟集合内第一个对象是否为空，为空则抛出异常
     * @param list
     * @throws Exception
     */
    private void checkNullObject(List<Object> list) throws Exception {
        if (CollectionUtils.isEmpty(list)) {
            throw new Exception(MSG_NULLOBJECT);
        } else {
            checkNullObject(list.get(0));
        }
    }

    /**
     * 解析Object中Field的getMethod方法，并保存
     * 对象a中存在子对象b(这里只判断class cn.gtmap中的)，属性获取用b.fieldName
     */
    private void initObjectMethodFactory() {
        methodHashMap = new HashMap<String, Method>();

        if (CollectionUtils.isEmpty(objectList))
            return;

        Object o = objectList.get(0);
        if (o == null)
            return;

        initObjectMethodFactory(o, null);
    }

    /**
     * 获取o中所有field存到methodHashMap
     * @param o 需要获取field的对象
     * @param parentName 所属父对象的名称
     */
    private void initObjectMethodFactory(Object o, String parentName) {
        if (o == null)
            return;

        List<Field> fieldList = findObjectFieldList(o);
        if (CollectionUtils.isNotEmpty(fieldList)) {
            for (Field field : fieldList) {
                String classType = field.getGenericType().toString();
                String fieldName = field.getName();

                // 获取method
                String getMethodName = "get" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
                Method method = getMethod(o, getMethodName);

                // 获取getMethod不为空，则将fieldName跟Method放到methodHashMap中
                if (method != null) {
                    // 若parentName不为空，fieldName需要带上parentName，中间用"."隔开
                    methodHashMap.put(StringUtils.isEmpty(parentName) ? fieldName : parentName + "." + fieldName, method);
                }

                // 如果获取的对象依然是对象，迭代获取get方法
                if (classType.startsWith("class " + CLASS_PATH)) {
                    try {
                        // 迭代将field对象的field放到methodHashMap中
                        // 若获取field对象报错，将忽略该对象
                        Object fieldObject = Class.forName(classType.substring(6)).newInstance();
                        initObjectMethodFactory(fieldObject, StringUtils.isEmpty(parentName) ? fieldName : parentName + "." + fieldName);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    /**
     * 根据methodName名称从o中获取Method
     * @param o
     * @param methodName
     * @return
     */
    private Method getMethod(Object o, String methodName) {
        Method method = null;
        try {
            // 根据methodName获取Method
            method = o.getClass().getMethod(methodName, new Class[] {});
        } catch (Exception e) {
            // 若获取Method异常，尝试将methodName中Field首字母改为大写，再获取Method
            methodName = "get" + methodName.substring(3, 4).toLowerCase() + methodName.substring(4, methodName.length());
            try {
                method = o.getClass().getMethod(methodName, new Class[] {});
            } catch (Exception e2) {
                e.printStackTrace();
            }
        }
        return method;
    }

    /**
     * 根据index获取Object,会对index做检查，判断是否索引超出范围
     * @param index
     * @return
     * @throws Exception
     */
    private Object getObject(Integer index) throws Exception {
        if (index == null || index < 0 || index >= objectList.size()) {
            throw new Exception(MSG_OUTBOUND);
        }

        Object o = objectList.get(index);
        return o;
    }

    /**
     * 将object转换成string
     * @param o
     * @return
     */
    private String objectToString(Object o) {
        String value = null;
        String fieldType = "";

        if (o == null) {
            return null;
        } else {
            fieldType = o.getClass().getSimpleName();
        }

        if (fieldType.equals("String")) {
            value = (String)o;
        } else if (fieldType.equals("int")) {
            int temp = (Integer)o;
            value = Integer.toString(temp);
        } else if (fieldType.equals("Boolean")) {
            Boolean temp = (Boolean)o;
            value = Boolean.toString(temp);
        } else if (fieldType.equals("float")) {
            float temp = (Float)o;
            value = Float.toString(temp);
        } else if (fieldType.equalsIgnoreCase("double")) {
            double temp = (Double)o;
            value = CommonUtil.DoubleToString(temp, 4, false);
        } else if (fieldType.equals("BigDecimal")) {
            BigDecimal temp = (BigDecimal)o;
            value = temp.toString();
        } else if (fieldType.equals("Long")) {
            long temp = (Long)o;
            value = Long.toString(temp);
        } else if (fieldType.equals("short")) {
            short temp = (Short)o;
            value = Short.toString(temp);
        } else if (fieldType.equals("byte")) {
            byte temp = (Byte)o;
            value = Byte.toString(temp);
        } else if (fieldType.equals("Integer")) {
            Integer tempInt = (Integer)o;
            value = String.valueOf(tempInt.intValue());
        } else if (fieldType.equals("Date") || fieldType.equals("Timestamp")) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            Date tempInt = (Date)o;
            value = String.valueOf(sdf.format(tempInt));
        } else {
            value = "";
        }

        return value;
    }

    /**
     * 根据o返回所有的Field，如果有extend，父类的Field也会找出来
     * @param o
     * @return
     */
    public static List<Field> findObjectFieldList(Object o) {
        // 如果object是空，直接返回null
        if (o == null) {
            return null;
        }

        List<Field> fieldList = new ArrayList<Field>();

        // 获取o的field
        Field[] fieldArr = null;
        fieldArr = o.getClass().getDeclaredFields();
        fieldList.addAll(Arrays.asList(fieldArr));

        // 循环获取o的父类的field
        Class superClazz = o.getClass().getSuperclass();
        while (superClazz != null) {
            fieldArr = superClazz.getDeclaredFields();
            fieldList.addAll(Arrays.asList(fieldArr));

            superClazz = superClazz.getSuperclass();
        }

        return  fieldList;
    }

    public HashMap<String, Method> getMethodHashMap() {
        return methodHashMap;
    }
}
