package cn.gtmap.landtax.print;

import cn.gtmap.landtax.util.ObjectValueManager;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import jxl.Workbook;
import jxl.format.Alignment;
import jxl.format.Border;
import jxl.format.BorderLineStyle;
import jxl.format.Colour;
import jxl.format.*;
import jxl.format.VerticalAlignment;
import jxl.write.*;
import jxl.write.Number;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.Serializable;
import java.lang.Boolean;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

/**
 * Created by zhouzhiwei on 2015-04-01.
 * 导出Excel的类
 */
public class PrintGridExcelUtil implements Serializable {
    /**
     * 导出Excel的Sheet表页索引号
     */
    private int sheetIndex;

    /**
     * 填写数据的索引号
     */
    private int rowIndex;

    /**
     * 导出数据的管理类，负责样式、数据等的管理
     */
    private ExportManager exportManager;

    /**
     * 创建WritableCellFormat的类，供CreateWritableCellFormat类调用
     */
    private class CellFormat {
        /**
         * 字体设置类
         */
        CellFont cellFont = new CellFont();
        private class CellFont {
            /**
             * 字体
             */
            String fontName = "宋体";

            /**
             * 字体大小
             */
            int fontSize = 9;

            /**
             * 是否加粗
             */
            boolean isBold = false;

            /**
             * 是否斜体
             */
            boolean isItalic = false;

            /**
             * 下划线类型
             */
            UnderlineStyle underlineStyle = UnderlineStyle.NO_UNDERLINE;

            /**
             * 字体颜色
             */
            Colour colour = Colour.BLACK;
        }

        /**
         * 单元格边框
         */
        CellBorder cellBorder = new CellBorder();
        private class CellBorder {
            BorderLineStyle left = BorderLineStyle.THIN;
            BorderLineStyle top = BorderLineStyle.THIN;
            BorderLineStyle right = BorderLineStyle.THIN;
            BorderLineStyle bottom = BorderLineStyle.THIN;

            /**
             * 设置统一的边框样式
             * @param bs
             */
            public void setBorderStyle(BorderLineStyle bs) {
                this.left = bs;
                this.top = bs;
                this.right = bs;
                this.bottom = bs;
            }

            /**
             * 设置四边的边框样式
             * @param leftBs
             * @param topBs
             * @param rightBs
             * @param bottomBs
             */
            public void setBorderStyle(BorderLineStyle leftBs, BorderLineStyle topBs, BorderLineStyle rightBs, BorderLineStyle bottomBs) {
                this.left = leftBs;
                this.top = topBs;
                this.right = rightBs;
                this.bottom = bottomBs;
            }
        }

        /**
         * 自动填充
         */
        boolean isAutoFill = false;

        /**
         * 自动换行
         */
        boolean isWrapText = false;

        /**
         * 水平对齐方式
         */
        Alignment alignment = Alignment.CENTRE;

        /**
         * 垂直对齐方式
         */
        VerticalAlignment verticalAlignment = VerticalAlignment.CENTRE;

        /**
         * 单元格背景颜色
         */
        Colour background = Colour.WHITE;

        /**
         * 构造函数，什么都没做
         */
        public CellFormat() {

        }

        /**
         * 通过对其方式名称设置水平对齐方式
         * @param aligStr
         */
        public void setAlignment(String aligStr) {
            if (aligStr != null) {
                if (aligStr.equalsIgnoreCase("left")) {
                    this.alignment = Alignment.LEFT;
                } else if (aligStr.equalsIgnoreCase("center")) {
                    this.alignment = Alignment.CENTRE;
                } else if (aligStr.equalsIgnoreCase("right")) {
                    this.alignment = Alignment.RIGHT;
                } else if (aligStr.equalsIgnoreCase("general")) {
                    this.alignment = Alignment.GENERAL;
                } else if (aligStr.equalsIgnoreCase("fill")) {
                    this.alignment = Alignment.FILL;
                } else if (aligStr.equalsIgnoreCase("justify")) {
                    this.alignment = Alignment.JUSTIFY;
                }
            }
        }

        /**
         * 通过对其方式名称设置垂直对齐方式
         * @param aligStr
         */
        public void setVerticalAlignment(String aligStr) {
            if (aligStr != null) {
                if (aligStr.equalsIgnoreCase("top")) {
                    this.verticalAlignment = VerticalAlignment.TOP;
                } else if (aligStr.equalsIgnoreCase("center")) {
                    this.verticalAlignment = VerticalAlignment.CENTRE;
                } else if (aligStr.equalsIgnoreCase("bottom")) {
                    this.verticalAlignment = VerticalAlignment.BOTTOM;
                } else if (aligStr.equalsIgnoreCase("justify")) {
                    this.verticalAlignment = VerticalAlignment.JUSTIFY;
                }
            }
        }

        /**
         * 创建WritableCellFormat
         * @return
         */
        public WritableCellFormat getCellFormat() {
            return getCellFormat(null, null);
        }

        /**
         * 创建WritableCellFormat
         * @param formatter 格式化
         * @return
         */
        public WritableCellFormat getCellFormat(String formatter, String formatterType) {
            WritableCellFormat cellFormat = null;
            try {
                // 新建WritableCellFormat
                if (StringUtils.isEmpty(formatter) || StringUtils.isEmpty(formatterType)) {
                    cellFormat = new WritableCellFormat(CreateFont());
                } else {
                    if ("number".equalsIgnoreCase(formatterType)) {
                        NumberFormat numberFormat = new NumberFormat(formatter);
                        cellFormat = new WritableCellFormat(CreateFont(), numberFormat);
                    } else if ("date".equalsIgnoreCase(formatterType)) {
                        DateFormat dateFormat = new DateFormat(formatter);
                        cellFormat = new WritableCellFormat(CreateFont(), dateFormat);
                    } else {
                        cellFormat = new WritableCellFormat(CreateFont());
                    }
                }

                // 应用是否自动换行设置
                cellFormat.setWrap(isWrapText);
                // 应用是否自动填充设置
                cellFormat.setShrinkToFit(isAutoFill);

                // 应用边框设置
                cellFormat.setBorder(Border.LEFT, cellBorder.left);
                cellFormat.setBorder(Border.RIGHT, cellBorder.right);
                cellFormat.setBorder(Border.BOTTOM, cellBorder.bottom);
                cellFormat.setBorder(Border.TOP, cellBorder.top);

                // 应用水平对齐方式设置
                cellFormat.setAlignment(alignment);
                // 应用垂直对齐方式设置
                cellFormat.setVerticalAlignment(verticalAlignment);

                // 应用单元格背景颜色设置
                cellFormat.setBackground(background);
            } catch (WriteException e) {
                e.printStackTrace();
            }

            return cellFormat;
        }

        /**
         * 创建WritableFont对象
         * @return
         */
        private WritableFont CreateFont() {
            WritableFont writableFont = null;
            if (cellFont.isBold) {
                writableFont = new WritableFont(WritableFont.createFont(cellFont.fontName), cellFont.fontSize, WritableFont.BOLD, cellFont.isItalic, cellFont.underlineStyle, cellFont.colour);
            } else {
                writableFont = new WritableFont(WritableFont.createFont(cellFont.fontName), cellFont.fontSize, WritableFont.NO_BOLD, cellFont.isItalic, cellFont.underlineStyle, cellFont.colour);
            }
            return writableFont;
        }
    }

    /**
     * 创建导出Excel需要的CellFormat
     */
    private class CreateWritableCellFormat {
        /**
         * 创建标题的CellFormat
         * @return
         */
        public WritableCellFormat TitleWritableCellFormat() {
            CellFormat cellFormat = new CellFormat();
            cellFormat.cellFont.isBold = true;
            cellFormat.cellFont.fontSize = 24;

            return cellFormat.getCellFormat();
        }

        /**
         * 创建列头的CellFormat
         * @return
         */
        public WritableCellFormat FieldWritableCellFormat() {
            CellFormat cellFormat = new CellFormat();
            cellFormat.background = Colour.GREY_25_PERCENT;
            cellFormat.cellFont.fontSize = 10;

            return cellFormat.getCellFormat();
        }

        /**
         * 创建列内容单元的CellFormat
         * @param alignment 对齐方式
         * @return
         */
        public WritableCellFormat FieldCellWritableCellFormat(String alignment) {
            CellFormat cellFormat = new CellFormat();
            cellFormat.setAlignment(alignment);

            return cellFormat.getCellFormat();
        }

        /**
         * 创建数字列内容单元的CellFormat
         * @param alignment 对齐方式
         * @param formatter 格式化
         * @return
         */
        public WritableCellFormat FieldCellWritableNumberCellFormat(String alignment, String formatter) {
            CellFormat cellFormat = new CellFormat();
            cellFormat.setAlignment(alignment);

            return cellFormat.getCellFormat(formatter, "number");
        }

        /**
         * 创建日期列内容单元的CellFormat
         * @param alignment 对齐方式
         * @param formatter 格式化
         * @return
         */
        public WritableCellFormat FieldCellWritableDateCellFormat(String alignment, String formatter) {
            CellFormat cellFormat = new CellFormat();
            cellFormat.setAlignment(alignment);

            return cellFormat.getCellFormat(formatter, "date");
        }
    }

    /**
     * 静态变量
     */
    private static class StaticParam {
        /**
         * 生成的Excel临时文件存放路径
         */
        private static final String xslPath = "\\static\\excel\\temp\\";

        /**
         * 序号列的列头名称
         */
        private static final String FIELD_XH = "序号";

        /**
         * 一页最多显示的记录数
         */
        private static final int OneSheetRows = 50000;

        /**
         * 内容填写的开始行
         */
        private static final int BEGIN_ROW = 1;

        /**
         * 内容填写的开始列
         */
        private static final int BEGIN_COL = 1;

        /**
         * 页面宽度与Excel宽度的比例
         * 页面宽度值(px) * WIDTH_SCALE = Excel实际宽度值
         */
        private static final double WIDTH_SCALE = 0.12;

        /**
         * 行高数值与设置数值之间的比例
         * 设置行高 = Excel实际行高值 * HEIGHT_SCALE
         */
        private static final double HEIGHT_SCALE = 20;

        /**
         * 标题单元格的格式化名称
         */
        private static final String FORMAT_TITLE = "excel_title";

        /**
         * 序号列的格式化名称
         */
        private static final String FORMAT_XH = "excel_xh";

        /**
         * 列头单元格的格式化名称
         */
        private static final String FORMAT_FIELDROW = "excel_field";

        /**
         * 首行行高，首行指从第0行到BEGIN_ROW之间的行
         */
        private static final int HEIGHT_FIRSTROW = 8;

        /**
         * 首列列宽，首列指从第0列到BEGIN_COL之间的列
         */
        private static final int WIDTH_FIRSTCOL = 9;

        /**
         * 标题行的行高
         */
        private static final int HEIGHT_TITLE = 48;

        /**
         * 列头行的行高
         */
        private static final int HEIGHT_FIELD = 24;

        /**
         * 内容行的行高
         */
        private static final int HEIGHT_ROW = 16;

        /**
         * 未设置列宽时，默认的列宽，单位是px
         */
        private static final int WIDTH_DEFAULT = 100;

        /**
         * 序号列的列宽，单位是px
         */
        private static final int WIDTH_XH = 72;
    }

    /**
     * 导出的数据列类型
     * HASHMAP：对应List<HashMap>类型数据
     * OBJECT：对应List<OBJECT>类型数据
     * NODATA：List为空或者长度为0
     */
    public enum DataType {
        HASHMAP, OBJECT, NODATA;
    }

    /**
     * 导出用的变量管理类
     */
    public class ExportManager {
        /**
         * 标题
         */
        private String title;

        /**
         * 列信息
         */
        private List<FieldVo> fieldList;

        /**
         * field索引号
         * String：fieldName
         * Integer：索引号
         */
        private HashMap<String, Integer> fieldIndexMap;

        /**
         * 记录列导出需要的信息
         */
        private class FieldVo {
            /**
             * 列名称，对应数据的列名称
             */
            public String name;

            /**
             * 列中文名称
             */
            public String text;

            /**
             * 列宽，由页面px宽度转为Excel的实际列宽
             */
            public Integer width;

            /**
             * 对齐方式，left、cernter、right
             */
            public String align;

            /**
             * 格式化类型，number、date
             */
            public String formatterType;

            /**
             * 格式化表达式
             */
            public String formatterExpress;
        }

        /**
         * 合并行
         */
        private List<MergeFieldVo> mergeFieldList;
        /**
         * 合并行时的字段记录信息
         */
        private class MergeFieldVo {
            /**
             * 列名称，对应数据的列名称
             */
            public String name;

            /**
             * 行根据字段合并
             */
            public String mergeByField;

            /**
             * 填写Excel数据时用，并且mergeByField不为空才用到
             * 记录当前行的数据应该根据哪个值合并
             */
            public String curMergeValue;

            /**
             * 填写Excel数据时用，并且mergeByField不为空才用到
             * 记录合并行从哪行开始
             */
            public Integer curMergeStartRow;
        }
        /**
         * 二级标题列信息
         */
        private List<SecFieldVo> secFieldList;

        /**
         * 二级标题的列头信息
         */
        private class SecFieldVo {
            /**
             * 标题中文信息
             */
            public String text;

            /**
             * 从哪列开始合并
             */
            public String startName;

            /**
             * 到哪个列结束
             */
            public String endName;

            /**
             * 合并多少列，该值是计算需导出的
             */
            public Integer mergeFieldCount;
        }

        /**
         * 记录Excel格式，标题、列、单元格
         */
        private HashMap<String, WritableCellFormat> formatMap;

        /**
         * 导出的数据列
         */
        private List dataList;

        /**
         * 数据类型，决定获取dataList中的数据时，采用哪种方式
         */
        private DataType dataType;

        private ObjectValueManager objectValueManager;
        /**
         * 构造函数
         * 初始化成员变量
         */
        public ExportManager() {
            title = "";
            fieldList = new ArrayList<FieldVo>();
            formatMap = new HashMap<String, WritableCellFormat>();
            secFieldList = new ArrayList<SecFieldVo>();
            mergeFieldList = new ArrayList<MergeFieldVo>();
        }

        /**
         * 构造函数，带gridOption参数
         * @param gridOptions 报表设置属性
         */
        public ExportManager(JSONObject gridOptions) {
            this();

            // 解析gridOptions
            LoadExportOptions(gridOptions);
        }

        /**
         * 构造函数，带gridOption和dataList参数
         * @param gridOptions 报表设置属性
         * @param dataList 导出的数据列表
         */
        public ExportManager(JSONObject gridOptions, List dataList) {
            this(gridOptions);

            setDataList(dataList);
        }

        /**
         * 解析gridOptions参数，若导出的不是JQGrid的报表，重写这个函数就好
         * 二级列secFieldList可以没有，
         * 列fieldList必须有，
         * 单元格格式可以加，标题跟列的格式要有
         * @param gridOptions
         */
        public void LoadExportOptions(JSONObject gridOptions) {
            // 获取二级列信息
            secFieldList.clear();
            JSONObject setgroupHeaders = gridOptions.getJSONObject("setgroupHeaders");
            JSONArray groupHeaders = null;
            if (setgroupHeaders != null) {
                groupHeaders = setgroupHeaders.getJSONArray("groupHeaders");
            }
            if (groupHeaders != null) {
                for (int j = 0; j < groupHeaders.size(); j++) {
                    SecFieldVo secFieldVo = new SecFieldVo();
                    secFieldVo.startName = groupHeaders.getJSONObject(j).getString("startColumnName");
                    secFieldVo.text = groupHeaders.getJSONObject(j).getString("titleText");
                    secFieldVo.mergeFieldCount = groupHeaders.getJSONObject(j).getInteger("numberOfColumns"); // 现在数值还不是显示的合计合并列
                    secFieldList.add(secFieldVo);
                }
            }

            // 遍历获取需要导出的列
            fieldList.clear();
            // 记录列是否需要导出
            List<HashMap<String, Object>> fieldneedExportMap = new ArrayList<HashMap<String, Object>>();
            JSONArray colModels = (JSONArray)gridOptions.get("colModel");
            JSONObject colModel = null;
            for (int i = 0; i < colModels.size(); i++) {
                colModel = colModels.getJSONObject(i);

                boolean needExport = false;
                if (colModel.get("hidden") != null && colModel.get("hidden").toString().equalsIgnoreCase("true")) { // 隐藏列
                    if (colModel.get("needExport") != null && colModel.get("needExport").toString().equalsIgnoreCase("true")) { // needExport不为空且needExport=true才需要导出
                        needExport = true;
                    }
                } else { // 非隐藏列
                    if (colModel.get("needExport") == null || colModel.get("needExport").toString().equalsIgnoreCase("true")) { // needExport为空或者needExport=true需要导出
                        needExport = true;
                    }
                }

                // 记录所有列的情况
                HashMap<String, Object> map = new HashMap<String, Object>();
                map.put("name", colModel.getString("name"));
                map.put("needExport", needExport);
                fieldneedExportMap.add(map);

                // 添加field信息
                if (needExport) {
                    FieldVo fieldVo = new FieldVo();

                    if (colModel.get("name") == null) {
                        continue;
                    }

                    fieldVo.name = colModel.getString("name");
                    fieldVo.width = colModel.getInteger("width");
                    fieldVo.align = colModel.getString("align");
                    fieldVo.text = ((JSONArray) gridOptions.get("colNames")).getString(i);
                    fieldVo.formatterType = colModel.getString("formatter");
                    if ("number".equalsIgnoreCase(fieldVo.formatterType)) {
                        String formatterExpress = "#,##0.00";
                        if (colModel.getJSONObject("formatoptions") != null) {
                            Integer decimalPlaces = 2;
                            String thousandsSeparator = ",";
                            if (colModel.getJSONObject("formatoptions").getInteger("decimalPlaces") != null)
                                decimalPlaces = colModel.getJSONObject("formatoptions").getInteger("decimalPlaces");
                            if (colModel.getJSONObject("formatoptions").getString("thousandsSeparator") != null)
                                thousandsSeparator = colModel.getJSONObject("formatoptions").getString("thousandsSeparator");

                            // 构建表达式
                            formatterExpress = "#" + thousandsSeparator + "##0";
                            if (decimalPlaces > 0) {
                                formatterExpress += ".";
                                for (int m = 1; m <= decimalPlaces; m++)
                                    formatterExpress += "0";
                            }
                        }
                        fieldVo.formatterExpress = formatterExpress;
                    } else if ("date".equalsIgnoreCase(fieldVo.formatterType)) {
                        String formatterExpress = "yyyy-MM-dd";
                        if (colModel.getJSONObject("formatoptions") != null) {
                            if (colModel.getJSONObject("formatoptions").getString("exportformat") != null)
                                formatterExpress = colModel.getJSONObject("formatoptions").getString("exportformat");
                        }
                        fieldVo.formatterExpress = formatterExpress;
                    }
                    fieldList.add(fieldVo);

                    if (colModel.getString("mergeBy") != null) {
                        MergeFieldVo mergeFieldVo = new MergeFieldVo();
                        mergeFieldVo.name = colModel.getString("name");
                        mergeFieldVo.mergeByField = colModel.getString("mergeBy");

                        mergeFieldList.add(mergeFieldVo);
                    }
                }
            }

            // 更新二级列的起始列结束列信息
            refreshSecFieldInfo(fieldneedExportMap);

            // 获取单元格格式
            CreateWritableCellFormat createWritableCellFormat = new CreateWritableCellFormat();
            // 添加标题样式
            formatMap.put(StaticParam.FORMAT_TITLE, createWritableCellFormat.TitleWritableCellFormat());
            // 添加列名样式
            formatMap.put(StaticParam.FORMAT_FIELDROW, createWritableCellFormat.FieldWritableCellFormat());
            // 添加序号列样式
            formatMap.put(StaticParam.FORMAT_XH, createWritableCellFormat.FieldCellWritableCellFormat(null));
            // 添加其它列单元格输入样式
            for (FieldVo fieldVo : fieldList) {
                if ("number".equalsIgnoreCase(fieldVo.formatterType)) {
                    formatMap.put(fieldVo.name, createWritableCellFormat.FieldCellWritableNumberCellFormat(fieldVo.align, fieldVo.formatterExpress));
                } else if ("date".equalsIgnoreCase(fieldVo.formatterType)) {
                    formatMap.put(fieldVo.name, createWritableCellFormat.FieldCellWritableDateCellFormat(fieldVo.align, fieldVo.formatterExpress));
                } else {
                    formatMap.put(fieldVo.name, createWritableCellFormat.FieldCellWritableCellFormat(fieldVo.align));
                }
            }
        }

        /**
         * 赋值导出的数据列，并检查数据是HashMap还是Object
         * @param dataList
         */
        public void setDataList(List dataList) {
            this.dataList = dataList;

            // 检查导出的DataList的类型（HashMap、Object，NoData，如果是Object类型，要初始化ObjectValueManager来获取值）
            CheckDataType();

            if (dataType == DataType.OBJECT) {
                try {
                    objectValueManager = new ObjectValueManager(this.dataList);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        /**
         * 根据rowIndex获取该索引号记录的fieldName字段的值
         * @param fieldName
         * @param rowIndex
         * @return
         */
        public String getData(String fieldName, int rowIndex) {
            return getData(fieldName, rowIndex, true);
        }

        /**
         * 根据rowIndex获取该索引号记录的fieldName字段的值
         * @param fieldName
         * @param rowIndex
         * @param autoFormat 是否自动格式化数值
         * @return
         */
        public String getData(String fieldName, int rowIndex, boolean autoFormat) {
            String value = null;
            if (dataType == DataType.OBJECT) {
                if (objectValueManager != null)
                    value = objectValueManager.getObjectValue(fieldName, rowIndex);
            } else if (dataType == DataType.HASHMAP) {
                HashMap dataMap = (HashMap)dataList.get(rowIndex);
                if (dataMap.get(fieldName) != null)
                    value = dataMap.get(fieldName).toString();
            }

            if (StringUtils.isNotEmpty(value) && autoFormat)
                value = formatValue(getFieldByName(fieldName), value);

            return value;
        }

        /**
         * 根据name获取FieldVo
         * @param name
         * @return
         */
        private FieldVo getFieldByName(String name) {
            FieldVo fieldVo = null;

            for (FieldVo fieldVoTemp : fieldList) {
                if (fieldVoTemp.name.equals(name)) {
                    fieldVo = fieldVoTemp;
                    break;
                }
            }
            return fieldVo;
        }
        /**
         * 格式化数值
         * @param fieldVo
         * @param value
         * @return
         */
        private String formatValue(FieldVo fieldVo, String value) {
            try {
                // 若fieldVo为空，不格式化值
                if (fieldVo == null) {
                    return value;
                }

                if (fieldVo.formatterType != null && StringUtils.isNotEmpty(value)) {
                    if (fieldVo.formatterType.equalsIgnoreCase("number")) {
                        DecimalFormat format = new DecimalFormat(fieldVo.formatterExpress);
                        value = format.format(Double.valueOf(value));
                    } else if (fieldVo.formatterType.equalsIgnoreCase("date")) {
                        SimpleDateFormat format = new SimpleDateFormat(fieldVo.formatterExpress);
                        value = format.format((new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")).parse(value));
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            return value;
        }

        /**
         * 合并行
         * @param ws
         * @param row
         * @param listIndex
         * @throws Exception
         */
        public void mergeField(WritableSheet ws, int row, int listIndex) throws Exception{
            // 如果没有需要合并的行，直接退出
            if (CollectionUtils.isEmpty(mergeFieldList))
                return;

            for (MergeFieldVo mergeFieldVo : mergeFieldList) {
                String mergeField = mergeFieldVo.name;
                String mergeByField = mergeFieldVo.mergeByField;
                Integer firstRow = mergeFieldVo.curMergeStartRow;

                Integer fieldCol = StaticParam.BEGIN_COL + 1 + getFieldIndex(mergeField);

                if (firstRow == null) {
                    mergeFieldVo.curMergeStartRow = row;
                } else {
                    String mergeKeyValue = getData(mergeByField, listIndex);
                    String mergeFirstKeyValue = mergeFieldVo.curMergeValue;

                    if (mergeKeyValue.equals(mergeFirstKeyValue)) { // 合并行，不做任何处理
                        if (listIndex % StaticParam.OneSheetRows == 0 || (listIndex + 1) >= dataList.size()) { // 一页数据结束或者数据结束，直接做合并处理
                            ws.mergeCells(fieldCol, firstRow, fieldCol, row);
                            mergeFieldVo.curMergeStartRow = null;
                            mergeFieldVo.curMergeValue = null;
                        }
                    } else { // 做合并处理
                        ws.mergeCells(fieldCol, firstRow, fieldCol, row - 1);
                        mergeFieldVo.curMergeStartRow = row;
                        mergeFieldVo.curMergeValue = mergeKeyValue;
                    }
                }

            }
        }

        /**
         * 获取根据名称获取field的索引号，计算col用
         * @param fieldName
         * @return
         */
        private Integer getFieldIndex(String fieldName) {
            if (fieldIndexMap == null) {
                fieldIndexMap = new HashMap<String, Integer>();
                for (int i = 0; i < fieldList.size(); i++) {
                    fieldIndexMap.put(fieldList.get(i).name, i);
                }
            }

            return fieldIndexMap.get(fieldName);
        }
        /**
         * 创建Excel表单头部信息
         * @param ws 新建的空表页
         * @return 返回填写数据的开始行
         * @throws Exception
         */
        private int initSheetHead (WritableSheet ws) throws Exception {
            int row = StaticParam.BEGIN_ROW;
            int fieldCount = fieldList.size();
            Label label = null;
            Integer width = null;

            // 设置首行首列宽高
            for (int i = 0; i < StaticParam.BEGIN_ROW; i++) {
                ws.setRowView(i, (int)(StaticParam.HEIGHT_FIRSTROW * StaticParam.HEIGHT_SCALE));
            }
            for (int i = 0; i < StaticParam.BEGIN_COL; i++) {
                ws.setColumnView(i, (int)(StaticParam.WIDTH_FIRSTCOL * StaticParam.WIDTH_SCALE));
            }

            // 填写表头标题
            label = new Label(StaticParam.BEGIN_COL, row, title, formatMap.get(StaticParam.FORMAT_TITLE));
            ws.addCell(label);
            ws.mergeCells(StaticParam.BEGIN_COL, row, StaticParam.BEGIN_COL + fieldCount, row);
            ws.setRowView(row, (int)(StaticParam.HEIGHT_TITLE * StaticParam.HEIGHT_SCALE));

            // 填写列名称
            if (CollectionUtils.isEmpty(secFieldList)) { // 没有二级列
                row++;

                // 填写列
                label = new Label(StaticParam.BEGIN_COL, row, StaticParam.FIELD_XH, formatMap.get(StaticParam.FORMAT_FIELDROW));
                ws.addCell(label);
                ws.setColumnView(StaticParam.BEGIN_COL, (int)(StaticParam.WIDTH_XH * StaticParam.WIDTH_SCALE));
                for (int i = 0; i < fieldCount; i++) {
                    label = new Label(StaticParam.BEGIN_COL + i + 1, row, fieldList.get(i).text, formatMap.get(StaticParam.FORMAT_FIELDROW));
                    ws.addCell(label);

                    if (fieldList.get(i).width != null) {
                        width = fieldList.get(i).width;
                        ws.setColumnView(StaticParam.BEGIN_COL + i + 1, (int)(width * StaticParam.WIDTH_SCALE));
                    } else {
                        ws.setColumnView(StaticParam.BEGIN_COL + i + 1, (int)(StaticParam.WIDTH_DEFAULT * StaticParam.WIDTH_SCALE));
                    }
                }
                ws.setRowView(row, (int)(StaticParam.HEIGHT_FIELD * StaticParam.HEIGHT_SCALE));
            } else { // 存在二级列
                row++;
                row++;

                // 填写列
                label = new Label(StaticParam.BEGIN_COL, row - 1, StaticParam.FIELD_XH, formatMap.get(StaticParam.FORMAT_FIELDROW));
                ws.addCell(label);
                ws.setColumnView(StaticParam.BEGIN_COL, (int)(StaticParam.WIDTH_XH * StaticParam.WIDTH_SCALE));
                ws.mergeCells(StaticParam.BEGIN_COL, row - 1, StaticParam.BEGIN_COL, row);

                int colIndex = StaticParam.BEGIN_COL;
                for (int i = 0; i < fieldCount; i++) {
                    SecFieldVo secFieldVo = null;
                    for (SecFieldVo secFieldVoTemp : secFieldList) {
                        if (secFieldVoTemp.startName.equals(fieldList.get(i).name)) {
                            secFieldVo = secFieldVoTemp;
                            break;
                        }
                    }

                    if (colIndex >= StaticParam.BEGIN_COL + i + 1) {
                        label = new Label(StaticParam.BEGIN_COL + i + 1, row, fieldList.get(i).text, formatMap.get(StaticParam.FORMAT_FIELDROW));
                        ws.addCell(label);
                    } else if (secFieldVo != null) {
                        label = new Label(StaticParam.BEGIN_COL + i + 1, row - 1, secFieldVo.text, formatMap.get(StaticParam.FORMAT_FIELDROW));
                        ws.addCell(label);
                        ws.mergeCells(StaticParam.BEGIN_COL + i + 1, row - 1, StaticParam.BEGIN_COL + i + 1 + secFieldVo.mergeFieldCount - 1, row - 1);
                        colIndex += secFieldVo.mergeFieldCount;

                        label = new Label(StaticParam.BEGIN_COL + i + 1, row, fieldList.get(i).text, formatMap.get(StaticParam.FORMAT_FIELDROW));
                        ws.addCell(label);
                    } else {
                        label = new Label(StaticParam.BEGIN_COL + i + 1, row - 1, fieldList.get(i).text, formatMap.get(StaticParam.FORMAT_FIELDROW));
                        ws.addCell(label);
                        ws.mergeCells(StaticParam.BEGIN_COL + i + 1, row - 1, StaticParam.BEGIN_COL + i + 1, row);
                        colIndex++;
                    }


                    if (fieldList.get(i).width != null) {
                        width = fieldList.get(i).width;
                        ws.setColumnView(StaticParam.BEGIN_COL + i + 1, (int) (width * StaticParam.WIDTH_SCALE));
                    } else {
                        ws.setColumnView(StaticParam.BEGIN_COL + i + 1, (int)(StaticParam.WIDTH_DEFAULT * StaticParam.WIDTH_SCALE));
                    }
                }
                ws.setRowView(row - 1, (int)(StaticParam.HEIGHT_FIELD * StaticParam.HEIGHT_SCALE));
                ws.setRowView(row, (int)(StaticParam.HEIGHT_FIELD * StaticParam.HEIGHT_SCALE));
            }

            return row;
        }

        /**
         * 检查数据是HashMap还是Object
         */
        private void CheckDataType() {
            if (CollectionUtils.isEmpty(dataList)) {
                dataType = DataType.NODATA;
            } else {
                Object data = dataList.get(0);
                if (data instanceof HashMap) {
                    dataType = DataType.HASHMAP;
                } else {
                    dataType = DataType.OBJECT;
                }
            }
        }

        /**
         * 更新二级列的起始列结束列信息
         * 原来配置的信息是针对所有的，需要计算成需要导出的列
         * @param fieldneedExportMap
         */
        private void refreshSecFieldInfo(List<HashMap<String, Object>> fieldneedExportMap) {
            // 计算二级标题的起始列以及合并多少列
            for (SecFieldVo secFieldVo : secFieldList) {
                int size = fieldneedExportMap.size();

                // 确定二级标题列的开始导出列以及结束导出列
                boolean bFindField = false;
                for (int i = 0; i < size; i++) {
                    HashMap hashMap = fieldneedExportMap.get(i);
                    if (!bFindField) {
                        if (secFieldVo.startName.equals(hashMap.get("name"))) {
                            bFindField = true;
                        }
                    }

                    // 找到开始列后计算结束列
                    if (bFindField) {
                        if ((Boolean) hashMap.get("needExport") == false) { // 如果开始列不需要导出，开始列往后推
                            if (i == size - 1) { // 如果是最后一列了，直接设置mergeFieldCount为0
                                secFieldVo.mergeFieldCount = 0;
                            } else {
                                secFieldVo.mergeFieldCount = secFieldVo.mergeFieldCount - 1;
                                secFieldVo.startName = fieldneedExportMap.get(i + 1).get("name").toString();
                            }
                        } else { // 如果开始列需要导出，直接计算结束列
                            if (i + secFieldVo.mergeFieldCount - 1 > size - 1) { // 如果结束列超过了最后一列，则用最后一列作为结束列
                                secFieldVo.mergeFieldCount = size - i;
                                secFieldVo.endName = fieldneedExportMap.get(size - 1).get("name").toString();
                            } else {
                                secFieldVo.endName = fieldneedExportMap.get(i + secFieldVo.mergeFieldCount - 1).get("name").toString();
                            }

                            // 退出循环
                            break;
                        }

                    }
                }

                // 计算结束的导出列
                bFindField = false;
                for (int i = size - 1; i >= 0; i--) {
                    HashMap hashMap = fieldneedExportMap.get(i);
                    if (!bFindField) {
                        if (secFieldVo.endName.equals(hashMap.get("name"))) {
                            bFindField = true;
                        }
                    }

                    // 找到结束列后计算合并多少列
                    if (bFindField) {
                        if ((Boolean) hashMap.get("needExport") == false) { // 如果结束列不需要导出，结束列往前推
                            if (i == 0) { // 如果是第一列了，直接设置mergeFieldCount为0
                                secFieldVo.mergeFieldCount = 0;
                            } else {
                                secFieldVo.mergeFieldCount = secFieldVo.mergeFieldCount - 1;
                                secFieldVo.startName = fieldneedExportMap.get(i - 1).get("name").toString();
                            }
                        } else { // 如果结束列需要导出，直接计算合并多少列
                            for (int j = i; j >= 0; j--) {
                                // 如果到了开始列，退出循环
                                if (secFieldVo.startName.equals(fieldneedExportMap.get(j).get("name"))) {
                                    break;
                                }

                                // 中间隐藏一列，和并列计数减一
                                if ((Boolean) fieldneedExportMap.get(j).get("needExport") == false) {
                                    secFieldVo.mergeFieldCount = secFieldVo.mergeFieldCount - 1;
                                }
                            }
                            // 退出循环
                            break;
                        }

                    }
                }
            }

            // 删除合并列是0的二级列信息
            for (int i = secFieldList.size() - 1; i >= 0; i--) {
                if (secFieldList.get(i).mergeFieldCount == 0) {
                    secFieldList.remove(i);
                }
            }
        }
    }

    public void setGridOptions(JSONObject gridOptions) {
        exportManager.LoadExportOptions(gridOptions);
    }

    public void setTitle(String title) {
        exportManager.title = title;
    }

    public PrintGridExcelUtil() {
        exportManager = new ExportManager();
    }

    public void printGridExcel(HttpServletRequest request, List list) throws Exception {
        String XlsName = new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + ".xls";
        String filePath = request.getSession().getServletContext().getRealPath("/") + StaticParam.xslPath + "\\" + XlsName;

        exportManager.setDataList(list);

        WritableWorkbook wwb = Workbook.createWorkbook(new File(filePath));
        WritableSheet ws = null;

        sheetIndex = -1;
        rowIndex = StaticParam.BEGIN_ROW;
        Label label = null;
        for (int i = 0; i < list.size(); i++) {
            if (i % StaticParam.OneSheetRows == 0) {
                // 创建新表页
                sheetIndex += 1;
                ws = wwb.createSheet("sheet" + String.valueOf(sheetIndex), sheetIndex);

                // 初始化表页
                rowIndex = exportManager.initSheetHead(ws);
            }

            rowIndex ++;
            // 填序号
            label = new Label(StaticParam.BEGIN_COL, rowIndex, String.valueOf(i + 1), exportManager.formatMap.get(StaticParam.FORMAT_XH));
            ws.addCell(label);

            String fieldName = "";
            String fieldValue = "";
            ExportManager.FieldVo fieldVo = null;
            for(int j = 0; j < exportManager.fieldList.size(); j++){
                fieldName = exportManager.fieldList.get(j).name;
                fieldValue = exportManager.getData(fieldName, i);
                fieldVo = exportManager.getFieldByName(fieldName);

                if ("number".equalsIgnoreCase(fieldVo.formatterType) && StringUtils.isNotEmpty(fieldValue)) {
                    Number number = new Number(StaticParam.BEGIN_COL + j + 1, rowIndex, Double.parseDouble(exportManager.getData(fieldName, i, false)), exportManager.formatMap.get(fieldName));
                    ws.addCell(number);
                } else if ("date".equalsIgnoreCase(fieldVo.formatterType) && StringUtils.isNotEmpty(fieldValue)) {
                    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    DateTime dateTime = new DateTime(StaticParam.BEGIN_COL + j + 1, rowIndex, dateFormat.parse(exportManager.getData(fieldName, i, false)), exportManager.formatMap.get(fieldName));
                    ws.addCell(dateTime);
                } else {
                    label = new Label(StaticParam.BEGIN_COL + j + 1, rowIndex, fieldValue, exportManager.formatMap.get(fieldName));
                    ws.addCell(label);
                }
            }

            exportManager.mergeField(ws, rowIndex, i);

        }
        // 释放数据内存
        list.clear();

        wwb.write();
        wwb.close();

        request.setAttribute("FileDownLoadURL", request.getContextPath()+"/" + (StaticParam.xslPath.replaceAll("\\\\", "/")) + "" + XlsName);
    }
}
