/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: FileServiceImpl.java
 *
 * LastModified: 2009-09-24 10:39:10
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.service.impl;

import com.gtis.fileCenter.CharsetDetector;
import com.gtis.fileCenter.dao.NodeDao;
import com.gtis.fileCenter.ex.FileIOException;
import com.gtis.fileCenter.ex.FileOverSizeException;
import com.gtis.fileCenter.file.CallbackFileChannel;
import com.gtis.fileCenter.file.FileStore;
import com.gtis.fileCenter.file.FileStreamListener;
import com.gtis.fileCenter.file.impl.WriteStreamListener;
import com.gtis.fileCenter.model.MicroFile;
import com.gtis.fileCenter.model.PreviewFile;
import com.gtis.fileCenter.service.FileStoreService;
import com.gtis.generic.util.ImageUtils;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.FileCopyUtils;

import javax.servlet.ServletOutputStream;
import java.io.*;
import java.net.URLEncoder;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.util.ArrayList;
import java.util.List;


/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-16
 */
public class FileStoreServiceImpl implements FileStoreService {

    private static final Logger logger = LoggerFactory.getLogger(FileStoreServiceImpl.class);

    private static final int MICRO_IMAGE_WIDTH = 96;
    private static final int MICRO_IMAGE_HEIGHT = 96;

    private static final int TRANSFER_SIZE = 16 * 1024;

    @Autowired
    FileStore fileStore;
    @Autowired
    NodeDao nodeDao;

    @Autowired
    private TaskExecutor taskExecutor;

    private String previewServerUrl;
    private boolean useDocServer;
    private String fcUrl;
    private int picturePreviewMaxSize;

    public void setPreviewServerUrl(String previewServerUrl) {
        this.previewServerUrl = previewServerUrl;
        this.useDocServer = previewServerUrl.contains("docserver");
    }

    public void setFcUrl(String fcUrl) {
        this.fcUrl = fcUrl;
    }

    public void setPicturePreviewMaxSize(int picturePreviewMaxSize) {
        this.picturePreviewMaxSize = picturePreviewMaxSize;
    }

    public FileStore getFileStore() {
        return fileStore;
    }

    public String getPreviewServerUrl() {
        return previewServerUrl;
    }

    public boolean exists(com.gtis.fileCenter.model.impl.File file) {
        return file.getStoreUrl() != null && fileStore.getFile(file.getStoreUrlObject()).exists();
    }

    public java.io.File getFile(com.gtis.fileCenter.model.impl.File file) {
        File f;
        if (file.getStoreUrl() == null) {
            f = fileStore.createFile(file);
            file.setStoreUrl(fileStore.getStoreUrl(f).toString());
            return f;
        } else {
            return fileStore.getFile(file.getStoreUrlObject());
        }
    }

    public FileChannel getFileChannel(com.gtis.fileCenter.model.impl.File file) throws FileIOException {
        return getFileChannel(file, 0);
    }

    public FileChannel getFileChannel(com.gtis.fileCenter.model.impl.File file, long skip) throws FileIOException {
        FileChannel fc;
        try {
            if (skip <= 0) {
                fc = new FileInputStream(getFile(file)).getChannel();
            } else {
                fc = new RandomAccessFile(getFile(file), "r").getChannel();
                fc.position(skip);
            }
            return fc;
        } catch (Throwable e) {
            throw new FileIOException("Failed to open FileChannel:" + file, e);
        }
    }

    public InputStream getFileInputStream(com.gtis.fileCenter.model.impl.File file) throws FileIOException {
        return new BufferedInputStream(Channels.newInputStream(getFileChannel(file)));
    }

    public InputStream getFileInputStream(com.gtis.fileCenter.model.impl.File file, long skip) throws FileIOException {
        return new BufferedInputStream(Channels.newInputStream(getFileChannel(file, skip)));
    }

    public String toString(com.gtis.fileCenter.model.impl.File file) {
        StringWriter writer = new StringWriter();
        BufferedReader reader = null;
        try {
            java.io.File f = getFile(file);
            reader = new BufferedReader(new InputStreamReader(getFileInputStream(file), CharsetDetector.detect(f)));
            char[] buffer = new char[TRANSFER_SIZE];
            int n;
            while (-1 != (n = reader.read(buffer))) {
                writer.write(buffer, 0, n);
            }
            reader.close();
        } catch (Exception e) {
            return "";
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ignored) {
                }
            }
        }
        return writer.toString();
    }

    public void transferTo(com.gtis.fileCenter.model.impl.File file, OutputStream os) throws FileIOException {
        transferTo(file, 0, os);
    }

    public void transferTo(com.gtis.fileCenter.model.impl.File file, long skip, OutputStream os) throws FileIOException {
        FileChannel in = null;
        try {
            in = getFileChannel(file);
            in.transferTo(skip, in.size() - skip, Channels.newChannel(os));
        } catch (IOException e) {
            throw new FileIOException("Failed to open transfer file:" + file, e);
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException ignored) {
            }
            if (os != null) {
                try {
                    os.close();
                } catch (IOException ignored) {
                }
            }
        }
    }

    public void transferZipContainedFileTo(com.gtis.fileCenter.model.impl.File file, String name, ServletOutputStream outputStream) throws FileIOException {
        try {
            ZipFile zfile = new ZipFile(getFile(file));
            ZipEntry zf = zfile.getEntry(name);
            if (zf != null)
                FileCopyUtils.copy(zfile.getInputStream(zf), outputStream);
        } catch (IOException e) {
            throw new FileIOException("Failed to transfer file:" + file, e);
        }
    }

    public FileChannel getWritableFileChannel(com.gtis.fileCenter.model.impl.File file) throws FileIOException {
        return getWritableFileChannel(file, 0);
    }

    public FileChannel getWritableFileChannel(com.gtis.fileCenter.model.impl.File file, long skip) throws FileIOException {
        FileChannel fc;
        try {
            if (skip <= 0) {
                fc = new FileOutputStream(getFile(file)).getChannel();
            } else {
                fc = new RandomAccessFile(getFile(file), "rw").getChannel();
                fc.position(skip);
            }
            List<FileStreamListener> listeners = new ArrayList<FileStreamListener>();
            listeners.add(new WriteStreamListener(this, file));
            return new CallbackFileChannel(fc, listeners);
        } catch (Throwable e) {
            throw new FileIOException("Failed to open FileChannel:" + file, e);
        }
    }

    public OutputStream getFileOutputStream(com.gtis.fileCenter.model.impl.File file) throws FileIOException {
        return getFileOutputStream(file, 0);
    }

    public OutputStream getFileOutputStream(com.gtis.fileCenter.model.impl.File file, long skip) throws FileIOException {
        return new BufferedOutputStream(Channels.newOutputStream(getWritableFileChannel(file, skip)));
    }

    public void save(com.gtis.fileCenter.model.impl.File file, InputStream is) throws FileIOException {
        ReadableByteChannel in = null;
        FileChannel out = null;
        try {
            in = Channels.newChannel(is);
            out = getWritableFileChannel(file);
            ByteBuffer buffer = ByteBuffer.allocateDirect(TRANSFER_SIZE);
            while (in.read(buffer) != -1) {
                buffer.flip();
                out.write(buffer);
                buffer.clear();
            }
        } catch (Exception e) {
            throw new FileIOException("Failed to save file:" + file, e);
        } finally {
            try {
                if (in != null)
                    in.close();
            } catch (IOException ignored) {
            }
            try {
                if (out != null)
                    out.close();
            } catch (IOException ignored) {
            }
        }
    }

    public void save(com.gtis.fileCenter.model.impl.File file, FileChannel fc) throws FileIOException {
        FileOutputStream fos = null;
        try {
            long size = fc.size();
            FileStreamListener listener = new WriteStreamListener(this, file);
            fos = new FileOutputStream(getFile(file));
            fc.transferTo(0, size, fos.getChannel());
            listener.contentStreamClosed(size);
        } catch (IOException e) {
            throw new FileIOException("Failed to save file:" + file, e);
        } finally {
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException ignored) {
                }
            }
            try {
                fc.close();
            } catch (IOException ignored) {
            }
        }
    }

    public void save(com.gtis.fileCenter.model.impl.File file, java.io.File diskFile) throws FileIOException {
        try {
            save(file, new FileInputStream(diskFile).getChannel());
        } catch (IOException e) {
            throw new FileIOException("Failed to save file:" + file, e);
        }
    }

    public void save(com.gtis.fileCenter.model.impl.File file, byte[] bytes) throws FileIOException {
        save(file, new ByteArrayInputStream(bytes));
    }

    public void save(com.gtis.fileCenter.model.impl.File file, String s) throws FileIOException {
        save(file, s.getBytes());
    }

    public void save(com.gtis.fileCenter.model.impl.File file, String s, String charset) throws FileIOException {
        try {
            save(file, s.getBytes(charset));
        } catch (UnsupportedEncodingException e) {
            throw new FileIOException("Failed to save file:" + file, e);
        }
    }


    public void copyTo(com.gtis.fileCenter.model.impl.File srcFile, com.gtis.fileCenter.model.impl.File destFile) throws FileIOException {
        destFile.setStoreUrl(srcFile.getStoreUrl());//just link to src file
        //save(destFile, getFileChannel(srcFile));
    }

    public void delete(final com.gtis.fileCenter.model.impl.File file) throws FileIOException {
        if (hasLinkFile(file))
            return;
        taskExecutor.execute(new Runnable() {
            public void run() {
                fileStore.delete(file.getStoreUrlObject());
                if (!(file instanceof MicroFile) && file.isImage()) {
                    fileStore.delete(new MicroFile(file).getStoreUrlObject());
                }
                if (!(file instanceof PreviewFile) && file.isDocument()) {
                    fileStore.delete(new PreviewFile(file).getStoreUrlObject());
                }
            }
        });
    }

    public void createMicroImage(MicroFile microFile) {
        createMicroImage(microFile,MICRO_IMAGE_WIDTH);
    }

    /**
     * 按照传递的尺寸生成图片缩略图
     * @param microFile 缩略图
     * @param fileSize
     */
    public void createMicroImage(MicroFile microFile,final int fileSize) {
        //该异常导致前台页面返回图片为红叉，故由抛出异常改为直接return
        if (microFile.getSize() > MicroFile.MAX_SIZE)
            return;
        //throw new FileOverSizeException(microFile.getFile().getName());
        final String srcPath = getFile(microFile.getFile()).getAbsolutePath();
        if (!new File(srcPath).exists())
            return;
        final String destPath = getFile(microFile).getAbsolutePath();
        try {
            if (microFile.getSize() > MicroFile.TASK_SIZE) {
                taskExecutor.execute(new Runnable() {
                    public void run() {
                        try {
                            ImageUtils.resizeImage(srcPath, destPath, fileSize, fileSize);
                        } catch (Exception e) {
                            logger.error("Unable to transform image:[" + srcPath + "] to [" + destPath + "]", e);
                        }
                    }
                });

            } else {
                ImageUtils.resizeImage(srcPath, destPath, fileSize, fileSize);
            }
        } catch (Exception ex) {
            logger.error("Unable to transform image:[" + srcPath + "] to [" + destPath + "]", ex);
        }
    }

    public void createPreviewFile(PreviewFile previewFile) throws FileOverSizeException {
        File srcFile = getFile(previewFile.getFile());
        if (!srcFile.exists() || srcFile.length() == 0) {
            return;
        }
        if (previewFile.getSize() > picturePreviewMaxSize)
            throw new FileOverSizeException(previewFile.getFile().getName());
        final StringBuilder sb = new StringBuilder(previewServerUrl);
        try {
            if (useDocServer) {
                String ext = previewFile.getFile().getExtension();
                if ("pdf".equals(ext) || "ceb".equals(ext)) {
                    return;//not support
                }
                String src = srcFile.getAbsolutePath();
                String target = getFile(previewFile).getAbsolutePath();
                sb.append("/convert?rsync=true&src=").append(URLEncoder.encode(src, "utf-8")).append("&target=").append(URLEncoder.encode(target, "utf-8"));
            } else {
                String url = fcUrl + "/file/preview.do?fid=" + previewFile.getId();
                url = URLEncoder.encode(url, "utf-8");
                sb.append("?fid=").append(previewFile.getId()).append("&url=").append(url).append("&filetype=").append(previewFile.getFile().getExtension());
            }
        } catch (UnsupportedEncodingException ignored) {
        }
        taskExecutor.execute(new Runnable() {
            public void run() {
                String url = sb.toString();
                try {
                    HttpClient httpClient = new HttpClient();
                    HttpMethod method = new GetMethod(url);
                    if (httpClient.executeMethod(method) != HttpStatus.SC_OK)
                        logger.error("error to connect {}", url);
                    else
                        logger.debug("get {} success", url);

                } catch (Exception e) {
                    logger.error("error to connect {}", url);
                }
            }
        });
    }

    public boolean hasLinkFile(com.gtis.fileCenter.model.impl.File file) {
        return nodeDao.hasLinkFile(file.getStoreUrl());
    }
}
