/*
 * Author: xyang
 *
 * Project: trunk
 *
 * File: NodeDirContext.java
 *
 * LastModified: 2010-01-15 10:27:49
 *
 * Copyright (c) 2010 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.fs.webdav;

import com.gtis.fileCenter.Constants;
import com.gtis.fileCenter.ex.NodeExistsException;
import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.*;
import com.gtis.fileCenter.model.impl.File;
import com.gtis.fileCenter.model.impl.NodeImpl;
import com.gtis.fileCenter.model.impl.PersonalSpace;
import com.gtis.fileCenter.service.FileStoreService;
import com.gtis.fileCenter.service.NodeService;
import org.apache.commons.lang.StringUtils;
import org.apache.naming.NamingContextBindingsEnumeration;
import org.apache.naming.NamingContextEnumeration;
import org.apache.naming.NamingEntry;
import org.apache.naming.resources.BaseDirContext;
import org.apache.naming.resources.Resource;
import org.apache.naming.resources.ResourceAttributes;

import javax.naming.NameAlreadyBoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.OperationNotSupportedException;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-1-15
 */
public class NodeDirContext extends BaseDirContext {

    protected Node base;

    private NodeService nodeService;
    private FileStoreService fileService;


    public NodeDirContext(NodeService nodeService, FileStoreService fileService) {
        this.nodeService = nodeService;
        this.fileService = fileService;
    }

    public void setDocBase(String docBase) {
        if (docBase == null)
            throw new IllegalArgumentException(sm.getString("resources.null"));
        base = nodeService.getNode(Integer.valueOf(docBase));
        if (base == null)
            throw new IllegalArgumentException(sm.getString("fileResources.base", docBase));
        super.setDocBase(docBase);
    }

    @Override
    public Object lookup(String name) throws NamingException {
        Object result;
        Node node = getNode(name);
        if (node == null)
            throw new NamingException(sm.getString("resources.notFound", name));
        if (node instanceof File) {
            try {
                result = new Resource(fileService.getFileInputStream((File) node));
            } catch (Exception e) {
                result = new Resource(new byte[]{});
            }
        } else {
            NodeDirContext tempContext = new NodeDirContext(nodeService, fileService);
            tempContext.setDocBase(node.getId().toString());
            result = tempContext;
        }
        return result;
    }

    @Override
    public void unbind(String name) throws NamingException {
        Node node = getNode(name);
        if (node == null)
            throw new NamingException(sm.getString("resources.notFound", name));
        nodeService.remove(node.getId());
    }

    @Override
    public void rename(String oldName, String newName) throws NamingException {
        Node node = getNode(oldName);
        if (node == null)
            throw new NamingException(sm.getString("resources.notFound", oldName));
        node.setName(newName);
        nodeService.save(node);
    }

    @Override
    public NamingEnumeration list(String name) throws NamingException {
        Node node = getNode(name);
        if (node == null)
            throw new NamingException(sm.getString("resources.notFound", name));
        return new NamingContextEnumeration(list(node).iterator());
    }

    @Override
    public NamingEnumeration listBindings(String name) throws NamingException {
        Node node = getNode(name);
        if (node == null)
            throw new NamingException(sm.getString("resources.notFound", name));
        return new NamingContextBindingsEnumeration(list(node).iterator(), this);
    }

    @Override
    public void destroySubcontext(String name) throws NamingException {
        unbind(name);
    }

    @Override
    public Object lookupLink(String name) throws NamingException {
        // Note : Links are not supported
        return lookup(name);
    }

    @Override
    public String getNameInNamespace() throws NamingException {
        return docBase;
    }

    @Override
    public Attributes getAttributes(String name, String[] strings) throws NamingException {
        // Building attribute list
        Node node = getNode(name);
        if (node == null)
            throw new NamingException(sm.getString("resources.notFound", name));
        return new NodeResourceAttributes(node);
    }

    @Override
    public void modifyAttributes(String name, int i, Attributes attributes) throws NamingException {

    }

    @Override
    public void modifyAttributes(String name, ModificationItem[] modificationItems) throws NamingException {

    }

    @Override
    public void bind(String name, Object obj, Attributes attributes) throws NamingException {
        Node node = getNode(name);
        if (node != null)
            throw new NameAlreadyBoundException(sm.getString("resources.alreadyBound", name));
        rebind(name, obj, attributes);
    }

    @Override
    public void rebind(String name, Object obj, Attributes attributes) throws NamingException {
        InputStream is = null;
        if (obj instanceof Resource) {
            try {
                is = ((Resource) obj).streamContent();
            } catch (IOException ignored) {
            }
        } else if (obj instanceof InputStream) {
            is = (InputStream) obj;
        }
        if (is == null)
            throw new NamingException(sm.getString("resources.bindFailed", name));
        int i = name.lastIndexOf('/', name.length() - 2) + 1;
        Node node = getNode(name.substring(0, i));
        if (node == null)
            throw new NamingException(sm.getString("resources.bindFailed", name));

        File file = new File();
        file.setParentId(node.getId());
        file.setName(name.substring(i));
        try {
            fileService.delete((File) nodeService.getChildNode(node.getId(), name.substring(i)));
        } catch (NodeNotFoundException ignored) {
        }
        try {
            fileService.save(file, is);
            nodeService.save(file);
        } catch (NodeExistsException e) {
            throw new NameAlreadyBoundException(sm.getString("resources.alreadyBound", name));
        } catch (Exception e) {
            NamingException ne = new NamingException(sm.getString("resources.bindFailed", e));
            ne.initCause(e);
            throw ne;
        }
    }

    @Override
    public DirContext createSubcontext(String name, Attributes attributes) throws NamingException {
        try {
            int i = name.lastIndexOf('/', name.length() - 2) + 1;
            Node node = getNode(name.substring(0, i));
            if (node != null) {
                nodeService.createNode(node.getId(), name.substring(i));
            } else
                throw new NamingException(sm.getString("resources.bindFailed", name));
        } catch (NodeExistsException e) {
            throw new NameAlreadyBoundException(sm.getString("resources.alreadyBound", name));
        }
        return (DirContext) lookup(name);
    }

    @Override
    public DirContext getSchema(String name) throws NamingException {
        throw new OperationNotSupportedException();
    }

    @Override
    public DirContext getSchemaClassDefinition(String name) throws NamingException {
        throw new OperationNotSupportedException();
    }

    @Override
    public NamingEnumeration search(String name, Attributes attributes, String[] strings) throws NamingException {
        return null;
    }

    @Override
    public NamingEnumeration search(String name, Attributes attributes) throws NamingException {
        return null;
    }

    @Override
    public NamingEnumeration search(String name, String name1, SearchControls searchControls) throws NamingException {
        return null;
    }

    @Override
    public NamingEnumeration search(String name, String name1, Object[] objects, SearchControls searchControls) throws NamingException {
        return null;
    }

    protected Node getNode(String name) {
        int i = name.indexOf('/', 1);
        if (i == -1)
            return null;
        int j = name.indexOf('/', i + 1);
        if (j == -1)
            j = name.length();
        String token = name.substring(1, i);
        String nid = name.substring(i + 1, j);
        String path = name.substring(j);
        if (StringUtils.isBlank(token) || StringUtils.isBlank(nid))
            return null;
        try {
            HttpSession session = WebdavSessionContext.getInstance().getSession(token);
            if (session == null) {
                Node node = null;
                try {
                    node = nodeService.getNode(Integer.parseInt(nid), path);
                } catch (NumberFormatException ignored) {
                }
                if (node == null)
                    return null;
                if (nodeService.hasPermission(token, node.getId()))
                    return node;
            } else {
                PersonalSpace space = (PersonalSpace) session.getAttribute(Constants.PERSONAL_SPACE);
                return nodeService.getNode(space.getId(), path);
            }
        } catch (NodeNotFoundException ignored) {
        }
        return null;
    }

    protected ArrayList<NamingEntry> list(Node node) {
        ArrayList<NamingEntry> entries = new ArrayList<NamingEntry>();
        if (node instanceof File)
            return entries;
        for (Node n : nodeService.getChildNodes(node.getId())) {
            Object obj;
            if (n instanceof File) {
                try {
                    obj = new Resource(fileService.getFileInputStream((File) n));
                } catch (Exception e) {
                    obj = new Resource(new byte[]{});
                }
            } else {
                NodeDirContext tempContext = new NodeDirContext(nodeService, fileService);
                tempContext.setDocBase(n.getId().toString());
                obj = tempContext;
            }
            NamingEntry entry = new NamingEntry(n.getName(), obj, NamingEntry.ENTRY);
            entries.add(entry);
        }
        return entries;
    }


    protected class NodeResourceAttributes extends ResourceAttributes {

        protected Node node;
        protected boolean accessed = false;
        protected String canonicalPath = null;

        public NodeResourceAttributes(Node node) {
            this.node = node;
            getCreation();
            getLastModified();
        }

        /**
         * Is collection.
         */
        public boolean isCollection() {
            if (!accessed) {
                collection = !(node instanceof File);
                accessed = true;
            }
            return super.isCollection();
        }


        /**
         * Get content length.
         *
         * @return content length value
         */
        public long getContentLength() {
            if (contentLength != -1L)
                return contentLength;
            if (node instanceof File) {
                contentLength = ((File) node).getSize();
            } else if (node instanceof Space) {
                contentLength = ((Space) node).getSize();
            } else
                contentLength = 0;
            return contentLength;
        }


        /**
         * Get creation time.
         *
         * @return creation time value
         */
        public long getCreation() {
            if (creation != -1L)
                return creation;
            creation = getLastModified();
            return creation;
        }


        /**
         * Get creation date.
         *
         * @return Creation date value
         */
        public Date getCreationDate() {
            if (creation == -1L) {
                creation = getCreation();
            }
            return super.getCreationDate();
        }


        /**
         * Get last modified time.
         *
         * @return lastModified time value
         */
        public long getLastModified() {
            if (lastModified != -1L)
                return lastModified;
            lastModified = node.getUpdateTime().getTime();
            return lastModified;
        }


        /**
         * Get lastModified date.
         *
         * @return LastModified date value
         */
        public Date getLastModifiedDate() {
            if (lastModified == -1L) {
                lastModified = getLastModified();
            }
            return super.getLastModifiedDate();
        }


        /**
         * Get name.
         *
         * @return Name value
         */
        public String getName() {
            if (name == null)
                name = node.getName();
            return name;
        }


        /**
         * Get resource type.
         *
         * @return String resource type
         */
        public String getResourceType() {
            if (!accessed) {
                collection = !(node instanceof File);
                accessed = true;
            }
            return super.getResourceType();
        }


        /**
         * Get canonical path.
         *
         * @return String the file's canonical path
         */
        public String getCanonicalPath() {
            if (canonicalPath == null) {
                canonicalPath = ((NodeImpl) node).getFullPath();
            }
            return canonicalPath;
        }

    }
}

