package com.gtis.fileCenter.fs.jlan;

import com.gtis.fileCenter.Constants;
import com.gtis.fileCenter.ex.NodeExistsException;
import com.gtis.fileCenter.fs.webdav.WebdavSessionContext;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.model.impl.File;
import com.gtis.fileCenter.model.impl.NodeImpl;
import com.gtis.fileCenter.model.impl.PersonalSpace;
import com.gtis.fileCenter.service.FileStoreService;
import com.gtis.fileCenter.service.NodeService;
import org.alfresco.jlan.server.SrvSession;
import org.alfresco.jlan.server.core.DeviceContext;
import org.alfresco.jlan.server.core.DeviceContextException;
import org.alfresco.jlan.server.filesys.*;
import org.alfresco.jlan.server.locking.FileLockingInterface;
import org.alfresco.jlan.server.locking.LockManager;
import org.alfresco.jlan.smb.server.disk.NIOJavaNetworkFile;
import org.alfresco.jlan.smb.server.disk.NIOLockManager;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.extensions.config.ConfigElement;
import org.springframework.util.FileCopyUtils;

import javax.servlet.http.HttpSession;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 11-10-11
 */
public class FileCenterDiskDriver implements DiskInterface, FileLockingInterface {

    private static final char DOS_SEPERATOR = '\\';

    private static LockManager _lockManager = new NIOLockManager();
    private static ApplicationContext applicationContext;
    protected static Date GLOBAL_UPDATEDATE = new Date();

    private NodeService nodeService;
    private FileStoreService fileService;

    public FileCenterDiskDriver() {
        nodeService = (NodeService) applicationContext.getBean("nodeService");
        fileService = (FileStoreService) applicationContext.getBean("fileService");
    }

    public static void setApplicationContext(ApplicationContext context) {
        applicationContext = context;
    }

    public void setNodeService(NodeService nodeService) {
        this.nodeService = nodeService;
    }

    public void setFileService(FileStoreService fileService) {
        this.fileService = fileService;
    }

    public void closeFile(SrvSession sess, TreeConnection tree, NetworkFile file) throws IOException {
        Node node = getNode(file.getFullName());
        file.closeFile();
        if (file.getWriteCount() > 0) {
            if (node instanceof File) {
                ((File) node).setSize(file.getFileSize());
            }
            nodeService.save(node);
        }
        if (file.hasDeleteOnClose()) {
            if (file.isDirectory())
                deleteDirectory(sess, tree, file.getFullName());
            else
                deleteFile(sess, tree, file.getFullName());
        }
    }

    public void createDirectory(SrvSession sess, TreeConnection tree, FileOpenParams params) throws IOException {
        String name = params.getPath();
        int i = name.lastIndexOf(DOS_SEPERATOR, name.length() - 2) + 1;
        try {
            Node node = getNode(name.substring(0, i));
            if (!node.isWriteable()) {
                throw new AccessDeniedException("no write ermission");
            }
            nodeService.createNode(node.getId(), name.substring(i));
        } catch (NodeExistsException e) {
            throw new FileExistsException("create directory, path exists " + name.substring(i));
        }
    }

    public NetworkFile createFile(SrvSession sess, TreeConnection tree, FileOpenParams params) throws IOException {
        String name = params.getPath();
        int i = name.lastIndexOf(DOS_SEPERATOR, name.length() - 2) + 1;
        try {
            Node node = getNode(name.substring(0, i));
            if (!node.isWriteable()) {
                throw new AccessDeniedException("no write ermission");
            }
            File file = new File();
            file.setName(name.substring(i));
            file.setParentId(node.getId());
            java.io.File f = fileService.getFile(file);
            FileWriter newFile = new FileWriter(f, false);
            newFile.close();
            nodeService.save(file);
            NIOJavaNetworkFile netFile = new NIOJavaNetworkFile(f, name);
            netFile.setGrantedAccess(NetworkFile.READWRITE);
            netFile.setFullName(name);
            return netFile;
        } catch (NodeExistsException e) {
            throw new FileExistsException("create file, name exists " + name.substring(i));
        }
    }

    public void deleteDirectory(SrvSession sess, TreeConnection tree, String dir) throws IOException {
        deleteFile(sess, tree, dir);
    }

    public void deleteFile(SrvSession sess, TreeConnection tree, String name) throws IOException {
        Node node = getNode(name);
        if (!node.isWriteable()) {
            throw new AccessDeniedException("no write ermission");
        }
        nodeService.remove(node.getId());
    }

    public int fileExists(SrvSession sess, TreeConnection tree, String name) {
        try {
            return getNode(name) instanceof File ? FileStatus.FileExists : FileStatus.DirectoryExists;
        } catch (IOException e) {
            return FileStatus.NotExist;
        }
    }

    public void flushFile(SrvSession sess, TreeConnection tree, NetworkFile file) throws IOException {
        file.flushFile();
    }

    public FileInfo getFileInformation(SrvSession sess, TreeConnection tree, String name) throws IOException {
        FileInfo info = new FileInfo();
        try {
            return nodeToFileInfo(getNode(name), info);
        } catch (IOException e) {
            return null;
        }
    }

    public boolean isReadOnly(SrvSession sess, DeviceContext ctx) throws IOException {
        return false;
    }

    public NetworkFile openFile(SrvSession sess, TreeConnection tree, FileOpenParams params) throws IOException {
        Node node = getNode(params.getPath());
        if (!node.isWriteable() && (params.isReadWriteAccess() || params.isWriteOnlyAccess()))
            throw new AccessDeniedException("File " + node.getName() + " is read-only");
        NetworkFile netFile;
        if (!(node instanceof File)) {
            netFile = new NIOJavaNetworkFile(new java.io.File("."), params.getPath());
            netFile.setAttributes(FileAttribute.Directory);
        } else {
            File file = (File) node;
            java.io.File srcFile = fileService.getFile(file);
            if (fileService.hasLinkFile(file)) { //use copy on write
                file.setStoreUrl(null);
                java.io.File copyedFile = fileService.getFile(file);
                FileCopyUtils.copy(srcFile, copyedFile);
                return new NIOJavaNetworkFile(copyedFile, params.getPath());
            } else {
                return new NIOJavaNetworkFile(srcFile, params.getPath());
            }
        }
        netFile.setGrantedAccess(params.isReadOnlyAccess() ? NetworkFile.READONLY : NetworkFile.READWRITE);
        netFile.setFullName(params.getPath());
        return netFile;
    }

    public int readFile(SrvSession sess, TreeConnection tree, NetworkFile file, byte[] buf, int bufPos, int siz, long filePos) throws IOException {
        if (file.isDirectory())
            throw new AccessDeniedException();
        int rdlen = file.readFile(buf, siz, bufPos, filePos);
        if (rdlen == -1)
            rdlen = 0;
        return rdlen;
    }

    public void renameFile(SrvSession sess, TreeConnection tree, String oldName, String newName) throws IOException {
        Node src = getNode(oldName);
        int i = newName.lastIndexOf(DOS_SEPERATOR, newName.length() - 2) + 1;
        String fName = newName.substring(i);
        String path = newName.substring(0, i);
        Node target = getNode(path);
        if (!src.isWriteable() || !target.isWriteable()) {
            throw new AccessDeniedException("no write ermission");
        }
        try {
            if (src.getParentId().equals(target.getId())) {
                src.setName(fName);
                nodeService.save(src);
            } else {
                nodeService.move(new Integer[]{src.getId()}, target.getId(), true);
            }
        } catch (NodeExistsException e) {
            throw new FileExistsException("Rename file, path exists " + newName);
        }
    }

    public long seekFile(SrvSession sess, TreeConnection tree, NetworkFile file, long pos, int typ) throws IOException {
        return file.seekFile(pos, typ);
    }

    public void setFileInformation(SrvSession sess, TreeConnection tree, String name, FileInfo info) throws IOException {
    }

    public SearchContext startSearch(SrvSession sess, TreeConnection tree, String name, int attrib) throws FileNotFoundException {
        int i = name.lastIndexOf(DOS_SEPERATOR, name.length() - 2) + 1;
        String fName = name.substring(i);
        List<Node> nodes;
        try {
            Node node = getNode(name.substring(0, i));
            if (node != null) {
                if (fName.indexOf('*') > -1) {
                    if ("*".equals(fName) || "*.*".equals(fName)) {
                        nodes = nodeService.getChildNodes(node.getId());
                    } else {
                        nodes = nodeService.search(node.getId(), StringUtils.replace(StringUtils.replace(fName, "*", "%"), "?", "_"));
                    }
                } else {
                    nodes = node.getId() == null ? Collections.<Node>emptyList() : Collections.singletonList(nodeService.getChildNode(node.getId(), fName));
                }
            } else
                throw new IOException();
            if (node.isWriteable())
                for (Node n : nodes) {
                    ((NodeImpl) n).setWriteable(true);
                }
        } catch (Exception e) {
            throw new FileNotFoundException("search file, not found " + fName);
        }

        FileCenterSearchContext sc = new FileCenterSearchContext();
        sc.initSearch(nodes.iterator());
        return sc;
    }

    public void truncateFile(SrvSession sess, TreeConnection tree, NetworkFile file, long siz) throws IOException {
        file.truncateFile(siz);
        file.flushFile();
    }

    public int writeFile(SrvSession sess, TreeConnection tree, NetworkFile file, byte[] buf, int bufoff, int siz, long fileoff) throws IOException {
        if (file.isDirectory())
            throw new AccessDeniedException();
        file.writeFile(buf, siz, bufoff, fileoff);
        return siz;
    }

    public DeviceContext createContext(String shareName, ConfigElement args) throws DeviceContextException {
        return new DiskDeviceContext("fileCenter");
    }

    public void treeOpened(SrvSession sess, TreeConnection tree) {
    }

    public void treeClosed(SrvSession sess, TreeConnection tree) {
    }

    public LockManager getLockManager(SrvSession sess, TreeConnection tree) {
        return _lockManager;
    }

    protected Node getNode(String name) throws IOException {
        int i = name.indexOf(DOS_SEPERATOR, 1);
        if (i == -1) {
            String token = name.substring(1);
            return newNode(StringUtils.isEmpty(token) ? "fileCenter" : token);
        }
        int j = name.indexOf(DOS_SEPERATOR, i + 1);
        if (j == -1)
            j = name.length();
        String token = name.substring(1, i);
        String nid = name.substring(i + 1, j);
        String path = name.substring(j);
        try {
            HttpSession session = WebdavSessionContext.getInstance().getSession(token);
            Node node = null;
            if (session == null) {
                try {
                    node = nodeService.getNode(Integer.parseInt(nid), path);
                } catch (NumberFormatException ignored) {
                }
                if (node != null && nodeService.hasPermission(token, node.getId())) {
                    ((NodeImpl) node).setWriteable(nodeService.isWriteable(token));
                    return node;
                }
            } else {
                PersonalSpace space = (PersonalSpace) session.getAttribute(Constants.PERSONAL_SPACE);
                node = nodeService.getNode(space.getId(), path);
                if (node instanceof PersonalSpace) {
                    node.setName(space.getId().toString());
                }
                ((NodeImpl) node).setWriteable(true);
                return node;
            }
        } catch (Exception ignored) {
        }
        throw new PathNotFoundException(name);
    }

    protected static Node newNode(String name) {
        NodeImpl node = new NodeImpl();
        node.setWriteable(true);
        node.setName(name);
        node.setUpdateTime(GLOBAL_UPDATEDATE);
        return node;
    }

    private static FileInfo nodeToFileInfo(final Node node, final FileInfo finfo) {
        int fattr = node.isWriteable() ? 0 : FileAttribute.ReadOnly;
        if (node instanceof File) {
            long flen = ((File) node).getSize();
            finfo.setSize(flen);
            finfo.setAllocationSize((flen + 512L) & 0xFFFFFFFFFFFFFE00L);
            finfo.setFileType(FileType.RegularFile);
        } else {
            fattr += FileAttribute.Directory;
            finfo.setFileType(FileType.Directory);
        }
        finfo.setFileName(node.getName());
        finfo.setFileAttributes(fattr);
        finfo.setFileId(node.hashCode());
        long fdate = node.getUpdateTime().getTime();
        finfo.setModifyDateTime(fdate);
        finfo.setCreationDateTime(fdate);
        finfo.setChangeDateTime(fdate);
        return finfo;
    }

    class FileCenterSearchContext extends SearchContext {

        private Iterator<Node> it;

        public void initSearch(Iterator<Node> it) {
            this.it = it;
        }

        @Override
        public int getResumeId() {
            return -1;
        }

        @Override
        public boolean hasMoreFiles() {
            return it.hasNext();
        }

        @Override
        public boolean nextFileInfo(FileInfo info) {
            if (hasMoreFiles()) {
                nodeToFileInfo(it.next(), info);
                return true;
            }
            return false;
        }

        @Override
        public String nextFileName() {
            return it.next().getName();
        }

        @Override
        public boolean restartAt(int resumeId) {
            return false;
        }

        @Override
        public boolean restartAt(FileInfo info) {
            return false;
        }
    }
}
