/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: FileStoreImpl.java
 *
 * LastModified: 2009-09-22 10:19:22
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.file.impl;

import com.gtis.fileCenter.ex.CapabilityNotEnoughException;
import com.gtis.fileCenter.ex.FileIOException;
import com.gtis.fileCenter.file.FileStore;
import com.gtis.fileCenter.model.impl.StoreUrl;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ResourceUtils;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import java.util.UUID;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-16
 */
public class FileStoreImpl implements FileStore {

    private static final Logger logger = LoggerFactory.getLogger(FileStoreImpl.class);

    public static final String PROP_FILE = "fileStore.properties";

    public static final SimpleDateFormat PATH_SDF = new SimpleDateFormat("yyyy/MM/dd/HH/");

    private String name;

    private File directory;

    private String absolutePath;

    private File properties;

    private long capability = 0L;

    private long used = 0L;

    public FileStoreImpl(String name, String path, String capability) throws Exception {
        properties = new File(ResourceUtils.getFile(path), PROP_FILE);
        directory = properties.getParentFile();
        if (!properties.exists()) {
            if (!directory.exists())
                directory.mkdirs();
            properties.createNewFile();
        }
        this.name = name;
        this.absolutePath = directory.getAbsolutePath();
        this.capability = Long.parseLong(capability);
    }

    public String getName() {
        return name;
    }

    public File getDirectory() {
        return directory;
    }

    public long getCapability() {
        return capability;
    }

    public long getUsed() {
        return used;
    }

    public boolean isAvailable() {
        return capability < 0 || capability > used;
    }

    public synchronized void increaseUsed(StoreUrl storeUrl, long size) {
        used = used + size;
    }

    public boolean exists(StoreUrl storeUrl) {
        return getFile(storeUrl).exists();
    }

    public File createFile(com.gtis.fileCenter.model.impl.File f) throws CapabilityNotEnoughException, FileIOException {
        File file = new File(directory, generateName(f));
        if (file.exists())
            logger.error("createFile error,path is exist:[" + file.getAbsolutePath() + "]");
        File dir = file.getParentFile();
        if (!dir.exists()) {
            dir.mkdirs();
        }
        try {
            if (!file.createNewFile())
                throw new FileIOException("createFile file " + file.getPath() + "error");
        } catch (IOException e) {
            throw new FileIOException("createFile file " + file.getPath() + "error");
        }
        return file;
    }

    protected String generateName(com.gtis.fileCenter.model.impl.File f) {
        StringBuilder sb = new StringBuilder(64);
        sb.append(PATH_SDF.format(new Date())).append(UUID.randomUUID().toString());
        return sb.toString();
    }

    public StoreUrl getStoreUrl(File file) {
        String path = file.getAbsolutePath();
        if (!path.startsWith(absolutePath))
            return null;
        return new StoreUrl(name, path.substring(absolutePath.length()).replace('\\', '/'));
    }

    public File getFile(StoreUrl storeUrl) {
        String relativeUrl = storeUrl.getPath();
        return new File(directory, relativeUrl);
    }

    public boolean delete(StoreUrl storeUrl) {
        File file = getFile(storeUrl);
        if (file.exists()) {
            long size = file.length();
            boolean deleted = file.delete();
            synchronized (this) {
                used = used - size;
            }
            return deleted;
        }
        return false;
    }

    public synchronized void init() throws Exception {
        if (this.properties != null) {
            Properties property = new Properties();
            FileInputStream fis = null;
            try {
                fis = new FileInputStream(this.properties);
                property.load(fis);
            } finally {
                if (fis != null) {
                    try {
                        fis.close();
                    } catch (IOException ignored) {
                    }
                }
            }
            String usedStr = property.getProperty(getName());
            try {
                this.used = Long.parseLong(usedStr);
            } catch (NumberFormatException ignore) {
            }
            if (capability != -1 && this.used == 0L) {
                logger.warn("used property in file fileStore.properties is zero,start to scan used size,may take some time");
                countUsed(directory);
                logger.info("scan complete used:" + this.used + "/" + this.capability);
            }
            if (capability != -1 && this.used > this.capability)
                throw new Exception("The used size has exceed,please extend the capability of the store " + this.name);
        }
    }

    public synchronized void save() throws Exception {
        if (this.properties != null) {
            Properties property = new Properties();
            InputStreamReader isr = null;
            FileOutputStream fos = null;
            try {
                isr = new InputStreamReader(new FileInputStream(this.properties), "UTF-8");
                property.load(isr);
                property.setProperty(getName(), Long.toString(this.used));
                isr.close();
                isr = null;
                fos = new FileOutputStream(this.properties);
                property.store(fos, "");
            } finally {
                if (isr != null) {
                    try {
                        isr.close();
                    } catch (IOException ignored) {
                    }
                }
                if (fos != null) {
                    try {
                        fos.close();
                    } catch (IOException ignored) {
                    }
                }
            }
        }
    }

    public synchronized void countUsed() throws Exception {
        if (capability != -1)
            return;
        used = 0L;
        countUsed(directory);
    }

    private void countUsed(File directory) {
        this.used = directory.getTotalSpace()-directory.getFreeSpace();
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }
}
