/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: GeneralNodeDaoImpl.java
 *
 * LastModified: 2009-09-22 02:00:50
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.dao.impl;

import com.alibaba.fastjson.JSON;
import com.gtis.common.pagination.PaginationSQLGetter;
import com.gtis.fileCenter.dao.NodeDao;
import com.gtis.fileCenter.ex.NestedRuntimeException;
import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.model.Space;
import com.gtis.fileCenter.model.impl.File;
import com.gtis.fileCenter.model.impl.NodeImpl;
import com.gtis.fileCenter.model.impl.PersonalSpace;
import com.gtis.fileCenter.model.impl.WorkSpace;
import com.gtis.fileCenter.service.MimeTypeService;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcTemplate;
import org.springframework.jdbc.support.incrementer.DataFieldMaxValueIncrementer;
import org.springframework.stereotype.Repository;

import javax.sql.DataSource;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-21
 */
@Repository("nodeDao")
public class GeneralNodeDaoImpl implements NodeDao {

    private static final Logger logger = LoggerFactory.getLogger(GeneralNodeDaoImpl.class);
    private SimpleJdbcTemplate simpleJdbcTemplate;
    @Autowired
    private MimeTypeService mimeTypeService;
    @Autowired
    private DataFieldMaxValueIncrementer nodeIdGenerator;
    @Autowired
    private PaginationSQLGetter paginationSQLGetter;

    @Autowired
    public void setDataSource(DataSource dataSource) {
        this.simpleJdbcTemplate = new SimpleJdbcTemplate(dataSource);
    }

    public void initRoot() {
        try {
            getNode(null, Node.TYPE_ROOT);
        } catch (NodeNotFoundException e) {
            logger.info("ROOT not exist create it");
            simpleJdbcTemplate.update("insert into t_fc_node (id,node_name,node_type,update_time,path) values (1,'ROOT',-1,'2011-01-01','/')");
        }
        try {
            getNode(null, Node.TYPE_PERSONAL_ROOT);
        } catch (NodeNotFoundException e) {
            logger.info("PERSONAL_ROOT not exist create it");
            simpleJdbcTemplate.update("insert into t_fc_node (id,parent_id,node_name,node_type,update_time,path) values (2,1,'PERSONALROOT',-2,'2011-01-01','/1/')");
        }
        try {
            getNode(null, Node.TYPE_WORK_ROOT);
        } catch (NodeNotFoundException e) {
            logger.info("WORK_ROOT not exist create it");
            simpleJdbcTemplate.update("insert into t_fc_node (id,parent_id,node_name,node_type,update_time,path) values (3,1,'WORKROOT',-3,'2011-01-01','/1/')");
        }
    }

    public Node save(Node node1) throws NodeNotFoundException {
        NodeImpl node = (NodeImpl) node1;
        if (node.getId() == null) {
            node.setId(nodeIdGenerator.nextIntValue());
            return insert(node);
        } else
            return update(node);
    }

    private Node insert(NodeImpl node) {
        final String sql = "insert into t_fc_node (id,parent_id,node_name,view_name,node_type,scope,description,owner,update_time,path,node_size,store_url,attributes) " +
                "values (:id,:parent_id,:name,:view_name,:type,:scope,:description,:owner,:update_time,:path,:size,:store_url,:attributes)";
        simpleJdbcTemplate.update(sql, createNodeParameterSource(node));
        logger.debug("insert node ({}) success", node);
        return completeNode(node);
    }

    private Node update(NodeImpl node) throws NodeNotFoundException {
        final String sql = "update t_fc_node set parent_id=:parent_id,node_name=:name,view_name=:view_name,node_type=:type,scope=:scope,description=:description,owner=:owner," +
                "update_time=:update_time,path=:path,node_size=:size,store_url=:store_url,attributes=:attributes where id=:id";
        if (simpleJdbcTemplate.update(sql, createNodeParameterSource(node)) == 0)
            throw new NodeNotFoundException(node.getId());
        logger.debug("update node ({}) success", node);
        return completeNode(node);
    }

    private MapSqlParameterSource createNodeParameterSource(NodeImpl node) {
        Map<String, String> attrs = node.getAttributes();
        Long size = null;
        String storeUrl = null;
        if (node instanceof Space) {
            size = ((Space) node).getSize();
        } else if (node instanceof File) {
            size = ((File) node).getSize();
            storeUrl = ((File) node).getStoreUrl();
        }
        return new MapSqlParameterSource()
                .addValue("id", node.getId())
                .addValue("parent_id", node.getParentId())
                .addValue("name", node.getName())
                .addValue("view_name", node.getViewName())
                .addValue("type", node.getType())
                .addValue("description", node.getDescription())
                .addValue("owner", node.getOwner())
                .addValue("update_time", node.getUpdateTime())
                .addValue("path", node.getPath())
                .addValue("scope", node.getScope())
                .addValue("size", size)
                .addValue("store_url", storeUrl)
                .addValue("attributes", attrs != null && attrs.size() > 0 ? JSON.toJSONString(attrs) : null);
    }

    public boolean exists(Integer nodeId) {
        final String sql = "select count(0) from t_fc_node where id=?";
        return simpleJdbcTemplate.queryForInt(sql, nodeId) > 0;
    }

    public void remove(Integer nodeId) {
        final String sql = "delete from t_fc_node where id=?";
        simpleJdbcTemplate.update(sql, nodeId);
        logger.debug("remove node,id:({})", nodeId);
    }

    public Node getParentNode(Integer nodeId) throws NodeNotFoundException {
        final String sql = "select t1.* from t_fc_node t1 left join t_fc_node t2 on t2.parent_id=t1.id where t2.id=?";
        try {
            return simpleJdbcTemplate.queryForObject(sql, new NodeRowMapper(), nodeId);
        } catch (EmptyResultDataAccessException e) {
            throw new NodeNotFoundException(nodeId);
        }
    }

    public Node getNode(String name, int type) throws NodeNotFoundException {
        String sql;
        try {
            if (type == Node.TYPE_PERSONAL_SPACE) {
                sql = "select * from t_fc_node where node_type=? and owner=?";
                return simpleJdbcTemplate.queryForObject(sql, new NodeRowMapper(), type, name);
            } else if (StringUtils.isNotBlank(name)) {
                sql = "select * from t_fc_node where node_type=? and node_name=?";
                return simpleJdbcTemplate.queryForObject(sql, new NodeRowMapper(), type, name);
            } else {
                sql = "select * from t_fc_node where node_type=?";
                return simpleJdbcTemplate.queryForObject(sql, new NodeRowMapper(), type);
            }
        } catch (EmptyResultDataAccessException e) {
            throw new NodeNotFoundException("type:[" + type + "] name:[" + name + "]");
        }
    }

    public Node getNode(Integer nodeId) throws NodeNotFoundException {
        final String sql = "select * from t_fc_node where id=?";
        try {
            return simpleJdbcTemplate.queryForObject(sql, new NodeRowMapper(), nodeId);
        } catch (EmptyResultDataAccessException e) {
            throw new NodeNotFoundException(nodeId);
        }
    }

    public Node getChildNode(Integer parentNodeId, String name) {
        final String sql = "select * from t_fc_node where parent_id=? and node_name=?";
        try {
            return simpleJdbcTemplate.queryForObject(sql, new NodeRowMapper(), parentNodeId, name);
        } catch (EmptyResultDataAccessException e) {        	
            throw new NodeNotFoundException(parentNodeId, name);
        } catch (IncorrectResultSizeDataAccessException e) {
            logger.error("node [{}] under [{}] duplicate", new Object[]{name, parentNodeId}, e);
            throw new NestedRuntimeException("node [" + name + "] under [" + parentNodeId + "] duplicate", e);
        }
    }

    public List<Node> search(String path, String keyword) {
        final String sql = "select * from t_fc_node where path like ? and (node_name like ? or description like ?) order by node_type asc,node_name asc";
        return simpleJdbcTemplate.query(sql, new NodeRowMapper(), path + "%", "%" + keyword + "%", "%" + keyword + "%");
    }

    public List<Node> getChildNodes(Integer nodeId) {
        final String sql = "select * from t_fc_node where parent_id=? order by node_type asc,node_name asc";
        final String countSql = "select t1.id,count(t2.id) as node_count from t_fc_node t1 left join t_fc_node t2 on t2.parent_id=t1.id where t1.parent_id=? group by t1.id";
        List<Node> nodes = simpleJdbcTemplate.query(sql, new NodeRowMapper(), nodeId);
        if (nodes.size() > 0) {
            final Map<Integer, Integer> map = new HashMap<Integer, Integer>(nodes.size());
            simpleJdbcTemplate.getJdbcOperations().query(countSql, new Object[]{nodeId}, new RowCallbackHandler() {
                public void processRow(ResultSet rs) throws SQLException {
                    map.put(rs.getInt("id"), rs.getInt("node_count"));
                }
            });
            for (Node node : nodes) {
                ((NodeImpl) node).setChildCount(map.get(node.getId()));
            }
        }
        return nodes;
    }

    public List<Node> getAllChildNodes(String path) {
        final String sql = "select * from t_fc_node where path like ? order by node_type asc,node_name asc";
        return simpleJdbcTemplate.query(sql, new NodeRowMapper(), path + "%");
    }

    public List<File> getAllChildFiles(String path, int start, int size) {
        final String sql = "select * from t_fc_node where node_type=1 and path like ? order by node_type asc,node_name asc";
        List<Node> nodes = simpleJdbcTemplate.query(paginationSQLGetter.getPaginationSQL(sql, start, size), new NodeRowMapper(), path + "%");
        List<File> files = new ArrayList<File>(nodes.size());
        for (Node node : nodes) {
            files.add((File) node);
        }
        return files;
    }

    public int getAllChildFilesCount(String path) {
        final String sql = "select count(0) from t_fc_node where node_type=1 and path like ?";
        return simpleJdbcTemplate.queryForInt(sql, path + "%");
    }

    public long getUsedSize(String path) {
        final String sql = "select sum(node_size) from t_fc_node where path like ?";
        return simpleJdbcTemplate.queryForLong(sql, path + "%");
    }

    public boolean hasLinkFile(String path) {
        final String sql = "select count(0) from t_fc_node where store_url=?";
        return simpleJdbcTemplate.queryForInt(sql, path) > 1;
    }

    @SuppressWarnings("unchecked")
    private class NodeRowMapper implements ParameterizedRowMapper<Node> {

        public NodeImpl mapRow(ResultSet rs, int rownum) throws SQLException {
            int type = rs.getInt("node_type");
            String name = rs.getString("node_name");
            NodeImpl node;
            switch (type) {
                case Node.TYPE_PERSONAL_SPACE:
                    node = new PersonalSpace();
                    ((PersonalSpace) node).setSize(rs.getInt("node_size"));
                    ((PersonalSpace) node).setUsed(getUsedSize(node.getFullPath()));
                    break;
                case Node.TYPE_WORK_SPACE:
                    node = new WorkSpace();
                    ((WorkSpace) node).setSize(rs.getInt("node_size"));
                    ((WorkSpace) node).setUsed(getUsedSize(node.getFullPath()));
                    break;
                case Node.TYPE_FILE:
                    node = new File();
                    ((File) node).setSize(rs.getInt("node_size"));
                    ((File) node).setStoreUrl(rs.getString("store_url"));
                    break;
                default:
                    node = new NodeImpl();
            }
            node.setId(rs.getInt("id"));
            node.setParentId(rs.getInt("parent_id"));
            node.setName(name);
            node.setType(type);
            node.setDescription(rs.getString("description"));
            node.setOwner(rs.getString("owner"));
            node.setUpdateTime(rs.getTimestamp("update_time"));
            node.setViewName(rs.getString("view_name"));
            node.setPath(rs.getString("path"));
            String attrs = rs.getString("attributes");
            if (StringUtils.isNotEmpty(attrs)) {
                try {
                    node.setAttributes(JSON.parseObject(attrs, Map.class));
                } catch (Exception e) {
                    logger.error("parse attributes [{}] to map error", attrs, e);
                }
            }
            completeNode(node);
            return node;
        }
    }

    private NodeImpl completeNode(NodeImpl node){
        if (node instanceof Space) {
            node.setIcon("space.gif");
        } else if (node instanceof File) {
            node.setIcon(mimeTypeService.getIcon(node.getName()));
        } else {
            node.setIcon("folder.gif");
        }
        return node;
    }
}
