/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: NodeDao.java
 *
 * LastModified: 2009-09-22 01:18:38
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.dao;

import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.model.impl.File;

import java.util.List;

/**
 * 节点持久层DAO
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-21
 */
public interface NodeDao {

    /**
     * 初始化根节点
     */
    void initRoot();

    /**
     * 保存节点
     *
     * @param node 节点
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node save(Node node) throws NodeNotFoundException;

    /**
     * 节点是否存在
     *
     * @param nodeId 节点id
     * @return 是否存在
     */
    boolean exists(Integer nodeId);

    /**
     * 删除节点
     *
     * @param nodeId 节点id
     */
    void remove(Integer nodeId);

    /**
     * 获取父节点
     *
     * @param nodeId 节点id
     * @return 父节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getParentNode(Integer nodeId) throws NodeNotFoundException;

    /**
     * 根据名称和类型获取节点
     *
     * @param name 名称
     * @param type 类型
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNode(String name, int type) throws NodeNotFoundException;

    /**
     * 获取节点
     *
     * @param nodeId 节点Id
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNode(Integer nodeId) throws NodeNotFoundException;

    /**
     * 根据父亲节点id和子节点名获取子节点
     *
     * @param parentNodeId 父节点id
     * @param name         子节点名
     * @return 子节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getChildNode(Integer parentNodeId, String name) throws NodeNotFoundException;

    /**
     * 检索节点
     *
     * @param path    路径
     * @param keyword 关键字
     * @return 节点列表
     */
    List<Node> search(String path, String keyword);

    /**
     * 获取子节点
     *
     * @param nodeId 节点Id
     * @return 子节点列表
     */
    List<Node> getChildNodes(Integer nodeId);

    /**
     * 获取所有子节点
     *
     * @param path 路径
     * @return 子节点列表
     */
    List<Node> getAllChildNodes(String path);

    /**
     * 分页获取所有子节点
     *
     * @param path  路径
     * @param start 开始
     * @param size  个数
     * @return
     */
    List<File> getAllChildFiles(String path, int start, int size);

    /**
     * 获取子节点文件数量
     *
     * @param path 路径
     * @return 子节点文件数量
     */
    int getAllChildFilesCount(String fullPath);

    /**
     * 获取节点的空间占用
     *
     * @param path 路径
     * @return 空间占用
     */
    long getUsedSize(String path);

    boolean hasLinkFile(String path);
}
