/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: NodeController.java
 *
 * LastModified: 2009-09-25 02:33:40
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.web;

import com.gtis.config.AppConfig;
import com.gtis.fileCenter.ComparatorNodes;
import com.gtis.fileCenter.Constants;
import com.gtis.fileCenter.NodeHelper;
import com.gtis.fileCenter.ex.NestedRuntimeException;
import com.gtis.fileCenter.ex.NoPermissionException;
import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.MicroFile;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.model.Space;
import com.gtis.fileCenter.model.impl.File;
import com.gtis.fileCenter.model.impl.NodeImpl;
import com.gtis.fileCenter.service.FileStoreService;
import com.gtis.generic.util.ImageUtils;
import com.gtis.generic.util.JsonUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.ServletRequestUtils;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-23
 */
public class NodeController extends BaseController {
    /** 文件中心的表空间 */
    public static final String WORK_FLOW_STUFF = "WORK_FLOW_STUFF";
    @Autowired
    private FileStoreService fileService;

    @RequestMapping
    public void list(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        String query = request.getParameter("query");
        String path = request.getParameter("path");
        String proid = request.getParameter("proid");

        Node node = new NodeImpl();
        if (nodeId == null && StringUtils.isNotBlank(proid)){
            Space space = nodeService.getWorkSpace(WORK_FLOW_STUFF,false);
            if (space != null){
                node = nodeService.getNode(space.getId(), proid, false);
            }
        }else {
            node = StringUtils.isNotBlank(path) ? nodeService.getNode(nodeId, path) : nodeService.getNode(nodeId);
        }
        if (node == null){
            node = new NodeImpl();
        }
        nodeId = node.getId();
        List<Node> nodes = StringUtils.isNotBlank(query) ? nodeService.search(nodeId, query) : nodeService.getChildNodes(nodeId);
        //            Fc.FcList文件列表按上传时间排序，不配置默认为按附件名称排序
        String viewFileListField= AppConfig.getProperty("viewFileList.field");
        if(StringUtils.isNotBlank(viewFileListField) && StringUtils.equals(viewFileListField,"update_time")){
            ComparatorNodes comparator=new ComparatorNodes();
            Collections.sort(nodes, comparator);
        }
        List<Map<String, Object>> list = NodeHelper.nodesToJson(nodes);
        Map<String, Object> map = new HashMap<String, Object>(4);
        map.put("items", list);
        map.put("total", list.size());

        Map<String, Object> currentNode = NodeHelper.nodeToJson(node);
        currentNode.put("path", getPathString(nodeService.getPath(nodeId)));
        map.put("node", currentNode);

        JsonUtils.out(response, map);
    }

    @RequestMapping
    public void all(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        String query = request.getParameter("query");
        String path = request.getParameter("path");
        String proid = request.getParameter("proid");

        Node node = new NodeImpl();
        if (nodeId == null && StringUtils.isNotBlank(proid)){
            Space space = nodeService.getWorkSpace(WORK_FLOW_STUFF,false);
            if (space != null){
                node = nodeService.getNode(space.getId(), proid, false);
            }
        }else {
            node = StringUtils.isNotBlank(path) ? nodeService.getNode(nodeId, path) : nodeService.getNode(nodeId);
        }

        if (node == null){
            node = new NodeImpl();
        }
        nodeId = node.getId();
        List<Node> nodes = nodeService.getAllChildNodes(nodeId);
        //            Fc.FcList文件列表按上传时间排序，不配置默认为按附件名称排序
        String viewFileListField= AppConfig.getProperty("viewFileList.field");
        if(StringUtils.isNotBlank(viewFileListField) && StringUtils.equals(viewFileListField,"update_time")){
            ComparatorNodes comparator=new ComparatorNodes();
            Collections.sort(nodes, comparator);
        }
        List<Map<String, Object>> list = NodeHelper.nodesToJson(nodes);
        Map<String, Object> map = new HashMap<String, Object>(4);
        map.put("items", list);
        map.put("total", list.size());

        Map<String, Object> currentNode = NodeHelper.nodeToJson(node);
        currentNode.put("path", getPathString(nodeService.getPath(nodeId)));
        map.put("node", currentNode);

        JsonUtils.out(response, map);
    }

    @RequestMapping
    public void tree(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        String proid = request.getParameter("proid");

        Node node = new NodeImpl();
        if (nodeId == null && StringUtils.isNotBlank(proid)){
            Space space = nodeService.getWorkSpace(WORK_FLOW_STUFF,false);
            if (space != null){
                node = nodeService.getNode(space.getId(), proid, false);
            }
        }else {
            node = nodeService.getNode(nodeId);
        }
        if (node == null){
            node = new NodeImpl();
        }
        nodeId = node.getId();
        List<Node> nodes = nodeService.getChildNodes(nodeId);
        //            Fc.FcList文件列表按上传时间排序，不配置默认为按附件名称排序
        String viewFileListField= AppConfig.getProperty("viewFileList.field");
        if(StringUtils.isNotBlank(viewFileListField) && StringUtils.equals(viewFileListField,"update_time")){
            ComparatorNodes comparator=new ComparatorNodes();
            Collections.sort(nodes, comparator);
        }
        List<Map<String, Serializable>> list = new ArrayList<Map<String, Serializable>>(nodes.size());
        for (Node childNode : nodes) {
            if (childNode instanceof File)
                continue;
            Map<String, Serializable> map = new HashMap<String, Serializable>();
            map.put("id", childNode.getId());
            map.put("iconCls", childNode instanceof Space ? "fc-space" : "fc-folder");
            map.put("text", childNode.getViewName() == null ? childNode.getName() : childNode.getViewName());
            map.put("name", childNode.getName());
            map.put("title", childNode.getDescription());
            map.put("leaf", Boolean.FALSE);
            list.add(map);
        }
        JsonUtils.out(response, list);
    }

    @RequestMapping
    public void preview(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        List<Node> nodes = nodeService.getChildNodes(nodeService.getNode(nodeId).getParentId());
        Document doc = DocumentHelper.createDocument();
        Element files = doc.addElement("files");
        for (Node node : nodes) {
            if (node instanceof File) {
                StringBuilder url = new StringBuilder(fileCenterUrl + "/file/get.do");
                String token = request.getParameter(Constants.TOKEN);
                if (StringUtils.isNotEmpty(token)) {
                    url.append("?token=").append(token).append("&");
                } else {
                    url.append(";jsessionid=").append(request.getSession().getId()).append("?");
                }
                url.append("fid=").append(node.getId()).append("&inline=true");
                Element file = files.addElement("file")
                        .addAttribute("name", node.getViewName() == null ? node.getName() : node.getViewName())
                        .addAttribute("url", url.toString());
                if (node.getId().equals(nodeId)) {
                    file.addAttribute("iscurrent", "true");
                }
            }
        }
        response.setContentType("text/xml;charset=UTF-8");
        response.getWriter().write(doc.asXML());
        response.getWriter().flush();
    }

    @RequestMapping
    public void rotate(HttpServletRequest request, HttpServletResponse response) throws Exception {
        int degree = ServletRequestUtils.getIntParameter(request, "degree");
        Integer nodeId = getNodeId(request);
        File file = (File) nodeService.getNode(nodeId);
        String path = fileService.getFile(file).getAbsolutePath();
        String to = path + "-1." + file.getExtension();
        ImageUtils.rotateImage(path, to, degree);
        java.io.File src = new java.io.File(path);
        src.delete();
        fileService.getFile(new MicroFile(file)).delete();
        FileUtils.moveFile(new java.io.File(to), src);
        JsonUtils.out(response, "{success:true}");
    }

    @RequestMapping
    public void path(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        String path = request.getParameter("path");
        Map<String, Integer> map = new HashMap<String, Integer>(1);
        if (StringUtils.isNotBlank(path)) {
            try {
                Node node = nodeService.getNode(nodeId, path);
                map.put(Constants.NODE_ID, node.getId());
            } catch (NodeNotFoundException ignored) {
            }
        }
        JsonUtils.out(response, map);
    }

    @RequestMapping
    public void edit(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        checkWriteable(request, nodeId);
        Integer pNodeId = NodeHelper.getIntegerParameter(request, Constants.PARENT_NODE_ID);
        NodeImpl node;
        if (nodeId != null)
            node = (NodeImpl) nodeService.getNode(nodeId);
        else if (pNodeId != null) {
            node = new NodeImpl();
            node.setParentId(pNodeId);
        } else
            throw new NestedRuntimeException("nodeId not found");
        String name = request.getParameter("name");
        String viewName = request.getParameter("viewName");
        String description = request.getParameter("description");
        boolean change = false;
        if (StringUtils.isNotBlank(name)) {
            node.setName(name);
            change = true;
        }
        if (StringUtils.isNotBlank(viewName)) {
            node.setViewName(viewName);
            change = true;
        }
        if (StringUtils.isNotBlank(description)) {
            node.setDescription(description);
            change = true;
        }
        if (change)
            nodeService.save(node);
        Map<String, Object> map = new HashMap<String, Object>(2);
        map.put("success", Boolean.TRUE);
        map.put("id", node.getId());
        JsonUtils.out(response, map);
    }

    @RequestMapping
    public void del(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer[] nodeIds = getNodeIds(request);
        checkWriteable(request, nodeIds);
        nodeService.remove(nodeIds);
        JsonUtils.out(response, "{success:true}");
    }

    @RequestMapping
    public void copy(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer[] nodeIds = getNodeIds(request);
        Integer destNodeId = getDestNodeId(request);
        checkWriteable(request, nodeIds);
        checkWriteable(request, destNodeId);
        boolean isCover = StringUtils.isNotBlank(request.getParameter("cover"));
        nodeService.copy(nodeIds, destNodeId, isCover);
        JsonUtils.out(response, "{success:true}");
    }

    @RequestMapping
    public void move(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer[] nodeIds = getNodeIds(request);
        Integer destNodeId = getDestNodeId(request);
        checkWriteable(request, nodeIds);
        checkWriteable(request, destNodeId);
        boolean isCover = StringUtils.isNotBlank(request.getParameter("cover"));
        nodeService.move(nodeIds, destNodeId, isCover);
        JsonUtils.out(response, "{success:true}");
    }

    @RequestMapping
    public void zip(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer[] nodeIds = getNodeIds(request);
        Integer destNodeId = getDestNodeId(request);
        checkWriteable(request, nodeIds);
        checkWriteable(request, destNodeId);
        String name = request.getParameter("name");
        nodeService.zip(nodeIds, destNodeId, name);
        JsonUtils.out(response, "{success:true}");
    }

    @RequestMapping
    public void unzip(HttpServletRequest request, HttpServletResponse response) throws Exception {
        Integer nodeId = getNodeId(request);
        checkWriteable(request, nodeId);
        String name = request.getParameter("name");
        boolean isCover = StringUtils.isNotBlank(request.getParameter("cover"));
        boolean isCurrent = StringUtils.isNotBlank(request.getParameter("current"));
        nodeService.unzip(nodeId, isCurrent ? null : name, isCover);
        JsonUtils.out(response, "{success:true}");
    }

    @RequestMapping
    public void config(HttpServletRequest request, HttpServletResponse response) {
        Map<String, Object> map = new HashMap<String, Object>(2);
        map.put("enablePreview", enablePreview);
        map.put("enableCifs", enableCifs);
        map.put("enableFtp", enableFtp);
        map.put("enableViewClient", enableViewClient);
        map.put("fileCenterUrl", fileCenterUrl);
        map.put("maxPreviewDialog", maxPreviewDialog);
        map.put("pictureFillToDialogSize", pictureFillToDialogSize);
        map.put("jsessionid", request.getSession().getId());
        map.put("fileTypes",fileTypes);
        map.put("fileTypeDescription", fileTypeDescription);
        JsonUtils.out(response, map);
    }
    
    @RequestMapping
    public void cache(HttpServletRequest request, HttpServletResponse response) {
    	nodeService.clearAll();
    }

    private static Integer getDestNodeId(HttpServletRequest request) {
        return NodeHelper.getIntegerParameter(request, Constants.DEST_NODE_ID);
    }

    private static Integer getNodeId(HttpServletRequest request) {
        return NodeHelper.getIntegerParameter(request, Constants.NODE_ID);
    }

    private static Integer[] getNodeIds(HttpServletRequest request) {
        return ArrayUtils.toObject(ServletRequestUtils.getIntParameters(request, Constants.NODE_ID));
    }

    private static String getPathString(List<Node> nodes) {
        StringBuilder sb = new StringBuilder("/");
        for (int i = nodes.size() - 1; i > -1; i--) {
            Node node = nodes.get(i);
            sb.append(node.getName()).append("/");
        }
        return sb.toString();
    }

    private void checkWriteable(HttpServletRequest request, Integer... nodeIds) {
        String token = request.getParameter(Constants.TOKEN);
        if (token != null && !nodeService.isWriteable(token)) {
            throw new NoPermissionException(Arrays.toString(nodeIds));
        }
    }
}
