/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: NodeServiceImpl.java
 *
 * LastModified: 2009-09-23 11:22:10
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.service.impl;

import com.gtis.fileCenter.Constants;
import com.gtis.fileCenter.dao.NodeDao;
import com.gtis.fileCenter.ex.NodeExistsException;
import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.model.PreviewFile;
import com.gtis.fileCenter.model.Space;
import com.gtis.fileCenter.model.impl.*;
import com.gtis.fileCenter.model.impl.File;
import com.gtis.fileCenter.service.FileStoreService;
import com.gtis.fileCenter.service.NodeService;
import com.gtis.generic.cache.Cache;
import com.gtis.generic.cache.CacheUtils;
import com.gtis.generic.cache.EntityLoader;
import com.gtis.generic.security.Helper;
import net.lingala.zip4j.model.FileHeader;
import net.sf.ehcache.Ehcache;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.*;
import java.util.*;
import java.util.zip.CRC32;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-17
 */
public class NodeServiceImpl implements NodeService {

    private static final Logger logger = LoggerFactory.getLogger(NodeServiceImpl.class);

    private static final int MAX_CACHE_NODE = 100;

    @Autowired
    private FileStoreService fileService;

    @Autowired
    private NodeDao nodeDao;

    private Cache<Node> nodeCache;

    private Ehcache tokenCache;

    private String superToken = "whosyourdaddy";

    private boolean enablePreview;

    public void setNodeCache(Ehcache nodeCache) {
        this.nodeCache = new Cache<Node>(nodeCache);
    }

    public void setTokenCache(Ehcache tokenCache) {
        this.tokenCache = tokenCache;
    }

    public void setSuperToken(String superToken) {
        this.superToken = superToken;
    }

    public void setEnablePreview(boolean enablePreview) {
        this.enablePreview = enablePreview;
    }

    public String getToken(Node node) {
        return getToken(node, null);
    }

    public String getToken(Node node, Integer ttl) {
        return getToken(node, true, null);
    }

    public String getToken(Node node, boolean writeable) {
        return getToken(node, writeable, null);
    }

    public Integer getTokenRoot(String token) {
        return CacheUtils.load(tokenCache, token);
    }

    public String getToken(Node node, boolean writeable, Integer ttl) {
        String key = CacheUtils.getKey(node.getId(), BooleanUtils.toInteger(writeable), ttl);
        String token = CacheUtils.load(tokenCache, key);
        if (token != null && CacheUtils.getExpirationTime(tokenCache, key) - DateUtils.MILLIS_PER_MINUTE * 30 > System.currentTimeMillis()) {
            return token;
        }
        token = (writeable ? Node.PERM_WRITE : Node.PERM_READ) + RandomStringUtils.randomAlphanumeric(12);
        CacheUtils.put(tokenCache, token, node.getId(), ttl);
        CacheUtils.put(tokenCache, CacheUtils.getKey(node.getId(), BooleanUtils.toInteger(writeable), ttl), token, ttl);
        logger.debug("create access token ({}) for node ({})", token, node);
        return token;
    }

    public boolean isWriteable(String token) {
        return token.charAt(0) - 48 == Node.PERM_WRITE || superToken.equals(token);
    }

    public boolean hasPermission(String token, Integer nodeId) {
        if (superToken.equals(token))
            return true;
        Integer tokenNodeId = CacheUtils.load(tokenCache, token);
        if (tokenNodeId != null) {
            try {
                return (tokenNodeId.equals(nodeId) || isChildNode(tokenNodeId, nodeId));
            } catch (NodeNotFoundException ignored) {
            }
        }
        return false;
    }

    public Node getRootNode() throws NodeNotFoundException {
        return getNodeByType(Node.TYPE_ROOT);
    }

    public Space getPersonalSpace(String userId) throws NodeNotFoundException {
        return (Space) getNodeByType(userId, Node.TYPE_PERSONAL_SPACE);
    }

    public Space getWorkSpace(final String name) throws NodeNotFoundException {
        return (Space) getNodeByType(name, Node.TYPE_WORK_SPACE);
    }

    public Space getWorkSpace(String name, boolean autoCreate) throws NodeNotFoundException {
        Space space;
        try {
            space = getWorkSpace(name);
        } catch (NodeNotFoundException e) {
            if (autoCreate) {
                try {
                    space = createWorkSpace(name);
                } catch (NodeExistsException e1) {
                    return getWorkSpace(name);
                }
            } else
                throw e;
        }
        return space;
    }

    public Space getSpace(Integer nodeId) throws NodeNotFoundException {
        Node node = getNode(nodeId);
        if (node instanceof Space)
            return (Space) node;
        for (String id : Constants.PATH_SEPARATOR.split(node.getPath()))
            if (StringUtils.isNotEmpty(id)) {
                node = getNode0(Integer.valueOf(id));
                if (node instanceof Space)
                    return (Space) node;
            }
        return null;
    }

    public boolean hasCapacity(Space space, long size) {
        return space.getSize() == -1 || space.getUsedSize() + size < space.getSize();
    }

    public Node getNodeByType(int type) throws NodeNotFoundException {
        return getNodeByType(null, type);
    }

    public Node getNodeByType(final String name, final int type) throws NodeNotFoundException {
        return clone(nodeCache.load(new EntityLoader<Node>() {
            public Node loadOnMiss() {
                Node node = nodeDao.getNode(name, type);
                return nodeCache.put(node, node.getId());
            }
        }, name, type));
    }

    protected NodeImpl getNode0(final Integer nodeId) throws NodeNotFoundException {
        Node node = nodeCache.load(nodeId);
        if (node == null) {
            node = nodeDao.getNode(nodeId);
            nodeCache.put(node, nodeId);
        }
        return (NodeImpl) node;
    }

    protected static Node clone(Node node) {
        try {
            return node.clone();
        } catch (CloneNotSupportedException e) {
            return null;
        }
    }

    public Node getNode(Integer nodeId) throws NodeNotFoundException {
        return clone(getNode0(nodeId));
    }

    public Node getNode(Integer parentNodeId, String path) throws NodeNotFoundException {
        return getNode(parentNodeId, path, false);
    }

    public Node getNode(Integer parentNodeId, String path, boolean autoCreate) {
        return getNode(parentNodeId,path,autoCreate,null);
    }

    public Node getNode(Integer parentNodeId, String path, boolean autoCreate, String userId) {
        Node node = getNode0(parentNodeId);
        if (StringUtils.isNotBlank(path))
            for (String name : Constants.PATH_SEPARATOR.split(path)) {
                if (node instanceof File) {
                    throw new NodeNotFoundException(parentNodeId, path);
                }
                if (StringUtils.isBlank(name))
                    continue;
                int nodeId = node.getId();
                try {
                    node = getChildNode(nodeId, name);
                } catch (NodeNotFoundException e) {
                    if (autoCreate) {
                        try {
                            node = createNode(nodeId, name, userId);
                        } catch (NodeExistsException e1) {
                            return getChildNode(nodeId, name);
                        }
                    } else
                        throw e;
                }
            }
        return node;
    }

    public List<Node> getNodes(Integer[] nodeIds) {
        List<Node> nodes = new ArrayList<Node>(nodeIds.length);
        for (Integer nodeId : nodeIds)
            nodes.add(getNode(nodeId));
        return nodes;
    }

    public Node getParentNode(Integer nodeId) throws NodeNotFoundException {
        return getNode(getNode0(nodeId).getParentId());
    }

    public Node getChildNode(final Integer nodeId, final String name) throws NodeNotFoundException {
        NodeImpl node = getNode0(nodeId);
        if (node.isChildrenUnSet()) {
            return clone(nodeCache.load(new EntityLoader<Node>() {
                public Node loadOnMiss() {
                    Node node = nodeDao.getChildNode(nodeId, name);
                    return nodeCache.put(node, node.getId());
                }
            }, nodeId, name));
        } else {
            Node n = node.getChild(name);
            if (n != null)
                return clone(n);
        }
        throw new NodeNotFoundException(nodeId, name);
    }

    public List<Node> getChildNodes(Integer nodeId) {
        NodeImpl node = getNode0(nodeId);
        List<Node> children = node.getChildren();
        if (children == null) {
            children = nodeDao.getChildNodes(nodeId);
            if (children.size() < MAX_CACHE_NODE) {
                node.setChildren(children);
                for (Node n : children) {
                    nodeCache.put(n, n.getId());
                }
            }
        }
        return children;
    }

    public List<Node> getAllChildNodes(Integer nodeId) {
        NodeImpl node = getNode0(nodeId);
        return nodeDao.getAllChildNodes(node.getFullPath());
    }

    public List<File> getAllChildFile(Integer nodeId, int start, int size) {
        NodeImpl node = getNode0(nodeId);
        return nodeDao.getAllChildFiles(node.getFullPath(), start, size);
    }

    public int getAllChildFilesCount(Integer nodeId) {
        NodeImpl node = getNode0(nodeId);
        return nodeDao.getAllChildFilesCount(node.getFullPath());
    }

    public List<Node> search(Integer nodeId, String keyword) {
        NodeImpl node = getNode0(nodeId);
        return nodeDao.search(node.getFullPath(), keyword);
    }

    public List<Node> getPath(Integer nodeId) {
        List<Node> path = new ArrayList<Node>();
        for (NodeImpl n = getNode0(nodeId); n.getParentId() != null; n = getNode0(n.getParentId())) {
            if (n.getType() == Node.TYPE_WORK_SPACE || n.getType() == Node.TYPE_PERSONAL_SPACE)
                break;
            path.add(n);
        }
        return path;
    }

    public boolean exists(Integer nodeId) {
        return nodeCache.exist(nodeId) || nodeDao.exists(nodeId);
    }

    public boolean isNodeNameUsed(Integer parentNodeId, String name) {
        try {
            return getChildNode(parentNodeId, name) != null;
        } catch (NodeNotFoundException e) {
            return false;
        }
    }

    public boolean isChildNode(Integer parentNodeId, Integer nodeId) {
        try {
            Node pNode = getNode0(parentNodeId);
            Node node = getNode0(nodeId);
            if (node.getPath().startsWith(pNode.getPath()) && !node.getParentId().equals(pNode.getParentId()))
                return true;
        } catch (NodeNotFoundException ignored) {
        }
        return false;
    }

    public Node save(Node node) throws NodeNotFoundException, NodeExistsException {
        return save(node, false);
    }

    public Node save(Node node1, boolean autoRename) throws NodeNotFoundException, NodeExistsException {
        NodeImpl node = (NodeImpl) node1;
        if (StringUtils.isBlank(node.getName()))
            throw new IllegalArgumentException("node name can not be null");

        if (node.getType() < Node.TYPE_NODE)
            throw new IllegalArgumentException("type must greater than 0");

        if (node instanceof PersonalSpace) {
            node.setParentId(getNodeByType(Node.TYPE_PERSONAL_ROOT).getId());
        } else if (node instanceof WorkSpace) {
            node.setParentId(getNodeByType(Node.TYPE_WORK_ROOT).getId());
        }
        Integer parentId = node.getParentId();

        boolean isCreate = node.getId() == null;
        boolean needCheckName = isCreate;
        String srcName = null;
        long srcSize = 0;
        synchronized (this) {
            if (!isCreate) {
                Node srcNode = getNode0(node.getId());
                srcName = srcNode.getName();
                needCheckName = !srcNode.getName().equals(node.getName());
                if (node instanceof File && srcNode instanceof File) {
                    File file = (File) node;
                    srcSize = file.getSize();
                }
            }
            if (needCheckName && isNodeNameUsed(parentId, node.getName())) {
                if (autoRename) {
                    node.setName(rename(parentId, node.getName()));
                } else {
                    throw new NodeExistsException(parentId, node.getName());
                }
            }

            node.setPath(getNode0(parentId).getFullPath());
            if (node.getOwner() == null)
                node.setOwner(Helper.getCurrentUserId());
            node.setUpdateTime(new Date());
            nodeDao.save(node);
            nodeCache.put(node, node.getId());
            NodeImpl pNode = getNode0(parentId);
            if (srcName != null) {
                pNode.removeChild(srcName);
                if (needCheckName) {
                    nodeCache.clear(node.getParentId(), srcName);
                }
            }
            pNode.addChild(node);
        }
        if (node instanceof File) {
            try {
                File file = (File) node;
                ((SpaceImpl) getSpace(node.getId())).setUsed(file.getSize() - srcSize);
                if (enablePreview && file.isDocument()) {
                    PreviewFile pf = new PreviewFile(file);
                    if (fileService.getFile(file).exists() && !fileService.getFile(pf).exists()) {
                        fileService.createPreviewFile(pf);
                    }
                }
            } catch (Exception e) {
                logger.error("process file error:", e);
            }
        }
        return clone(node);
    }

    public Node createNode(Integer parentId, String name, String viewName, String description, Integer scope, String owner, Map<String, String> attributes) {
        NodeImpl node = new NodeImpl();
        node.setParentId(parentId);
        node.setName(name);
        node.setViewName(viewName);
        node.setDescription(description);
        if (scope != null)
            node.setScope(scope);
        node.setOwner(owner);
        node.setAttributes(attributes);
        return save(node);
    }

    public Node createNode(Integer parentId, String name) {
        return createNode(parentId, name, null, null, null, null, null);
    }

    public Node createNode(Integer parentId, String name, String userId) {
        return createNode(parentId, name, null, null, null, userId, null);
    }

    public Space createWorkSpace(String name, String viewName, Long size, String description, Map<String, String> attributes) {
        WorkSpace node = new WorkSpace();
        node.setName(name);
        node.setViewName(viewName);
        node.setDescription(description);
        node.setAttributes(attributes);
        node.setSize(size);
        return (Space) save(node);
    }

    public Space createWorkSpace(String name) {
        return createWorkSpace(name, null, -1l, null, null);
    }

    private String rename(Integer nodeId, String fileName) {
        int index = fileName.lastIndexOf('.');
        String name = index > -1 ? fileName.substring(0, index) : fileName;
        String ext = index > -1 ? fileName.substring(index) : "";
        String name1 = name + "_1" + ext;
        for (int i = 1; isNodeNameUsed(nodeId, name1); i++) {
            name1 = name + "_" + i + ext;
        }
        return name1;
    }

    public void remove(Integer nodeId) {
        Node node = getNode0(nodeId);
        deleteSingleNode(node);
        if (!(node instanceof File)) {
            for (Node n : getAllChildNodes(nodeId))  //删除所有子节点
                deleteSingleNode(n);
        }
        getNode0(node.getParentId()).removeChild(node.getName());
    }

    protected void deleteSingleNode(Node node) {
        try {
            if (node instanceof File) {
                File file = (File) node;
                fileService.delete(file);
                ((SpaceImpl) getSpace(file.getId())).setUsed(-file.getSize());
            }
            nodeCache.clear(node.getId());
            nodeCache.clear(node.getParentId(), node.getName());
            nodeDao.remove(node.getId());
        } catch (Exception ignored) {
        }
    }

    public void remove(Integer[] nodeIds) {
        for (Integer nodeId : nodeIds)
            remove(nodeId);
    }

    public void copy(Integer[] nodeIds, Integer destNodeId, boolean cover) {
        for (Integer nodeId : nodeIds) {
            try {
                copyNode(getNode0(nodeId), getNode0(destNodeId), cover);
            } catch (NodeNotFoundException ignored) {
            }
        }
    }

    private void copyNode(Node node, final Node destNode, boolean cover) {
        if (node.getId().equals(destNode.getId()) || node.getParentId().equals(destNode.getId()) || isChildNode(node.getId(), destNode.getId()))
            return;
        boolean isFile = node instanceof File;
        NodeImpl sameNameNode = null;
        NodeImpl newNode = (NodeImpl) clone(node);
        //取在目标文件夹里有相同的文件
        try {
            sameNameNode = (NodeImpl) getChildNode(destNode.getId(), node.getName());
        } catch (NodeNotFoundException ignored) {
        }
        //覆盖文件处理
        if (sameNameNode != null && cover) {
            if (isFile) {
                if (!(sameNameNode instanceof File)) { //目标是文件夹,则不允许文件覆盖
                    return;
                }
                remove(sameNameNode.getId());
            } else {
                if (sameNameNode instanceof File) {//如果是文件,则不允许文件夹覆盖
                    return;
                }
                sameNameNode.setViewName(node.getViewName());
                sameNameNode.setOwner(node.getOwner());
                sameNameNode.setScope(node.getScope());
                sameNameNode.setDescription(node.getDescription());
                sameNameNode.setAttributes(node.getAttributes());
                newNode = sameNameNode;
            }
        }
        if (isFile) {
            File file = (File) newNode;
            file.setStoreUrl(null);
            fileService.copyTo((File) node, (File) newNode);
        } else {
            newNode.setChildren(Collections.<Node>emptyList());
        }
        newNode.setId(null);
        newNode.setParentId(destNode.getId());

        if(!cover)
            //不覆盖时，如果目标文件夹有相同文件，重命名文件保存。如果没有目标文件夹没有相同文件，直接保存
            save(newNode,true);
        else
            //直接保存
            save(newNode);
        if (!isFile) {
            for (Node n : getChildNodes(node.getId())) {
                copyNode(n, sameNameNode != null && !(sameNameNode instanceof File) ? sameNameNode : newNode, cover);
            }
        }
    }

    public void move(Integer[] nodeIds, Integer destNodeId, boolean cover) {
        for (Integer nodeId : nodeIds) {
            try {
                moveNode(getNode0(nodeId), getNode0(destNodeId), cover);
            } catch (NodeNotFoundException ignored) {
            }
        }
    }

    private void moveNode(Node node, final Node destNode, boolean cover) {
        if (node.getId().equals(destNode.getId()) || node.getParentId().equals(destNode.getId()) || isChildNode(node.getId(), destNode.getId()))
            return;
        boolean isFile = node instanceof File;
        NodeImpl sameNameNode = null;
        try {
            sameNameNode = (NodeImpl) getChildNode(destNode.getId(), node.getName());
        } catch (NodeNotFoundException ignored) {
        }
        if (sameNameNode != null) {
            if (isFile) {
                if (!(sameNameNode instanceof File) || !cover) { //目标是文件夹,则不允许文件覆盖
                    return;
                }
                remove(sameNameNode.getId());
            } else {
                if (sameNameNode instanceof File || !cover) {//如果是文件,则不允许文件夹覆盖
                    return;
                }
                sameNameNode.setViewName(node.getViewName());
                sameNameNode.setOwner(node.getOwner());
                sameNameNode.setScope(node.getScope());
                sameNameNode.setDescription(node.getDescription());
                sameNameNode.setAttributes(node.getAttributes());
                node = sameNameNode;
            }

        }
        NodeImpl pNode = getNode0(node.getParentId());
        ((NodeImpl) node).setParentId(destNode.getId());
        save(node);
        pNode.removeChild(node.getName());
        if (!isFile) {
            for (Node n : getChildNodes(node.getId())) {
                moveNode(n, sameNameNode != null && !(sameNameNode instanceof File) ? sameNameNode : node, cover);
            }
        }
    }

    public void zip(Integer[] nodeIds, Integer destNodeId, String name) throws NodeExistsException {
        if (isNodeNameUsed(destNodeId, name)) {
            throw new NodeExistsException(destNodeId, name);
        }
        File file = new File();
        file.setParentId(destNodeId);
        file.setName(name);
        ZipOutputStream zos = null;
        try {
            zos = new ZipOutputStream(fileService.getFileOutputStream(file));
            zos.setEncoding("GBK");  //这行代码就是解决中文文件名的关键
            for (Integer nodeId : nodeIds) {
                Node topNode = getNode0(nodeId);
                if (topNode instanceof File) {
                    addNodeToZip((File) topNode, topNode.getName(), zos);
                    continue;
                }
                List<Node> nodes = getAllChildNodes(nodeId);
                for (Node node : nodes) {
                    if (node instanceof File)
                        addNodeToZip((File) node, getAbsFileName(node.getId(), nodeId), zos);
                }
            }
        } catch (Exception e) {
            logger.error("create zip file :[{}] error", file);
        } finally {
            try {
                if (zos != null)
                    zos.close();
            } catch (IOException ignored) {
            }
        }
        save(file);
    }

    private void addNodeToZip(File file, String zipEntryName, ZipOutputStream zos) {
        InputStream is = null;
        try {
            zos.setEncoding("GBK");  //这行代码就是解决中文文件名的关键
            ZipEntry ze = new ZipEntry(zipEntryName);
            // 创建一个ZipEntry，并设置Name和其它的一些属性
            ze.setSize(file.getSize());
            ze.setTime(file.getUpdateTime().getTime());
            // 设置CRC校验
            ze.setCrc(0);
            CRC32 crc = new CRC32();
            crc.reset();
            // 将ZipEntry加到zos中，再写入实际的文件内容
            zos.putNextEntry(ze);
            is = fileService.getFileInputStream(file);
            byte[] buf = new byte[5120];
            int readLen;
            while ((readLen = is.read(buf, 0, 5120)) != -1) {
                zos.write(buf, 0, readLen);
                crc.update(buf, 0, readLen);
            }
            ze.setCrc(crc.getValue());
        } catch (IOException e) {
            logger.error("zip error", e);
        } finally {
            if (is != null)
                try {
                    is.close();
                } catch (IOException ignored) {
                }
        }
    }

    private String getAbsFileName(Integer nodeId, Integer topNodeId) {
        List<String> paths = new ArrayList<String>();
        Node topNode = getNode0(topNodeId);
        if (!nodeId.equals(topNodeId))
            for (Node node = getNode0(nodeId); !node.getId().equals(topNodeId); node = getNode0(node.getParentId()))
                paths.add(node.getName());
        StringBuilder sb = new StringBuilder(topNode.getName());
        for (int i = paths.size() - 1; i > -1; i--)
            sb.append("/").append(paths.get(i));
        return sb.toString();
    }

    public void unzip(Integer nodeId, String name, boolean cover) {
        File file = (File) getNode0(nodeId);
        Node topNode;
        if (name != null) {
            try {
                Node sameNameNode = getChildNode(file.getParentId(), name);
                if (cover && sameNameNode instanceof File) {
                    remove(sameNameNode.getId());
                    throw new NodeNotFoundException(sameNameNode.getId());
                } else {
                    topNode = sameNameNode;
                }
            } catch (NodeNotFoundException ignored) {
                topNode = new NodeImpl();
                ((NodeImpl) topNode).setParentId(file.getParentId());
                topNode.setName(name);
                save(topNode);
            }
        } else
            topNode = getNode0(file.getParentId());
        try {
            java.io.File f = fileService.getFile(file);
            //采用zjp4j的方式进行解压，避免ant解压带来的乱码问题
            net.lingala.zip4j.core.ZipFile zipFile = new net.lingala.zip4j.core.ZipFile(f);
            zipFile.setFileNameCharset("GBK");
            List<FileHeader> headerList = zipFile.getFileHeaders();
            for(FileHeader fileHeader : headerList) {
                saveZipEntry(topNode,zipFile,fileHeader,cover);
            }

           /* ZipFile zfile = new ZipFile(f);
            Enumeration zList = zfile.getEntries();
            while (zList.hasMoreElements()) {
                ZipEntry ze = (ZipEntry) zList.nextElement();
                if (!ze.isDirectory()) {
                    saveZipEntry(topNode, zfile, ze, cover);
                }
            }
            zfile.close();*/
        } catch (Exception e) {
            logger.error("unzip error", e);
        }
    }

    private void saveZipEntry(final Node topNode,net.lingala.zip4j.core.ZipFile zipFile,FileHeader fileHeader, boolean cover){
        try {
            String path = fileHeader.getFileName();
            int index = path.lastIndexOf("/");
            String fileName = path.substring(index + 1);
            Node parent = index == -1 ? topNode : getNode(topNode.getId(), path.substring(0, index), true);
            try {
                Node sameNameNode = getChildNode(parent.getId(), fileName);
                if (cover && sameNameNode instanceof File) {
                    remove(sameNameNode.getId());
                } else
                    return;
            } catch (NodeNotFoundException ignored) {
            }
            File file = new File();
            file.setParentId(parent.getId());
            file.setName(fileName);
            fileService.save(file, zipFile.getInputStream(fileHeader));
            save(file);
        } catch (Exception e) {
            logger.error("unzip error", e);
        }
    }

    private void saveZipEntry(final Node topNode, ZipFile zipFile, ZipEntry zipEntry, boolean cover) {
        try {
            String path = zipEntry.getName();
            int index = path.lastIndexOf("/");
            String fileName = path.substring(index + 1);
            Node parent = index == -1 ? topNode : getNode(topNode.getId(), path.substring(0, index), true);
            try {
                Node sameNameNode = getChildNode(parent.getId(), fileName);
                if (cover && sameNameNode instanceof File) {
                    remove(sameNameNode.getId());
                } else
                    return;
            } catch (NodeNotFoundException ignored) {
            }
            File file = new File();
            file.setParentId(parent.getId());
            file.setName(fileName);
            file.setSize(zipEntry.getSize());
            fileService.save(file, zipFile.getInputStream(zipEntry));
            save(file);
        } catch (Exception e) {
            logger.error("unzip error", e);
        }
    }

    public void init() {
        nodeDao.initRoot();
    }
    
    public void clearAll(){
    	nodeCache.clearAll();
    }
}
