/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: FileService.java
 *
 * LastModified: 2009-09-23 09:15:40
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.service;

import com.gtis.fileCenter.ex.FileIOException;
import com.gtis.fileCenter.ex.FileOverSizeException;
import com.gtis.fileCenter.file.FileStore;
import com.gtis.fileCenter.model.impl.File;
import com.gtis.fileCenter.model.MicroFile;
import com.gtis.fileCenter.model.PreviewFile;

import javax.servlet.ServletOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.FileChannel;

/**
 * 虚拟文件操作服务
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-16
 */
public interface FileStoreService {

    /**
     * 获取预览服务器地址
     *
     * @return 服务器地址
     */
    String getPreviewServerUrl();

    /**
     * 获取使用的文件存储
     *
     * @return 文件存储
     */
    FileStore getFileStore();

    /**
     * 物理文件是否存在
     *
     * @param file 文件
     * @return 是否存在
     */
    boolean exists(File file);

    /**
     * 获取物理文件
     *
     * @param file 文件
     * @return 物理文件
     */
    java.io.File getFile(File file);

    /**
     * 获取文件读取channel
     *
     * @param file 文件
     * @return 文件channel
     * @throws FileIOException 文件访问异常
     */
    FileChannel getFileChannel(File file) throws FileIOException;

    /**
     * 获取文件读取channel
     *
     * @param file 文件
     * @param skip 跳过的字节
     * @return 文件channel
     * @throws com.gtis.fileCenter.ex.FileIOException
     *          文件访问异常
     */
    FileChannel getFileChannel(File file, long skip) throws FileIOException;

    /**
     * 获取文件读取得输入流
     *
     * @param file 文件
     * @return 文件输入流
     * @throws FileIOException 文件访问异常
     */
    InputStream getFileInputStream(File file) throws FileIOException;

    /**
     * 获取文件读取得输入流
     *
     * @param file 文件
     * @param skip 跳过的字节
     * @return 文件输入流
     * @throws FileIOException 文件访问异常
     */
    InputStream getFileInputStream(File file, long skip) throws FileIOException;


    /**
     * 将文件转换为字符串
     *
     * @param file 文件
     * @return 文件字符串
     */
    String toString(File file);

    /**
     * 传输文件到输出流
     *
     * @param file 文件
     * @param os   输入流
     * @throws FileIOException 文件访问异常
     */
    void transferTo(File file, OutputStream os) throws FileIOException;

    /**
     * 传输文件到输出流
     *
     * @param file 文件
     * @param skip 跳过的字节
     * @param os   输入流
     * @throws FileIOException 文件访问异常
     */
    void transferTo(File file, long skip, OutputStream os) throws FileIOException;

    /**
     * 传输zip包内的指定文件
     *
     * @param file         文件
     * @param name         zip内部的文件名
     * @param outputStream 文件输出流
     */
    void transferZipContainedFileTo(File file, String name, ServletOutputStream outputStream);

    /**
     * 获取文件写入channel
     *
     * @param file 文件
     * @return 可写文件channel
     * @throws FileIOException 文件访问异常
     */
    FileChannel getWritableFileChannel(File file) throws FileIOException;

    /**
     * 获取文件写入channel
     *
     * @param file 文件
     * @param skip 跳过的字节
     * @return 可写文件channel
     * @throws FileIOException 文件访问异常
     */
    FileChannel getWritableFileChannel(File file, long skip) throws FileIOException;

    /**
     * 获取文件输出流
     *
     * @param file 文件
     * @return 文件输出流
     * @throws FileIOException 文件访问异常
     */
    OutputStream getFileOutputStream(File file) throws FileIOException;

    /**
     * 获取文件输出流
     *
     * @param file 文件
     * @param skip 跳过的字节
     * @return 文件输出流
     * @throws FileIOException 文件访问异常
     */
    OutputStream getFileOutputStream(File file, long skip) throws FileIOException;

    /**
     * 保存指定输入流到文件
     *
     * @param file 文件
     * @param is   输入流
     * @throws FileIOException 文件访问异常
     */
    void save(File file, InputStream is) throws FileIOException;

    /**
     * 保存指定文件channel到文件
     *
     * @param file 文件
     * @param fc   文件channel
     * @throws FileIOException 文件访问异常
     */
    void save(File file, FileChannel fc) throws FileIOException;

    /**
     * 保存指定磁盘文件到文件
     *
     * @param file     文件
     * @param diskFile 磁盘文件
     * @throws FileIOException 文件访问异常
     */
    void save(File file, java.io.File diskFile) throws FileIOException;

    /**
     * 保存指定字节数组到文件
     *
     * @param file  文件
     * @param bytes 字节数组
     * @throws FileIOException 文件访问异常
     */
    void save(File file, byte[] bytes) throws FileIOException;

    /**
     * 保存指定字符串到文件
     *
     * @param file 文件
     * @param s    字符串
     * @throws FileIOException 文件访问异常
     */
    void save(File file, String s) throws FileIOException;

    /**
     * 保存指定编码的字符串到文件
     *
     * @param file    文件
     * @param s       字符串
     * @param charset 编码
     * @throws FileIOException 文件访问异常
     */
    void save(File file, String s, String charset) throws FileIOException;

    /**
     * 复制文件到目标文件
     *
     * @param srcFile  src文件
     * @param destFile dest文件
     * @throws FileIOException 文件访问异常
     */
    void copyTo(File srcFile, File destFile) throws FileIOException;

    /**
     * 删除文件
     *
     * @param file 文件
     * @throws FileIOException 文件访问异常
     */
    void delete(File file) throws FileIOException;

    /**
     * 创建图片缩微图
     *
     * @param microFile 缩略图
     */
    void createMicroImage(MicroFile microFile);

    /**
     * 按照传递的尺寸生成图片缩略图
     * @param microFile 缩略图
     */
    void createMicroImage(MicroFile microFile,final int fileSize);

    /**
     * 创建web预览文件
     *
     * @param previewFile 预览文件
     * @throws FileOverSizeException 文件过大
     */
    void createPreviewFile(PreviewFile previewFile) throws FileOverSizeException;

    boolean hasLinkFile(File file);
}
