/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: CallbackFileChannel.java
 *
 * LastModified: 2009-09-17 03:29:08
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.file;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-17
 */
public class CallbackFileChannel extends FileChannel {

    /**
     * the channel to route all calls to
     */
    private FileChannel delegate;
    /**
     * listeners waiting for the stream close
     */
    private List<FileStreamListener> listeners;

    /**
     * @param delegate  the channel that will perform the work
     * @param listeners listeners for events coming from this channel
     */
    public CallbackFileChannel(
            FileChannel delegate,
            List<FileStreamListener> listeners) {
        if (delegate == null) {
            throw new IllegalArgumentException("FileChannel delegate is required");
        }
        if (delegate instanceof CallbackFileChannel) {
            throw new IllegalArgumentException("FileChannel delegate may not be a CallbackFileChannel");
        }

        this.delegate = delegate;
        this.listeners = listeners;
    }

    /**
     * Closes the channel and makes the callbacks to the listeners
     */
    protected void implCloseChannel() throws IOException {
        fireChannelClosed();
        delegate.close();
    }

    /**
     * Helper method to notify stream listeners
     *
     * @throws java.io.IOException IOException
     */
    private void fireChannelClosed() throws IOException {
        if (listeners.size() == 0) {
            // nothing to do
            return;
        }
        for (FileStreamListener listener : listeners) {
            listener.contentStreamClosed(size());
        }
    }

    public void force(boolean metaData) throws IOException {
        delegate.force(metaData);
    }

    public FileLock lock(long position, long size, boolean shared) throws IOException {
        return delegate.lock(position, size, shared);
    }

    public MappedByteBuffer map(MapMode mode, long position, long size) throws IOException {
        return delegate.map(mode, position, size);
    }

    public long position() throws IOException {
        return delegate.position();
    }

    public FileChannel position(long newPosition) throws IOException {
        return delegate.position(newPosition);
    }

    public int read(ByteBuffer dst) throws IOException {
        return delegate.read(dst);
    }

    public int read(ByteBuffer dst, long position) throws IOException {
        return delegate.read(dst, position);
    }

    public long read(ByteBuffer[] dsts, int offset, int length) throws IOException {
        return delegate.read(dsts, offset, length);
    }

    public long size() throws IOException {
        return delegate.size();
    }

    public long transferFrom(ReadableByteChannel src, long position, long count) throws IOException {
        return delegate.transferFrom(src, position, count);
    }

    public long transferTo(long position, long count, WritableByteChannel target) throws IOException {
        return delegate.transferTo(position, count, target);
    }

    public FileChannel truncate(long size) throws IOException {
        return delegate.truncate(size);
    }

    public FileLock tryLock(long position, long size, boolean shared) throws IOException {
        return delegate.tryLock(position, size, shared);
    }

    public int write(ByteBuffer src) throws IOException {
        return delegate.write(src);
    }

    public int write(ByteBuffer src, long position) throws IOException {
        return delegate.write(src, position);
    }

    public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
        return delegate.write(srcs, offset, length);
    }
}

