package com.gits.etl.service.impl;

import com.alibaba.fastjson.JSON;
import com.gits.etl.Constants;
import com.gits.etl.model.*;
import com.gits.etl.service.JobManager;
import com.gits.etl.service.JobReportManager;
import com.gits.etl.util.CfgUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.osgi.framework.*;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.support.CronTrigger;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 11-12-28
 */
public class JobManagerImpl implements JobManager, JobListener {
    protected final Log LOG = LogFactory.getLog(JobManagerImpl.class);
    @Autowired
    private JobReportManager jobReportManager;
    private Map<String, Job> jobs = new ConcurrentHashMap<String, Job>();
    private Map<String, JobFuture> scheduledjobs = new ConcurrentHashMap<String, JobFuture>();
    private Map<String, JobInstance> jobInstances = new ConcurrentHashMap<String, JobInstance>();
    private TaskScheduler taskScheduler;
    private BundleContext bundleContext;
    private File cfgRoot;

    public void setTaskScheduler(TaskScheduler taskScheduler) {
        this.taskScheduler = taskScheduler;
    }

    public void setBundleContext(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    public void setCfgRoot(Resource cfgRoot) throws IOException {
        this.cfgRoot = cfgRoot.getFile();
    }

    public Job getJob(String jobName) {
        try {
            return jobs.get(jobName).clone();
        } catch (CloneNotSupportedException ignored) {
        }
        return null;
    }

    public Job saveJob(Job job) {
        if (job.getPreferences().isEmpty()) {
            loadJobCfg(job);
        } else {
            saveJobCfg(job);
        }
        Job srcJob = jobs.get(job.getName());
        if (srcJob != null) {
            boolean crontabChange = job.getCrontab() != null && !job.getCrontab().equals(srcJob.getCrontab());
            if (srcJob.isEnable()) {
                if (crontabChange) {
                    changeJobSchedule(job, false);
                }
            }
            if (job.isEnable()) {
                if (crontabChange || !srcJob.isEnable()) {
                    changeJobSchedule(job, true);
                }
            }
        } else {
            if (job.isEnable()) {
                changeJobSchedule(job, true);
            }
        }
        jobs.put(job.getName(), job);
        return job;
    }

    private void saveJobCfg(Job job) {
        Properties props = new Properties();
        Properties defProps = loadDefaultCfg(job);
        for (Map.Entry<String, String> entry : job.getPreferences().entrySet()) {
            String defValue = defProps.getProperty(entry.getKey());
            String value = entry.getValue();
            if (StringUtils.isNotBlank(value) && !value.equals(defValue)) {
                props.put(entry.getKey(), value);
            } else {
                entry.setValue(defValue);
            }
        }
        if (job.getCrontab() != null)
            props.put(Constants.CRONTAB, job.getCrontab());
        props.put(Constants.ENABLE, String.valueOf(job.isEnable()));
        CfgUtils.save(props, getCfgFile(job));
    }

    private File getCfgFile(Job job) {
        return new File(cfgRoot, job.getName() + ".cfg");
    }

    private void loadJobCfg(Job job) {
        Properties props = loadDefaultCfg(job);
        File cfg = getCfgFile(job);
        if (cfg.exists()) {
            CfgUtils.load(props, getCfgFile(job));
        }
        Map<String, String> map = job.getPreferences();
        for (Map.Entry entry : props.entrySet()) {
            String name = (String) entry.getKey();
            String value = (String) entry.getValue();
            if (Constants.ENABLE.equals(name)) {
                job.setEnable(Boolean.valueOf(value));
            } else if (Constants.CRONTAB.equals(name)) {
                job.setCrontab(value);
            } else {
                map.put(name, value);
            }
        }
    }

    private Properties loadDefaultCfg(Job job) {
        Properties props = new Properties();
        try {
            JobInstance instance = new JobInstance(job.getJobClass().newInstance());
            return CfgUtils.load(props, job.getJobClass().getResourceAsStream("contexts/" + instance.getContextStr() + ".properties"));
        } catch (Exception e) {
            LOG.error("Load job " + job.getName() + " props error", e);
        }
        return props;
    }

    public void removeJob(String jobName) {
        for (Bundle bundle : bundleContext.getBundles()) {
            if (bundle.getSymbolicName().equals(jobName)) {
                try {
                    bundle.uninstall();
                } catch (BundleException e) {
                    LOG.error("Remove job " + jobName + " error", e);
                }
            }
        }
    }

    public boolean installJob(String name, InputStream inputStream) {
        try {
            Bundle bundle = bundleContext.installBundle(name, inputStream);
            bundle.start();
            Thread.sleep(200);
            return true;
        } catch (BundleException e) {
            LOG.error("install job " + name + " error", e);
        } catch (InterruptedException ignored) {
        }
        return false;
    }

    public List<Job> getJobs() {
        List<Job> list = new ArrayList<Job>(jobs.values());
        Collections.sort(list);
        return list;
    }

    public List<JobFuture> getJobFutures() {
        List<JobFuture> list = new ArrayList<JobFuture>(scheduledjobs.values());
        Collections.sort(list);
        return list;
    }

    public String runJob(String jobName, Map<String, String> params) {
        JobTask task = new JobTask(params);
        JobFuture future = new JobFuture(jobName, taskScheduler.schedule(task, new Date(System.currentTimeMillis() + 10)));
        task.setFutureId(future.getId());
        registerJobFuture(future);
        return future.getId();
    }

    public void stopJob(String futureId) {
        ScheduledFuture future = scheduledjobs.get(futureId);
        future.cancel(true);
        scheduledjobs.remove(futureId);
        jobInstances.remove(futureId);
    }

    public JobInstance getJobInstance(String futureId) {
        return jobInstances.get(futureId);
    }

    public Map<String, JobInstance> getJobInstances() {
        return jobInstances;
    }

    public void setJobEnable(String jobName, boolean enable) {
        Job job = jobs.get(jobName);
        if (job.isEnable() != enable) {
            changeJobSchedule(job, enable);
            job.setEnable(enable);
            saveJobCfg(job);
        }
    }

    protected void changeJobSchedule(Job job, boolean enable) {
        if (enable) {
            if (job.getCrontab() != null) {
                JobTask task = new JobTask();
                JobFuture future = new JobFuture(job.getName(), taskScheduler.schedule(task, new CronTrigger(job.getCrontab())));
                task.setFutureId(future.getId());
                registerJobFuture(future);
            }
        } else {
            stopJobByName(job.getName());
        }
    }

    protected void stopJobByName(String jobName) {
        Iterator<Map.Entry<String, JobFuture>> it = scheduledjobs.entrySet().iterator();
        while (it.hasNext()) {
            JobFuture future = it.next().getValue();
            if (jobName.equals(future.getJobName())) {
                future.cancel(false);
                it.remove();
                jobInstances.remove(future.getId());
            }
        }
    }

    public void init() {
        LOG.info("init calling, creating and opening ServiceTracker...");
        Filter filter = null;
        try {
            filter = bundleContext.createFilter("(objectClass=routines.system.api.TalendJob)");
        } catch (InvalidSyntaxException e) {
            LOG.error("create jobs filter error ", e);
        }
        new ServiceTracker(bundleContext, filter, new Customizer(this)).open();
    }

    private JobFuture registerJobFuture(JobFuture jobFuture) {
        scheduledjobs.put(jobFuture.getId(), jobFuture);
        return jobFuture;
    }

    public void jobAdded(Object object, String name) {
        JobInstance instance = new JobInstance(object);
        Job job = new Job();
        job.setName(name);
        job.setVersion(instance.getJobVersion());
        job.setJobClass(object.getClass());
        saveJob(job);
    }

    public void jobRemoved(Object object, String name) {
        JobInstance instance = new JobInstance(object);
        jobs.remove(instance.getJobName());
        stopJobByName(name);
    }

    private class JobTask implements Runnable {

        private String futureId;
        private Map<String, String> params;

        private JobTask() {
        }

        public JobTask(Map<String, String> params) {
            this.params = params;
        }

        public void setFutureId(String futureId) {
            this.futureId = futureId;
        }

        public void run() {
            JobFuture future = scheduledjobs.get(futureId);
            Job job = jobs.get(future.getJobName());
            if (!future.hasNextTime()) {
                scheduledjobs.remove(futureId);
            }
            JobReport report = new JobReport();
            JobInstance instance = null;
            try {
                Map<String, String> map = new HashMap<String, String>(job.getPreferences());
                if (params != null) {
                    map.putAll(params);
                }
                report.setId(futureId);
                report.setName(job.getName());
                instance = new JobInstance(job.getJobClass().newInstance());
                jobInstances.put(futureId, instance);
                instance.setReport(report);
                report.setStartAt(new Date());
                report.setRet(instance.runJob(map));
            } catch (Exception e) {
                LOG.error("Run job " + job.getName() + " error", e);
            } finally {
                jobInstances.remove(futureId);
                report.setFinishAt(new Date());
                Map<String, Object> detail = new LinkedHashMap<String, Object>();
                if (instance != null) {
                    detail.put("global", instance.getGlobalMap());
                    detail.put("ok", instance.getOkHash());
                    detail.put("end", instance.getEndHash());
                    detail.put("start", instance.getStartHash());
                    if (report.getRet() != 0) {
                        detail.put("errorMessage", instance.getErrorMessage());
                    }
                    report.setDetail(JSON.toJSONString(detail));
                }
                jobReportManager.saveJobReport(report);
            }
        }
    }

    private class Customizer implements ServiceTrackerCustomizer {

        private JobListener jobListener;

        private Customizer(JobListener jobListener) {
            this.jobListener = jobListener;
        }

        public Object addingService(ServiceReference reference) {
            LOG.info("Service with reference " + reference + " added");
            Object service = bundleContext.getService(reference);
            if (service != null) {

                jobListener.jobAdded(service, (String) reference.getProperty("name"));
            }
            return service;
        }

        public void modifiedService(ServiceReference reference, Object job) {
            LOG.info("Service " + job + " modified");
        }

        public void removedService(ServiceReference reference, Object service) {
            LOG.info("Service " + service + " removed");
            jobListener.jobRemoved(service, (String) reference.getProperty("name"));
            bundleContext.ungetService(reference);
        }
    }

}
