/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: NodeService.java
 *
 * LastModified: 2009-09-23 11:22:10
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.fileCenter.service;

import com.gtis.fileCenter.ex.NodeExistsException;
import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.model.Space;
import com.gtis.fileCenter.model.impl.File;

import java.util.List;
import java.util.Map;

/**
 * 文件中心服务类.
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-15
 */
public interface NodeService {

    /**
     * 获取节点的访问令牌
     *
     * @param node 节点
     * @return 令牌
     */
    String getToken(Node node);

    /**
     * 获取节点的访问令牌
     *
     * @param node 节点
     * @param ttl  token存活时间
     * @return 令牌
     */
    String getToken(Node node, Integer ttl);

    /**
     * 获取节点的访问令牌
     *
     * @param node      节点
     * @param writeable 是否可写
     * @return 令牌
     */
    String getToken(Node node, boolean writeable);

    /**
     * 获取节点的访问令牌
     *
     * @param node      节点
     * @param writeable 是否可写
     * @param ttl       token存活时间
     * @return 令牌
     */
    String getToken(Node node, boolean writeable, Integer ttl);

    /**
     * 获取token对应的节点Id
     *
     * @param token 令牌
     * @return 节点id
     */
    Integer getTokenRoot(String token);

    /**
     * 根据令牌判断是否有权限访问节点
     *
     * @param token  令牌
     * @param nodeId 节点id
     * @return 是否有权限
     */
    boolean hasPermission(String token, Integer nodeId);

    /**
     * 根据令牌判断是否有写权限
     *
     * @param token 令牌
     * @return 是否有写权限
     */
    public boolean isWriteable(String token);

    /**
     * 得到根节点
     *
     * @return 根节点
     * @throws NodeNotFoundException 找不到根节点
     */
    Node getRootNode() throws NodeNotFoundException;

    /**
     * 获取用户的个人空间节点
     *
     * @param userId 用户id
     * @return 个人空间，如果不存在，返回null
     * @throws NodeNotFoundException 找不到节点
     */
    Space getPersonalSpace(String userId) throws NodeNotFoundException;

    /**
     * 获取业务工作区空间节点
     *
     * @param name 业务名称
     * @return 业务工作区空间，如果不存在，返回null
     * @throws NodeNotFoundException 找不到节点
     */
    Space getWorkSpace(String name) throws NodeNotFoundException;

    /**
     * 获取业务工作区空间节点
     *
     * @param name       业务名称
     * @param autoCreate 自动创建路径
     * @return 业务工作区空间
     * @throws NodeNotFoundException 找不到节点
     */
    Space getWorkSpace(String name, boolean autoCreate) throws NodeNotFoundException;

    /**
     * 获取节点对应的空间
     *
     * @param nodeId 节点id
     * @return 空间
     * @throws NodeNotFoundException 找不到节点
     */
    Space getSpace(Integer nodeId) throws NodeNotFoundException;

    /**
     * 空间是否空余指定大小容量
     *
     * @param space 空间
     * @param size  大小
     * @return 是否拥有
     */
    boolean hasCapacity(Space space, long size);

    /**
     * 获取节点
     *
     * @param nodeId 节点Id
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNode(Integer nodeId) throws NodeNotFoundException;

    /**
     * 根据类型获取节点
     *
     * @param type 类型
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNodeByType(int type) throws NodeNotFoundException;

    /**
     * 根据名称和类型获取节点
     *
     * @param name 名称
     * @param type 类型
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNodeByType(String name, int type) throws NodeNotFoundException;


    /**
     * 根据父节点和路径获取节点
     *
     * @param parentNodeId 父节点id
     * @param path         路径
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNode(Integer parentNodeId, String path) throws NodeNotFoundException;

    /**
     * 根据父节点和路径获取节点
     *
     * @param parentNodeId 父节点id
     * @param path         路径
     * @param autoCreate   自动创建路径
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getNode(Integer parentNodeId, String path, boolean autoCreate);

    /**
     * 根据id列表获取一批节点
     *
     * @param nodeIds id列表
     * @return 节点列表
     */
    List<Node> getNodes(Integer[] nodeIds);

    /**
     * 获取父亲节点
     *
     * @param nodeId 节点Id
     * @return 父节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getParentNode(Integer nodeId) throws NodeNotFoundException;


    /**
     * 获取子节点
     *
     * @param parentNodeId 父节点Id
     * @param nodeId       节点Id
     * @param name         名称
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     */
    Node getChildNode(Integer nodeId, String name) throws NodeNotFoundException;

    /**
     * 获取下级子节点
     *
     * @param nodeId 节点Id
     * @return 子节点列表
     */
    List<Node> getChildNodes(Integer nodeId);

    /**
     * 获取所有子节点
     *
     * @param nodeId 节点Id
     * @return 子节点列表
     */
    List<Node> getAllChildNodes(Integer nodeId);

    /**
     * 分页获取所有文件子节点
     *
     * @param nodeId 节点Id
     * @param start  开始
     * @param size   个数
     * @return 子节点文件列表
     */
    List<File> getAllChildFile(Integer nodeId, int start, int size);

    /**
     * 获取子节点文件数量
     *
     * @param nodeId 节点Id
     * @return 子节点文件数量
     */
    int getAllChildFilesCount(Integer nodeId);

    /**
     * 检索节点
     *
     * @param nodeId  节点Id
     * @param keyword 关键字
     * @return 节点列表
     */
    List<Node> search(Integer nodeId, String keyword);

    /**
     * 获取节点路径
     *
     * @param nodeId 节点Id
     * @return 路径列表
     */
    List<Node> getPath(Integer nodeId);

    /**
     * 节点是否存在
     *
     * @param nodeId 节点Id
     * @return 是否存在
     */
    boolean exists(Integer nodeId);

    /**
     * 节点名是否已存在
     *
     * @param parentNodeId 父节点id
     * @param name         名称
     * @return 是否存在
     */
    boolean isNodeNameUsed(Integer parentNodeId, String name);

    /**
     * 判断一个节点是否另外一个的子节点
     *
     * @param parentNodeId 节点id
     * @param nodeId       节点id
     * @return 是否为子节点
     */
    boolean isChildNode(Integer parentNodeId, Integer nodeId);

    /**
     * 保存节点
     *
     * @param node 节点
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     * @throws NodeExistsException   节点已存在
     */
    Node save(Node node) throws NodeNotFoundException, NodeExistsException;

    /**
     * 保存节点
     *
     * @param node       节点
     * @param autoRename 是否自动重命名
     * @return 节点
     * @throws NodeNotFoundException 找不到节点
     * @throws NodeExistsException   节点已存在
     */
    Node save(Node node, boolean autoRename) throws NodeNotFoundException, NodeExistsException;

    /**
     * 创建节点
     *
     * @param parentId    父节点Id
     * @param name        名称
     * @param viewName    显示名称
     * @param description 说明
     * @param scope       可见范围
     * @param owner       所有者
     * @param attributes  属性
     * @return 节点
     */
    Node createNode(Integer parentId, String name, String viewName, String description, Integer scope, String owner, Map<String, String> attributes);

    /**
     * 创建节点
     *
     * @param parentId 父节点Id
     * @param name     名称
     * @return 节点
     */
    Node createNode(Integer parentId, String name);

    /**
     * 创建工作空间
     *
     * @param name        空间名 必须唯一
     * @param viewName    显示名
     * @param size        空间大小
     * @param description 说明
     * @param attributes  属性
     * @return 创建的空间
     */
    Space createWorkSpace(String name, String viewName, Long size, String description, Map<String, String> attributes);

    /**
     * 创建工作空间
     *
     * @param name 空间名 必须唯一
     * @return 创建的空间
     */
    Space createWorkSpace(String name);

    /**
     * 删除节点，提醒，会递归删除所属子节点
     *
     * @param nodeId 节点Id
     */
    void remove(Integer nodeId);

    /**
     * 删除节点，提醒，会递归删除所属子节点
     *
     * @param nodeIds 节点Id列表
     */
    void remove(Integer[] nodeIds);

    /**
     * 复制一批节点到目标节点
     *
     * @param nodeIds    节点列表
     * @param destNodeId 目标节点
     * @param cover      是否覆盖
     */
    void copy(Integer[] nodeIds, Integer destNodeId, boolean cover);

    /**
     * 移动一批节点到目标节点
     *
     * @param nodeIds    节点列表
     * @param destNodeId 目标节点
     * @param cover      是否覆盖
     */
    void move(Integer[] nodeIds, Integer destNodeId, boolean cover);

    /**
     * 压缩一批节点到指定文件
     *
     * @param nodeIds    节点列表
     * @param destNodeId 文件位置
     * @param name       文件名
     * @throws NodeExistsException 同名文件已存在
     */
    void zip(Integer[] nodeIds, Integer destNodeId, String name) throws NodeExistsException;

    /**
     * 解压zip文件到指定位置
     *
     * @param nodeId zip文件
     * @param name   目录名
     * @param cover  是否覆盖
     */
    void unzip(Integer nodeId, String name, boolean cover);
}
