/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: CookieUtils.java
 *
 * LastModified: 2009-09-23 11:17:01
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.generic.util;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-9-23
 */
public class CookieUtils {

    /**
     * Returns the specified cookie, or <tt>null</tt> if the cookie
     * does not exist. Note: because of the way that cookies are implemented
     * it's possible for multiple cookies with the same name to exist (but with
     * different domain values). This method will return the first cookie that
     * has a name match.
     *
     * @param request the servlet request.
     * @param name    the name of the cookie.
     * @return the Cookie object if it exists, otherwise <tt>null</tt>.
     */
    public static Cookie get(HttpServletRequest request, String name) {
        Cookie cookies[] = request.getCookies();
        // Return null if there are no cookies or the name is invalid.
        if (cookies == null || name == null || name.length() == 0) {
            return null;
        }
        // Otherwise, we  do a linear scan for the cookie.
        Cookie cookie = null;
        for (Cookie cooky : cookies) {
            // If the current cookie name matches the one we're looking for, we've
            // found a matching cookie.
            if (cooky.getName().equals(name)) {
                cookie = cooky;
                // The best matching cookie will be the one that has the correct
                // domain name. If we've found the cookie with the correct domain name,
                // return it. Otherwise, we'll keep looking for a better match.
                if (request.getServerName().equals(cookie.getDomain())) {
                    break;
                }
            }
        }
        return cookie;
    }

    public static String getValue(HttpServletRequest request, String name) {
        Cookie ci = get(request, name);
        return ci == null ? null : ci.getValue();
    }

    /**
     * Deletes the specified cookie.
     *
     * @param request  the servlet request.
     * @param response the servlet response.
     * @param cookie   the cookie object to be deleted.
     */
    public static void remove(HttpServletRequest request, HttpServletResponse response, Cookie cookie) {
        if (cookie != null) {
            // Invalidate the cookie
            String path = request.getContextPath() == null ? "/" : request.getContextPath();
            if ("".equals(path)) {
                path = "/";
            }
            cookie.setPath(path);
            cookie.setValue("");
            cookie.setMaxAge(0);
            response.addCookie(cookie);
        }
    }

    /**
     * Stores a value in a cookie. By default this cookie will persist for 30 days.
     *
     * @param request  the servlet request.
     * @param response the servlet response.
     * @param name     a name to identify the cookie.
     * @param value    the value to store in the cookie.
     * @see #set(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, String, String, int)
     */
    public static void set(HttpServletRequest request, HttpServletResponse response, String name, String value) {
        // Save the cookie value for 1 month
        set(request, response, name, value, 60 * 60 * 24 * 30);
    }

    /**
     * Stores a value in a cookie. This cookie will persist for the amount
     * specified in the <tt>saveTime</tt> parameter.
     *
     * @param request  the servlet request.
     * @param response the servlet response.
     * @param name     a name to identify the cookie.
     * @param value    the value to store in the cookie.
     * @param maxAge   the time (in seconds) this cookie should live.
     * @see #set(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, String, String)
     */
    public static void set(HttpServletRequest request, HttpServletResponse response, String name, String value, int maxAge) {
        // Check to make sure the new value is not null (appservers like Tomcat
        // 4 blow up if the value is null).
        if (value == null) {
            value = "";
        }
        String path = request.getContextPath() == null ? "/" : request.getContextPath();
        if ("".equals(path)) {
            path = "/";
        }
        Cookie cookie = new Cookie(name, value);
        cookie.setMaxAge(maxAge);
        cookie.setPath(path);
        response.addCookie(cookie);
    }
}

