package com.gtis.archive.core.impl;

import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.Field;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.environment.Environment;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.util.CriteriaCallback;
import com.gtis.common.Page;
import com.gtis.support.hibernate.HibernateDao;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.hibernate.*;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.hibernate3.SessionFactoryUtils;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-10
 */
@SuppressWarnings("unchecked")
public class EntityServiceImpl extends HibernateDao implements EntityService {
    private static final Logger logger = LoggerFactory.getLogger(EntityServiceImpl.class);

    private ModelService modelService;

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void cleanCache() {
        SessionFactoryUtils.getSession(sessionFactory, true).clear();
    }

    public String getModelName(Object entity) {
        return modelService.getModelName(entity);
    }

    public Model getModel(String modelName) {
        return modelService.getModel(modelName);
    }

    public <T> T load(String modelName, String id) {
        return (T) load(getModelClass(modelName), id);
    }

    @Transactional
    public <T> T load(Class clazz, String id) {
//        return (T)getSession().get(clazz,id) ;
        List<T> t = find("from" +"\t" + clazz.getName() + "\t"+" c where c.id=?", new Object[]{id}) ;
        return t.size() == 0 ? null : t.get(0) ;
    }

    public <T> T loadState(Class clazz, String archiveId) {
        List<T> t = find("from" +"\t" + clazz.getName() + "\t"+" c where c.archive_id=?", new Object[]{archiveId}) ;
        return t.size() == 0 ? null : t.get(0) ;
    }

    public <T> T newInstance(String modelName) {
        return (T) newInstance(getModelClass(modelName));
    }

    public <T> T newInstance(Class clazz) {
        try {
            Object obj = clazz.newInstance();
            processDefaultValue(obj);
            return (T) obj;
        } catch (Exception e) {
            throw new RuntimeException("model:[" + clazz.getName() + "] instance error.");
        }
    }

    @Transactional
    public void remove(String modelName, String[] ids) {
        remove(getModelClass(modelName), ids);
    }

    @Transactional
    public void remove(Class clazz, String[] ids) {
        try{
            for (String id : ids) {
                getSession().delete(load(clazz, id));
            }
        }catch (Exception ex){
              logger.error(ex.toString());
               ex.printStackTrace();
        }
    }

    public <T> T query(String modelName, Criterion... criterions) {
        return query(getModelClass(modelName), criterions);
    }

    public <T> T query(Class clazz, Criterion... criterions) {
        Criteria criteria = getSession().createCriteria(clazz);
        if (criterions != null)
            for (Criterion c : criterions) {
                if (c != null)
                    criteria.add(c);
            }
        return (T) criteria.uniqueResult();
    }

    @Transactional
    public <T> T save(Object entity) {
        processDefaultValue(entity);
        getSession().saveOrUpdate(entity);
        return (T) entity;
    }

    public <X> Page<X> search(CriteriaCallback callback, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        Criteria criteria = callback.create(getSession());
        if (criterions != null)
            for (Criterion c : criterions) {
                if (c != null)
                    criteria.add(c);
            }
        if (orders != null)
            for (Order order : orders) {
                if (order != null)
                    criteria.addOrder(order);
            }
        return search(criteria, start, limit);
    }

    public <T> Page<T> search(CriteriaCallback callback, int start, int limit) {
        return search(callback, null, null, start, limit);
    }

    public <T> Page<T> search(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        return search(getModelClass(modelName), criterions, orders, start, limit);
    }

    public <T> T query(String hql, Object... values) {
        return findUnique(hql, values);
    }

    public <T> T query(String hql, Map<String, ?> values) {
        return findUnique(hql, values);
    }

    public <T> Page<T> search(String hql, String countHql, int start, int limit, Object... values) {
        return doSearch(hql, countHql, start, limit, values);
    }

    public <T> Page<T> search(String hql, String countHql, int start, int limit, Map<String, ?> values) {
        return doSearch(hql, countHql, start, limit, values);
    }

    public <T> Page<T> search(String hql, int start, int limit, Object... values) {
        return search(hql, "select count(*) " + hql, start, limit, values);
    }

    public <T> Page<T> search(String hql, int start, int limit, Map<String, ?> values) {
        return search(hql, "select count(*) " + hql, start, limit, values);
    }

    private <T> Page<T> doSearch(String hql, String countHql, int start, int limit, Object values) {
        Page page = new Page(Page.toIndex(start, limit), limit);
        Query query;
        if (countHql == null || limit == Page.ALL) {
            query = createQuery(hql, values);
            page.setItems(query.list());
        } else {
            query = createQuery(countHql, values);
            int totalCount = toInt(((Long) query.uniqueResult()),0);
            page.setTotalCount(totalCount);
            if (totalCount > 0) {
                query = createQuery(hql, values).setFirstResult(start).setMaxResults(limit);
                page.setItems(query.list());
            }
        }
        return page;
    }

    public int getCount(String modelName, List<? extends Criterion> criterions) {
        return getCount(getModelClass(modelName), criterions);
    }

    public int getCount(Class clazz, List<? extends Criterion> criterions) {
        try {
            Criteria criteria = getSession().createCriteria(clazz);
            if (criterions != null)
                for (Criterion c : criterions) {
                    criteria.add(c);
                }
            return toInt(((Long) criteria.setProjection(Projections.rowCount()).uniqueResult()),0);
        } catch (Exception e) {
            logger.error("getCount for model" + clazz + " error", e);
            return 0;
        }
    }

    private Class getModelClass(String modelName) {
        try {
            return modelService.getClass(modelName);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("model:[" + modelName + "] not found.");
        }
    }

    private void processDefaultValue(Object entity) {
        Model model = modelService.getModel(entity);
        Environment env = new Environment(entity, model.getEnv());
        for (Field field : model.getInheritfieldsMap().values()) {
            String prop = field.getName();
            try {
                Object v = PropertyUtils.getSimpleProperty(entity, prop);
                if ((v == null || "".equals(v)|| "0".equals(v.toString()))) {
                    if (field.getDefaultValue() != null) {
                        String defaultValue = env.getExpr(field.getDefaultValue());
                        if (defaultValue != null) {
                            if(field.getType()== Field.Type.DATE){//日期型默认值
                                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
                                Date date = sdf.parse(defaultValue);
                                BeanUtils.setProperty(entity, prop, date);
                            }else{
                                BeanUtils.setProperty(entity, prop, defaultValue);
                            }
                        }
                    }
                }
            } catch (Exception e) {
                logger.info("error to set field:" + prop + "default value", e);
            }
        }
        try {
            String id = (String) PropertyUtils.getSimpleProperty(entity, "id");
            if (StringUtils.isBlank(id)) {
                PropertyUtils.setSimpleProperty(entity, "id", null);
            }
        } catch (Exception ignored) {
            logger.error(ignored.getLocalizedMessage());
        }
    }

}
