package com.gtis.archive.service.impl;

import com.gtis.archive.Constants;
import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.dict.DictService;
import com.gtis.archive.core.dict.Item;
import com.gtis.archive.core.support.hibernate.UUIDHexGenerator;
import com.gtis.archive.entity.*;
import com.gtis.archive.service.ArchiveService;
import com.gtis.archive.service.OriginalService;
import com.gtis.archive.service.PermissionService;
import com.gtis.archive.util.Struts2Utils;
import com.gtis.common.Page;
import com.gtis.plat.service.SysUserService;
import com.gtis.support.hibernate.HibernateDao;
import com.gtis.web.SessionUtil;
import org.apache.commons.lang.StringUtils;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.Collections;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-10-26
 */
public class ArchiveServiceImpl extends HibernateDao implements ArchiveService {
    private static final Logger logger = LoggerFactory.getLogger(ArchiveServiceImpl.class);

    @Autowired
    private EntityService entityService;
    @Autowired
    private ModelService modelService;
    @Autowired
    private OriginalService originalService;
    @Autowired
    private SysUserService userService;
    @Autowired
    private DictService dictService;
    @Autowired
    private PermissionService permissionService;

    private boolean dwdmEnable = true;
    private boolean jiangyinEnable = false;

    public void setDwdmEnable(boolean dwdmEnable) {
        this.dwdmEnable = dwdmEnable;
    }

    public void setJiangyinEnable(boolean jiangyinEnable) {
        this.jiangyinEnable = jiangyinEnable;
    }

    public Archive getArchive(String modelName, String id) {
        return entityService.load(Document.toArchiveModelName(modelName), id);
    }

    public Document getDocument(String modelName, String id) {
        return entityService.load(Archive.toDocumentModelName(modelName), id);
    }

    public Archive getSimpleArchive(String id) {
        return (Archive) getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.id=?").addEntity(Archive.class).setParameter(0, id).uniqueResult();
    }

    public Archive getSimpleArchiveByProId(String proid) {
        return (Archive) getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.pro_id=?").addEntity(Archive.class).setParameter(0, proid).uniqueResult();
    }

    public Document getSimpleDocument(String id) {
        return (Document) getSession().createSQLQuery("select t.*,0 clazz_ from t_document t where t.id=?").addEntity(Document.class).setParameter(0, id).uniqueResult();
    }

    public Archive getDocumentArchive(String modelName, String documentId) {
        return getArchive(modelName, getDocument(modelName, documentId).getId());
    }

    public Archive getDocumentArchive(Document document) {
        return getArchive(document.getModelName(), document.getArchiveId());
    }

    public List<Document> getArchiveDocuments(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.asc("sxh"));
        orders.add(Order.asc("id"));
        return entityService.<Document>search(Archive.toDocumentModelName(modelName), Collections.singletonList(Restrictions.eq("archiveId", archiveId)), orders, 0, Page.ALL).getItems();
    }

    public List<Document> getArchiveDocuments(Archive archive) {
        return getArchiveDocuments(archive.getModelName(), archive.getId());
    }

    public Model getArchiveModel(String modelName) {
        return modelService.getModel(Document.toArchiveModelName(modelName));
    }

    public Model getDocumentModel(String modelName) {
        return modelService.getModel(Archive.toDocumentModelName(modelName));
    }

    public Archive newArchive(String modelName) {
        return (Archive) entityService.newInstance(Document.toArchiveModelName(modelName));
    }

    public Document newDocument(String modelName) {
        return (Document) entityService.newInstance(Archive.toDocumentModelName(modelName));
    }

    @Transactional
    public void removeArchive(String modelName, String[] ids) {
        for (String id : ids) {
            List<Document> docs = getArchiveDocuments(modelName, id);
            String[] dids = new String[docs.size()];
            for (int i = 0, size = docs.size(); i < size; i++) {
                dids[i] = docs.get(i).getId();
            }
            removeDocument(modelName, dids);
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);
        }
        entityService.remove(modelName, ids);
        logger.debug("remove archive [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    public void removeDocument(String modelName, String[] ids) {
        for (String id : ids) {
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);
        }
        entityService.remove(Archive.toDocumentModelName(modelName), ids);
        logger.debug("remove document [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    public Archive saveArchive(Archive archive) {
        if (dwdmEnable) {
            if (StringUtils.isBlank(archive.getDwdm()))
                archive.setDwdm(getDwdm());
            if (jiangyinEnable) setQzhByDwdm(archive);
        }
        if (StringUtils.isEmpty(archive.getId())) {
            archive.setId(UUIDHexGenerator.generate());
        }
        return entityService.save(archive);
    }

    @Transactional
    public Document saveDocument(Document document) {
        if (dwdmEnable) {
            document.setDwdm(getDwdm());
        }
        return entityService.save(document);
    }

    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);

        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin() && !hasPermissionToViewAllArchive()) {
            list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", getDwdm())));
        }
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    public Page<Document> searchDocument(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin()) {
            list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", getDwdm())));
        }
        return entityService.search(Archive.toDocumentModelName(modelName), list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    public int getArchiveCount(String modelName, List<? extends Criterion> criterions) {
        return entityService.getCount(modelName, criterions);
    }

    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, String dwdm, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", dwdm)));
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    @Override
    public List<Object> searchArchiveList(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit).getItems();
    }

    @Override
    public String flhjs(String flhjs, String modelName) {
        String hql = "select flh from Archive where flh like :flhcx and modelName= :modelName group by flh order by flh ";
        List<Object> list = null;
        Map map = new HashMap();
        String flhcs = "%" + flhjs + "%";
        map.put("flhcx", flhcs);
        map.put("modelName", modelName);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            e.printStackTrace();
        }
        StringBuffer flhsz = new StringBuffer();
        for (int i = 0; i < list.size(); i++) {
            try {
                flhsz.append(list.get(i).toString());
                flhsz.append(";  ");
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        String flhtj = flhsz.toString();
        return flhtj;
    }

    private String getDwdm() {
        String dwdm = (String) Struts2Utils.getSessionAttribute("__dwdm");
        if (dwdm == null) {
            dwdm = userService.getUserRegionCode(SessionUtil.getCurrentUserId());
            Struts2Utils.getSession().setAttribute("__dwdm", dwdm);
        }
        return dwdm;
    }

    /**
     * 根据档号获取档案信息
     *
     * @param dh 档号
     * @return
     */
    public Archive getArchiveByDh(String dh) {
        String hql = "FROM Archive where dh =:dh";
        Map map = new HashMap();
        List<Archive> list = new ArrayList<Archive>();
        map.put("dh", dh);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error("根据档号查询档案出现异常" + e.toString());
        }
        return list.get(0);
    }

    /**
     * 根据档号获取档案信息
     *
     * @param ajh 案卷号
     * @return
     */
    public Archive getArchiveByAjh(Integer ajh, String mlh) {
        String hql = "FROM Archive where ajh=:ajh and mlh=:mlh ";
        Map map = new HashMap();
        List<Archive> list = new ArrayList<Archive>();
        map.put("dh", ajh);
        map.put("mh", mlh);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error("根据档号查询档案出现异常" + e.toString());
        }
        return list.get(0);
    }

    /**
     * 获取ajhs
     *
     * @param room
     * @param carbinet
     * @param side
     * @param cell
     * @param box
     * @return
     */
    public String getAjhs(String room, String carbinet, Integer side, Integer cell, Integer box) {
        String hql = "FROM Archive where roomId =:room and cabinetSno =:carbinet and sideId =:side and cellId =:cell and boxId =:box";
        List<Archive> list = new ArrayList<Archive>();
        StringBuffer stringBuffer = new StringBuffer();
        Map map = new HashMap();
        map.put("room", room);
        map.put("carbinet", carbinet);
        map.put("side", side);
        map.put("cell", cell);
        map.put("box", box);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error("查询ajhs异常" + e.toString());
        }
        for (Archive archive : list) {
            stringBuffer.append(archive.getAjh().toString());
            stringBuffer.append(",");
        }
        String ajhs = stringBuffer.toString();
        return ajhs;
    }

    /**
     * 判断是否可以看到所有档案
     *
     * @return
     */
    private boolean isViewAllArchive() {
        try {
            List<Item> dicts = dictService.getItems(Constants.VIEW_ALL_ARCHIVE);
            String currentDwdm = getDwdm();

            for (Item item : dicts) {
                if (item.getName().equals(currentDwdm)) {
                    return true;
                }
            }
        } catch (Exception ex) {
            logger.error(ex.getMessage());
//            ex.printStackTrace();
        }
        return false;
    }

    /**
     * 检查是否有权限查看所有的档案
     *
     * @return
     */
    private boolean hasPermissionToViewAllArchive() {
        boolean flag = false;
        flag = permissionService.hasPermission("sys", "viewAllArchive");
        return flag;
    }

    /**
     * 可选功能：根据单位代码设置全宗号，其对应关系由字典表中qzhdwdm决定
     *
     * @param archive
     */
    private void setQzhByDwdm(Archive archive) {
        if (StringUtils.isBlank(archive.getDwdm())) return;
        String dwdm = archive.getDwdm();
        try {
            List<Item> items = dictService.getItems(Constants.QZH_DWDM);
            for (Item item : items) {
                if (dwdm.equals(item.getName())) {
                    archive.setQzh(item.getValue());
                    break;
                }
            }
        } catch (Exception ex) {
            logger.info("未配置[{}]单位代码全宗号字典项", Constants.QZH_DWDM);
        }

    }

    /**
     * 获取数据库中顺序号最大的卷内档案。
     *
     * @param modelName
     * @param archiveId
     * @return
     */
    public Document getMaxSxhDocuments(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.desc("sxh"));
        // orders.add(Order.asc("id"));
        List<Document> documents = entityService.<Document>search(Archive.toDocumentModelName(modelName), Collections.singletonList(Restrictions.eq("archiveId", archiveId)), orders, 0, Page.ALL)
                .getItems();
        return documents.size() == 0 ? null : documents.get(0);
    }

    /**
     * 获取档案最大柜号
     */
    public Integer getArchivePosition() {
        List<String> list = getSession().createSQLQuery("select t.cabinet_sno from t_archive t where t.cabinet_sno is not null  group by t.cabinet_sno order by t.cabinet_sno desc").list();
        if (list == null || list.size() == 0) {
            return 1;
        }

        return Integer.valueOf(list.get(0));
    }

    /**
     * 根据柜号获取柜号中最大格号
     *
     * @param setCabinetSno
     * @return
     */
    public Integer getMaxCellId(Integer setCabinetSno) {
        Object object = getSession().createSQLQuery("select  max (a.cell_id) from t_archive a where a.cabinet_sno  = " + setCabinetSno).uniqueResult();
        return object == null ? 1 : ((BigDecimal) object).intValue();
    }

    /**
     * 根据格号、柜号获取最大的盒号
     * @param cellId
     * @param cabinetSno
     * @return
     */
    public Integer getMaxBox(Integer cellId, Integer cabinetSno) {
        Object object = getSession().createSQLQuery("select  max (a.box_id) from t_archive a where a.cell_id = " + cellId + " and a.cabinet_sno = " + cabinetSno).uniqueResult();
        return object == null ? 1 : ((BigDecimal) object).intValue();
    }

    /**
     * 判断盒子是否已满
     * @param boxId
     * @param cellId
     * @param cabinetSno
     * @return
     */
    public boolean isBoxFull(Integer boxId, Integer cellId, Integer cabinetSno) {
        Integer count = ((BigDecimal) (getSession().createSQLQuery("select count(t.id) from t_archive t where t.cell_id = " + cellId + " and t.box_id = " + boxId + " and t.cabinet_sno = " + cabinetSno).uniqueResult())).intValue();
        return count < 15 ? false : true;
    }

    /**
     * 判断格子是否已满
     * @param cellId
     * @param cabinetSno
     * @return
     */
    public boolean isCellFull(Integer cellId, Integer cabinetSno) {
        Integer count = ((BigDecimal) (getSession().createSQLQuery("select count(t.id) from t_archive t where t.cell_id = " + cellId + " and t.cabinet_sno = " + cabinetSno).uniqueResult())).intValue();
        return count < 180 ? false : true;
    }
}
