package com.gtis.archive.service.impl;

import com.gtis.archive.Constants;
import com.gtis.archive.Switch;
import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.dict.DictService;
import com.gtis.archive.core.dict.Item;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.core.support.hibernate.UUIDHexGenerator;
import com.gtis.archive.entity.*;
import com.gtis.archive.service.ArchiveService;
import com.gtis.archive.service.OriginalService;
import com.gtis.archive.util.Struts2Utils;
import com.gtis.common.Page;
import com.gtis.plat.service.SysUserService;
import com.gtis.support.hibernate.HibernateDao;
import com.gtis.web.SessionUtil;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-10-26
 */
public class ArchiveServiceImpl extends HibernateDao implements ArchiveService {
    private static final Logger logger = LoggerFactory.getLogger(ArchiveServiceImpl.class);

    @Autowired
    private EntityService entityService;
    @Autowired
    private ModelService modelService;
    @Autowired
    private OriginalService originalService;
    @Autowired
    private SysUserService userService;
    @Autowired
    private DictService dictService;
    private boolean dwdmEnable = false;
    private boolean jiangyinEnable = false;

    public void setDwdmEnable(boolean dwdmEnable) {
        this.dwdmEnable = dwdmEnable;
    }

    public void setJiangyinEnable(boolean jiangyinEnable) {
        this.jiangyinEnable = jiangyinEnable;
    }

    public Archive getArchive(String modelName, String id) {
        return entityService.load(Document.toArchiveModelName(modelName), id);
    }

    public Document getDocument(String modelName, String id) {
        return entityService.load(Archive.toDocumentModelName(modelName), id);
    }

    public Archive getSimpleArchive(String id) {
        return (Archive) getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.id=?").addEntity(Archive.class).setParameter(0, id).uniqueResult();
        //return entityService.load(Archive.class,id);
    }

    public Document getSimpleDocument(String id) {
        return (Document) getSession().createSQLQuery("select t.*,0 clazz_ from t_document t where t.id=?").addEntity(Document.class).setParameter(0, id).uniqueResult();
        //return entityService.load(Document.class,id);
    }

    public Archive getDocumentArchive(String modelName, String documentId) {
        return getArchive(modelName, getDocument(modelName, documentId).getId());
    }

    public Archive getDocumentArchive(Document document) {
        return getArchive(document.getModelName(), document.getArchiveId());
    }

    public List<Document> getArchiveDocuments(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.asc("sxh"));
        orders.add(Order.asc("id"));
        return entityService.<Document>search(Archive.toDocumentModelName(modelName), Collections.singletonList(Restrictions.eq("archiveId", archiveId)), orders, 0, Page.ALL).getItems();
    }

    @Override
    public List<Document> getArchiveDocuments(String modelName, String archiveId, int currentSxh, String insertId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.asc("sxh"));
        orders.add(Order.asc("id"));
        List<Criterion> criterions = new ArrayList<Criterion>();
        criterions.add(Restrictions.eq("archiveId", archiveId));
        criterions.add(Restrictions.ne("id", insertId));
        criterions.add(Restrictions.ge("sxh", currentSxh));
        return entityService.<Document>search(Archive.toDocumentModelName(modelName), criterions, orders, 0, Page.ALL).getItems();
    }

    @Override
    public Integer getArchiveDocumentLastSxh(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.asc("sxh"));
        orders.add(Order.asc("id"));
        List<Criterion> criterions = new ArrayList<Criterion>();
        criterions.add(Restrictions.eq("archiveId", archiveId));
        criterions.add(Restrictions.isNotNull("sxh"));
        List<Document> list = entityService.<Document>search(Archive.toDocumentModelName(modelName), criterions, orders, 0, Page.ALL).getItems();
        if(list!=null&&list.size()>0)
            return list.get(list.size()-1).getSxh();
        return 0;
    }

    public List<Document> getArchiveDocuments(Archive archive) {
        return getArchiveDocuments(archive.getModelName(), archive.getId());
    }

    public Model getArchiveModel(String modelName) {
        return modelService.getModel(Document.toArchiveModelName(modelName));
    }

    public Model getDocumentModel(String modelName) {
        return modelService.getModel(Archive.toDocumentModelName(modelName));
    }

    public Archive newArchive(String modelName) {
        return (Archive) entityService.newInstance(Document.toArchiveModelName(modelName));
    }

    public Document newDocument(String modelName) {
        return (Document) entityService.newInstance(Archive.toDocumentModelName(modelName));
    }

    @Transactional
    public void removeArchive(String modelName, String[] ids) {
        for (String id : ids) {
            List<Document> docs = getArchiveDocuments(modelName, id);
            String[] dids = new String[docs.size()];
            for (int i = 0, size = docs.size(); i < size; i++) {
                dids[i] = docs.get(i).getId();
            }
            removeDocument(modelName, dids);
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);
        }
        entityService.remove(modelName, ids);
        logger.debug("remove archive [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    public void removeDocument(String modelName, String[] ids) {
        for (String id : ids) {
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);
        }
        entityService.remove(Archive.toDocumentModelName(modelName), ids);
        logger.debug("remove document [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    public Archive saveArchive(Archive archive) {
        if (dwdmEnable) {
            if (StringUtils.isBlank(archive.getDwdm()))
                archive.setDwdm(getDwdm());
            if (jiangyinEnable) setQzhByDwdm(archive);
        }
        if (StringUtils.isEmpty(archive.getId())) {
            archive.setId(UUIDHexGenerator.generate());
        }
        return entityService.save(archive);
    }

    @Transactional
    public Document saveDocument(Document document) {
        if (dwdmEnable) {
            document.setDwdm(getDwdm());
        }
        return entityService.save(document);
    }

    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin()) {
//            if(EnvHolder.isEnable(Switch.JIANGYIN)){
//                try{
//                    boolean hasPrimDwdm= PropertyUtils.getProperty(newArchive(modelName),"primdwdm")==null;
//                    list.add(Restrictions.or(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", getDwdm())),Restrictions.eq("primdwdm",getDwdm())));
//                }catch (Exception ex){
//                    list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", getDwdm())));
//                }
//            }else{
            list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", getDwdm())));
//            }

        }
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    public Page<Document> searchDocument(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin()) {
            list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", getDwdm())));
        }
        return entityService.search(Archive.toDocumentModelName(modelName), list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    public int getArchiveCount(String modelName, List<? extends Criterion> criterions) {
        return entityService.getCount(modelName, criterions);
    }

    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, String dwdm, boolean flag, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        if (flag) {
//            list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", dwdm)));
            list.add(Restrictions.eq("dwdm", dwdm));
            list.add(Restrictions.isNull("primdwdm"));
        } else {
            list.add(Restrictions.eq("primdwdm", dwdm));
        }
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    @Override
    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, String dwdm, int start, int limit) {

        List<Criterion> list = new ArrayList<Criterion>(criterions);
        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin()) {
            if(StringUtils.isNotBlank(dwdm))
                list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", dwdm)));
        }
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    private String getDwdm() {
        String dwdm = (String) Struts2Utils.getSessionAttribute("__dwdm");
        if (dwdm == null) {
            dwdm = userService.getUserRegionCode(SessionUtil.getCurrentUserId());
            Struts2Utils.getSession().setAttribute("__dwdm", dwdm);
        }
        return dwdm;
    }

    /**
     * 可选功能：根据单位代码设置全宗号，其对应关系由字典表中qzhdwdm决定
     *
     * @param archive
     */
    private void setQzhByDwdm(Archive archive) {
        if (StringUtils.isBlank(archive.getDwdm())) return;
        String dwdm = archive.getDwdm();
        try {
            List<Item> items = dictService.getItems(Constants.QZH_DWDM);
            for (Item item : items) {
                if (dwdm.equals(item.getName())) {
                    archive.setQzh(item.getValue());
                    break;
                }
            }
        } catch (Exception ex) {
            logger.info("未配置[{}]单位代码全宗号字典项", Constants.QZH_DWDM);
        }

    }
}
