package com.gtis.archive.service;

import com.gtis.archive.core.Model;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.entity.Document;
import com.gtis.common.Page;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;

import java.util.List;

/**
 * 档案服务.
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-10-26
 */
public interface ArchiveService {
    /**
     * 获取案卷
     *
     * @param modelName 案卷模型名
     * @param id        案卷id
     * @return 案卷
     */
    Archive getArchive(String modelName, String id);

    /**
     * 获取卷内
     *
     * @param modelName 卷内模型名
     * @param id        卷内id
     * @return 卷内
     */
    Document getDocument(String modelName, String id);

    /**
     * 获取只有基本信息的案卷对象
     * @param id  案卷id
     * @return  案卷
     */
    Archive getSimpleArchive(String id);

    /**
     *  获取只有基本信息的卷内对象
     * @param id 卷内id
     * @return  案卷
     */
    Document getSimpleDocument(String id);

    /**
     * 获取卷内的案卷信息
     *
     * @param modelName  卷内模型名
     * @param documentId 卷内id
     * @return 案卷
     */
    Archive getDocumentArchive(String modelName, String documentId);

    /**
     * 获取卷内的案卷信息
     *
     * @param document 卷内
     * @return 案卷
     */
    Archive getDocumentArchive(Document document);

    /**
     * 获取案卷的卷内列表
     *
     * @param modelName 案卷模型名
     * @param archiveId 案卷id
     * @return 卷内列表
     */
    List<Document> getArchiveDocuments(String modelName, String archiveId);

    /**
     * 插入卷内
     * @param modelName
     * @param archiveId
     * @param currentSxh  被插入的卷内的顺序号
     * @param insertId     插入后生成的documnetId
     * @return
     */
    List<Document> getArchiveDocuments(String modelName, String archiveId, int currentSxh, String insertId);

    Integer getArchiveDocumentLastSxh(String modelName, String archiveId);

    /**
     * 获取案卷的卷内列表
     *
     * @param archive 案卷
     * @return 卷内列表
     */
    List<Document> getArchiveDocuments(Archive archive);

    /**
     * 根据模型名获取案卷模型
     *
     * @param modelName 模型名
     * @return 字段列表
     */
    Model getArchiveModel(String modelName);

    /**
     * 根据模型名获取卷内模型
     *
     * @param modelName 模型名
     * @return 字段列表
     */
    Model getDocumentModel(String modelName);

    /**
     * 根据模型名创建一个案卷实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Archive newArchive(String modelName);

    /**
     * 根据模型名创建一个卷内实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Document newDocument(String modelName);

    /**
     * 删除案卷
     *
     * @param modelName 案卷的模型名
     * @param ids       案卷id列表
     */
    void removeArchive(String modelName, String[] ids);

    /**
     * 删除卷内
     *
     * @param modelName 案卷的模型名
     * @param ids       卷内id列表
     */
    void removeDocument(String modelName, String[] ids);

    /**
     * 保存案卷
     *
     * @param archive 案卷
     * @return 安敢
     */
    Archive saveArchive(Archive archive);

    /**
     * 保存卷内
     *
     * @param document 卷内
     * @return 卷内
     */
    Document saveDocument(Document document);

    /**
     * 检索案卷
     *
     * @param modelName  案卷的模型名
     * @param criterions 检索条件
     * @param orders     排序
     * @param start      位置
     * @param limit      分页大小
     * @return 案卷列表page对象
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 检索卷内
     *
     * @param modelName  案卷的模型名
     * @param criterions 检索条件
     * @param orders     排序
     * @param start      位置
     * @param limit      分页大小
     * @return 卷内列表page对象
     */
    Page<Document> searchDocument(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 获取案卷数量
     *
     * @param modelName  案卷的模型名
     * @param criterions 查询条件
     * @return 数量
     */
    int getArchiveCount(String modelName, List<? extends Criterion> criterions);

    /**
     *
     * @param modelName
     * @param criterions
     * @param orders
     * @param dwdm
     * @param start
     * @param limit
     * @param flag true @param dwdm值对应dwdm字段，false @param dwdm值对应primdwdm
     * @return
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders,String dwdm ,boolean  flag,int start, int limit);


    /**
     * 根据传入的单位代码检索
     * @param modelName
     * @param criterions
     * @param orders
     * @param dwdm
     * @param start
     * @param limit
     * @return
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders,String dwdm,int start, int limit);
}
