package com.gtis.archive.core.support.xstream;

import com.thoughtworks.xstream.core.util.FastStack;
import com.thoughtworks.xstream.io.StreamException;
import com.thoughtworks.xstream.io.naming.NameCoder;
import com.thoughtworks.xstream.io.xml.AbstractXmlWriter;
import com.thoughtworks.xstream.io.xml.Dom4JXmlWriter;
import com.thoughtworks.xstream.io.xml.XmlFriendlyNameCoder;
import com.thoughtworks.xstream.io.xml.XmlFriendlyReplacer;
import org.dom4j.CDATA;
import org.dom4j.Element;
import org.dom4j.io.XMLWriter;
import org.dom4j.tree.DefaultCDATA;
import org.dom4j.tree.DefaultElement;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import java.io.IOException;
import java.util.regex.Pattern;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-11-3
 */
public class CDATASupportDom4JXmlWriter extends AbstractXmlWriter {
    private final XMLWriter writer;
    private final FastStack elementStack;
    private AttributesImpl attributes;
    private boolean started;
    private boolean children;
    private Pattern escapePattern = Pattern.compile("[<>&]");

    public CDATASupportDom4JXmlWriter(XMLWriter writer) {
        this(writer, new XmlFriendlyNameCoder());
    }

    /**
     * @since 1.4
     */
    public CDATASupportDom4JXmlWriter(XMLWriter writer, NameCoder nameCoder) {
        super(nameCoder);
        this.writer = writer;
        this.elementStack = new FastStack(16);
        this.attributes = new AttributesImpl();
        try {
            writer.startDocument();
        } catch (SAXException e) {
            throw new StreamException(e);
        }
    }

    /**
     * @since 1.2
     * @deprecated As of 1.4 use {@link Dom4JXmlWriter#Dom4JXmlWriter(XMLWriter, NameCoder)} instead.
     */
    public CDATASupportDom4JXmlWriter(XMLWriter writer, XmlFriendlyReplacer replacer) {
        this(writer, (NameCoder) replacer);
    }

    public void startNode(String name) {
        if (elementStack.size() > 0) {
            try {
                startElement();
            } catch (SAXException e) {
                throw new StreamException(e);
            }
            started = false;
        }
        elementStack.push(encodeNode(name));
        children = false;
    }

    public void setValue(String text) {
        text = text.replaceAll("\\t", "");
        if (text.length() > 0) {
            try {
                startElement();
                if (text.length() > 10 && escapePattern.matcher(text).find()) {
                    CDATA cdata = new DefaultCDATA(text);
                    writer.write(cdata);
                } else {
                    writer.characters(text.toCharArray(), 0, text.length());
                }
            } catch (SAXException e) {
                throw new StreamException(e);
            } catch (IOException e) {
                throw new StreamException(e);
            }
            children = true;
        }
    }

    public void addAttribute(String key, String value) {
        attributes.addAttribute("", "", encodeAttribute(key), "string", value);
    }

    public void endNode() {
        try {
            if (!children) {
                Element element = new DefaultElement((String) elementStack.pop());
                for (int i = 0; i < attributes.getLength(); ++i) {
                    element.addAttribute(attributes.getQName(i), attributes.getValue(i));
                }
                writer.write(element);
                attributes.clear();
                children = true;   // node just closed is child of node on top of stack
                started = true;
            } else {
                startElement();
                writer.endElement("", "", (String) elementStack.pop());
            }
        } catch (SAXException e) {
            throw new StreamException(e);
        } catch (IOException e) {
            throw new StreamException(e);
        }
    }

    public void flush() {
        try {
            writer.flush();
        } catch (IOException e) {
            throw new StreamException(e);
        }
    }

    public void close() {
        try {
            writer.endDocument();
        } catch (SAXException e) {
            throw new StreamException(e);
        }
    }

    private void startElement() throws SAXException {
        if (!started) {
            writer.startElement("", "", (String) elementStack.peek(), attributes);
            attributes.clear();
            started = true;
        }
    }
}
