package com.gtis.archive.service;

import com.gtis.archive.core.Model;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.entity.Document;
import com.gtis.common.Page;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;

import java.util.List;
import java.util.Map;

/**
 * 档案服务.
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-10-26
 */
public interface ArchiveService {
    /**
     * 获取案卷
     *
     * @param modelName 案卷模型名
     * @param id        案卷id
     * @return 案卷
     */
    Archive getArchive(String modelName, String id);

    /**
     * 获取卷内
     *
     * @param modelName 卷内模型名
     * @param id        卷内id
     * @return 卷内
     */
    Document getDocument(String modelName, String id);

    /**
     * 获取只有基本信息的案卷对象
     * @param id  案卷id
     * @return  案卷
     */
    Archive getSimpleArchive(String id);

    /**
     * 根据架号、排号、盒号获取案卷集合
     * @param jh 架号
     * @param ph 排号
     * @param hh 盒号
     * @return 案卷集合
     */
    List getArchive(String jh, String ph, String hh);

    /**
     *  获取只有基本信息的卷内对象
     * @param id 卷内id
     * @return  案卷
     */
    Document getSimpleDocument(String id);

    /**
     * 获取卷内的案卷信息
     *
     * @param modelName  卷内模型名
     * @param documentId 卷内id
     * @return 案卷
     */
    Archive getDocumentArchive(String modelName, String documentId);

    /**
     * 获取卷内的案卷信息
     *
     * @param document 卷内
     * @return 案卷
     */
    Archive getDocumentArchive(Document document);

    /**
     * 获取案卷的卷内列表
     *
     * @param modelName 案卷模型名
     * @param archiveId 案卷id
     * @return 卷内列表
     */
    List<Document> getArchiveDocuments(String modelName, String archiveId);

    /**
     * 获取案卷的卷内列表
     *
     * @param archive 案卷
     * @return 卷内列表
     */
    List<Document> getArchiveDocuments(Archive archive);

    /**
     * 根据模型名获取案卷模型
     *
     * @param modelName 模型名
     * @return 字段列表
     */
    Model getArchiveModel(String modelName);

    /**
     * 根据模型名获取卷内模型
     *
     * @param modelName 模型名
     * @return 字段列表
     */
    Model getDocumentModel(String modelName);

    /**
     * 根据模型名创建一个案卷实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Archive newArchive(String modelName);

    /**
     * 根据模型名创建一个卷内实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Document newDocument(String modelName);

    /**
     * 删除案卷
     *
     * @param modelName 案卷的模型名
     * @param ids       案卷id列表
     */
    void removeArchive(String modelName, String[] ids);

    /**
     * 删除卷内
     *
     * @param modelName 案卷的模型名
     * @param ids       卷内id列表
     */
    void removeDocument(String modelName, String[] ids);

    /**
     * 保存案卷
     *
     * @param archive 案卷
     * @return 安敢
     */
    Archive saveArchive(Archive archive);

    /**
     * 保存卷内
     *
     * @param document 卷内
     * @return 卷内
     */
    Document saveDocument(Document document);

    /**
     * 检索案卷
     *
     * @param modelName  案卷的模型名
     * @param criterions 检索条件
     * @param orders     排序
     * @param start      位置
     * @param limit      分页大小
     * @return 案卷列表page对象
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 检索卷内
     *
     * @param modelName  案卷的模型名
     * @param criterions 检索条件
     * @param orders     排序
     * @param start      位置
     * @param limit      分页大小
     * @return 卷内列表page对象
     */
    Page<Document> searchDocument(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 获取案卷数量
     *
     * @param modelName  案卷的模型名
     * @param criterions 查询条件
     * @return 数量
     */
    int getArchiveCount(String modelName, List<? extends Criterion> criterions);

    /**
     *
     * @param modelName
     * @param criterions
     * @param orders
     * @param dwdm
     * @param start
     * @param limit
     * @return
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders,String dwdm ,int start, int limit);

    /**
     * 根据架号、排号、盒号获取案卷集合
     * @param jh
     * @param ph
     * @param hh
     * @param start
     * @param limit
     * @return
     */
    Page<Archive> searchArchive(String jh, String ph, String hh, int start, int limit);

    /**
     * 获取满足条件的档案序列
     *
     * @return
     */
    List<Object> searchArchiveList(String modelName, List<? extends Criterion> criterions, List<Order> orders ,int start, int limit);

    /**
     * 获取分类号种类
     *
     * @return
     */
    String flhjs(String flhjs,String modelName);

    /**
     * 获取档案信息通过受理号
     * @return
     */
    Archive getArchiveBySlh(String slh);

    /**
     * 获取上一手或者下一手的档案信息
     * @param slh
     * @type pre | after
     * @return
     */
    Page<Archive> getPreOrAfterArchiveInfo(String slh, String type, int start, int limit);

    /**
     * 查询案卷号获取档案号后六位
     * @param ajhtodah
     * @return
     */
    String getdzdah(int ajhtodah);

    /**
     * 添加保存卷内信息
     * @param modelName
     * @param ownerId
     * @param list
     */
    void saveDoc(String modelName,String ownerId,List<List> list);

    /**
     * 材料名称数字转换
     * @return
     */
    Map getDictforsave();

    /**
     * id转为用户名展示
     * @param page
     * @return
     */
    Page<Archive> getNameByid(Page<Archive> page);

    /**
     * 保存档案信息
     * @param jh （架号）
     * @param ph （排号）
     * @param hh （盒号）
     * @param flh （材料名称）
     * @param dzdah （电子档案号）
     * @param das （档案数）
     * @return
     */
    boolean saveArchiveInfo(String jh, String ph, String hh, String flh, String dzdah, String das);

    /**
     * 退回材料
     * @param ids
     * @param modelName
     * @return
     */
    Map backToBdcdj(String[] ids, String info, String modelName);
}
