package com.gtis.archive.core.impl;

import com.gtis.archive.core.Field;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelChangeEvent;
import com.gtis.archive.core.ModelManager;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.core.environment.Environment;
import com.gtis.archive.core.environment.EnvironmentConverter;
import com.gtis.archive.core.ex.ModelNotFoundException;
import com.gtis.archive.core.support.xstream.CDATASupportDom4JDriver;
import com.thoughtworks.xstream.XStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.core.io.Resource;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-5
 */
public class ModelManagerImpl implements ModelManager, ApplicationEventPublisherAware {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private XStream sm;
    private Resource location;
    private Map<String, Model> models;
    private ApplicationEventPublisher publisher;

    public void setLocation(Resource modelslocation) {
        this.location = modelslocation;
    }

    public void setApplicationEventPublisher(ApplicationEventPublisher publisher) {
        this.publisher = publisher;
    }

    public Model getModel(String modelName) {
        Model model = models.get(modelName);
        if (model == null)
            throw new ModelNotFoundException(modelName);
        return model;
    }

    public Model getModelAllowNull(String modelName) {
        Model model = models.get(modelName);
        return model;
    }

    public Collection<Model> getModels() {
        return models.values();
    }

    public void removeModel(String modelName) {
        models.remove(modelName);
        save();
    }

    public void saveModel(Model model) {
        models.put(model.getName(), model);
        save();
    }

    @SuppressWarnings("unchecked")
    public synchronized void init() {
        models = new LinkedHashMap<String, Model>();
        sm = new XStream(new CDATASupportDom4JDriver());
        sm.alias("environment", Environment.class);
        sm.alias("model", Model.class);
        sm.alias("field", Field.class);
        sm.autodetectAnnotations(true);
        sm.registerConverter(new EnvironmentConverter(sm.getMapper()));
        InputStream is = null;
        try {
            is = location.getInputStream();
            List<Model> list = (List<Model>) sm.fromXML(is);
            for (Model model : list) {
                models.put(model.getName(), model);
            }
        } catch (Exception e) {
            logger.error("load custom models file [" + location + "],error:{}", e);
        } finally {
            if (is != null)
                try {
                    is.close();
                } catch (IOException ignored) {
                }
        }
        for (Model model : models.values()) {
            if (model.getEnv() == null) {
                model.setEnv(new Environment());
            }
            if (model.getTemplates() == null) {
                model.setTemplates(new LinkedHashMap<String, String>());
            }
            if (model.getFields() == null) {
                model.setFields(new LinkedHashSet<Field>());
            }
            if (model.getParentName() != null && models.containsKey(model.getParentName()) && !model.getName().equals(model.getParentName())) {
                model.setParent(models.get(model.getParentName()));
            }
            if (model.getParent() != null) {
                model.getEnv().setParent(model.getParent().getEnv());
            } else {
                model.getEnv().setParent(EnvHolder.getAppEnv());
            }
        }
    }

    private synchronized void save() {
        OutputStream os = null;
        List<Model> list = new ArrayList<Model>();
        for (Model model : models.values()) {
            list.add(model);
        }
        try {
            os = new FileOutputStream(location.getFile());
            sm.toXML(list, os);
        } catch (IOException e) {
            logger.error("store models file [" + list + "],error:{}", e);
        } finally {
            if (os != null)
                try {
                    os.close();
                } catch (IOException ignored) {
                }
        }
        publisher.publishEvent(new ModelChangeEvent(models));
    }
}
