package com.allcam.acs.json;

import com.allcam.acs.http.HttpService;
import com.allcam.acs.utils.JsonUtil;
import com.allcam.common.base.BaseRequest;
import com.allcam.common.base.BaseResponse;
import com.allcam.common.base.ErrorCode;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.SocketException;

/**
 * RYBAndroid
 * <p>
 * Created by huyuwen on 2017/8/11.
 */

public class HttpJsonHandler
{
    private static final Logger LOG = LoggerFactory.getLogger(HttpJsonHandler.class);

    private HttpService httpService;

    public HttpJsonHandler(HttpService httpService)
    {
        this.httpService = httpService;
    }

    public <T extends BaseResponse> T post(String apiUrl, BaseRequest reqBean, Class<T> respCls)
    {
        if (null == reqBean)
        {
            LOG.debug("doPost reqBean is null");
            return optErrorBean(ErrorCode.ERROR_COMMON_FAIL, "request bean is null.", respCls);
        }

        String jsonString = JsonUtil.toJson(reqBean);
        try (Response response = httpService.restHttp(apiUrl, jsonString))
        {
            if (!response.isSuccessful())
            {
                return optErrorBean(response.code(), "request fail.", respCls);
            }

            ResponseBody respBody = response.body();
            String bodyStr = respBody == null ? null : respBody.string();
            if (StringUtils.isEmpty(bodyStr))
            {
                LOG.error("Response Body is Empty.");
                return optErrorBean(ErrorCode.ERROR_COMMON_FAIL, "response body is empty.", respCls);
            }

            LOG.debug("Response Body: {}", bodyStr);
            return JsonUtil.fromJson(bodyStr, respCls);
        }
        catch (IOException e)
        {
            LOG.error("http post json fail.", e);
            if (handleTimeout(e))
            {
                return optErrorBean(ErrorCode.ERROR_TIMEOUT, "http post time out.", respCls);
            }
            else
            {
                return optErrorBean(ErrorCode.ERROR_COMMON_FAIL, "http post json fail.", respCls);
            }
        }
    }

    private boolean handleTimeout(Throwable e)
    {
        if (null == e)
        {
            return false;
        }

        String errMsg = e.getMessage();
        if (e instanceof SocketException
            || e instanceof InterruptedIOException
            || (null != errMsg && errMsg.toLowerCase().contains("timed out")))
        {
            LOG.debug("TIME_OUT");
            return true;
        }
        return false;
    }

    private <T extends BaseResponse> T optErrorBean(int code, String msg, Class<T> respCls)
    {
        try
        {
            T respBean = respCls.newInstance();
            respBean.setResultCode(code);
            respBean.setResultDesc(msg);
            return respBean;
        }
        catch (InstantiationException | IllegalAccessException e)
        {
            LOG.error("instance of response class fail.", e);
            return null;
        }
    }
}
