package com.allcam.acs.http;

import com.allcam.acs.utils.StringUtil;
import com.allcam.common.base.ErrorCode;
import okhttp3.*;
import okio.BufferedSink;
import okio.Okio;
import okio.Source;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

public class HttpService
{
    private static final Logger LOG = LoggerFactory.getLogger(HttpService.class);

    private static final MediaType JSON_TYPE = MediaType.parse("application/json;charset=UTF-8");

    private OkHttpClient client;

    private HttpConfig httpConfig;

    public HttpService(HttpConfig config)
    {
        updateConfig(config);

        OkHttpClient.Builder builder = new OkHttpClient.Builder()
            .connectTimeout(config.timeout, TimeUnit.SECONDS)
            .writeTimeout(config.timeout, TimeUnit.SECONDS)
            .readTimeout(config.timeout * 2, TimeUnit.SECONDS)
            .authenticator(new HttpAuthenticator(httpConfig))
            .retryOnConnectionFailure(true);

        if (null != httpConfig.sslSocketFactory)
        {
            builder.sslSocketFactory(httpConfig.sslSocketFactory);
        }
        if (null != httpConfig.hostnameVerifier)
        {
            builder.hostnameVerifier(httpConfig.hostnameVerifier);
        }
        client = builder.build();
    }

    public void updateConfig(HttpConfig config)
    {
        if (null == this.httpConfig)
        {
            this.httpConfig = config;
        }
        else
        {
            this.httpConfig.updateFrom(config);
        }
    }

    public boolean isAuthorized()
    {
        return null != httpConfig
            && !StringUtil.isEmpty(httpConfig.userName)
            && !StringUtil.isEmpty(httpConfig.password);
    }

    public String get(String url)
    {
        LOG.error("url is: {}", url);
        if (!StringUtil.isEmpty(url))
        {
            throw new IllegalArgumentException("url is empty.");
        }

        Request request = new Request.Builder().url(url)
            .addHeader("User-Agent", "allcam")
            .get().build();

        try (Response response = client.newCall(request).execute())
        {
            if (response.isSuccessful())
            {
                ResponseBody body = response.body();
                return null == body ? "" : body.string();
            }
        }
        catch (IOException e)
        {
            LOG.error("", e);
        }
        return "";
    }

    /**
     * 你需要使用JsonHandler的post方法来发送http请求
     */
    public Response postHttp(String jsonStr)
        throws IOException
    {
        return post(httpConfig.httpUrl, jsonStr);
    }

    public Response restHttp(String path, String jsonStr)
        throws IOException
    {
        return post(httpConfig.httpUrl + path, jsonStr);
    }

    /**
     * 你需要使用JsonHandler的post方法来发送https请求
     */
    public Response postHttps(String jsonStr)
        throws IOException
    {
        if (StringUtil.isEmpty(httpConfig.httpsUrl))
        {
            return postHttp(jsonStr);
        }
        else
        {
            return post(httpConfig.httpsUrl, jsonStr);
        }
    }

    public Response post(String url, String jsonStr)
        throws IOException
    {
        if (null == client)
        {
            LOG.error("http service not init.");
            throw new IllegalStateException("http service not init.");
        }

        if (null == jsonStr)
        {
            LOG.error("body is null");
            throw new IllegalArgumentException("http post body is null");
        }

        if (StringUtil.isEmpty(url))
        {
            LOG.error("url is empty.");
            throw new IllegalArgumentException("url is empty.");
        }

        RequestBody requestBody = RequestBody.create(JSON_TYPE, jsonStr);
        Request request = new Request.Builder()
            .url(url)
            .post(requestBody)
            .build();

        LOG.info("post url[{}] \n and body[{}]", url, jsonStr);

        return client.newCall(request).execute();
    }

    public int upload(String url, String path, MediaType mediaType, HttpProgressCallback callback)
    {
        if (null == client)
        {
            LOG.error("http service not init.");
            return ErrorCode.ERROR_COMMON_FAIL;
        }

        if (StringUtil.isEmpty(url))
        {
            LOG.error("context or url is null");
            return ErrorCode.ERROR_COMMON_FAIL;
        }

        if (StringUtil.isEmpty(path))
        {
            LOG.error("path is empty");
            return ErrorCode.ERROR_COMMON_FAIL;
        }

        File upFile = new File(path);
        RequestBody fileBody = new ProgressRequestBody(upFile, mediaType, callback);
        RequestBody reqBody = new MultipartBody.Builder()
            .setType(MultipartBody.FORM)
            .addFormDataPart("file", upFile.getName(), fileBody)
            .build();

        Request request = new Request.Builder()
            .url(url)
            .post(reqBody)
            .build();

        LOG.info("url is: {}", url);
        LOG.info("path is: {}", path);
        LOG.info("mediaType is: {}", mediaType);

        try
        {
            client.newCall(request).enqueue(callback);
            return ErrorCode.SUCCESS;
        }
        catch (Exception e)
        {
            LOG.error("upload fail: ", e);
        }
        return ErrorCode.ERROR_COMMON_FAIL;
    }

    private class ProgressRequestBody extends RequestBody
    {
        private static final int SEGMENT_SIZE = 2048;

        private File file;

        private MediaType mediaType;

        private long contentLength;

        private ProgressCallback callback;

        ProgressRequestBody(File file, MediaType mediaType, ProgressCallback callback)
        {
            this.file = file;
            this.mediaType = mediaType;
            this.contentLength = file.length();
            this.callback = callback;
        }

        @Override
        public MediaType contentType()
        {
            return mediaType;
        }

        @Override
        public long contentLength()
            throws IOException
        {
            return this.contentLength;
        }

        @Override
        public void writeTo(BufferedSink sink)
            throws IOException
        {
            Source source = Okio.source(this.file);
            long total = 0L;

            while (total < this.contentLength)
            {
                long remain = this.contentLength - total;
                long toRead = Math.min(remain, SEGMENT_SIZE);
                long read = source.read(sink.buffer(), toRead);
                if (read == -1L)
                {
                    break;
                }

                total += read;
                sink.flush();
                if (this.callback != null)
                {
                    this.callback.onProgress(total, this.contentLength);
                }
            }

            source.close();
            this.callback = null;
        }
    }

}
