package cn.gtmap.ruleengine.util;

import cn.gtmap.ruleengine.event.EventName;
import cn.gtmap.ruleengine.model.RuleVo;
import cn.gtmap.ruleengine.operator.OperatorName;
import cn.gtmap.ruleengine.service.RuleEngineService;
import com.ql.util.express.DefaultContext;
import com.ql.util.express.DynamicParamsUtil;
import com.ql.util.express.ExpressRunner;
import com.ql.util.express.Operator;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;

/**
 * 规则引擎执行器
 * @author 杨红杰
 * @date 2020-3-11
 */
@Slf4j
public class RuleExecutor {
    private RuleExecutor(){}

    private static final ExpressRunner RUNNER = new ExpressRunner();

    static {
        try {
            // 定义逻辑
            addLogic();
            //添加操作符
            addOperator(OperatorName.class.getPackage().getName());
            //添加事件
            addEvent(EventName.class.getPackage().getName());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * 定义逻辑
     * @throws Exception
     */
    private static void addLogic() throws Exception {
        RUNNER.addOperatorWithAlias("如果", "if", null);
        RUNNER.addOperatorWithAlias("则", "then", null);
        RUNNER.addOperatorWithAlias("否则", "else", null);
        RUNNER.addOperatorWithAlias("并且", "and", null);
        RUNNER.addOperatorWithAlias("或者", "or", null);
        
        RUNNER.addOperatorWithAlias("返回", "return", null);
    }

    /**
     * 添加操作符
     * @throws Exception
     */
    private static void addOperator(String operatorPackage) throws Exception {
        // 获取特定包下所有的类
        List<Class<?>> clsList = ClassUtil.getClasses(operatorPackage);
        if (!CollectionUtils.isEmpty(clsList)) {
            for (Class<?> cls : clsList) {
                // 选出包含注解@OperatorName的类
                if (cls.isAnnotationPresent(OperatorName.class)) {
                    OperatorName operatorName = cls.getAnnotation(OperatorName.class);
                    RUNNER.addOperator(operatorName.value(), (Operator) cls.newInstance());
                }
            }
        }
    }

    /**
     * 添加事件
     * @throws Exception
     */
    private static void addEvent(String eventPackage) throws Exception {
        // 获取特定包下所有的类
        List<Class<?>> clsList = ClassUtil.getClasses(eventPackage);
        if (!CollectionUtils.isEmpty(clsList)) {
            for (Class<?> cls : clsList) {
                // 选出包含注解@EventName的类
                if (cls.isAnnotationPresent(EventName.class)) {
                    EventName eventName = cls.getAnnotation(EventName.class);
                    RUNNER.addFunctionOfServiceMethod(eventName.value(), cls.newInstance(), "deal", new Class[]{ Object[].class }, null);
                }
            }
        }
    }

    /**
     * 添加自定义操作符、事件
     * @param basePackage   操作符、事件包路径
     * @throws Exception
     */
    public static void addCustomOperatorAndEvent(String basePackage) throws Exception {
        // 添加自定义操作符
        addOperator(basePackage);
        // 添加自定义事件
        addEvent(basePackage);
    }

    /**
     * 获取规则列表
     * @param ruleDriver    规则引擎名
     * @return
     */
    public static List<RuleVo> getRuleList(String ruleDriver) {
        RuleEngineService ruleEngineService = SpringUtil.getBean(RuleEngineService.class);
        return ruleEngineService.getRuleList(ruleDriver);
    }

    /**
     * 执行规则
     * @param express   规则表达式
     * @param map       参数集合
     * @return
     * @throws Exception
     */
    public static Object execute(String express, Map<String, Object>... map) throws Exception {
        DefaultContext<String, Object> context = new DefaultContext<String, Object>();
        if (map.length > 0 && map[0]!= null) {
            for (Map.Entry<String, Object> entry : map[0].entrySet()) {
                context.put(entry.getKey(), entry.getValue());
            }
        }
        //不定参数的使用
        DynamicParamsUtil.supportDynamicParams = true;
        return RUNNER.execute(express, context, null, false, false, null);
    }
}
