package cn.gtmap.realestate.supervise.portal.utils;

import cn.gtmap.realestate.supervise.entity.UserAuthDTO;
import cn.gtmap.realestate.supervise.portal.model.LoginUserCache;
import com.gtis.config.AppConfig;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;

import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;

/**
 * @author hqz
 * @version 1.0, 2017/6/8
 * @description 登录用户处理工具类
 */
public class LoginUserUtil {

    /**
     * 超时时间 application.properties 配置文件中的 session.time.out 配置项
     */
    private static Integer timeout = AppConfig.getIntProperty("session.time.out");

    /**
     * 根据cookie ID获取当前登录用户
     *
     * @param uuid cookie ID
     * @return 用户信息
     */
    public static UserAuthDTO getLoginUserMap(String uuid) {
        UserAuthDTO userAuthDTO = null;
        if (StringUtils.isNotBlank(uuid)) {
            LoginUserCache loginUserCache = LoginUserCache.getInstance();
            ConcurrentMap loginMap = loginUserCache.getLoginUser();
            if (loginMap.containsKey(uuid)) {
                userAuthDTO = (UserAuthDTO) loginMap.get(uuid);
                updateUserStatusDate(loginMap, uuid);
            }
        }
        return userAuthDTO;
    }

    /**
     * 登录成功后缓存当前登录用户
     *
     * @param uuid        cookieId
     * @param userAuthDTO 用户信息
     */
    public static void loginSuccessHandle(String uuid, UserAuthDTO userAuthDTO) {
        if (null != userAuthDTO && StringUtils.isNotBlank(uuid)) {
            userAuthDTO.setStatusDate(new Date());
            LoginUserCache loginUserCache = LoginUserCache.getInstance();
            ConcurrentMap loginMap = loginUserCache.getLoginUser();
            loginMap.put(uuid, userAuthDTO);
            loginUserCache.setLoginUser(loginMap);
            LoginUserCache.getInstance();
            updateUserStatusDate(loginMap, uuid);
        }
    }

    /**
     * 退出登录后处理事件
     *
     * @param uuid cookieId
     */
    public static void loginOutHandle(String uuid) {
        if (StringUtils.isNotBlank(uuid)) {
            LoginUserCache loginUserCache = LoginUserCache.getInstance();
            ConcurrentMap loginMap = loginUserCache.getLoginUser();
            if (loginMap.containsKey(uuid)) {
                loginMap.remove(uuid);
            }
        }
    }

    /**
     * 提出用户
     *
     * @param username
     */
    public static void eliminateUser(String username) {
        LoginUserCache loginUserCache = LoginUserCache.getInstance();
        ConcurrentMap loginMap = loginUserCache.getLoginUser();
        Iterator<ConcurrentMap.Entry> entries = loginMap.entrySet().iterator();
        while (entries.hasNext()) {
            Map.Entry entry = entries.next();
            if (StringUtils.equals(username, ((UserAuthDTO) entry.getValue()).getUsername())) {
                loginOutHandle((String) entry.getKey());
            }
        }
    }

    /**
     * 更新用户状态时间
     *
     * @param loginMap
     * @param uuid
     */
    public static boolean updateUserStatusDate(ConcurrentMap loginMap, String uuid) {
        boolean flag = false;
        DateTime statusDate = new DateTime(((UserAuthDTO) loginMap.get(uuid)).getStatusDate().getTime());
        DateTime now = new DateTime(new Date());
        statusDate = statusDate.plusMinutes(timeout);
        if (statusDate.isAfter(now)) {
            ((UserAuthDTO) loginMap.get(uuid)).setStatusDate(new Date());
        } else {
            loginOutHandle(uuid);
        }
        return flag;
    }

    /**
     * 取消用户密码过于简单
     *
     * @param username
     */
    public static void cleanUserPasswordTooSimple(String username) {
        LoginUserCache loginUserCache = LoginUserCache.getInstance();
        ConcurrentMap loginMap = loginUserCache.getLoginUser();
        Iterator<ConcurrentMap.Entry> entries = loginMap.entrySet().iterator();
        while (entries.hasNext()) {
            Map.Entry entry = entries.next();
            if (StringUtils.equals(username, ((UserAuthDTO) entry.getValue()).getUsername())) {
                ((UserAuthDTO) entry.getValue()).setPasswordTooSimple(false);
            }
        }
    }
}
