package cn.gtmap.realestate.supervise.utils;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.HashMap;
import java.util.Map;

/**
 * @author hqz
 * @version 1.0, 2017/10/09
 * @description token池
 */
public class TokenPool {
    //token对象（单例）
    private static TokenPool tokenPool = null;
    //token集合
    private List<TokenEntry> tokenPools = new ArrayList<TokenEntry>();
    //token加密的公钥
    private final static String publickey = "gtmap_wechat";
    //默认1天的有效期
    private final static int validMinute = 30;

    //redis 表名
    private final static String REDIS_TOKEN = "TOKEN";

    private static final Logger LOGGER = LoggerFactory.getLogger(TokenPool.class);

    private TokenPool() {
    }

    public static TokenPool getTokenPoolInstance() {
        if (null == tokenPool) {
            tokenPool = new TokenPool();
        }
        return tokenPool;
    }

    /**
     * 根据提供的用户ID加密生成token并返回
     *
     * @param userId 用户ID
     */
    public synchronized String getToken(String userId) {
        String token = null;
        if (StringUtils.isNotBlank(userId)) {
            token = validUserHasToken(userId);
            if (StringUtils.isNotBlank(token)) {
                return token;
            }
            Date now = new Date(System.currentTimeMillis());
            SimpleDateFormat sf = new SimpleDateFormat("yyyyMMddHHmmss");
            String strNow = sf.format(now);
            String content = userId + "|" + strNow;
            byte[] bytes = AESEncrypter.encrypt(content, publickey);
            token = byteArrayToHexStr(bytes);
            Jedis jedis = JedisPoolUtil.getJedis();
            Map<String, String> map = new HashMap<>();
            map.put("token_" + userId, token);
            map.put("time_" + userId, strNow);
            map.put("userId_" + userId, userId);
            jedis.hmset(REDIS_TOKEN, map);
            jedis.close();
        }
        return token;
    }


    /**
     * 验证token是否过期
     *
     * @param token
     * @return
     */
    public synchronized boolean validToken(String token) {
        boolean isflag = false;
        if (StringUtils.isNotBlank(token)) {
            Jedis jedis = JedisPoolUtil.getJedis();
            Map<String, String> maps = jedis.hgetAll(REDIS_TOKEN);
            if (maps.containsValue(token)) {
                String tokenKey = "";
                String timeKey = "";
                String userIdKey = "";
                for (Map.Entry<String, String> entry : maps.entrySet()) {
                    if (token.equals(entry.getValue()) && StringUtils.isBlank(tokenKey)) {
                        tokenKey = entry.getKey();
                        break;
                    }
                }
                timeKey = tokenKey.replace("token", "time");
                userIdKey = tokenKey.replace("token", "userId");
                String time = jedis.hget(REDIS_TOKEN, timeKey);
                Date oldDate = TimeUtils.strToDateTime(time);
                boolean isValid = validDate(oldDate);
                if (!isValid) {
                    jedis.hdel(REDIS_TOKEN, new String[]{userIdKey, timeKey, tokenKey});
                }
                isflag = isValid;
            }
            jedis.close();
        }
        return isflag;
    }

    /**
     * 验证是否过期
     *
     * @param initDate
     * @return
     */
    private boolean validDate(Date initDate) {
        Calendar cd = Calendar.getInstance();
        Calendar now = Calendar.getInstance();
        cd.setTime(initDate);
        cd.add(Calendar.MINUTE, validMinute);
        if (now.before(cd)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 验证用户是否存在有效的token
     *
     * @param userId
     * @return
     */
    private String validUserHasToken(String userId) {
        String token = null;
        if (StringUtils.isNotBlank(userId)) {
            Jedis jedis = JedisPoolUtil.getJedis();
            String userIdKey = "userId_" + userId;
            String timeKey = "time_" + userId;
            String tokenKey = "token_" + userId;
            String userIdStr = jedis.hget(REDIS_TOKEN, userIdKey);
            if (StringUtils.isNotEmpty(userIdStr)) {
                if (StringUtils.equals(userId, userIdStr)) {
                    String time = jedis.hget(REDIS_TOKEN, timeKey);
                    if (StringUtils.isNotEmpty(time)) {
                        Date oldDate = TimeUtils.strToDateTime(time);
                        boolean isValid = validDate(oldDate);
                        if (isValid) {
                            token = jedis.hget(REDIS_TOKEN, tokenKey);
                            Map<String, String> map = new HashMap<>();
                            map.put(timeKey, TimeUtils.dateToStr(new Date(System.currentTimeMillis())));
                            jedis.hmset(REDIS_TOKEN, map);//刷新生命周期
                        } else {
                            jedis.hdel(REDIS_TOKEN, new String[]{userIdKey, timeKey, tokenKey});
                        }
                    }
                }
            }
            jedis.close();
        }
        return token;
    }

    public static String byteArrayToHexStr(byte[] byteArray) {
        if (byteArray == null) {
            return null;
        }
        char[] hexArray = "0123456789ABCDEF".toCharArray();
        char[] hexChars = new char[byteArray.length * 2];
        for (int j = 0; j < byteArray.length; j++) {
            int v = byteArray[j] & 0xFF;
            hexChars[j * 2] = hexArray[v >>> 4];
            hexChars[j * 2 + 1] = hexArray[v & 0x0F];
        }
        return new String(hexChars);
    }

    public class TokenEntry {
        String token;
        Date initDate;
        String initUser;

        public String getToken() {
            return token;
        }

        public void setToken(String token) {
            this.token = token;
        }

        public Date getInitDate() {
            return initDate;
        }

        public void setInitDate(Date initDate) {
            this.initDate = initDate;
        }

        public String getInitUser() {
            return initUser;
        }

        public void setInitUser(String initUser) {
            this.initUser = initUser;
        }
    }

    /**
     * @param token 安全令牌
     * @return username 用户名称
     * @description 根据token获取用户名称
     * @author <a href="mailto:liuqiang@gtmap.cn">liuqiang</a>
     */
    public static String getUsernameByToken(String token) {
        String username = "";
        if (StringUtils.isNotBlank(token)) {
            Jedis jedis = JedisPoolUtil.getJedis();
            try {
                Map<String, String> maps = jedis.hgetAll(REDIS_TOKEN);
                if (maps.containsValue(token)) {
                    String tokenKey = "";
                    for (Map.Entry<String, String> entry : maps.entrySet()) {
                        if (token.equals(entry.getValue()) && StringUtils.isBlank(tokenKey)) {
                            tokenKey = entry.getKey();
                            break;
                        }
                    }
                    if (StringUtils.isNotBlank(tokenKey)) {
                        username = tokenKey.replace("token_", "");
                    }
                }
            } catch (Exception e) {
                LOGGER.error("");
            } finally {
                jedis.close();
            }
        }
        return username;
    }

}
