package cn.gtmap.realestate.supervise.utils;


import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.util.MultiValueMap;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.multipart.commons.CommonsMultipartResolver;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.*;

/**
 * @author <a href="mailto:huming@gtmap.cn">huming</a>
 * @description form-data 请求 xss处理
 */
public class XssAndSqlFormDataWrapper extends HttpServletRequestWrapper implements MultipartHttpServletRequest {

    private Map<String, String[]> params = new HashMap<>();

    private MultipartHttpServletRequest multReq;

    public XssAndSqlFormDataWrapper(HttpServletRequest request) {
        super(request);
        CommonsMultipartResolver resolver = new CommonsMultipartResolver();
        this.multReq = resolver.resolveMultipart(request);
        this.params.putAll(multReq.getParameterMap());
    }

    @Override
    public Map<String, MultipartFile> getFileMap() {
        return this.multReq.getFileMap();
    }

    @Override
    public MultiValueMap<String, MultipartFile> getMultiFileMap() {
        return this.multReq.getMultiFileMap();
    }

    @Override
    public String getMultipartContentType(String paramOrFileName) {
        return this.multReq.getMultipartContentType(paramOrFileName);
    }

    @Override
    public Iterator<String> getFileNames() {
        return this.multReq.getFileNames();
    }

    @Override
    public MultipartFile getFile(String name) {
        return this.multReq.getFile(name);
    }

    @Override
    public List<MultipartFile> getFiles(String name) {
        return this.multReq.getFiles(name);
    }

    @Override
    public String[] getParameterValues(String name) {
        // 如果文件参数，返回null或者空数组
        if (this.multReq.getFile(name) != null) {
            return null;
        }
        String a = null;
        try {
            a = new String(params.get(name)[0].getBytes("ISO-8859-1"), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        return new String[] {clearXss(a)};
    }

    @Override
    public Enumeration<String> getParameterNames() {
        // 合并普通参数和文件参数的名称
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.putAll(super.getParameterMap());
        paramMap.putAll(getFileMap());
        return Collections.enumeration(paramMap.keySet());
    }


    @Override
    public String getParameter(String name) {//重写getParameter，代表参数从当前类中的map获取
        String[] values = params.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }


    public void addParameter(String name, Object value) {//增加参数
        if (value != null) {
            if (value instanceof String[]) {
                params.put(name, (String[]) value);
            } else if (value instanceof String) {
                params.put(name, new String[]{(String) value});
            } else {
                params.put(name, new String[]{String.valueOf(value)});
            }
        }
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return this.multReq.getReader();
    }

    /**
     * 在使用@RequestBody注解的时候，其实框架是调用了getInputStream()方法，所以我们要重写这个方法
     *
     * @return
     * @throws IOException
     */
    @Override
    public ServletInputStream getInputStream() throws IOException {
        return this.multReq.getInputStream();
    }

    @Override
    public HttpMethod getRequestMethod() {
        return this.multReq.getRequestMethod();
    }

    @Override
    public HttpHeaders getRequestHeaders() {
        return this.multReq.getRequestHeaders();
    }

    @Override
    public HttpHeaders getMultipartHeaders(String paramOrFileName) {
        return this.multReq.getMultipartHeaders(paramOrFileName);
    }

    /**
     * @param value
     * @return java.lang.String
     * @Title: 特殊字符处理（转义或删除）
     * @methodName: clearXss
     * @Description:
     */
    private String clearXss(String value) {

        if (StringUtils.isEmpty(value)) {
            return value;
        }

        return XssFilterUtil.stripXss(value, this.getRequest());
    }

    /**
     * 将容易引起xss漏洞的半角字符直接替换成全角字符 在保证不删除数据的情况下保存
     *
     * @param value
     * @return 过滤后的值
     */
    private static String xssEncode(String value) {
        if (value == null || value.isEmpty()) {
            return value;
        }
        value = value.replaceAll("eval\\((.*)\\)", "");
        value = value.replaceAll("<","&lt;");
        value = value.replaceAll(">","&gt;");
        value = value.replaceAll("'","&apos;");
        value = value.replaceAll("[\\\"\\\'][\\s]*javascript:(.*)[\\\"\\\']", "\"\"");
        value = value.replaceAll("(?i)<script.*?>.*?<script.*?>", "");
        value = value.replaceAll("(?i)<script.*?>.*?</script.*?>", "");
        value = value.replaceAll("(?i)<.*?javascript:.*?>.*?</.*?>", "");
        value = value.replaceAll("(?i)<.*?\\s+on.*?>.*?</.*?>", "");
        return value;
    }

}