package cn.gtmap.realestate.supervise.utils;

import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import redis.clients.jedis.Jedis;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;

/**
 * @version 1.0 2019-1-7
 * @auther <a href="mailtoxutao@gtmap.cn">xutao</a>
 * @description
 */
public class OpenIdPool {

    // openId 单例
    private static OpenIdPool openIdPool = null;

    // openId 加密的公钥
    private final static String publickey = "gtmap_exchange_openId";

    // 默认60分钟的有效期
    private final static int validMinute = 60;

    // redis 表名
    private final static String REDIS_OPENID = "OPENID";

    private OpenIdPool() {
    }

    public static OpenIdPool getOpenIdPoolInstance() {
        if (null == openIdPool) {
            openIdPool = new OpenIdPool();
        }
        return openIdPool;
    }

    /**
     * 根据提供的用户ID加密生成openId并返回
     * 每次调用都产生一个新的operId
     *
     * @param userId 用户ID
     */
    public synchronized String getOpenId(String userId) {
        String openId = null;
        if (StringUtils.isNotBlank(userId)) {
            Date now = new Date(System.currentTimeMillis());
            SimpleDateFormat sf = new SimpleDateFormat("yyyyMMddHHmmss");
            String strNow = sf.format(now);
            String content = userId + "|" + strNow;
            byte[] bytes = AESEncrypter.encrypt(content, publickey);
            openId = byteArrayToHexStr(bytes);
            Jedis jedis = JedisPoolUtil.getJedis();
            Map<String, String> map = Maps.newHashMap();
            map.put("openId_" + userId, openId);
            map.put("time_" + userId, strNow);
            jedis.hmset(REDIS_OPENID, map);
            jedis.close();
        }
        return openId;
    }

    /**
     * 验证openId是否过期
     *
     * @param openId
     * @return
     */
    public synchronized boolean validOpenId(String openId) {
        boolean isflag = false;
        if (StringUtils.isNotBlank(openId)) {
            Jedis jedis = JedisPoolUtil.getJedis();
            Map<String, String> maps = jedis.hgetAll(REDIS_OPENID);
            if (maps.containsValue(openId)) {
                String openIdKey = "";
                for (Map.Entry<String, String> entry : maps.entrySet()) {
                    if (openId.equals(entry.getValue()) && StringUtils.isBlank(openIdKey)) {
                        openIdKey = entry.getKey();
                        break;
                    }
                }
                String timeKey = openIdKey.replace("openId", "time");
                String time = jedis.hget(REDIS_OPENID, timeKey);
                Date oldDate = TimeUtils.strToDateTime(time);
                boolean isValid = validDate(oldDate);
                if (!isValid) {
                    jedis.hdel(REDIS_OPENID, new String[]{timeKey, openIdKey});
                }
                isflag = isValid;
            }
            jedis.close();
        }
        return isflag;
    }

    /**
     * 当某一次请求已经结束，去掉这次的openId
     *
     * @param openId
     */
    public synchronized void removeOpenId(String openId) {
        if (StringUtils.isNotBlank(openId)) {
            Jedis jedis = JedisPoolUtil.getJedis();
            Map<String, String> maps = jedis.hgetAll(REDIS_OPENID);
            if (maps.containsValue(openId)) {
                String openIdKey = "";
                for (Map.Entry<String, String> entry : maps.entrySet()) {
                    if (openId.equals(entry.getValue()) && StringUtils.isBlank(openIdKey)) {
                        openIdKey = entry.getKey();
                        break;
                    }
                }
                String timeKey = openIdKey.replace("openId", "time");
                jedis.hdel(REDIS_OPENID, new String[]{timeKey, openIdKey});
            }
            jedis.close();
        }
    }

    /**
     * 验证是否过期
     *
     * @param initDate
     * @return
     */
    private boolean validDate(Date initDate) {
        Calendar cd = Calendar.getInstance();
        Calendar now = Calendar.getInstance();
        cd.setTime(initDate);
        cd.add(Calendar.MINUTE, validMinute);
        if (now.before(cd)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 加密后处理为16进制表示
     *
     * @param byteArray
     * @return
     */
    public static String byteArrayToHexStr(byte[] byteArray) {
        if (byteArray == null) {
            return null;
        }
        char[] hexArray = "0123456789ABCDEF".toCharArray();
        char[] hexChars = new char[byteArray.length * 2];
        for (int j = 0; j < byteArray.length; j++) {
            int v = byteArray[j] & 0xFF;
            hexChars[j * 2] = hexArray[v >>> 4];
            hexChars[j * 2 + 1] = hexArray[v & 0x0F];
        }
        return new String(hexChars);
    }
}
