package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.dao.SpatialDao;
import cn.gtmap.onemap.platform.dao.impl.ArcSDEDaoImpl;
import cn.gtmap.onemap.platform.dao.impl.OracleSpatialDaoImpl;
import cn.gtmap.onemap.platform.entity.Configuration;
import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.entity.dict.Dict;
import cn.gtmap.onemap.platform.entity.dict.Item;
import cn.gtmap.onemap.platform.event.GISDaoException;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.support.spring.ApplicationContextHelper;
import cn.gtmap.onemap.platform.utils.ArrayUtils;
import cn.gtmap.onemap.platform.utils.EnumUtils;
import cn.gtmap.onemap.platform.utils.Utils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.Polygon;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.collections.map.MultiKeyMap;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.geometry.jts.Geometries;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.crs.GeographicCRS;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.util.*;
import java.util.List;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-20 下午5:25
 */
public class GISServiceImpl extends BaseLogger implements GISService {

    private final String SE_SHAPE_FIELD = Constant.SE_SHAPE_FIELD;

    private final String SE_SHAPE_AREA = Constant.SE_SHAPE_AREA;

    private final String OG_PRO_PERFIX = "OG_PRO_";

    private final String SEARCH_LAYERS = "layers";

    private final String EXPORT_TYPE_GH = "gh";

    private final String EXPORT_TYPE_XZ = "xz";


    /**
     * 规划图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_E = "_E_";

    /**
     * 现状图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_H = "_H_";

    /**
     * 土地用途总体规划审查
     */
    private enum GHSC {

        TDYTQ("土地用途分区"),
        JSYDGZQ("建设用地管制区"),
        GHJBNTTZ("规划基本农田调整");
//        MZZDJSXM("重点建设项目");

        private String label;

        private GHSC(String value) {
            this.label = value;
        }

        private String getLabel() {
            return label;
        }
    }

    /**
     * 土地利用现状
     */
    private enum TDXZ {
        DLTB, XZDW
    }

    private enum Tag {
        YES, NO;
    }

    private SpatialDao spatialDao;

    @Autowired
    private GeometryService geometryService;
    @Autowired
    private AgsGeoemtryService agsGeoemtryService;

    @Autowired
    private MapService mapService;
    @Autowired
    private WebMapService webMapService;

    private Map searchConfig;

    private Map exportConfig;

    public void setSearchConfig(Resource path) {
        try {
            searchConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" search config can't found ");
        }
    }

    public void setExportConfig(Resource path) {
        try {
            exportConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" export config can't found ");
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
    }


    /**
     * 初始化
     */
    @Override
    public void initialize(Constant.SpatialType type) {
        switch (type) {
            case ARC_SDE:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(ArcSDEDaoImpl.class);
                break;
            case ORACLE_SPATIAL:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(OracleSpatialDaoImpl.class);
                break;
        }
    }

    /**
     * 属性查询
     *
     * @param layerName
     * @param where
     * @param columns
     * @param returnGeometry
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String where, String[] columns, boolean returnGeometry, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(where, getMessage("query.where.notnull"));
        List result = null;
        try{
            result = spatialDao.query(layerName, where, columns, returnGeometry, dataSource);
        }catch (Exception e){
            logger.info(getMessage("query.result.null"),e.getLocalizedMessage());
        }
        return result;
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param wkt
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String wkt, String[] columns, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(wkt, getMessage("query.geometry.notnull"));
        return spatialDao.query(layerName, wkt, columns, dataSource);
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param geometry
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, Geometry geometry, String[] columns, String dataSource) {
        return query(layerName, geometry.toText(), columns, dataSource);
    }

    /**
     * 空间查询(进行投影转换)
     *
     * @param layerName
     * @param feature
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, SimpleFeature feature, String[] columns, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null && !sourceCRS.equals(layerCRS))
            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        return query(layerName, geometry.toText(), columns, dataSource);
    }


    /**
     * 相交分析
     *
     * @param layerName
     * @param wktPlygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, String wktPlygon, String[] returnFields, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        return spatialDao.intersect(layerName, wktPlygon, returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param polygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Polygon polygon, String[] returnFields, String dataSource) {
        return intersect(layerName, polygon.toText(), returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, CoordinateReferenceSystem sourceCRS,
                             String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null) geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        List<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    result.addAll((Collection<? extends Map<String, Object>>)
                            intersect(layerName, geo.toText(), outFields, dataSource));
                    addGeoProperty2List(result, geo);
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            result.addAll((Collection<? extends Map<String, Object>>)
                    intersect(layerName, geometry.toText(), outFields, dataSource));
            addGeoProperty2List(result, geometry);
        }
        if (sourceCRS != null) {
            for (Map<String, Object> item : result) {
                Geometry geo = geometryService.readWKT((String) item.get(SE_SHAPE_FIELD));
                item.put(Constant.SE_SHAPE_FIELD, geometryService.project(geo, layerCRS, sourceCRS).toText());
            }
        }
        return result;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem sourceCRS = null;
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    List queryResults = query(layerName, geo, outFields, dataSource);


                    if (layerCRS instanceof GeographicCRS) {
                        if (isNull(sourceCRS) && queryResults.size() > 0)
                            sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        geo = geometryService.project(geo, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
                    }
                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            if(isNull(sourceCRS))
                                sourceCRS = layerCRS;
                            geometryResult=geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),geo.toText(),sourceCRS.toWKT()));
//                            geometryResult = geo1.intersection(geo);
                        } catch (Exception e) {
                            geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geo);
                            logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            List queryResults = query(layerName, geometry, outFields, dataSource);

            if (layerCRS instanceof GeographicCRS) {
                if (isNull(sourceCRS) && queryResults.size() > 0)
                    sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                geometry = geometryService.project(geometry, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                }
                Geometry geometryResult = null;
                try {
                    geometryResult=geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),geometry.toText(),sourceCRS.toWKT()));
                } catch (Exception e) {
                    logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                    geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geometry);
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                results.add(result);
            }
        }

        return results;
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersectByGeoJSON(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCRS = null;
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
            results = intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            results = addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            FeatureCollection collection = (FeatureCollection) geo;
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                results.addAll(addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature));
            }
        }
        return results;
    }


    /**
     *  改进的相交分析，为了减少土地利用现状分析的误差，需考虑将原始的分析区域的坐标保留，便于后续在平面参考下进行相切
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    public List<?> intersect4(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = bufferIntersect(layerName, geometry, layerCrs, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) bufferIntersect(layerName, geometry, sourceCrs, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) bufferIntersect(layerName, geometry, sourceCrs, outField, dataSource), feature));

            }
        }
        return results;
    }

    /**
     * 为了提高土地利用现状分析的精确度，将经纬度进行相交搜索的时候，做了buffer，扩大搜索范围
     * @param layerName
     * @param sourceGeometry
     * @param sourceCRS
     * @param outFields
     * @param dataSource
     * @return
     */
    public List<?> bufferIntersect(String layerName, Geometry sourceGeometry, CoordinateReferenceSystem sourceCRS, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (sourceGeometry instanceof GeometryCollection) {
            for (int i = 0; i < sourceGeometry.getNumGeometries(); i++) {
                try {
                    Geometry beforeGeo = sourceGeometry.getGeometryN(i);
                    Geometry afterGeometry=null;
                    if (!isNull(layerCRS))
                        afterGeometry = geometryService.project(beforeGeo, sourceCRS,layerCRS);
                    else
                        afterGeometry= beforeGeo;
                    Geometry tmpGeo =afterGeometry.buffer(0.002);
                    List queryResults = query(layerName, tmpGeo, outFields, dataSource);

                    CoordinateReferenceSystem tmpProjectedCRS = null;
                    if (sourceCRS instanceof GeographicCRS && layerCRS instanceof GeographicCRS) {

                        if (queryResults.size() > 0)
                            tmpProjectedCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        beforeGeo = geometryService.project(beforeGeo, layerCRS, isNull(tmpProjectedCRS) ? layerCRS : tmpProjectedCRS);
                    }

                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),beforeGeo.toText(),sourceCRS.toWKT()));
                        } catch (Exception e) {
                            logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                            geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(beforeGeo);
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            Geometry afterGeometry=null;
            if (!isNull(layerCRS))
                afterGeometry = geometryService.project(sourceGeometry, sourceCRS,layerCRS);
            else
                afterGeometry= sourceGeometry;
            Geometry tmpGeo =afterGeometry.buffer(0.002);
            List queryResults = query(layerName, tmpGeo, outFields, dataSource);

            CoordinateReferenceSystem tmpProjectedCRS = null;
            if (sourceCRS instanceof GeographicCRS && layerCRS instanceof GeographicCRS) {
                if (queryResults.size() > 0)
                    tmpProjectedCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                sourceGeometry = geometryService.project(sourceGeometry, layerCRS, isNull(tmpProjectedCRS) ? layerCRS : tmpProjectedCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS);
                }
                Geometry geometryResult = null;
                try {
                    geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),sourceGeometry.toText(),sourceCRS.toWKT()));
                } catch (Exception e) {
                    logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                    geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(sourceGeometry);
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                results.add(result);
            }
        }

        return results;
    }


    /**
     * @param layerName
     * @param feature
     * @param outField
     * @param dataSource
     * @return
     */
    private List<?> intersectBySimpleFeature(String layerName, SimpleFeature feature, String[] outField, String dataSource) {
        CoordinateReferenceSystem sourceCRS = null;
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (!geometry.isValid())
            geometry = createValidGeometry(geometry);
        setFeaturePros2Geo(feature, geometry);
        try {
            sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            return intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } catch (Exception e) {
            logger.error("intersect analysis - feature json crs not define [{}]", e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return GeoJSON 格式要素集 {"type":"FeatureCollection","features":[{"type":"Feature","geometry":{"type":"Polygon","coordinates":[[]]},"properties":{}}}
     */
    @Override
    public List<?> intersect2(String layerName, String geoJson, String[] outField, String dataSource) {
        List results = intersectByGeoJSON(layerName, geoJson, outField, dataSource);
//        FeatureCollection collection = geometryService.list2FeatureCollection(results, null, null);
        return results;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect3(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = intersect(layerName, geometry, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            if (!isNull(sourceCrs) && !isNull(layerCrs))
                geometry = geometryService.project(geometry, sourceCrs, layerCrs);
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                if (!isNull(sourceCrs) && !isNull(layerCrs))
                    geometry = geometryService.project(geometry, sourceCrs, layerCrs);
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature));
//                results.addAll(intersect(layerName, geometry, outField, dataSource));
            }
        }
        return results;
    }

    /**
     * 创建新的有效的geometry
     *
     * @param geometry
     * @return
     */
    private Geometry createValidGeometry(Geometry geometry) {
        try {
            if (geometry instanceof Polygon) {
                Polygon polygon = (Polygon) geometry;
                GeometryFactory factory = geometry.getFactory();
                List<Polygon> polygons = new ArrayList<Polygon>();
                Polygon exteriorPolygon = new Polygon((LinearRing) polygon.getExteriorRing(), null, factory);
                polygons.add(exteriorPolygon);
                for (int i = 0; i < polygon.getNumInteriorRing(); i++) {
                    LinearRing interiorRing = (LinearRing) polygon.getInteriorRingN(i);
                    Polygon interiorPolygon = new Polygon(interiorRing, null, factory);
                    polygons.add(interiorPolygon);
                }
                List<Polygon> newPolygons = new ArrayList<Polygon>();
                List<Polygon> excludePolygons = new ArrayList<Polygon>();
                for (Polygon temp : polygons) {
                    temp = getPolygonWithHoles(polygons, excludePolygons, temp);
                    if (!excludePolygons.contains(temp))
                        newPolygons.add(temp);
                }
                return new MultiPolygon(newPolygons.toArray(new Polygon[0]), factory);
            } else if (geometry instanceof MultiPolygon) {
                MultiPolygon multiPolygon = (MultiPolygon) geometry;
                List<Polygon> polygonList = new ArrayList<Polygon>();
                for (int j = 0; j < multiPolygon.getNumGeometries(); j++) {

                    Polygon polygon = (Polygon) multiPolygon.getGeometryN(j);
                    if (!polygon.isValid()) {
                        MultiPolygon tempMultiPolygon = (MultiPolygon) createValidGeometry(polygon);
                        for (int k = 0; k < tempMultiPolygon.getNumGeometries(); k++) {
                            polygonList.add((Polygon) tempMultiPolygon.getGeometryN(k));
                        }
                    } else {
                        polygonList.add(polygon);

                    }
                }
                return new MultiPolygon(polygonList.toArray(new Polygon[0]), multiPolygon.getFactory());
            } else {
                logger.info("geometryType has not been supported yet");
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 判断一个polygon在polygon组中是否是含有环的polygon
     *
     * @param srcPolygons     原始polygon组
     * @param excludePolygons 内环polygon需排除
     * @param polygon
     * @return
     */
    private Polygon getPolygonWithHoles(List<Polygon> srcPolygons, List<Polygon> excludePolygons, Polygon polygon) {
        List<LinearRing> holes = new ArrayList<LinearRing>();
        for (Polygon item : srcPolygons) {
            if (polygon.equals(item) || excludePolygons.contains(polygon))
                continue;
            if (polygon.contains(item)) {
                holes.add((LinearRing) item.getExteriorRing());
                excludePolygons.add(item);
            }
        }
        if (holes.size() > 0)
            return new Polygon((LinearRing) polygon.getExteriorRing(), holes.toArray(new LinearRing[0]), polygon.getFactory());
        else
            return polygon;
    }


    /**
     * @param value
     * @param feature
     * @return
     */
    private List addFeaturePros2List(List<Map> value, SimpleFeature feature) {
        for (Map item : value) {
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                item.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }
        }
        return value;
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param columns    包含图形、属性数据
     * @param dataSource
     * @return
     */
    @Override
    public String insert(String layerName, Map<String, Object> columns, String dataSource) {
        LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
        if (!isNull(layerRegion.getRegionField())) {
            boolean contains = false;
            for (Map.Entry entry : columns.entrySet()) {
                if (layerRegion.getRegionField().equalsIgnoreCase((String) entry.getKey())
                        && geometryService.containsRegionValue(String.valueOf(entry.getValue()))) {
                    contains = true;
                    break;
                }
            }
            if (!contains)
                throw new RuntimeException(getMessage("insert.region.field.not.null", layerName, layerRegion.getRegionField()));
        }
        return spatialDao.insert(layerName, columns, dataSource);
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param geoJSON    GeoJSON 格式
     * @param dataSource
     * @return primaryKey
     */
    @Override
    public String insert(String layerName, String geoJSON, String dataSource){
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        try{
            if (geo instanceof FeatureCollection) {
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    columns = geometryService.simpleFeature2Map(feature); //转map
                    for (Map.Entry<String, Object> entry : columns.entrySet()) {
                        if (entry.getValue() instanceof Geometry) {
                            Geometry geometry = (Geometry) entry.getValue();
                            if (layerCRS != null && sourceCRS != null)
                                geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                            columns.put(SE_SHAPE_FIELD, geometry.toText());
                            columns.remove(entry.getKey());
                        }
                    }
                    if (featureIterator.hasNext())
                        insert(layerName, columns, dataSource);
                    else
                        return insert(layerName, columns, dataSource);
                }
            }
        }catch (Exception e){
            logger.info(getMessage("insert.sde.false"),e.getLocalizedMessage());
        }
        return insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 插入数据（返回geojson）
     *
     * @param layerName
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
            resultFc.put("type", "FeatureCollection");
            List<LinkedHashMap> resultFeatures = new ArrayList();
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                properties.put("OBJECTID", objectId);
                resultFeatures.add(featureMap);
            }
            resultFc.put("features", resultFeatures);
            return JSON.toJSONString(resultFc);

        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
            JSONObject properties = (JSONObject) featureMap.get("properties");
            String objectId = insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
            properties.put("OBJECTID", objectId);

            return JSON.toJSONString(featureMap);
        }
        return null;

    }

    /**
     * @param layerName
     * @param geoJSON
     * @param check      是否执行拓扑检查
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, boolean check, String dataSource) {

        if (check) {
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
            if (geo instanceof FeatureCollection) {

                LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
                resultFc.put("type", "FeatureCollection");
                List<LinkedHashMap> resultFeatures = new ArrayList();
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    List list = query(layerName, feature, null, dataSource);
//                    List list = query(layerName, feature.getAttribute("geometry").toString(), null, dataSource);
                    if (list.size() > 0) {
                        logger.error("图形已在此图层中存在,插入失败!【{}】", geometryService.toFeatureJSON(feature).toString());
                        if (featureIterator.hasNext())
                            continue;
                        else
                            throw new GISDaoException(GISDaoException.Method.INSERT, "图形已在此图层中存在,插入失败!", GISDaoException.Type.ARC_SDE);
                    }
                    CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    columns = geometryService.simpleFeature2Map(feature);
                    try {
                        for (String key : columns.keySet()) {
                            if (columns.get(key) instanceof Geometry) {
                                Geometry geometry = (Geometry) columns.get(key);
                                if (layerCRS != null && sourceCRS != null)
                                    geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                                columns.put(SE_SHAPE_FIELD, geometry.toText());
                                columns.remove(key);
                                break;
                            }
                        }
                    } catch (Exception e) {
                        logger.error(e.getLocalizedMessage());
                        throw new RuntimeException(e.getLocalizedMessage());
                    }
                    String objectId = insert(layerName, columns, dataSource);
                    LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                    JSONObject properties = (JSONObject) featureMap.get("properties");
                    properties.put("OBJECTID", objectId);
                    resultFeatures.add(featureMap);
                }
                resultFc.put("features", resultFeatures);
                return JSON.toJSONString(resultFc);

            } else if (geo instanceof SimpleFeature) {

                SimpleFeature feature = (SimpleFeature) geo;
                List list = query(layerName, feature, null, dataSource);
//                List list = query(layerName, feature.getAttribute("geometry").toString(), null, dataSource);
                if (list.size() > 0) {
                    logger.error("图形已在此图层中存在,插入失败!【{}】", geoJSON);
                    throw new GISDaoException(GISDaoException.Method.INSERT, "图形已在此图层中存在,插入失败!", GISDaoException.Type.ARC_SDE);
                }
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                String objectId = insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
                properties.put("OBJECTID", objectId);

                return JSON.toJSONString(featureMap);
            }

        } else {
            return insert2(layerName, geoJSON, dataSource);
        }


        return null;
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, Map<String, Object> columns, String dataSource) {
        return spatialDao.update(layerName, primaryKey, columns, dataSource);
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        return update(layerName, primaryKey, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 删除
     *
     * @param layerName
     * @param primaryKey
     * @param dataSource
     * @return
     */
    @Override
    public boolean delete(String layerName, String primaryKey, String dataSource) {
        return spatialDao.delete(layerName, primaryKey, dataSource);
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry      GeoJSON format
     * @param outFields
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String dltbLayerName, String xzdwLayerName, String geometry, String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        Map<String, Double> analysisResult = new HashMap<String, Double>();
        Map<String, Double> analysisTemp = new HashMap<String, Double>();
        /**
         * 地类图斑相交
         */
        String[] dltbOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "ZLDWDM", "TBBH"});
        List<Map<String, Object>> results = (List<Map<String, Object>>) intersectByGeoJSON(dltbLayerName, geometry, dltbOutFields, dataSource);
        for (Map<String, Object> result : results) {
            Double area = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (area > 0.4) {
                String dlmc = String.valueOf(result.get("DLMC"));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + area);
                analysisTemp.put(String.valueOf(result.get("ZLDWDM")) + String.valueOf(result.get("TBBH")), area);
            }
        }
        /**
         * 现状地物相交并扣除
         */
        String[] xzdwOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "KD", "KCTBDWDM2", "KCTBDWDM1", "KCTBBH1", "KCTBBH2"});
        List<Map<String, Object>> xzResults = (List<Map<String, Object>>) intersectByGeoJSON(xzdwLayerName, geometry, xzdwOutFields, dataSource);
        for (Map<String, Object> result : xzResults) {
            Double length = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (length > 0.08) {
                String dlmc = String.valueOf(result.get("DLMC"));
                Double kd = Double.parseDouble(String.valueOf(result.get("KD")));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + length * kd);
                //
                String where = "ZLDWDM='" + result.get("KCTBDWDM1") + "' and TBBH='" + result.get("KCTBBH1") + "'";
                List<Map<String, Object>> temps = new ArrayList<Map<String, Object>>();
                try {
                    temps = (List<Map<String, Object>>) query(dltbLayerName, where,
                            ArrayUtils.add2Arrays(outFields, new String[]{"DLMC"}), true, dataSource);
                } catch (Exception e) {
                    logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                }
                String kctb1Key = String.valueOf(result.get("KCTBDWDM1")) + String.valueOf(result.get("KCTBBH1"));
                //
                if (StringUtils.isBlank((String) result.get("KCTBDWDM2"))) {
                    for (Map<String, Object> temp : temps) {
                        String tmpDlmc = String.valueOf(temp.get("DLMC"));
                        analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - length * kd);
                        analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - length * kd);
                    }
                } else {
                    //
                    Tag tag = Tag.NO;
                    String tmpDlmc = "";
                    String tmpDlmc1 = "";
                    Double tmpXzdwKd = 0.0;
                    String tmpXzdwKctbdwdm1 = "";
                    String tmpXzdwKctbbh1 = "";
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb1Key)) {
                            Double tmpValue = analysisTemp.get(kctb1Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                tmpXzdwKd = (length / 2.0) * kd;
                                tmpDlmc1 = tmpDlmc;
                                tmpXzdwKctbbh1 = String.valueOf(result.get("KCTBBH1"));
                                tmpXzdwKctbdwdm1 = String.valueOf(result.get("KCTBDWDM1"));
                                analysisTemp.put(tmpXzdwKctbdwdm1 + tmpXzdwKctbbh1, tmpValue - (length / 2.0) * kd);
                            } else
                                tag = Tag.YES;
                        } else
                            tag = Tag.YES;
                    }
                    //
                    where = "ZLDWDM='" + result.get("KCTBDWDM2") + "' and TBBH='" + result.get("KCTBBH2") + "'";
                    try {
                        temps = (List<Map<String, Object>>) query(dltbLayerName, where, new String[]{"DLMC"}, true, dataSource);
                    } catch (Exception e) {
                        logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        temps.clear();
                    }
                    String kctb2Key = String.valueOf(result.get("KCTBDWDM2")) + String.valueOf(result.get("KCTBBH2"));
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb2Key)) {
                            Double tmpValue = analysisTemp.get(kctb2Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                if (!Tag.YES.equals(tag)) {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                } else {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                }
                            } else {
                                tmpDlmc = tmpDlmc1;
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                                analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                            }
                        } else {
                            tmpDlmc = tmpDlmc1;
                            analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                            analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                        }
                    }
                }
            }
        }
        return analysisResult;
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param regionCode
     * @param geometry   GeoJSON format
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String regionCode, String geometry, String dataSource) {
        regionCode = Utils.formatRegionCode(regionCode);
        String dltb = TDXZ.DLTB.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        String xzdw = TDXZ.XZDW.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        return tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
    }

    @Override
    public Map tdlyxzAnalysis2(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        if (StringUtils.isBlank(dltbLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "地类图斑图层名称为空！"));
        if (StringUtils.isBlank(xzdwLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "线状地物图层名称为空！"));
        if (StringUtils.isBlank(geometry))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "分析地块坐标为空！"));
//        Double geometryArea = geometryService.readGeometryAera(geometry);
//        if (geometryArea > 200000 || geometryArea < 100) {
//            throw new RuntimeException("分析地块面积超过20公顷或者小于100平米！");
//        }
        Map<String, Double> dlMap = new HashMap<String, Double>();   //分析地类面积结果
        Map<String, Double> bhMap = new HashMap<String, Double>();
        MultiKeyMap analysisDetailMap = MultiKeyMap.decorate(new HashedMap()); //分析地类面积明细
        String[] columns = {"DLMC"};
        List<Map<String, Object>> xzdwDltbResult = new ArrayList<Map<String, Object>>();
        List<Map<String, Object>> dltbAnalysisResult = new ArrayList<Map<String, Object>>();
        List<Map<String, Object>> xzdwAnalysisResult = new ArrayList<Map<String, Object>>();
        //与地类图斑层进行相交分析
        try {
//            dltbAnalysisResult = (List<Map<String, Object>>) intersectByGeoJSON(dltbLayerName, geometry, "DLMC;ZLDWDM;TBBH;ZLDWMC;QSXZ;DLBM;TBMJ".split(";"), dataSource);
            dltbAnalysisResult = (List<Map<String, Object>>) intersect3(dltbLayerName, geometry, "DLMC;ZLDWDM;TBBH;ZLDWMC;QSXZ;DLBM;TBMJ;QSDWDM;QSDWMC".split(";"), dataSource);
        } catch (Exception e) {
            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
        }

        if (dltbAnalysisResult.size() > 0) {
            for (int i = 0; i < dltbAnalysisResult.size(); i++) {
                Double tmpArea = Double.parseDouble(String.valueOf(dltbAnalysisResult.get(i).get("SHAPE_AREA"))); //相交部分的面积
                if (tmpArea > 0.4) {
                    if (dlMap.containsKey(dltbAnalysisResult.get(i).get("DLMC")))
                        dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), dlMap.get(dltbAnalysisResult.get(i).get("DLMC")) + tmpArea);
                    else
                        dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), tmpArea);
                    bhMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")) + String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), tmpArea);

                    //图斑明细
                    if (analysisDetailMap.containsKey(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")))) {
                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpArea);
                    } else {
                        Map<String, Object> detailValue = new HashMap<String, Object>();
                        detailValue.put("ZLDWDM", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")));
                        detailValue.put("ZLDWMC", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWMC")));
                        detailValue.put("TBBH", String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                        detailValue.put("QSXZ", String.valueOf(dltbAnalysisResult.get(i).get("QSXZ")));
                        detailValue.put("DLMC", String.valueOf(dltbAnalysisResult.get(i).get("DLMC")));
                        detailValue.put("DLBM", String.valueOf(dltbAnalysisResult.get(i).get("DLBM")));
                        detailValue.put("TBMJ", String.valueOf(dltbAnalysisResult.get(i).get("TBMJ")));
                        detailValue.put("QSDWDM", String.valueOf(dltbAnalysisResult.get(i).get("QSDWDM")));
                        detailValue.put("QSDWMC", String.valueOf(dltbAnalysisResult.get(i).get("QSDWMC")));
                        detailValue.put("CCMJ", tmpArea);
                        analysisDetailMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), detailValue);
                    }

                }

            }
        }

        //与线状地物层进行相交分析，并扣除
        try {
            xzdwAnalysisResult = (List<Map<String, Object>>) intersect3(xzdwLayerName, geometry, "DLMC;KD;KCTBDWDM2;KCTBDWDM1;KCTBBH1;KCTBBH2;QSDWDM1;XZDWBH;QSXZ;DLBM;XZDWMJ;QSDWMC1".split(";"), dataSource);
        } catch (Exception e) {
            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
        }

        if (xzdwAnalysisResult.size() > 0) {
            for (int i = 0; i < xzdwAnalysisResult.size(); i++) {
                Double tmpLength = Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("SHAPE_AREA")));
                if (tmpLength > 0.08) {
                    if (dlMap.containsKey(xzdwAnalysisResult.get(i).get("DLMC"))) {
                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) + (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                    } else
                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                    //线物明细
                    if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))) {
                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                    } else {
                        Map<String, Object> detailValue = new HashMap<String, Object>();
                        detailValue.put("ZLDWDM", String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH"))));
                        detailValue.put("ZLDWMC", String.valueOf(xzdwAnalysisResult.get(i).get("")));
                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                        detailValue.put("QSXZ", String.valueOf(xzdwAnalysisResult.get(i).get("QSXZ")));
                        detailValue.put("DLMC", String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")));
                        detailValue.put("DLBM", String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")));
                        detailValue.put("TBMJ", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWMJ")));
                        detailValue.put("QSDWDM1", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")));
                        detailValue.put("QSDWMC1", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWMC1")));
                        detailValue.put("CCMJ", tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                        analysisDetailMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), detailValue);
                    }
                    if (StringUtils.isBlank(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2"))) || String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")).equalsIgnoreCase("null")) {
                        String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {
                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        String tmpKey = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                        if (xzdwDltbResult.size() > 0&&bhMap.containsKey(tmpKey)) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                dlMap.put(String.valueOf(xzdwDltbResult.get(j).get("DLMC")), dlMap.get(String.valueOf(xzdwDltbResult.get(j).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
//                                String tmpKey = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                bhMap.put(tmpKey, bhMap.get(tmpKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                //扣除线物明细
                                if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                } else {
                                    logger.error((getMessage("analysis.tdlyxz.kc.error", "ERROR", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))))));
//                                    throw new RuntimeException("扣除线物明细异常，扣除图斑单位代码1:[ " + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + " ]");
//                                    Map<String, Object> detailValue = new HashMap<String, Object>();
//                                    detailValue.put("ZLDWDM", "error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                    analysisDetailMap.put("error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                }
                            }
                        }
                    } else {
                        //记录扣除状态
                        String kouchuTag = "no";
                        //记录扣除面积
                        double kouchuMj = 0.0;
                        //记录扣除地类
                        String kouchuDlmc = "";
                        //扣除ZLDWDM
                        String kouchuZldwdm = "";
                        //扣除tbbh
                        String kouchuTbbh = "";
                        String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {

                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                    Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        kouchuMj = (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")));
                                        kouchuDlmc = tmpDlmcKey;
                                        kouchuZldwdm = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1"));
                                        kouchuTbbh = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                        bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        //扣除线物明细
                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);
                                    } else {
                                        kouchuTag = "yes";
//                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积过小", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                        throw new RuntimeException("扣除线物明细异常:[面积过小，扣除图斑单位代码1:[ " + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + " ]]");
                                        Map<String, Object> detailValue = new HashMap<String, Object>();
                                        detailValue.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("TBMJ", tmpValue);
                                        detailValue.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                    }
                                } else {
                                    kouchuTag = "yes";
                                     //RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有占用", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                    logger.error(getMessage("analysis.tdlyxz.kc.error", "没有占用", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
//                                    Map<String, Object> detailValue = new HashMap<String, Object>();
//                                    detailValue.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                    analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                }
                            }
                            if (xzdwDltbResult.size() == 0) {
                                kouchuTag = "yes";
//                                throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有找到", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                logger.error(getMessage("analysis.tdlyxz.kc.error", "没有找到", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                        String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                        , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
//                                Map<String, Object> detailValue = new HashMap<String, Object>();
//                                detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                            }
                        } else {
                            kouchuTag = "yes";
//                            throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有找到", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                            logger.error(getMessage("analysis.tdlyxz.kc.error", "没有找到", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                    String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                    , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
//                            Map<String, Object> detailValue = new HashMap<String, Object>();
//                            detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                            detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                            analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                        }

                        tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM2") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH2") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {
                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")))) {
                                    Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                    if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                        if (!kouchuTag.equals("yes")) {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        } else {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        }
                                    } else {
                                        try{
                                            tmpDlmcKey = kouchuDlmc;
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                        }catch (Exception ex){
                                            throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积扣除错误，出现无法扣除情况，请检查，请用实地勘测坐标叠加到图形上！!", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                    String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                    , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        }

                                        bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);

                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

//                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积过小", 2, xzdwAnalysisResult.get(i).get("KCTBDWDM2")));

                                        Map<String, Object> detailValue1 = new HashMap<String, Object>();
                                        detailValue1.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                        detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                        detailValue1.put("TBMJ", tmpValue);
                                        detailValue1.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);

                                    }
                                } else {
                                    try{
                                        tmpDlmcKey = kouchuDlmc;
                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                    }catch (Exception ex){
                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "土地利用现状底图中现状地物扣除有误，请检查!", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                    }

                                    bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);

                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

//                                    throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有占用", 2, xzdwAnalysisResult.get(i).get("KCTBDWDM2")));

                                    logger.error(getMessage("analysis.tdlyxz.kc.error", "没有占用", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

//                                    Map<String, Object> detailValue1 = new HashMap<String, Object>();
//                                    detailValue1.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
//                                    detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
//                                    analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);
                                }
                            }
                        }
                    }

                }
            }
        }
        Map resultMap = new HashMap();
        resultMap.put("analysisArea", dlMap);
        List resultDetailList = new ArrayList();
        Iterator keyIterator = analysisDetailMap.keySet().iterator();
        while (keyIterator.hasNext()) {
            Object tmpKey = keyIterator.next();
            resultDetailList.add(analysisDetailMap.get(tmpKey));
        }
        resultMap.put("analysisAreaDetail", resultDetailList);
//        return JSON.toJSONString(resultMap);
        return resultMap;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> tdghscAnalysis(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        if (StringUtils.isBlank(layerType))
            throw new RuntimeException(getMessage("layer.type.notnull"));
        String layerName = layerType.concat(LAYER_MIDDLE_FIX_E).concat(Utils.formatRegionCode(regionCode));
        return intersect3(layerName, geometry, outFields, dataSource);
//        return intersectByGeoJSON(layerName, geometry, outFields, dataSource);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public String tdghscAnalysis2(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        List result = tdghscAnalysis(layerType, regionCode, geometry, outFields, dataSource);
        FeatureCollection collection = geometryService.list2FeatureCollection(result, null, null);
        return geometryService.toFeatureJSON(collection);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis2(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis2(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 组织土地规划分析的结果（所有图层类型）
     *
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(Map analysisResult) {

        Map resultMap = new HashMap();
        try {
            for (GHSC item : GHSC.values()) {
                List list = new ArrayList();
//                List detailList = new ArrayList();
                Map total = new HashMap();
                Map ghMap = new HashMap();
                double totalArea = 0;
                JSONObject singleObj = JSON.parseObject(analysisResult.get(item.getLabel()).toString());
                JSONArray features = JSON.parseArray(singleObj.get("features").toString());
                totalArea = getTotalArea(features);
                total.put("LXMC", "地块总面积");
                total.put("AREA", totalArea);
                list.add(total);
                switch (item.ordinal()) {
                    case 0:
                        for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name() + area);
                        }
                        break;
                    case 1:
                        for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name() + area);
                        }
                        break;
                    case 2:
                        for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("TZLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name());
                        }
                        break;
                    case 3:
                        for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);

                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("XMLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.getLxdm());
                        }
                        break;
                }
//                ghMap.put("detail", detailList);
                ghMap.put("info", list);
                resultMap.put(item.getLabel(), ghMap);
            }
        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());

        }

        return resultMap;
    }

    /**
     * 组织土地规划分析的结果（单个图层类型）
     *
     * @param layerType
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(String layerType, String analysisResult) {
        Map resultMap = new HashMap();
        try {

            JSONObject fc = JSON.parseObject(analysisResult);
            JSONArray features = (JSONArray) fc.get("features");

            List list = new ArrayList();
            Map total = new HashMap();
            Map ghMap = new HashMap();
            double totalArea = 0;
            totalArea = getTotalArea(features);
            total.put("LXMC", "地块总面积");
            total.put("AREA", totalArea);
            list.add(total);
            for (GHSC item : GHSC.values()) {

                if (item.name().equals(layerType)) {
                    switch (item.ordinal()) {
                        case 0:
                            for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    Map detailMap = new HashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name() + area);
                            }
                            break;
                        case 1:
                            for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
//                                        detailMap.put("ghsc", item.getLabel());
                                        detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name() + area);
                            }
                            break;
                        case 2:
                            for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TZLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name());
                            }
                            break;
                        case 3:
                            for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);

                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("XMLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.getLxdm());
                            }
                            break;
                    }
                    ghMap.put("info", list);
                    resultMap.put(item.getLabel(), ghMap);
                }
            }

        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultMap;
    }

    @Override
    public LinkedHashMap tdghscExcelData(Map result) {
        LinkedHashMap map = new LinkedHashMap();
        for (Object key : result.keySet()) {
            List list = new ArrayList();
            List keyList = (List) ((Map) result.get(key)).get("info");
            for (int i = 1; i < keyList.size(); i++) {
                Map temp = (Map) keyList.get(i);
                List tempList = new ArrayList();
                tempList.add(temp.get("LXMC"));
                tempList.add(Math.round(Double.valueOf(temp.get("AREA").toString())));
                list.add(tempList);
            }
            map.put(key, list);
        }
        return map;
    }

    /**
     * 组织土地利用现状分析的结果
     *
     * @param analysisResult
     * @return
     */
    @Override
    public List<Map> tdlyxzResult(Map analysisResult, String tpl) {
        List<Map> resultList = new ArrayList<Map>();
        try {
            List detailList = (List) analysisResult.get("analysisAreaDetail");
            List<Map> dwdmList = new ArrayList<Map>();
            for (int i = 0; i < detailList.size(); i++) {
                Map temp = new HashMap();
                List list = new ArrayList();
                int tag = 0;

                Map map = (Map) detailList.get(i);
                String xzqdm = map.get("ZLDWDM").toString().substring(0, 6);
                list.add(map);

                temp.put("xzqdm", xzqdm);
                temp.put("detail", list);

                if (dwdmList.size() > 0) {
                    for (Map dwdmMap : dwdmList) {
                        if (dwdmMap.get("xzqdm").toString().equals(xzqdm)) {
                            List tempList = (List) dwdmMap.get("detail");
                            tempList.add(map);
                            dwdmMap.put("detail", tempList);
                            tag = 1;
                        }
                    }
                }
                if (tag == 0) {
                    dwdmList.add(temp);
                }
            }

            for (int j = 0; j < dwdmList.size(); j++) {
                List<Map> groupList = getTdlyxzDictList(tpl);
                Map temp = new HashMap();
                Map map = dwdmList.get(j);
                List<Map> list = (List) map.get("detail");
                for (Map detailMap : list) {
                    String dlbm = detailMap.get("DLBM").toString();
                    double area = Double.valueOf(detailMap.get("CCMJ").toString());
                    for (Map groupMap : groupList) {
                        List<Map> childList = new ArrayList<Map>();
                        if (groupMap.containsKey("detail"))
                            childList = (List<Map>) groupMap.get("detail");
                        if (groupMap.get("dlbm").toString().equals(dlbm.substring(0, 2))) {
                            double newArea = Double.valueOf(groupMap.get("area").toString()) + area;
                            groupMap.put("area", newArea);
                            childList = add2List(childList, detailMap);
                            groupMap.put("detail", childList);
                        }
                    }
                }

                temp.put("xzqdm", map.get("xzqdm"));
                temp.put("sumArea", getSumArea(groupList));
                temp.put("categoryA", groupList);
                temp.put("categoryB", getCategory(groupList));

                resultList.add(temp);
            }

        } catch (Exception e) {
             logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultList;
    }

    @Override
    public LinkedHashMap tdlyxzExcelData(List<Map> result) {
        LinkedHashMap map = new LinkedHashMap();
        if(result.size()>0)
        {
            List list1 = new ArrayList();
            List list2 = new ArrayList();
            map.put("一级分类", list1);
            map.put("二级分类", list2);
            for (Map item : result) {
                List temp1 = new ArrayList();
                List temp2 = new ArrayList();
                temp1.add(item.get("xzqdm"));
                temp2.add(item.get("xzqdm"));
                temp2.add(item.get("sumArea"));
                List<Map> categoryA = (List<Map>) item.get("categoryA");
                Map categoryB = (Map) item.get("categoryB");
                for (Map tempMap : categoryA) {
                    temp1.add(tempMap.get("area"));
                }
                temp2.add(isNull(categoryB.get("01")) ? 0 : categoryB.get("01"));
                temp2.add(isNull(categoryB.get("011")) ? 0 : categoryB.get("011"));
                temp2.add(isNull(categoryB.get("012")) ? 0 : categoryB.get("012"));
                temp2.add(isNull(categoryB.get("013")) ? 0 : categoryB.get("013"));
                temp2.add(isNull(categoryB.get("02")) ? 0 : categoryB.get("02"));
                temp2.add(isNull(categoryB.get("021")) ? 0 : categoryB.get("021"));
                temp2.add(isNull(categoryB.get("022")) ? 0 : categoryB.get("022"));
                temp2.add(isNull(categoryB.get("023")) ? 0 : categoryB.get("023"));
                temp2.add(isNull(categoryB.get("03")) ? 0 : categoryB.get("03"));
                temp2.add(isNull(categoryB.get("031")) ? 0 : categoryB.get("031"));
                temp2.add(isNull(categoryB.get("032")) ? 0 : categoryB.get("032"));
                temp2.add(isNull(categoryB.get("033")) ? 0 : categoryB.get("033"));
                temp2.add(isNull(categoryB.get("04")) ? 0 : categoryB.get("04"));
                temp2.add(isNull(categoryB.get("041")) ? 0 : categoryB.get("041"));
                temp2.add(isNull(categoryB.get("042")) ? 0 : categoryB.get("042"));
                temp2.add(isNull(categoryB.get("043")) ? 0 : categoryB.get("043"));
                temp2.add(isNull(categoryB.get("20")) ? 0 : categoryB.get("20"));
                temp2.add(isNull(categoryB.get("201")) ? 0 : categoryB.get("201"));
                temp2.add(isNull(categoryB.get("202")) ? 0 : categoryB.get("202"));
                temp2.add(isNull(categoryB.get("203")) ? 0 : categoryB.get("203"));
                temp2.add(isNull(categoryB.get("204")) ? 0 : categoryB.get("204"));
                temp2.add(isNull(categoryB.get("205")) ? 0 : categoryB.get("205"));
                temp2.add(isNull(categoryB.get("10")) ? 0 : categoryB.get("10"));
                temp2.add(isNull(categoryB.get("101")) ? 0 : categoryB.get("101"));
                temp2.add(isNull(categoryB.get("102")) ? 0 : categoryB.get("102"));
                temp2.add(isNull(categoryB.get("104")) ? 0 : categoryB.get("104"));
                temp2.add(isNull(categoryB.get("105")) ? 0 : categoryB.get("105"));
                temp2.add(isNull(categoryB.get("106")) ? 0 : categoryB.get("106"));
                temp2.add(isNull(categoryB.get("107")) ? 0 : categoryB.get("107"));
                temp2.add(isNull(categoryB.get("11")) ? 0 : categoryB.get("11"));
                temp2.add(isNull(categoryB.get("111")) ? 0 : categoryB.get("111"));
                temp2.add(isNull(categoryB.get("112")) ? 0 : categoryB.get("112"));
                temp2.add(isNull(categoryB.get("113")) ? 0 : categoryB.get("113"));
                temp2.add(isNull(categoryB.get("114")) ? 0 : categoryB.get("114"));
                temp2.add(isNull(categoryB.get("115")) ? 0 : categoryB.get("115"));
                temp2.add(isNull(categoryB.get("116")) ? 0 : categoryB.get("116"));
                temp2.add(isNull(categoryB.get("117")) ? 0 : categoryB.get("117"));
                temp2.add(isNull(categoryB.get("118")) ? 0 : categoryB.get("118"));
                temp2.add(isNull(categoryB.get("119")) ? 0 : categoryB.get("119"));
                temp2.add(isNull(categoryB.get("12")) ? 0 : categoryB.get("12"));
                temp2.add(isNull(categoryB.get("122")) ? 0 : categoryB.get("122"));
                temp2.add(isNull(categoryB.get("123")) ? 0 : categoryB.get("123"));
                temp2.add(isNull(categoryB.get("124")) ? 0 : categoryB.get("124"));
                temp2.add(isNull(categoryB.get("125")) ? 0 : categoryB.get("125"));
                temp2.add(isNull(categoryB.get("126")) ? 0 : categoryB.get("126"));
                temp2.add(isNull(categoryB.get("127")) ? 0 : categoryB.get("127"));

                list1.add(temp1);
                list2.add(temp2);

            }
            List temp = new ArrayList();
            temp = (ArrayList) list1.get(0);
            for (int i = 0; i < temp.size(); i++) {
                Object value = temp.get(i);
                if (value instanceof Double) {
                    temp.remove(i);
                    value = Math.round((Double) value * 100) / 100.00;
                    temp.add(i, value);
                }
            }
            temp = (ArrayList) list2.get(0);
            for (int j = 0; j < temp.size(); j++) {
                Object value = temp.get(j);
                if (value instanceof Double) {
                    temp.remove(j);
                    value = Math.round((Double) value * 100) / 100.00;
                    temp.add(j, value);
                }
            }
        }
        return map;
    }

    @Override
    public List<LinkedHashMap> analysisExcelData(List list) {
        List<LinkedHashMap> result = new ArrayList<LinkedHashMap>();
        for (int i = 0; i < list.size(); i++) {
            LinkedHashMap map = new LinkedHashMap();
            List headerList = new ArrayList();
            List dataList = new ArrayList();
            JSONObject obj = (JSONObject) list.get(i);
            JSONArray children = (JSONArray) obj.get("children");
            for (int j = 0; j < children.size(); j++) {
                JSONArray child = (JSONArray) children.get(j);
                List temp = new ArrayList();

                for (int m = 0; m < child.size(); m++) {
                    JSONObject field = (JSONObject) child.get(m);
                    if (j == 0)
                        headerList.add(field.get("alias") == null ? field.get("name") : field.get("alias"));
                    if (field.get("value") instanceof BigDecimal)
                    temp.add(processValue(((BigDecimal) field.get("value")).doubleValue(), 3));
                    else
                        temp.add(field.get("value"));
                }
                dataList.add(temp);
//                JSONObject child = (JSONObject) children.get(j);
//                List temp = new ArrayList();
//                if (j == 0) {
//                    for (String key : child.keySet()) {
//                        headerList.add(key);
//                    }
//                }
//                if (child.get(headerList.get(0)) instanceof BigDecimal)
//                    temp.add(processValue(((BigDecimal) child.get(headerList.get(0))).doubleValue(), 3));
//                else if (child.get(headerList.get(1)) instanceof BigDecimal)
//                    temp.add(processValue(((BigDecimal) child.get(headerList.get(1))).doubleValue(), 3));
//                dataList.add(temp);
            }
            map.put("name", obj.get("alias"));
            map.put("header", headerList);
            map.put("data", dataList);
            result.add(map);
        }
        return result;
    }

    /**
     *
     * @param list
     * @return
     */
    @Override
    public List<Map> gzqydExcelData(List list) {
        try {
            List<Map> result=new ArrayList<Map>();
            for (int i = 0; i < list.size(); i++) {
                Map tempMap = new HashMap();
                List<List>  tempList = new ArrayList<List>();
                Map item = (Map) list.get(0);
                List<Map> values = (List<Map>) item.get("value");
                for(Map value:values){
                    List temp = new ArrayList(8);
                    temp.add(0,String.valueOf(value.get("OG_PRO_XZQMC")));
                    temp.add(1,String.valueOf(value.get("OG_PRO_REGIONCODE")));
                    temp.add(2,String.valueOf(value.get("OG_PRO_PRONAME")));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("011"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("012"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("020"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("030"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("040"))));
                    tempList.add(temp);
                }
                tempMap.put("value",tempList);
                tempMap.put("type",(String) item.get("type"));
                result.add(tempMap);
            }
            return result;
        } catch (NumberFormatException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     *
     * @param data
     * @return
     */
    @Override
    public Map<String, Object> dtExcelData(Map<String,Object> data) {
        Map<String,Object> result = new HashMap();
        if(isNull(data))return null;
        try {
            for(Object key:data.keySet())
            {
                Map map = (Map) data.get(key);
                if(isNull(map))return result;
                List<Map> info = new ArrayList<Map>();
                List<Map>  detail =new ArrayList<Map>();
                Map<String,List> sheet = new HashMap();
                 if(key.equals("xz"))
                 {
                     if(!isNull(map.get("detail")))
                     {
                         detail = (List) map.get("detail");
                         List<List> xzList = new ArrayList();
                         for(Map xzMap:detail){
                             List xzTmpList = new ArrayList();
                             xzTmpList.add(isNull(xzMap.get("qsmc"))?"":xzMap.get("qsmc"));
                             xzTmpList.add(isNull(xzMap.get("nydZmj"))?"":xzMap.get("nydZmj"));
                             xzTmpList.add(isNull(xzMap.get("gdMj"))?"":xzMap.get("gdMj"));
                             xzTmpList.add(isNull(xzMap.get("011"))?"":xzMap.get("011"));
                             xzTmpList.add(isNull(xzMap.get("012"))?"":xzMap.get("012"));
                             xzTmpList.add(isNull(xzMap.get("013"))?"":xzMap.get("013"));
                             xzTmpList.add(isNull(xzMap.get("021"))?"":xzMap.get("021"));
                             xzTmpList.add(isNull(xzMap.get("022"))?"":xzMap.get("022"));
                             xzTmpList.add(isNull(xzMap.get("023"))?"":xzMap.get("023"));
                             xzTmpList.add(isNull(xzMap.get("031"))?"":xzMap.get("031"));
                             xzTmpList.add(isNull(xzMap.get("032"))?"":xzMap.get("032"));
                             xzTmpList.add(isNull(xzMap.get("033"))?"":xzMap.get("033"));
                             xzTmpList.add(isNull(xzMap.get("041"))?"":xzMap.get("041"));
                             xzTmpList.add(isNull(xzMap.get("042"))?"":xzMap.get("042"));
                             xzTmpList.add(isNull(xzMap.get("122"))?"":xzMap.get("122"));
                             xzTmpList.add(isNull(xzMap.get("104"))?"":xzMap.get("104"));
                             xzTmpList.add(isNull(xzMap.get("114"))?"":xzMap.get("114"));
                             xzTmpList.add(isNull(xzMap.get("117"))?"":xzMap.get("117"));
                             xzTmpList.add(isNull(xzMap.get("123"))?"":xzMap.get("123"));
                             xzTmpList.add(isNull(xzMap.get("jsydZmj"))?"":xzMap.get("jsydZmj"));
                             xzTmpList.add(isNull(xzMap.get("201"))?"":xzMap.get("123"));
                             xzTmpList.add(isNull(xzMap.get("202"))?"":xzMap.get("202"));
                             xzTmpList.add(isNull(xzMap.get("203"))?"":xzMap.get("203"));
                             xzTmpList.add(isNull(xzMap.get("204"))?"":xzMap.get("204"));
                             xzTmpList.add(isNull(xzMap.get("205"))?"":xzMap.get("205"));
                             xzTmpList.add(isNull(xzMap.get("101"))?"":xzMap.get("101"));
                             xzTmpList.add(isNull(xzMap.get("102"))?"":xzMap.get("102"));
                             xzTmpList.add(isNull(xzMap.get("105"))?"":xzMap.get("105"));
                             xzTmpList.add(isNull(xzMap.get("106"))?"":xzMap.get("106"));
                             xzTmpList.add(isNull(xzMap.get("107"))?"":xzMap.get("107"));
                             xzTmpList.add(isNull(xzMap.get("113"))?"":xzMap.get("113"));
                             xzTmpList.add(isNull(xzMap.get("118"))?"":xzMap.get("118"));
                             xzTmpList.add(isNull(xzMap.get("wlydZmj"))?"":xzMap.get("wlydZmj"));
                             xzTmpList.add(isNull(xzMap.get("043"))?"":xzMap.get("043"));
                             xzTmpList.add(isNull(xzMap.get("124"))?"":xzMap.get("124"));
                             xzTmpList.add(isNull(xzMap.get("125"))?"":xzMap.get("125"));
                             xzTmpList.add(isNull(xzMap.get("126"))?"":xzMap.get("126"));
                             xzTmpList.add(isNull(xzMap.get("127"))?"":xzMap.get("127"));
                             xzTmpList.add(isNull(xzMap.get("121"))?"":xzMap.get("121"));
                             xzTmpList.add(isNull(xzMap.get("111"))?"":xzMap.get("111"));
                             xzTmpList.add(isNull(xzMap.get("112"))?"":xzMap.get("112"));
                             xzTmpList.add(isNull(xzMap.get("115"))?"":xzMap.get("115"));
                             xzTmpList.add(isNull(xzMap.get("116"))?"":xzMap.get("116"));
                             xzTmpList.add(isNull(xzMap.get("119"))?"":xzMap.get("119"));
                             xzTmpList.add(isNull(xzMap.get("zmj"))?"":xzMap.get("zmj"));
                             xzList.add(xzTmpList);
                         }
                         if(xzList.size()>0)
                             sheet.put("Sheet1",xzList);
                     }
                     result.put(String.valueOf(key),sheet);

                 }else if(key.equals("bp"))
                 {
                     List list1 = new ArrayList();
                     List list2 = new ArrayList();
                     if (!isNull(map.get("info")))
                         info = (List) map.get("info");
                     if (!isNull(map.get("detail")))
                         detail = (List) map.get("detail");
                     if(info.size()>0&&detail.size()>0)
                     {
                         for(Map temp:info)
                         {
                             List tempList = new ArrayList();
                             tempList.add(temp.get("label"));
                             tempList.add(temp.get("area_pfm"));
                             tempList.add(temp.get("area_gq"));
                             tempList.add(temp.get("area_m"));
                             list1.add(tempList);
                         }
                         List<String> keyList = new ArrayList();
                         for(Object detailKey:detail.get(0).keySet())
                         {
                             keyList.add(String.valueOf(detailKey));
                         }
                         list2.add(keyList);
                         for(Map detailTmp:detail)
                         {
                             List tempList = new ArrayList();
                             for(String key1:keyList){
                                 tempList.add(detailTmp.get(key1));
                             }
                             list2.add(tempList);
                         }
                         sheet.put("报批汇总",list1);
                         sheet.put("报批详情",list2);
                     }
                     result.put(String.valueOf(key),sheet);
                 }else
                 {
                     List<Map> list = new ArrayList<Map>();
                     List<List> bpList = new ArrayList<List>();
                     if(!isNull(map.get("detail")))
                     {
                         detail = (List) map.get("detail");
                         List firstRow = new ArrayList();
                         List secondRow = new ArrayList();
                         firstRow.add("用途");
                         secondRow.add("面积");
                         for(Map ghMap:detail)
                         {
                             firstRow.add(ghMap.get("label"));
                             secondRow.add(ghMap.get("area"));
                         }
                         bpList.add(secondRow);
                         sheet.put("header",firstRow);
                         sheet.put("data",bpList);
                         list.add(sheet);
                     }
                     result.put(String.valueOf(key),list);
                 }
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
           throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }


    /**
     * 处理double类型数据,返回处理后的新数据
     *
     * @param value
     * @param precision
     * @return
     */
    public double processValue(Double value, int precision) {

        switch (precision) {
            case 1:
                return Math.round(value * 100) / 100.0;
            case 2:
                return Math.round(value * 100) / 100.00;
            case 3:
                return Math.round(value * 100) / 100.000;
            default:
                return Math.round(value * 100) / 100;

        }
    }


    public double getSumArea(List<Map> list) {

        double area = 0;
        for (Map map : list) {
            area += Double.valueOf(map.get("area").toString());
        }
        return area;
    }

    public Map getCategory(List<Map> list) {
        Map result = new HashMap();
        try {
            for (Map map : list) {
                result.put(map.get("dlbm"), map.get("area"));
                if (map.containsKey("detail")) {
                    List<Map> detailList = (List<Map>) map.get("detail");
                    for (Map detail : detailList) {
                        result.put(detail.get("dlbm"), detail.get("area"));
                    }
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }

    public List<Map> add2List(List<Map> list, Map map) {
        int tag = 0;
        String dlbm = map.get("DLBM").toString();
        double area = Double.valueOf(map.get("CCMJ").toString());
        Map temp = new HashMap();
        try {
            for (Map item : list) {

                if (item.get("dlbm").equals(dlbm)) {
                    double newArea = Double.valueOf(item.get("area").toString()) + area;
                    item.put("area", newArea);
                    tag = 1;
                }
            }
            if (tag == 0) {
                temp.put("area", area);
                temp.put("dlbm", dlbm);
                list.add(temp);
            }
        } catch (NumberFormatException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return list;
    }


    public List<Map> tdlyxzResult2(Map analysisResult, String tpl) {
        List<Map> resultList = new ArrayList<Map>();
        List<Map> list = (List) analysisResult.get("analysisAreaDetail");
        List<Map> groupList = getTdlyxzDictList(tpl);
        for (Map map : list) {
            String dlbm = map.get("DLBM").toString();
            double area = Double.valueOf(map.get("CCMJ").toString());
            for (Map groupMap : groupList) {
                if (groupMap.get("dlbm").toString().equals(dlbm.substring(0, 2))) {
                    double newArea = Double.valueOf(groupMap.get("area").toString()) + area;
                    ((List) groupMap.get("detail")).add(map);
                    groupMap.put("area", newArea);
                }
            }
        }
        for (int i = 0; i < groupList.size(); i++) {
            Map map = groupList.get(i);
            Map temp = new HashMap();
            double area = Double.valueOf(map.get("area").toString());
            if (area > 0) {
                temp.put("dlmc", map.get("dlmc"));
                temp.put("area", map.get("area"));
                temp.put("detail", map.get("detail"));
                resultList.add(temp);
            }
        }
        return resultList;
    }

    /**
     * 快速检索
     *
     * @param value
     * @return
     */
    @Override
    public List search(String value, boolean returnGeometry) {
        if (searchConfig == null) throw new RuntimeException(getMessage("search.config.not.found"));
        List result = new ArrayList();
        List<Map> layers = (List) searchConfig.get(SEARCH_LAYERS);
        for (Map layer : layers) {
            String name = (String) layer.get("layer");
            String[] fields = ((String) layer.get("fields")).split(",");
            String where = getWhereClause(fields, value);
            result.addAll(spatialDao.query(name, where, fields, returnGeometry, 10, null));
        }
        return result;
    }

    /**
     * 根据相关条件输出图片
     * @param where 查询子句
     * @param mapScale 出图的比例尺
     * @param width 图片的width
     * @param height 图片的height
     * @param type 出图类型(xz,gh)
     * @return
     */
    @Override
    public InputStream exportMap(String where, String mapScale, int width,int height, String type) {
        if (exportConfig == null) {
            logger.info("无配置信息");
            return null;
        }
        int bgWidth = width +10;
        int bgHeight = height+10;
        String imageSize = width+","+height;
        double scale = 0;
        if(StringUtils.isNotBlank(mapScale))
            scale = Double.valueOf(mapScale);
        InputStream inputStream;
        List<BufferedImage> baseImages=new ArrayList<BufferedImage>();
        BufferedImage topImage = null;
        BufferedImage resultImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_BGR);
        Envelope envelope =null;
        Map operaLayer = (Map) exportConfig.get("operaLayer");
        List<Map> ghLayers = (List<Map>) exportConfig.get("ghLayers");
        List<Map> xzLayers = (List<Map>) exportConfig.get("xzLayers");
        String layerName = (String) operaLayer.get("layerName");
        String dataSource = (String) operaLayer.get("dataSource");
        String url = (String) operaLayer.get("url");
        List list = query(layerName,where,null,true,dataSource);
        if(list.size()>0){
            FeatureCollection featureCollection =  geometryService.list2FeatureCollection(list, null, null);
            envelope = featureCollection.getBounds();
            topImage = getExportImage(envelope,url,imageSize,scale);

            if(type.equals(EXPORT_TYPE_GH)){
                for (Map item:ghLayers){
                    BufferedImage tempImage = getExportImage(envelope,(String) item.get("url"),bgWidth+","+bgHeight,scale);
                    baseImages.add(tempImage);
                }
            } else if (type.equals(EXPORT_TYPE_XZ)) {
                for (Map item : xzLayers) {
                    BufferedImage tempImage = getExportImage(envelope, (String) item.get("url"), bgWidth+","+bgHeight, scale);
                    baseImages.add(tempImage);
                }
            } else {
                logger.error("type is not found");
            }
            Graphics2D graphics2D = resultImage.createGraphics();
            resultImage = graphics2D.getDeviceConfiguration().createCompatibleImage(width, height, Transparency.TRANSLUCENT);
            graphics2D.dispose();
            graphics2D = resultImage.createGraphics();
            if (baseImages.size() > 0) {
                for (BufferedImage tempImage : baseImages) {
                    graphics2D.drawImage(tempImage, 0, 0, tempImage.getWidth(), tempImage.getHeight(), null);
                }
            }
            graphics2D.drawImage(topImage, 0, 0, topImage.getWidth(), topImage.getHeight(), null);
            graphics2D.dispose();
        }else
        {
            logger.info("未查询到地块!");
            return null;
//            Graphics2D g2d = resultImage.createGraphics();
//            Font font = new Font("微软雅黑", Font.BOLD, 25);
//            String s = "未查询到地块";
//            g2d.setBackground(Color.white);
//            g2d.clearRect(0, 0, width, height);
//            g2d.setColor(Color.red);
//            FontRenderContext context = g2d.getFontRenderContext();
//            Rectangle2D bounds = font.getStringBounds(s, context);
//            double x = (width - bounds.getWidth()) / 2;
//            double y = (height - bounds.getHeight()) / 2;
//            g2d.drawString(s, (int)x, (int)y);
//            g2d.dispose();
//            resultImage.flush();
        }
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ImageIO.write(resultImage, "png", out);
            inputStream = new ByteArrayInputStream(out.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return inputStream;
    }

    private enum JCTB {
        BPDK,GDDK,DLTB,XZDW,JSYDGZQ,TDYTQ
    }

    private enum JTag{

        JC_GD_AREA,BP_AREA,BP_GD_AREA,YG_AREA,WG_AREA,WPYJ_AREA,WPYJ_GD_AREA,YXJSQ_AREA,JBNT_AREA,YBNTQ_AREA,PCMC,NZYPW,GDBH

    }

    private final double ACRES_CONV=0.0015;

    public List jctbExcelData(List<Map> list){
        List result = null;
        if(isNull(list))return null;
        result = new ArrayList();
        for(Map item:list){
            List temp = new ArrayList();
            temp.add(list.indexOf(item)+1);
            temp.add(item.get("OG_PRO_JCBH"));
            temp.add(item.get("OG_PRO_TBLX"));
            temp.add(item.get("OG_PRO_BSM"));
            temp.add(item.get("OG_PRO_XZMC"));
            temp.add(item.get("OG_PRO_XMMC"));
            temp.add(item.get("OG_PRO_JSDW"));
            temp.add(item.get("OG_PRO_JCMJ"));
            temp.add(item.get("JC_GD_AREA"));
            temp.add(item.get(JTag.BP_AREA.name()));
            temp.add(item.get(JTag.BP_GD_AREA.name()));
            temp.add(item.get(JTag.YG_AREA.name()));
            temp.add(item.get(JTag.WG_AREA.name()));
            temp.add(item.get(JTag.WPYJ_AREA.name()));
            temp.add(item.get(JTag.WPYJ_GD_AREA.name()));
            temp.add(item.get(JTag.YXJSQ_AREA.name()));
            temp.add(item.get(JTag.JBNT_AREA.name()));
            temp.add(item.get(JTag.YBNTQ_AREA.name()));
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(item.get(JTag.PCMC.name()));
            temp.add(item.get(JTag.NZYPW.name()));
            temp.add(item.get(JTag.GDBH.name()));
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            result.add(temp);
        }
       return result;
    }

    /**
     * jctb analysis
     * @param geometry
     *
     * @param analysisLayers
     * @return
     */
    @Override
    public List<Map> jctbAnalysis(String geometry, List analysisLayers,String dataSource) {
        if (isNull(analysisLayers)) throw new RuntimeException("no analysis Layers");
        if (isNull(geometry)) throw new JSONMessageException("geometry is null");
        List results = null;
        try {
            Object geo = geometryService.readUnTypeGeoJSON(geometry);
            if (geo instanceof FeatureCollection) {
                results = new ArrayList();
                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) iterator.next();
                    Map map = singleAnalysisOfJctb(feature, analysisLayers, dataSource);
                    results.add(map);
                }
            }
        } catch (GeometryServiceException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION,e.getLocalizedMessage());
        } catch (NoSuchElementException e) {
           throw new RuntimeException(e.getLocalizedMessage());
        }
        return results;
    }

    /**
     *
     * @param feature
     * @param analysisLayers
     * @param dataSource
     * @return
     */
    public Map singleAnalysisOfJctb(SimpleFeature feature,List analysisLayers,String dataSource){
        Map result=null;
        Map<String,List<Map>> analysisMap=new HashMap<String, List<Map>>();
        try {
            for (int i = 0; i < analysisLayers.size(); i++) {
                Map layer = (Map) analysisLayers.get(i);
                String lyrId = layer.get("id").toString();
                String lyrName = (String) layer.get("layerName");
                String outFields = layer.get("fields").toString();
                String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                List<Map> analysisList = (List<Map>)intersectBySimpleFeature(lyrName, feature, fields, dataSource);
                analysisMap.put(lyrId.toUpperCase(),analysisList);
            }
            List<Map> bpList = analysisMap.get(JCTB.BPDK.name());
            List<Map> gdList = analysisMap.get(JCTB.GDDK.name());
            List<Map> dltbList = analysisMap.get(JCTB.DLTB.name());
            List<Map> xzdwList = analysisMap.get(JCTB.XZDW.name());
            List<Map> jsydgzqList = analysisMap.get(JCTB.JSYDGZQ.name());
            List<Map> tdytqList = analysisMap.get(JCTB.TDYTQ.name());

            double geoArea = getAreaByFeature(feature, ACRES_CONV);

            double dltb_gd_area =getAreaByField(dltbList,"DLBM","011",1)+getAreaByField(dltbList,"DLBM","012",1)+getAreaByField(dltbList,"DLBM","013",1);
            double xzdw_gd_area = getXzdwAreaByField(xzdwList,"DLBM","011",1)+getXzdwAreaByField(xzdwList,"DLBM","012",1)+ getXzdwAreaByField(xzdwList,"DLBM","013",1);

            result = new HashMap();
            result.put(JTag.JC_GD_AREA.name(),(dltb_gd_area-xzdw_gd_area)<0?0:(dltb_gd_area-xzdw_gd_area)*ACRES_CONV);
            result.put(JTag.BP_AREA.name(), getAreaByList(bpList, ACRES_CONV));
            result.put(JTag.BP_GD_AREA.name(),getDouValueByField(bpList, "GD"));
            result.put(JTag.YG_AREA.name(), getAreaByList(gdList, ACRES_CONV));
            result.put(JTag.WG_AREA.name(),(geoArea-(Double)result.get(JTag.YG_AREA.name()))<0?0:(geoArea-(Double)result.get(JTag.YG_AREA.name())));
            result.put(JTag.WPYJ_AREA.name(),(geoArea-(Double)result.get(JTag.BP_AREA.name()))<0?0:geoArea-(Double)result.get(JTag.BP_AREA.name()));
            result.put(JTag.WPYJ_GD_AREA.name(),(Double)result.get(JTag.JC_GD_AREA.name())-(Double)result.get(JTag.BP_GD_AREA.name())<0?0:(Double)result.get(JTag.JC_GD_AREA.name())-(Double)result.get(JTag.BP_GD_AREA.name()));
            result.put(JTag.YXJSQ_AREA.name(),getAreaByField(jsydgzqList,"GZQLXDM","010",ACRES_CONV));
            result.put(JTag.JBNT_AREA.name(),getAreaByField(tdytqList,"TDYTQLXDM","010",ACRES_CONV));
            result.put(JTag.YBNTQ_AREA.name(),getAreaByField(tdytqList,"TDYTQLXDM","020",ACRES_CONV));

            result.put(JTag.PCMC.name(), getStrValueByField(bpList, "PCJC"));
            result.put(JTag.NZYPW.name(), getStrValueByField(bpList, "PZWH"));
            result.put(JTag.GDBH.name(), getStrValueByField(gdList, "XMBH"));

            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")||p.getName().getLocalPart().equals("crs")) continue;
                result.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }


        } catch (Exception e) {
           throw new RuntimeException(e.getLocalizedMessage());
        }

        return result;
    }

    /**
     * sum area of list
     * @param list
     * @return
     */
    public double getAreaByList(List<Map> list, double conv){
        if(isNull(conv))conv=1;
        double area=0;
        for(Map map:list){
            double value =(Double)map.get(SE_SHAPE_AREA);
            area += value;
        }
        return area*conv;
    }
    /**
     * get feature's area
     * @param feature
     * @return
     */
    public double getAreaByFeature(SimpleFeature feature, double conv){
        if(isNull(conv))conv=1;
        double area = 0;
        try {
            CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            CoordinateReferenceSystem targetCrs = geometryService.parseUndefineSR("2364");
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (sourceCRS != null && sourceCRS instanceof GeographicCRS) {
                geometry = geometryService.project((Geometry) feature.getDefaultGeometry(), sourceCRS, targetCrs);
            }
            if (isPolygon(geometry))
                area = geometry.getArea();
            else
                area = geometry.getLength();
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area*conv;
    }

    /**
     * get area by some field
     * @param list
     * @param field
     * @param value
     * @param conv
     * @return
     */
    public double getAreaByField(List<Map> list,String field,String value,double conv){
        if(isNull(conv))conv=1;
        assert field !=null;
        assert value!=null;
        double area=0;
        for(Map map:list){
            if (value.equals(map.get(field)))
                area += (Double) map.get(SE_SHAPE_AREA);
        }
        return area*conv;
    }

    /**
     * get string value by some field
     * @param list
     * @param field
     * @return
     */
    public String getStrValueByField(List<Map> list, String field){
        StringBuilder sb=new StringBuilder();
        assert field !=null;
        for(Map map:list){
            if(map.containsKey(field))
            {
                if(StringUtils.isNotBlank(sb.toString()))
                    sb.append(",");
                if(!isNull(map.get(field)))
                    sb.append(map.get(field));
            }
        }
        return sb.toString();
    }

    /**
     * get double value by some field
     * @param list
     * @param field
     * @return
     */
    public double getDouValueByField(List<Map> list, String field){
        double value=0;
        assert field !=null;
        for(Map map:list){
               if(isNull(map.get(field)))continue;
            value += Double.valueOf(map.get(field).toString());
        }
        return value;
    }


    /**
     * get xzdw area by field
     * @param list
     * @param field
     * @param value
     * @param conv
     * @return
     */
    public double getXzdwAreaByField(List<Map> list,String field,String value,double conv){
        if(isNull(conv))conv=1;
        double area=0;
        for(Map map:list){
            if (value.equals(map.get(field)))
            {
                if(isNull(map.get("XZDWMJ")))continue;
                area += Double.valueOf(map.get("XZDWMJ").toString());
            }
        }
        return area*conv;
    }


    /**
     * 根据bbox范围和相关参数,获取输出图片
     * @param envelope
     * @param url
     * @param imageSize
     * @param mapScale
     * @return
     */
    private BufferedImage getExportImage(Envelope envelope,String url,String imageSize,double mapScale) {
        BufferedImage image = null;
        String bbox = "bbox="+ envelope.getMinX()+","+envelope.getMinY()+","+envelope.getMaxX()+","+envelope.getMaxY();
        String size = "size="+imageSize;
        StringBuilder exportUrl = new StringBuilder();
        exportUrl.append(url);
        exportUrl.append("/export?");
        exportUrl.append(bbox);
        exportUrl.append("&");
        exportUrl.append(size);
        exportUrl.append("&");
        exportUrl.append("&");
        if (mapScale > 0)
            exportUrl.append("mapScale=" + mapScale);
        exportUrl.append("&");
        exportUrl.append("transparent=true&");
        exportUrl.append("format=png&");
        exportUrl.append("f=image");
        HttpClient httpClient = new HttpClient();
        GetMethod getMethod = new GetMethod(exportUrl.toString());
        try {
            httpClient.executeMethod(getMethod);
            InputStream inputStream = new ByteArrayInputStream(getMethod.getResponseBody());
            if(inputStream.available()>0)
                image = ImageIO.read(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return image;
    }


    /**
     * @param features
     * @return
     */
    private double getTotalArea(JSONArray features) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {
            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
        }
        return area;
    }

    /**
     * @param field
     * @param features
     * @param lxdm
     * @return
     */
    private double getAreaByLxdm(String field, JSONArray features, String lxdm) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {

            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            if (property.get(field) == null)
//                throw new RuntimeException("字段【"+field+"】不存在！");
                return 0;
            else {
                if (property.get(field).equals(lxdm))
                    area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
            }
        }
        return area;
    }

    /**
     * @param fields
     * @param keyword
     * @return
     */
    private String getWhereClause(String[] fields, String keyword) {
        String or = " or ";
        StringBuilder b = new StringBuilder();
        for (String field : fields) {
            b.append(field);
            b.append(" like '");
            b.append(keyword);
            b.append("%'");
            b.append(or);
        }
        b.delete(b.length() - or.length(), b.length());
        return b.toString();
    }


    /**
     * GeoJSON to Map
     *
     * @param geoJSON single geometry
     * @return
     */
    private Map<String, Object> geoJSON2Map(String geoJSON, CoordinateReferenceSystem targetCRS) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        Map<String, Object> columns = null;
        if (geo instanceof Geometry) {
            columns = new HashMap<String, Object>();
            columns.put(SE_SHAPE_FIELD, ((Geometry) geo).toText());
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geoJSON);
            columns = geometryService.simpleFeature2Map(feature);
            try {
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (targetCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, targetCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
            } catch (Exception e) {
                logger.error(e.getLocalizedMessage());
                throw new RuntimeException(e.getLocalizedMessage());
            }
        } else if (geo instanceof GeometryCollection) {
            throw new RuntimeException("current omp version don't support GeometryCollection ");
        } else if (geo instanceof FeatureCollection) {
            throw new RuntimeException("current omp version don't support FeatureCollection ");
        } else {
            throw new RuntimeException(getMessage("geometry.undefined"));
        }
        return columns;
    }

    /**
     * feature to geometry
     *
     * @param feature
     * @param geometry
     * @return
     */
    private Geometry setFeaturePros2Geo(SimpleFeature feature, Geometry geometry) {
        Map<String, Object> map = null;
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                Geometry geo = geometry.getGeometryN(i);
                map = new HashMap<String, Object>();
                for (Property p : feature.getProperties()) {
                    if (p.getName().equals("geometry")) continue;
                    map.put(p.getName().getLocalPart(), p.getValue());
                }
                geo.setUserData(map);
            }
        } else {
            map = new HashMap<String, Object>();
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                map.put(p.getName().getLocalPart(), p.getValue());
            }
            geometry.setUserData(map);
        }
        return geometry;
    }

    /**
     * add pros 2 map  list
     *
     * @param list
     * @param geometry
     * @return
     */
    private List<?> addGeoProperty2List(List<Map<String, Object>> list, Geometry geometry) {
        if (geometry.getUserData() == null) return list;
        for (Map<String, Object> item : list) {
            Map<String, Object> pros = (Map<String, Object>) geometry.getUserData();
            for (Map.Entry entry : pros.entrySet()) {
                item.put("G_".concat((String) entry.getKey()), entry.getValue());
            }
        }
        return list;
    }

    /**
     * @param tpl
     * @return
     */
    private List<Map> getTdlyxzDictList(String tpl) {
        List<Map> groupList = new ArrayList<Map>();
        Configuration configuration = webMapService.getConfig(tpl);
        List<Dict> dicts = configuration.getDicts();
        for (Dict dict : dicts) {
            if (dict.getName().toString().equals("tdlyxz")) {
                List<Item> dictItems = dict.getDictItems();
                for (Item item : dictItems) {
                    Map temp = new HashMap();
                    temp.put("dlbm", item.getName());
                    temp.put("dlmc", item.getTitle());
                    temp.put("area", 0);
//                    temp.put("detail",new ArrayList<Map>());
                    groupList.add(temp);
                }
            }
        }
        if (groupList.size() == 0)
            throw new RuntimeException(getMessage("get.dict.error",tpl));
        return groupList;
    }

    /**
     * check field is in layer
     *
     * @param field
     * @param layerName
     * @return
     */
    private boolean checkFieldInLayer(String field, String layerName, String dataSource) {
        return ArrayUtils.contains(spatialDao.getLayerColumns(layerName, dataSource), field, true);
    }


    private boolean isPolygon(Geometry geometry) {
        String geometryType = geometry.getGeometryType();
        if(geometryType.equals(Geometries.POLYGON.getName())||geometryType.equals(Geometries.MULTIPOLYGON.getName()))
            return true;
        else if(geometryType.equals(Geometries.LINESTRING.getName())||geometryType.equals(Geometries.MULTILINESTRING.getName()))
            return false;
        else if(geometryType.equals(Geometries.GEOMETRYCOLLECTION.getName())){
            return geometry.getArea()>0?true:false;
        }else
            return false;
//        return (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName())) ? true : false;
    }

}
