package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.dao.SpatialDao;
import cn.gtmap.onemap.platform.dao.impl.ArcSDEDaoImpl;
import cn.gtmap.onemap.platform.dao.impl.OracleSpatialDaoImpl;
import cn.gtmap.onemap.platform.entity.Configuration;
import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.entity.dict.Dict;
import cn.gtmap.onemap.platform.entity.dict.Item;
import cn.gtmap.onemap.platform.event.GISDaoException;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.support.spring.ApplicationContextHelper;
import cn.gtmap.onemap.platform.utils.ArrayUtils;
import cn.gtmap.onemap.platform.utils.EnumUtils;
import cn.gtmap.onemap.platform.utils.Utils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.gtis.config.AppConfig;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.geom.Polygon;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.collections.map.MultiKeyMap;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.codehaus.xfire.client.Client;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.geometry.jts.Geometries;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.crs.GeographicCRS;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.net.URL;
import java.text.DecimalFormat;
import java.util.*;
import java.util.List;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-20 下午5:25
 */
public class GISServiceImpl extends BaseLogger implements GISService {

    private final String SE_SHAPE_FIELD = Constant.SE_SHAPE_FIELD;

    private final String SE_SHAPE_AREA = Constant.SE_SHAPE_AREA;

    private final String OG_PRO_PERFIX = "OG_PRO_";

    private final String SEARCH_LAYERS = "layers";

    private final String EXPORT_TYPE_GH = "gh";

    private final String EXPORT_TYPE_XZ = "xz";


    /**
     * 规划图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_E = "_E_";

    /**
     * 现状图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_H = "_H_";

    /**
     * 土地用途总体规划审查
     */
    private enum GHSC {

        TDYTQ("土地用途分区"),
        JSYDGZQ("建设用地管制区"),
        GHJBNTTZ("规划基本农田调整"),
        MZZDJSXM("重点建设项目");

        private String label;

        private GHSC(String value) {
            this.label = value;
        }

        private String getLabel() {
            return label;
        }
    }

    /**
     * wcf用到的tag
     */
    private enum WCFTag {
        AnalyseGeoJSON, TBLayerName, XWLayerName, LXLayerName, TBDLMJ, XZDWMJ, LXDWMJ,
        Summary,LXFeatures,XWFeatures,TBFeatures,
        DLBM,QSDWMC,QSDWMC1,QSDWDM,QSDWDM1,QSXZ,AREA,ZLDWDM,ZLDWMC
    }

    /**
     * 土地利用现状
     */
    private enum TDXZ {
        DLTB, XZDW
    }

    private enum Tag {
        YES, NO;
    }

    private enum JCTB {
        BPDK,GDDK,DLTB,XZDW,JSYDGZQ,TDYTQ
    }

    private enum JTag{

        JC_GD_AREA,BP_AREA,BP_GD_AREA,YG_AREA,WG_AREA,WPYJ_AREA,WPYJ_GD_AREA,YXJSQ_AREA,JBNT_AREA,YBNTQ_AREA,PCMC,NZYPW,GDBH

    }

    /**
     * 亩和平方米的换算值
     */
    private final double ACRES_CONV=0.0015;

    private SpatialDao spatialDao;

    @Autowired
    private GeometryService geometryService;
    @Autowired
    private AgsGeoemtryService agsGeoemtryService;

    @Autowired
    private MapService mapService;
    @Autowired
    private WebMapService webMapService;

    private Map searchConfig;

    private Map exportConfig;

    public void setSearchConfig(Resource path) {
        try {
            searchConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" search config can't found ");
        }
    }

    public void setExportConfig(Resource path) {
        try {
            exportConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" export config can't found ");
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
    }


    /**
     * 初始化
     */
    @Override
    public void initialize(Constant.SpatialType type) {
        switch (type) {
            case ARC_SDE:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(ArcSDEDaoImpl.class);
                break;
            case ORACLE_SPATIAL:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(OracleSpatialDaoImpl.class);
                break;
        }
    }

    /**
     * 属性查询
     *
     * @param layerName
     * @param where
     * @param columns
     * @param returnGeometry
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String where, String[] columns, boolean returnGeometry, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(where, getMessage("query.where.notnull"));
        return spatialDao.query(layerName, where, columns, returnGeometry, dataSource);
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param wkt
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String wkt, String[] columns, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(wkt, getMessage("query.geometry.notnull"));
        return spatialDao.query(layerName, wkt, columns, dataSource);
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param geometry
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, Geometry geometry, String[] columns, String dataSource) {
        return query(layerName, geometry.toText(), columns, dataSource);
    }

    /**
     * 空间查询(进行投影转换)
     *
     * @param layerName
     * @param feature
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, SimpleFeature feature, String[] columns, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null && !sourceCRS.equals(layerCRS))
            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        return query(layerName, geometry.toText(), columns, dataSource);
    }


    /**
     * 相交分析
     *
     * @param layerName
     * @param wktPlygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, String wktPlygon, String[] returnFields, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        return spatialDao.intersect(layerName, wktPlygon, returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param polygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Polygon polygon, String[] returnFields, String dataSource) {
        return intersect(layerName, polygon.toText(), returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, CoordinateReferenceSystem sourceCRS,
                             String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null) geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        List<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    result.addAll((Collection<? extends Map<String, Object>>)
                            intersect(layerName, geo.toText(), outFields, dataSource));
                    addGeoProperty2List(result, geo);
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            result.addAll((Collection<? extends Map<String, Object>>)
                    intersect(layerName, geometry.toText(), outFields, dataSource));
            addGeoProperty2List(result, geometry);
        }
        if (sourceCRS != null) {
            for (Map<String, Object> item : result) {
                Geometry geo = geometryService.readWKT((String) item.get(SE_SHAPE_FIELD));
                item.put(Constant.SE_SHAPE_FIELD, geometryService.project(geo, layerCRS, sourceCRS).toText());
            }
        }
        return result;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem sourceCRS = null;
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    List queryResults = query(layerName, geo, outFields, dataSource);


                    if (layerCRS instanceof GeographicCRS) {
                        if (isNull(sourceCRS) && queryResults.size() > 0)
                            sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        geo = geometryService.project(geo, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
                    }
                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            if(isNull(sourceCRS))
                                sourceCRS = layerCRS;
                            geometryResult=geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),geo.toText(),sourceCRS.toWKT()));
//                            geometryResult = geo1.intersection(geo);
                        } catch (Exception e) {
                            geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geo);
                            logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            List queryResults = query(layerName, geometry, outFields, dataSource);

            if (layerCRS instanceof GeographicCRS) {
                if (isNull(sourceCRS) && queryResults.size() > 0)
                    sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                geometry = geometryService.project(geometry, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                }
                Geometry geometryResult = null;
                try {
                    if(isNull(sourceCRS))
                        sourceCRS = layerCRS;
                    geometryResult=geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),geometry.toText(),sourceCRS.toWKT()));
                } catch (Exception e) {
                    logger.info(" geometry is invalid,[{}]", e.getLocalizedMessage());
                    geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geometry);
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                results.add(result);
            }
        }

        return results;
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersectByGeoJSON(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCRS = null;
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
            results = intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            results = addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            FeatureCollection collection = (FeatureCollection) geo;
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                results.addAll(addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature));
            }
        }
        return results;
    }


    /**
     *  改进的相交分析，为了减少土地利用现状分析的误差，需考虑将原始的分析区域的坐标保留，便于后续在平面参考下进行相切
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    public List<?> intersect4(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = bufferIntersect(layerName, geometry, layerCrs, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) bufferIntersect(layerName, geometry, sourceCrs, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) bufferIntersect(layerName, geometry, sourceCrs, outField, dataSource), feature));

            }
        }
        return results;
    }

    /**
     * 为了提高土地利用现状分析的精确度，将经纬度进行相交搜索的时候，做了buffer，扩大搜索范围
     * @param layerName
     * @param sourceGeometry
     * @param sourceCRS
     * @param outFields
     * @param dataSource
     * @return
     */
    public List<?> bufferIntersect(String layerName, Geometry sourceGeometry, CoordinateReferenceSystem sourceCRS, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (sourceGeometry instanceof GeometryCollection) {
            for (int i = 0; i < sourceGeometry.getNumGeometries(); i++) {
                try {
                    Geometry beforeGeo = sourceGeometry.getGeometryN(i);
                    Geometry afterGeometry=null;
                    if (!isNull(layerCRS))
                        afterGeometry = geometryService.project(beforeGeo, sourceCRS,layerCRS);
                    else
                        afterGeometry= beforeGeo;
                    Geometry tmpGeo =afterGeometry.buffer(0.002);
                    List queryResults = query(layerName, tmpGeo, outFields, dataSource);

                    CoordinateReferenceSystem tmpProjectedCRS = null;
                    if (sourceCRS instanceof GeographicCRS && layerCRS instanceof GeographicCRS) {

                        if (queryResults.size() > 0)
                            tmpProjectedCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        beforeGeo = geometryService.project(beforeGeo, layerCRS, isNull(tmpProjectedCRS) ? layerCRS : tmpProjectedCRS);
                    }

                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),beforeGeo.toText(),sourceCRS.toWKT()));
                        } catch (Exception e) {
                            logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                            geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(beforeGeo);
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            Geometry afterGeometry=null;
            if (!isNull(layerCRS))
                afterGeometry = geometryService.project(sourceGeometry, sourceCRS,layerCRS);
            else
                afterGeometry= sourceGeometry;
            Geometry tmpGeo =afterGeometry.buffer(0.002);
            List queryResults = query(layerName, tmpGeo, outFields, dataSource);

            CoordinateReferenceSystem tmpProjectedCRS = null;
            if (sourceCRS instanceof GeographicCRS && layerCRS instanceof GeographicCRS) {
                if (queryResults.size() > 0)
                    tmpProjectedCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                sourceGeometry = geometryService.project(sourceGeometry, layerCRS, isNull(tmpProjectedCRS) ? layerCRS : tmpProjectedCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS);
                }
                Geometry geometryResult = null;
                try {
                    geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(),sourceGeometry.toText(),sourceCRS.toWKT()));
                } catch (Exception e) {
                    logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                    geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(sourceGeometry);
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS)?sourceCRS:tmpProjectedCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                results.add(result);
            }
        }

        return results;
    }


    /**
     * @param layerName
     * @param feature
     * @param outField
     * @param dataSource
     * @return
     */
    private List<?> intersectBySimpleFeature(String layerName, SimpleFeature feature, String[] outField, String dataSource) {
        CoordinateReferenceSystem sourceCRS = null;
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (!geometry.isValid())
            geometry = createValidGeometry(geometry);
        setFeaturePros2Geo(feature, geometry);
        try {
            sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            return intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } catch (Exception e) {
            logger.error("intersect analysis - feature json crs not define [{}]", e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return GeoJSON 格式要素集 {"type":"FeatureCollection","features":[{"type":"Feature","geometry":{"type":"Polygon","coordinates":[[]]},"properties":{}}}
     */
    @Override
    public List<?> intersect2(String layerName, String geoJson, String[] outField, String dataSource) {
        List results = intersectByGeoJSON(layerName, geoJson, outField, dataSource);
//        FeatureCollection collection = geometryService.list2FeatureCollection(results, null, null);
        return results;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect3(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = intersect(layerName, geometry, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            if (!isNull(sourceCrs) && !isNull(layerCrs))
                geometry = geometryService.project(geometry, sourceCrs, layerCrs);
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                if (!isNull(sourceCrs) && !isNull(layerCrs))
                    geometry = geometryService.project(geometry, sourceCrs, layerCrs);
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature));
//                results.addAll(intersect(layerName, geometry, outField, dataSource));
            }
        }
        return results;
    }

    /**
     * 创建新的有效的geometry
     *
     * @param geometry
     * @return
     */
    private Geometry createValidGeometry(Geometry geometry) {
        try {
            if (geometry instanceof Polygon) {
                Polygon polygon = (Polygon) geometry;
                GeometryFactory factory = geometry.getFactory();
                List<Polygon> polygons = new ArrayList<Polygon>();
                Polygon exteriorPolygon = new Polygon((LinearRing) polygon.getExteriorRing(), null, factory);
                polygons.add(exteriorPolygon);
                for (int i = 0; i < polygon.getNumInteriorRing(); i++) {
                    LinearRing interiorRing = (LinearRing) polygon.getInteriorRingN(i);
                    Polygon interiorPolygon = new Polygon(interiorRing, null, factory);
                    polygons.add(interiorPolygon);
                }
                List<Polygon> newPolygons = new ArrayList<Polygon>();
                List<Polygon> excludePolygons = new ArrayList<Polygon>();
                for (Polygon temp : polygons) {
                    temp = getPolygonWithHoles(polygons, excludePolygons, temp);
                    if (!excludePolygons.contains(temp))
                        newPolygons.add(temp);
                }
                return new MultiPolygon(newPolygons.toArray(new Polygon[0]), factory);
            } else if (geometry instanceof MultiPolygon) {
                MultiPolygon multiPolygon = (MultiPolygon) geometry;
                List<Polygon> polygonList = new ArrayList<Polygon>();
                for (int j = 0; j < multiPolygon.getNumGeometries(); j++) {

                    Polygon polygon = (Polygon) multiPolygon.getGeometryN(j);
                    if (!polygon.isValid()) {
                        MultiPolygon tempMultiPolygon = (MultiPolygon) createValidGeometry(polygon);
                        for (int k = 0; k < tempMultiPolygon.getNumGeometries(); k++) {
                            polygonList.add((Polygon) tempMultiPolygon.getGeometryN(k));
                        }
                    } else {
                        polygonList.add(polygon);

                    }
                }
                return new MultiPolygon(polygonList.toArray(new Polygon[0]), multiPolygon.getFactory());
            } else {
                logger.info("geometryType has not been supported yet");
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 判断一个polygon在polygon组中是否是含有环的polygon
     *
     * @param srcPolygons     原始polygon组
     * @param excludePolygons 内环polygon需排除
     * @param polygon
     * @return
     */
    private Polygon getPolygonWithHoles(List<Polygon> srcPolygons, List<Polygon> excludePolygons, Polygon polygon) {
        List<LinearRing> holes = new ArrayList<LinearRing>();
        for (Polygon item : srcPolygons) {
            if (polygon.equals(item) || excludePolygons.contains(polygon))
                continue;
            if (polygon.contains(item)) {
                holes.add((LinearRing) item.getExteriorRing());
                excludePolygons.add(item);
            }
        }
        if (holes.size() > 0)
            return new Polygon((LinearRing) polygon.getExteriorRing(), holes.toArray(new LinearRing[0]), polygon.getFactory());
        else
            return polygon;
    }


    /**
     * @param value
     * @param feature
     * @return
     */
    private List addFeaturePros2List(List<Map> value, SimpleFeature feature) {
        for (Map item : value) {
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                item.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }
        }
        return value;
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param columns    包含图形、属性数据
     * @param dataSource
     * @return
     */
    @Override
    public String insert(String layerName, Map<String, Object> columns, String dataSource) {
        LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
        if (!isNull(layerRegion.getRegionField())) {
            boolean contains = false;
            for (Map.Entry entry : columns.entrySet()) {
                if (layerRegion.getRegionField().equalsIgnoreCase((String) entry.getKey())
                        && geometryService.containsRegionValue(String.valueOf(entry.getValue()))) {
                    contains = true;
                    break;
                }
            }
            if (!contains)
                throw new RuntimeException(getMessage("insert.region.field.not.null", layerName, layerRegion.getRegionField()));
        }
        return spatialDao.insert(layerName, columns, dataSource);
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param geoJSON    GeoJSON 格式
     * @param dataSource
     * @return primaryKey
     */
    @Override
    public String insert(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (Map.Entry<String, Object> entry : columns.entrySet()) {
                    if (entry.getValue() instanceof Geometry) {
                        Geometry geometry = (Geometry) entry.getValue();
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(entry.getKey());
                    }
                }
                if (featureIterator.hasNext())
                    insert(layerName, columns, dataSource);
                else
                    return insert(layerName, columns, dataSource);
            }
        }
        return insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 插入数据（返回geojson）
     *
     * @param layerName
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
            resultFc.put("type", "FeatureCollection");
            List<LinkedHashMap> resultFeatures = new ArrayList();
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                properties.put("OBJECTID", objectId);
                resultFeatures.add(featureMap);
            }
            resultFc.put("features", resultFeatures);
            return JSON.toJSONString(resultFc);

        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
            JSONObject properties = (JSONObject) featureMap.get("properties");
            String objectId = insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
            properties.put("OBJECTID", objectId);

            return JSON.toJSONString(featureMap);
        }
        return null;

    }

    /**
     * @param layerName
     * @param geoJSON
     * @param check      是否执行拓扑检查
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, boolean check, String dataSource) {

        if (check) {
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
            if (geo instanceof FeatureCollection) {

                LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
                resultFc.put("type", "FeatureCollection");
                List<LinkedHashMap> resultFeatures = new ArrayList();
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    List list = query(layerName, feature, null, dataSource);
//                    List list = query(layerName, feature.getAttribute("geometry").toString(), null, dataSource);
                    if (list.size() > 0) {
                        logger.error("图形已在此图层中存在,插入失败!【{}】", geometryService.toFeatureJSON(feature).toString());
                        if (featureIterator.hasNext())
                            continue;
                        else
                            throw new GISDaoException(GISDaoException.Method.INSERT, "图形已在此图层中存在,插入失败!", GISDaoException.Type.ARC_SDE);
                    }
                    CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    columns = geometryService.simpleFeature2Map(feature);
                    try {
                        for (String key : columns.keySet()) {
                            if (columns.get(key) instanceof Geometry) {
                                Geometry geometry = (Geometry) columns.get(key);
                                if (layerCRS != null && sourceCRS != null)
                                    geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                                columns.put(SE_SHAPE_FIELD, geometry.toText());
                                columns.remove(key);
                                break;
                            }
                        }
                    } catch (Exception e) {
                        logger.error(e.getLocalizedMessage());
                        throw new RuntimeException(e.getLocalizedMessage());
                    }
                    String objectId = insert(layerName, columns, dataSource);
                    LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                    JSONObject properties = (JSONObject) featureMap.get("properties");
                    properties.put("OBJECTID", objectId);
                    resultFeatures.add(featureMap);
                }
                resultFc.put("features", resultFeatures);
                return JSON.toJSONString(resultFc);

            } else if (geo instanceof SimpleFeature) {

                SimpleFeature feature = (SimpleFeature) geo;
                List list = query(layerName, feature, null, dataSource);
//                List list = query(layerName, feature.getAttribute("geometry").toString(), null, dataSource);
                if (list.size() > 0) {
                    logger.error("图形已在此图层中存在,插入失败!【{}】", geoJSON);
                    throw new GISDaoException(GISDaoException.Method.INSERT, "图形已在此图层中存在,插入失败!", GISDaoException.Type.ARC_SDE);
                }
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                String objectId = insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
                properties.put("OBJECTID", objectId);

                return JSON.toJSONString(featureMap);
            }

        } else {
            return insert2(layerName, geoJSON, dataSource);
        }


        return null;
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, Map<String, Object> columns, String dataSource) {
        return spatialDao.update(layerName, primaryKey, columns, dataSource);
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        return update(layerName, primaryKey, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 删除
     *
     * @param layerName
     * @param primaryKey
     * @param dataSource
     * @return
     */
    @Override
    public boolean delete(String layerName, String primaryKey, String dataSource) {
        return spatialDao.delete(layerName, primaryKey, dataSource);
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry      GeoJSON format
     * @param outFields
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String dltbLayerName, String xzdwLayerName, String geometry, String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        Map<String, Double> analysisResult = new HashMap<String, Double>();
        Map<String, Double> analysisTemp = new HashMap<String, Double>();
        /**
         * 地类图斑相交
         */
        String[] dltbOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "ZLDWDM", "TBBH"});
        List<Map<String, Object>> results = (List<Map<String, Object>>) intersectByGeoJSON(dltbLayerName, geometry, dltbOutFields, dataSource);
        for (Map<String, Object> result : results) {
            Double area = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (area > 0.4) {
                String dlmc = String.valueOf(result.get("DLMC"));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + area);
                analysisTemp.put(String.valueOf(result.get("ZLDWDM")) + String.valueOf(result.get("TBBH")), area);
            }
        }
        /**
         * 现状地物相交并扣除
         */
        String[] xzdwOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "KD", "KCTBDWDM2", "KCTBDWDM1", "KCTBBH1", "KCTBBH2"});
        List<Map<String, Object>> xzResults = (List<Map<String, Object>>) intersectByGeoJSON(xzdwLayerName, geometry, xzdwOutFields, dataSource);
        for (Map<String, Object> result : xzResults) {
            Double length = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (length > 0.08) {
                String dlmc = String.valueOf(result.get("DLMC"));
                Double kd = Double.parseDouble(String.valueOf(result.get("KD")));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + length * kd);
                //
                String where = "ZLDWDM='" + result.get("KCTBDWDM1") + "' and TBBH='" + result.get("KCTBBH1") + "'";
                List<Map<String, Object>> temps = new ArrayList<Map<String, Object>>();
                try {
                    temps = (List<Map<String, Object>>) query(dltbLayerName, where,
                            ArrayUtils.add2Arrays(outFields, new String[]{"DLMC"}), true, dataSource);
                } catch (Exception e) {
                    logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                }
                String kctb1Key = String.valueOf(result.get("KCTBDWDM1")) + String.valueOf(result.get("KCTBBH1"));
                //
                if (StringUtils.isBlank((String) result.get("KCTBDWDM2"))) {
                    for (Map<String, Object> temp : temps) {
                        String tmpDlmc = String.valueOf(temp.get("DLMC"));
                        analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - length * kd);
                        analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - length * kd);
                    }
                } else {
                    //
                    Tag tag = Tag.NO;
                    String tmpDlmc = "";
                    String tmpDlmc1 = "";
                    Double tmpXzdwKd = 0.0;
                    String tmpXzdwKctbdwdm1 = "";
                    String tmpXzdwKctbbh1 = "";
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb1Key)) {
                            Double tmpValue = analysisTemp.get(kctb1Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                tmpXzdwKd = (length / 2.0) * kd;
                                tmpDlmc1 = tmpDlmc;
                                tmpXzdwKctbbh1 = String.valueOf(result.get("KCTBBH1"));
                                tmpXzdwKctbdwdm1 = String.valueOf(result.get("KCTBDWDM1"));
                                analysisTemp.put(tmpXzdwKctbdwdm1 + tmpXzdwKctbbh1, tmpValue - (length / 2.0) * kd);
                            } else
                                tag = Tag.YES;
                        } else
                            tag = Tag.YES;
                    }
                    //
                    where = "ZLDWDM='" + result.get("KCTBDWDM2") + "' and TBBH='" + result.get("KCTBBH2") + "'";
                    try {
                        temps = (List<Map<String, Object>>) query(dltbLayerName, where, new String[]{"DLMC"}, true, dataSource);
                    } catch (Exception e) {
                        logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        temps.clear();
                    }
                    String kctb2Key = String.valueOf(result.get("KCTBDWDM2")) + String.valueOf(result.get("KCTBBH2"));
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb2Key)) {
                            Double tmpValue = analysisTemp.get(kctb2Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                if (!Tag.YES.equals(tag)) {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                } else {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                }
                            } else {
                                tmpDlmc = tmpDlmc1;
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                                analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                            }
                        } else {
                            tmpDlmc = tmpDlmc1;
                            analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                            analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                        }
                    }
                }
            }
        }
        return analysisResult;
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param regionCode
     * @param geometry   GeoJSON format
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String regionCode, String geometry, String dataSource) {
        regionCode = Utils.formatRegionCode(regionCode);
        String dltb = TDXZ.DLTB.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        String xzdw = TDXZ.XZDW.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        return tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
    }

    @Override
    public Map tdlyxzAnalysis2(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        if (StringUtils.isBlank(dltbLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "地类图斑图层名称为空！"));
        if (StringUtils.isBlank(xzdwLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "线状地物图层名称为空！"));
        if (StringUtils.isBlank(geometry))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "分析地块坐标为空！"));
        boolean useWcf = AppConfig.getBooleanProperty("analysis.useWcf");
        if(useWcf)
        {
            String wcfUrl = AppConfig.getProperty("wcfUrl");
            String wcfMethod = AppConfig.getProperty("wcfMethod");
            if(StringUtils.isBlank(wcfUrl))
                throw new RuntimeException(getMessage("analysis.wcf.url.null"));
            if(StringUtils.isBlank(wcfMethod))
                throw new RuntimeException(getMessage("analysis.wcf.method.null"));
            return tdlyxzAnalysisByWcf(dltbLayerName,xzdwLayerName,geometry,wcfUrl,wcfMethod);
        }else
        {
            Map<String, Double> dlMap = new HashMap<String, Double>();   //分析地类面积结果
            Map<String, Double> bhMap = new HashMap<String, Double>();
            MultiKeyMap analysisDetailMap = MultiKeyMap.decorate(new HashedMap()); //分析地类面积明细
            String[] columns = {"DLMC"};
            List<Map<String, Object>> xzdwDltbResult = new ArrayList<Map<String, Object>>();
            List<Map<String, Object>> dltbAnalysisResult = new ArrayList<Map<String, Object>>();
            List<Map<String, Object>> xzdwAnalysisResult = new ArrayList<Map<String, Object>>();
            //与地类图斑层进行相交分析
            try {
                dltbAnalysisResult = (List<Map<String, Object>>) intersect3(dltbLayerName, geometry, "DLMC;ZLDWDM;TBBH;ZLDWMC;QSXZ;DLBM;TBMJ;QSDWDM;QSDWMC".split(";"), dataSource);
            } catch (Exception e) {
                logger.error("tdlyxzAnalysis dltb error [{}] ", e.getLocalizedMessage());
            }

            if (dltbAnalysisResult.size() > 0) {
                for (int i = 0; i < dltbAnalysisResult.size(); i++) {
                    Double tmpArea = Double.parseDouble(String.valueOf(dltbAnalysisResult.get(i).get("SHAPE_AREA"))); //相交部分的面积
                    if (tmpArea > 0.4) {
                        if (dlMap.containsKey(dltbAnalysisResult.get(i).get("DLMC")))
                            dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), dlMap.get(dltbAnalysisResult.get(i).get("DLMC")) + tmpArea);
                        else
                            dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), tmpArea);
                        bhMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")) + String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), tmpArea);

                        //图斑明细
                        if (analysisDetailMap.containsKey(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")))) {
                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
//                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpArea);
                            detailValue.put("CCMJ", MapUtils.getDouble(detailValue,"CCMJ",0.0) + tmpArea);
                        } else {
                            Map<String, Object> detailValue = new HashMap<String, Object>();
                            detailValue.put("ZLDWDM", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")));
                            detailValue.put("ZLDWMC", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWMC")));
                            detailValue.put("TBBH", String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                            detailValue.put("QSXZ", String.valueOf(dltbAnalysisResult.get(i).get("QSXZ")));
                            detailValue.put("DLMC", String.valueOf(dltbAnalysisResult.get(i).get("DLMC")));
                            detailValue.put("DLBM", String.valueOf(dltbAnalysisResult.get(i).get("DLBM")));
                            detailValue.put("TBMJ", String.valueOf(dltbAnalysisResult.get(i).get("TBMJ")));
                            detailValue.put("QSDWDM", String.valueOf(dltbAnalysisResult.get(i).get("QSDWDM")));
                            detailValue.put("QSDWMC", String.valueOf(dltbAnalysisResult.get(i).get("QSDWMC")));
                            detailValue.put("CCMJ", tmpArea);
                            analysisDetailMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), detailValue);
                        }

                    }

                }
            }

            //与线状地物层进行相交分析，并扣除
            try {
                xzdwAnalysisResult = (List<Map<String, Object>>) intersect3(xzdwLayerName, geometry, "DLMC;KD;KCTBDWDM2;KCTBDWDM1;KCTBBH1;KCTBBH2;QSDWDM1;XZDWBH;QSXZ;DLBM;XZDWMJ;QSDWMC1".split(";"), dataSource);
            } catch (Exception e) {
                logger.error("tdlyxzAnalysis xzdw error [{}] ", e.getLocalizedMessage());
            }

            if (xzdwAnalysisResult.size() > 0) {
                for (int i = 0; i < xzdwAnalysisResult.size(); i++) {
                    Double tmpLength = Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("SHAPE_AREA")));
//                if (tmpLength > 0.08) {
                    if (dlMap.containsKey(xzdwAnalysisResult.get(i).get("DLMC"))) {
                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) + (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                    } else
                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                    //线物明细
                    if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))) {
                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                    } else {
                        Map<String, Object> detailValue = new HashMap<String, Object>();
                        detailValue.put("ZLDWDM", String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH"))));
                        detailValue.put("ZLDWMC", String.valueOf(xzdwAnalysisResult.get(i).get("")));
                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                        detailValue.put("QSXZ", String.valueOf(xzdwAnalysisResult.get(i).get("QSXZ")));
                        detailValue.put("DLMC", String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")));
                        detailValue.put("DLBM", String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")));
                        detailValue.put("TBMJ", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWMJ")));
                        detailValue.put("QSDWDM1", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")));
                        detailValue.put("QSDWMC", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWMC1")));
                        detailValue.put("CCMJ", tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                        analysisDetailMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), detailValue);
                    }
                    if (StringUtils.isBlank(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2"))) || String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")).equalsIgnoreCase("null")) {
                        String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {
                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        String tmpKey = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                        if (xzdwDltbResult.size() > 0&&bhMap.containsKey(tmpKey)) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                dlMap.put(String.valueOf(xzdwDltbResult.get(j).get("DLMC")), dlMap.get(String.valueOf(xzdwDltbResult.get(j).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
//                                String tmpKey = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                bhMap.put(tmpKey, bhMap.get(tmpKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                //扣除线物明细
                                if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                } else {
                                    logger.error((getMessage("analysis.tdlyxz.kc.error", "ERROR", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))))));
//                                    throw new RuntimeException("扣除线物明细异常，扣除图斑单位代码1:[ " + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + " ]");
//                                    Map<String, Object> detailValue = new HashMap<String, Object>();
//                                    detailValue.put("ZLDWDM", "error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                    analysisDetailMap.put("error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                }
                            }
                        }
                    } else {
                        //记录扣除状态
                        String kouchuTag = "no";
                        //记录扣除面积
                        double kouchuMj = 0.0;
                        //记录扣除地类
                        String kouchuDlmc = "";
                        //扣除ZLDWDM
                        String kouchuZldwdm = "";
                        //扣除tbbh
                        String kouchuTbbh = "";
                        String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {

                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                    Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        kouchuMj = (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")));
                                        kouchuDlmc = tmpDlmcKey;
                                        kouchuZldwdm = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1"));
                                        kouchuTbbh = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                        bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        //扣除线物明细
                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);
                                    } else {
                                        kouchuTag = "yes";
//                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积过小", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                        throw new RuntimeException("扣除线物明细异常:[面积过小，扣除图斑单位代码1:[ " + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + " ]]");
                                        Map<String, Object> detailValue = new HashMap<String, Object>();
                                        detailValue.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("TBMJ", tmpValue);
                                        detailValue.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                    }
                                } else {
                                    kouchuTag = "yes";
                                    //RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有占用", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                    logger.error(getMessage("analysis.tdlyxz.kc.error", "没有占用", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                    Map<String, Object> detailValue = new HashMap<String, Object>();
                                    detailValue.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                    detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                }
                            }
                            if (xzdwDltbResult.size() == 0) {
                                kouchuTag = "yes";
//                                throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有找到", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                logger.error(getMessage("analysis.tdlyxz.kc.error", "没有找到", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                        String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                        , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                Map<String, Object> detailValue = new HashMap<String, Object>();
                                detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                            }
                        } else {
                            kouchuTag = "yes";
//                            throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有找到", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                            logger.error(getMessage("analysis.tdlyxz.kc.error", "没有找到", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                    String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                    , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                            Map<String, Object> detailValue = new HashMap<String, Object>();
                            detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                            detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                            analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                        }

                        tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM2") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH2") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {
                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")))) {
                                    Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                    if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                        if (!kouchuTag.equals("yes")) {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        } else {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        }
                                    } else {
                                        try{
                                            tmpDlmcKey = kouchuDlmc;
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                        }catch (Exception ex){
                                            throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积扣除错误，出现无法扣除情况，请检查，请用实地勘测坐标叠加到图形上！!", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                    String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                    , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        }

                                        bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);

                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

//                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积过小", 2, xzdwAnalysisResult.get(i).get("KCTBDWDM2")));

                                        Map<String, Object> detailValue1 = new HashMap<String, Object>();
                                        detailValue1.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                        detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                        detailValue1.put("TBMJ", tmpValue);
                                        detailValue1.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);

                                    }
                                } else {
                                    try{
                                        tmpDlmcKey = kouchuDlmc;
                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                    }catch (Exception ex){
                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "土地利用现状底图中现状地物扣除有误，请检查!", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                    }

                                    bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);

                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

//                                    throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有占用", 2, xzdwAnalysisResult.get(i).get("KCTBDWDM2")));

                                    logger.error(getMessage("analysis.tdlyxz.kc.error", "没有占用", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")),String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"),xzdwAnalysisResult.get(i).get("KCTBDWDM2"),xzdwAnalysisResult.get(i).get("XZDWMJ"),tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                    Map<String, Object> detailValue1 = new HashMap<String, Object>();
                                    detailValue1.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                    detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                    analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);
                                }
                            }
                        }
                    }

//                }
                }
            }
            Map resultMap = new HashMap();
            resultMap.put("analysisArea", dlMap);
            List resultDetailList = new ArrayList();
            Iterator keyIterator = analysisDetailMap.keySet().iterator();
            while (keyIterator.hasNext()) {
                Object tmpKey = keyIterator.next();
                resultDetailList.add(analysisDetailMap.get(tmpKey));
            }
            resultMap.put("analysisAreaDetail", resultDetailList);
            return resultMap;
        }

    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> tdghscAnalysis(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        if (StringUtils.isBlank(layerType))
            throw new RuntimeException(getMessage("layer.type.notnull"));
        String layerName = layerType.concat(LAYER_MIDDLE_FIX_E).concat(Utils.formatRegionCode(regionCode));
        return intersect3(layerName, geometry, outFields, dataSource);
//        return intersectByGeoJSON(layerName, geometry, outFields, dataSource);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public String tdghscAnalysis2(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        List result = tdghscAnalysis(layerType, regionCode, geometry, outFields, dataSource);
        FeatureCollection collection = geometryService.list2FeatureCollection(result, null, null);
        return geometryService.toFeatureJSON(collection);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis2(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis2(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 组织土地规划分析的结果（所有图层类型）
     *
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(Map analysisResult) {

        Map resultMap = new HashMap();
        try {
            for (GHSC item : GHSC.values()) {
                List list = new ArrayList();
//                List detailList = new ArrayList();
                Map total = new HashMap();
                Map ghMap = new HashMap();
                double totalArea = 0;
                JSONObject singleObj = JSON.parseObject(analysisResult.get(item.getLabel()).toString());
                JSONArray features = JSON.parseArray(singleObj.get("features").toString());
                totalArea = getTotalArea(features);
                total.put("LXMC", "地块总面积");
                total.put("AREA", totalArea);
                list.add(total);
                switch (item.ordinal()) {
                    case 0:
                        for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name() + area);
                        }
                        break;
                    case 1:
                        for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name() + area);
                        }
                        break;
                    case 2:
                        for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("TZLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name());
                        }
                        break;
                    case 3:
                        for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);

                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("XMLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.getLxdm());
                        }
                        break;
                }
//                ghMap.put("detail", detailList);
                ghMap.put("info", list);
                resultMap.put(item.getLabel(), ghMap);
            }
        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());

        }

        return resultMap;
    }

    /**
     * 组织土地规划分析的结果（单个图层类型）
     *
     * @param layerType
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(String layerType, String analysisResult) {
        Map resultMap = new HashMap();
        try {

            JSONObject fc = JSON.parseObject(analysisResult);
            JSONArray features = (JSONArray) fc.get("features");

            List list = new ArrayList();
            Map total = new HashMap();
            Map ghMap = new HashMap();
            double totalArea = 0;
            totalArea = getTotalArea(features);
            total.put("LXMC", "地块总面积");
            total.put("AREA", totalArea);
            list.add(total);
            for (GHSC item : GHSC.values()) {

                if (item.name().equals(layerType)) {
                    switch (item.ordinal()) {
                        case 0:
                            for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    Map detailMap = new HashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name() + area);
                            }
                            break;
                        case 1:
                            for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
//                                        detailMap.put("ghsc", item.getLabel());
                                        detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name() + area);
                            }
                            break;
                        case 2:
                            for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TZLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name());
                            }
                            break;
                        case 3:
                            for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);

                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("XMLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.getLxdm());
                            }
                            break;
                    }
                    ghMap.put("info", list);
                    resultMap.put(item.getLabel(), ghMap);
                }
            }

        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultMap;
    }

    @Override
    public LinkedHashMap tdghscExcelData(Map result) {
        LinkedHashMap map = new LinkedHashMap();
        for (Object key : result.keySet()) {
            List list = new ArrayList();
            List keyList = (List) ((Map) result.get(key)).get("info");
            for (int i = 1; i < keyList.size(); i++) {
                Map temp = (Map) keyList.get(i);
                List tempList = new ArrayList();
                tempList.add(temp.get("LXMC"));
                tempList.add(Math.round(Double.valueOf(temp.get("AREA").toString())));
                list.add(tempList);
            }
            map.put(key, list);
        }
        return map;
    }

    /**
     * 组织土地利用现状分析的结果
     *
     * @param analysisResult
     * @return
     */
    @Override
    public List<Map> tdlyxzResult(Map analysisResult, String tpl) {
        List<Map> resultList = new ArrayList<Map>();
        try {
            List detailList = (List) analysisResult.get("analysisAreaDetail");
            List<Map> dwdmList = new ArrayList<Map>();

            for (int i = 0; i < detailList.size(); i++)
            {
                Map temp = new HashMap();
                List list = new ArrayList();
                int tag = 0;

                Map map = (Map) detailList.get(i);
                String xzqmc = null;
                if (map.containsKey("QSDWMC")) {
                    xzqmc = String.valueOf(map.get("QSDWMC"));
                    list.add(map);
                    temp.put("xzqmc", xzqmc);
                    temp.put("qsxz",map.get("QSXZ"));
                    temp.put("detail", list);
                } else
                    continue;
//                if (map.containsKey("ZLDWDM")) {
//                    xzqdm = String.valueOf(map.get("ZLDWDM")).substring(0, 6);
//                    list.add(map);
//                    temp.put("xzqdm", xzqdm);
//                    temp.put("detail", list);
//                } else
//                    continue;
                if (dwdmList.size() > 0)
                {
                    for (Map dwdmMap : dwdmList) {
                        if (String.valueOf(dwdmMap.get("xzqmc")).equals(xzqmc)) {
                            List tempList = (List) dwdmMap.get("detail");
                            tempList.addAll((List) temp.get("detail"));
                            tag = 1;
                            break;
                        }
                    }
                }
                if (tag == 0)
                {
                    dwdmList.add(temp);
                }
            }
            List<Map> groupList = getTdlyxzDictList(tpl);

            for (int j = 0; j < dwdmList.size(); j++) {

                Map dwMap = dwdmList.get(j);
                List<Map> detailMaps = (List<Map>) dwMap.get("detail");
                List<Map> categoryA = new ArrayList<Map>();

                for(Map item:groupList)
                {
                    Map cMap = new HashMap();
                    cMap.put("dlmc",item.get("dlmc"));
                    cMap.put("dlbm",item.get("dlbm"));
                    cMap.put("detail",new ArrayList<Map>());
                    cMap.put("area",0);
                    for (Map detailMap : detailMaps)
                    {
                        String dlbm = detailMap.get("DLBM").toString();
                        double ccmj = Double.valueOf(detailMap.get("CCMJ").toString());
                         if(item.get("dlbm").toString().equals(dlbm.substring(0,2)))
                         {
                             cMap.put("area",MapUtils.getDouble(cMap,"area",0.0)+ccmj);
                             add2List((List<Map>)cMap.get("detail"), detailMap);
                         }
                    }
                  categoryA.add(cMap);
                }
                Map temp = new HashMap();
                temp.put("xzqmc",String.valueOf(dwMap.get("xzqmc")));
                temp.put("qsxz",String.valueOf(dwMap.get("qsxz")));
                temp.put("area", getSumArea(categoryA));
                temp.put("categoryA", categoryA);
                temp.put("categoryB", getCategoryB(categoryA));
                resultList.add(temp);
            }
            Map totalMap = new  HashMap();
            totalMap.put("xzqmc","合计");
            totalMap.put("area",getSumArea(resultList));
            totalMap.put("categoryA",getSumCategoryA(resultList));
            totalMap.put("categoryB", getSumCategoryB(resultList));

            resultList.add(totalMap);
        } catch (Exception e) {
             logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultList;
    }

    /**
     *
     * 获取合计面积一级分类
     * @param list
     * @return
     */
   public List<Map> getSumCategoryA(List<Map> list)
    {
        assert list!=null;
        List<Map> result=null;
        if(list.size()>0)
        {
            result=new ArrayList<Map>();
            List<Map> firstList  = (List<Map>) list.get(0).get("categoryA");
            for(Map map:firstList)
            {
                Map temp = new HashMap();
                temp.put("dlbm",map.get("dlbm"));
                temp.put("dlmc",map.get("dlmc"));
                temp.put("area", 0);
                for(Map item:list)
                {
                    List<Map> groupList = (List<Map>) item.get("categoryA");
                    for(Map groupMap:groupList)
                    {
                        if(temp.get("dlbm").equals(groupMap.get("dlbm")))
                            temp.put("area",MapUtils.getDouble(temp,"area",0.0)+MapUtils.getDouble(groupMap,"area",0.0));
                    }
                }
                result.add(temp);
            }
        }
        return result;
    }

    /**
     *  获取合计面积二级分类
     * @param list
     * @return
     */
   public Map getSumCategoryB(List<Map> list)
    {
        assert list!=null;
        Map result=null;
        if(list.size()>0)
        {
            result=new HashMap();
            for(Map item:list)
            {
                Map categoryB = (Map) item.get("categoryB");
                for(Object key:categoryB.keySet())
                {
                     result.put(key,MapUtils.getDouble(result,key,0.0)+MapUtils.getDouble(categoryB,key,0.0));
                }
            }
        }
        return result;
    }

    public double  getXzAreaByKey(List<Map> list,String key)
    {
          assert key!=null;
        double area=0;
        for(Map map:list){
            if(map.containsKey(key))
                area += MapUtils.getDouble(map,key,0.0);
        }
        return area;
    }


    public double getXzAreaByField(List<Map> list,String field,String value,double conv){
        if(isNull(conv))conv=1;
        assert field !=null;
        assert value!=null;
        double area=0;
        List values = Arrays.asList(value.split(","));
        for(Map map:list){
            if(values.contains(map.get(field)))
                area += (Double) map.get("area");
        }
        return area*conv;
    }



    @Override
    public LinkedHashMap tdlyxzExcelData(List<Map> result) {
        LinkedHashMap map = new LinkedHashMap();
        DecimalFormat df = new DecimalFormat("0.00");
        if(result.size()>0)
        {
            List list1 = new ArrayList();
            List list2 = new ArrayList();
            map.put("一级分类", list1);
            map.put("二级分类", list2);
            for (Map item : result) {
                List temp1 = new ArrayList();
                List temp2 = new ArrayList();
                temp1.add(item.get("xzqmc"));
                temp1.add(df.format(MapUtils.getDouble(item,"area",0.0)));
                temp2.add(item.get("xzqmc"));
                temp2.add(df.format(MapUtils.getDouble(item,"area",0.0)));
                List<Map> categoryA = (List<Map>) item.get("categoryA");
                Map categoryB = (Map) item.get("categoryB");
                for (Map cMap : categoryA) {
                    temp1.add(df.format(MapUtils.getDouble(cMap,"area",0.0)));
                }
                temp2.add(df.format(MapUtils.getDouble(categoryB,"01",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"011",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"012",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"013",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"02",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"021",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"022",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"023",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"03",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"031",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"032",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"033",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"04",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"041",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"042",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"043",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"20",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"201",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"202",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"203",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"204",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"205",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"10",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"101",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"102",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"104",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"105",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"106",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"107",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"11",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"111",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"112",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"113",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"114",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"115",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"116",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"117",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"118",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"119",0.0)));

                temp2.add(df.format(MapUtils.getDouble(categoryB,"12",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"122",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"123",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"124",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"125",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"126",0.0)));
                temp2.add(df.format(MapUtils.getDouble(categoryB,"127",0.0)));

                list1.add(temp1);
                list2.add(temp2);

            }
//            List temp = new ArrayList();
//            temp = (ArrayList) list1.get(0);
//            for (int i = 0; i < temp.size(); i++) {
//                Object value = temp.get(i);
//                if (value instanceof Double) {
//                    temp.remove(i);
//                    value = Math.round((Double) value * 100) / 100.00;
//                    temp.add(i, value);
//                }
//            }
//            temp = (ArrayList) list2.get(0);
//            for (int j = 0; j < temp.size(); j++) {
//                Object value = temp.get(j);
//                if (value instanceof Double) {
//                    temp.remove(j);
//                    value = Math.round((Double) value * 100) / 100.00;
//                    temp.add(j, value);
//                }
//            }
        }
        return map;
    }

    @Override
    public List<LinkedHashMap> analysisExcelData(List list) {
        List<LinkedHashMap> result = new ArrayList<LinkedHashMap>();
        for (int i = 0; i < list.size(); i++) {
            LinkedHashMap map = new LinkedHashMap();
            List headerList = new ArrayList();
            List dataList = new ArrayList();
            JSONObject obj = (JSONObject) list.get(i);
            JSONArray children = (JSONArray) obj.get("children");
            for (int j = 0; j < children.size(); j++) {
                JSONArray child = (JSONArray) children.get(j);
                List temp = new ArrayList();

                for (int m = 0; m < child.size(); m++) {
                    JSONObject field = (JSONObject) child.get(m);
                    if (j == 0)
                        headerList.add(field.get("alias"));
                    if (field.get("value") instanceof BigDecimal)
                    temp.add(processValue(((BigDecimal) field.get("value")).doubleValue(), 3));
                    else
                        temp.add(field.get("value"));
                }
                dataList.add(temp);
//                JSONObject child = (JSONObject) children.get(j);
//                List temp = new ArrayList();
//                if (j == 0) {
//                    for (String key : child.keySet()) {
//                        headerList.add(key);
//                    }
//                }
//                if (child.get(headerList.get(0)) instanceof BigDecimal)
//                    temp.add(processValue(((BigDecimal) child.get(headerList.get(0))).doubleValue(), 3));
//                else if (child.get(headerList.get(1)) instanceof BigDecimal)
//                    temp.add(processValue(((BigDecimal) child.get(headerList.get(1))).doubleValue(), 3));
//                dataList.add(temp);
            }
            map.put("name", obj.get("alias"));
            map.put("header", headerList);
            map.put("data", dataList);
            result.add(map);
        }
        return result;
    }

    /**
     *
     * @param list
     * @return
     */
    @Override
    public List<Map> gzqydExcelData(List list) {
        try {
            List<Map> result=new ArrayList<Map>();
            for (int i = 0; i < list.size(); i++) {
                Map tempMap = new HashMap();
                List<List>  tempList = new ArrayList<List>();
                Map item = (Map) list.get(0);
                List<Map> values = (List<Map>) item.get("value");
                for(Map value:values){
                    List temp = new ArrayList(8);
                    temp.add(0,String.valueOf(value.get("OG_PRO_XZQMC")));
                    temp.add(1,String.valueOf(value.get("OG_PRO_REGIONCODE")));
                    temp.add(2,String.valueOf(value.get("OG_PRO_PRONAME")));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("011"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("012"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("020"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("030"))));
                    temp.add(3,Double.valueOf(String.valueOf(value.get("040"))));
                    tempList.add(temp);
                }
                tempMap.put("value",tempList);
                tempMap.put("type",(String) item.get("type"));
                result.add(tempMap);
            }
            return result;
        } catch (NumberFormatException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     *
     * @param data
     * @return
     */
    @Override
    public Map<String, Object> dtExcelData(Map<String,Object> data) {
        Map<String,Object> result = new HashMap();
        if(isNull(data))return null;
        try {
            for(Object key:data.keySet())
            {
                Map map = (Map) data.get(key);
                if(isNull(map))return result;
                List<Map> info = new ArrayList<Map>();
                List<Map>  detail =new ArrayList<Map>();
                Map<String,List> sheet = new HashMap();
                 if(key.equals("xz"))
                 {
                     if(!isNull(map.get("detail")))
                     {
                         detail = (List) map.get("detail");
                         List<List> xzList = new ArrayList();
                         for(Map xzMap:detail){
                             List xzTmpList = new ArrayList();
                             xzTmpList.add(isNull(xzMap.get("qsmc"))?"":xzMap.get("qsmc"));
                             xzTmpList.add(isNull(xzMap.get("nydZmj"))?"":xzMap.get("nydZmj"));
                             xzTmpList.add(isNull(xzMap.get("gdMj"))?"":xzMap.get("gdMj"));
                             xzTmpList.add(isNull(xzMap.get("011"))?"":xzMap.get("011"));
                             xzTmpList.add(isNull(xzMap.get("012"))?"":xzMap.get("012"));
                             xzTmpList.add(isNull(xzMap.get("013"))?"":xzMap.get("013"));
                             xzTmpList.add(isNull(xzMap.get("021"))?"":xzMap.get("021"));
                             xzTmpList.add(isNull(xzMap.get("022"))?"":xzMap.get("022"));
                             xzTmpList.add(isNull(xzMap.get("023"))?"":xzMap.get("023"));
                             xzTmpList.add(isNull(xzMap.get("031"))?"":xzMap.get("031"));
                             xzTmpList.add(isNull(xzMap.get("032"))?"":xzMap.get("032"));
                             xzTmpList.add(isNull(xzMap.get("033"))?"":xzMap.get("033"));
                             xzTmpList.add(isNull(xzMap.get("041"))?"":xzMap.get("041"));
                             xzTmpList.add(isNull(xzMap.get("042"))?"":xzMap.get("042"));
                             xzTmpList.add(isNull(xzMap.get("122"))?"":xzMap.get("122"));
                             xzTmpList.add(isNull(xzMap.get("104"))?"":xzMap.get("104"));
                             xzTmpList.add(isNull(xzMap.get("114"))?"":xzMap.get("114"));
                             xzTmpList.add(isNull(xzMap.get("117"))?"":xzMap.get("117"));
                             xzTmpList.add(isNull(xzMap.get("123"))?"":xzMap.get("123"));
                             xzTmpList.add(isNull(xzMap.get("jsydZmj"))?"":xzMap.get("jsydZmj"));
                             xzTmpList.add(isNull(xzMap.get("201"))?"":xzMap.get("123"));
                             xzTmpList.add(isNull(xzMap.get("202"))?"":xzMap.get("202"));
                             xzTmpList.add(isNull(xzMap.get("203"))?"":xzMap.get("203"));
                             xzTmpList.add(isNull(xzMap.get("204"))?"":xzMap.get("204"));
                             xzTmpList.add(isNull(xzMap.get("205"))?"":xzMap.get("205"));
                             xzTmpList.add(isNull(xzMap.get("101"))?"":xzMap.get("101"));
                             xzTmpList.add(isNull(xzMap.get("102"))?"":xzMap.get("102"));
                             xzTmpList.add(isNull(xzMap.get("105"))?"":xzMap.get("105"));
                             xzTmpList.add(isNull(xzMap.get("106"))?"":xzMap.get("106"));
                             xzTmpList.add(isNull(xzMap.get("107"))?"":xzMap.get("107"));
                             xzTmpList.add(isNull(xzMap.get("113"))?"":xzMap.get("113"));
                             xzTmpList.add(isNull(xzMap.get("118"))?"":xzMap.get("118"));
                             xzTmpList.add(isNull(xzMap.get("wlydZmj"))?"":xzMap.get("wlydZmj"));
                             xzTmpList.add(isNull(xzMap.get("043"))?"":xzMap.get("043"));
                             xzTmpList.add(isNull(xzMap.get("124"))?"":xzMap.get("124"));
                             xzTmpList.add(isNull(xzMap.get("125"))?"":xzMap.get("125"));
                             xzTmpList.add(isNull(xzMap.get("126"))?"":xzMap.get("126"));
                             xzTmpList.add(isNull(xzMap.get("127"))?"":xzMap.get("127"));
                             xzTmpList.add(isNull(xzMap.get("121"))?"":xzMap.get("121"));
                             xzTmpList.add(isNull(xzMap.get("111"))?"":xzMap.get("111"));
                             xzTmpList.add(isNull(xzMap.get("112"))?"":xzMap.get("112"));
                             xzTmpList.add(isNull(xzMap.get("115"))?"":xzMap.get("115"));
                             xzTmpList.add(isNull(xzMap.get("116"))?"":xzMap.get("116"));
                             xzTmpList.add(isNull(xzMap.get("119"))?"":xzMap.get("119"));
                             xzTmpList.add(isNull(xzMap.get("zmj"))?"":xzMap.get("zmj"));
                             xzList.add(xzTmpList);
                         }
                         if(xzList.size()>0)
                             sheet.put("Sheet1",xzList);
                     }
                     result.put(String.valueOf(key),sheet);

                 }else if(key.equals("bp"))
                 {
                     List list1 = new ArrayList();
                     List list2 = new ArrayList();
                     if (!isNull(map.get("info")))
                         info = (List) map.get("info");
                     if (!isNull(map.get("detail")))
                         detail = (List) map.get("detail");
                     if(info.size()>0&&detail.size()>0)
                     {
                         for(Map temp:info)
                         {
                             List tempList = new ArrayList();
                             tempList.add(temp.get("label"));
                             tempList.add(temp.get("area_pfm"));
                             tempList.add(temp.get("area_gq"));
                             tempList.add(temp.get("area_m"));
                             list1.add(tempList);
                         }
                         List<String> keyList = new ArrayList();
                         for(Object detailKey:detail.get(0).keySet())
                         {
                             keyList.add(String.valueOf(detailKey));
                         }
                         list2.add(keyList);
                         for(Map detailTmp:detail)
                         {
                             List tempList = new ArrayList();
                             for(String key1:keyList){
                                 tempList.add(detailTmp.get(key1));
                             }
                             list2.add(tempList);
                         }
                         sheet.put("报批汇总",list1);
                         sheet.put("报批详情",list2);
                     }
                     result.put(String.valueOf(key),sheet);
                 }else
                 {
                     List<Map> list = new ArrayList<Map>();
                     List<List> bpList = new ArrayList<List>();
                     if(!isNull(map.get("detail")))
                     {
                         detail = (List) map.get("detail");
                         List firstRow = new ArrayList();
                         List secondRow = new ArrayList();
                         firstRow.add("用途");
                         secondRow.add("面积");
                         for(Map ghMap:detail)
                         {
                             firstRow.add(ghMap.get("label"));
                             secondRow.add(ghMap.get("area"));
                         }
                         bpList.add(secondRow);
                         sheet.put("header",firstRow);
                         sheet.put("data",bpList);
                         list.add(sheet);
                     }
                     result.put(String.valueOf(key),list);
                 }
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
           throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }


    /**
     * 处理double类型数据,返回处理后的新数据
     *
     * @param value
     * @param precision
     * @return
     */
    public double processValue(Double value, int precision) {

        switch (precision) {
            case 1:
                return Math.round(value * 100) / 100.0;
            case 2:
                return Math.round(value * 100) / 100.00;
            case 3:
                return Math.round(value * 100) / 100.000;
            default:
                return Math.round(value * 100) / 100;

        }
    }


    public double getSumArea(List<Map> list) {
        assert list!=null;
        double area = 0;
        if(list.size()>0)
        {
            for (Map map : list) {
                area += Double.valueOf(String.valueOf(map.get("area")).toString());
            }
        }
        return area;
    }

    public Map getCategoryB(List<Map> list) {
        Map result = new HashMap();
        try {
            for (Map map : list) {
                result.put(map.get("dlbm"), map.get("area"));
                if (map.containsKey("detail")) {
                    List<Map> detailList = (List<Map>) map.get("detail");
                    for (Map detail : detailList) {
                        result.put(detail.get("dlbm"), detail.get("area"));
                    }
                }
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }

    public List<Map> add2List(List<Map> list, Map map) {
        int tag = 0;
        String dlbm = map.get("DLBM").toString();
        double area = Double.valueOf(map.get("CCMJ").toString());
        Map temp = new HashMap();
        try {
            for (Map item : list) {
                if (item.get("dlbm").equals(dlbm)) {
                    double newArea = Double.valueOf(item.get("area").toString()) + area;
                    item.put("area", newArea);
                    tag = 1;
                }
            }
            if (tag == 0) {
                temp.put("area", area);
                temp.put("dlbm", dlbm);
                list.add(temp);
            }
        } catch (NumberFormatException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return list;
    }


    public List<Map> tdlyxzResult2(Map analysisResult, String tpl) {
        List<Map> resultList = new ArrayList<Map>();
        List<Map> list = (List) analysisResult.get("analysisAreaDetail");
        List<Map> groupList = getTdlyxzDictList(tpl);
        for (Map map : list) {
            String dlbm = map.get("DLBM").toString();
            double area = Double.valueOf(map.get("CCMJ").toString());
            for (Map groupMap : groupList) {
                if (groupMap.get("dlbm").toString().equals(dlbm.substring(0, 2))) {
                    double newArea = Double.valueOf(groupMap.get("area").toString()) + area;
                    ((List) groupMap.get("detail")).add(map);
                    groupMap.put("area", newArea);
                }
            }
        }
        for (int i = 0; i < groupList.size(); i++) {
            Map map = groupList.get(i);
            Map temp = new HashMap();
            double area = Double.valueOf(map.get("area").toString());
            if (area > 0) {
                temp.put("dlmc", map.get("dlmc"));
                temp.put("area", map.get("area"));
                temp.put("detail", map.get("detail"));
                resultList.add(temp);
            }
        }
        return resultList;
    }

    /**
     * 快速检索
     *
     * @param value
     * @return
     */
    @Override
    public List search(String value, boolean returnGeometry) {
        if (searchConfig == null) throw new RuntimeException(getMessage("search.config.not.found"));
        List result = new ArrayList();
        List<Map> layers = (List) searchConfig.get(SEARCH_LAYERS);
        for (Map layer : layers) {
            String name = (String) layer.get("layer");
            String[] fields = ((String) layer.get("fields")).split(",");
            String where = getWhereClause(fields, value);
            result.addAll(spatialDao.query(name, where, fields, returnGeometry, 10, null));
        }
        return result;
    }

    /**
     * 根据相关条件输出图片
     * @param where 查询子句
     * @param mapScale 出图的比例尺
     * @param width 图片的width
     * @param height 图片的height
     * @param type 出图类型(xz,gh)
     * @return
     */
    @Override
    public InputStream exportMap(String where, String mapScale, int width,int height, String type) {
        if (exportConfig == null) {
            logger.info("无配置信息");
            return null;
        }
        int bgWidth = width +10;
        int bgHeight = height+10;
        String imageSize = width+","+height;
        double scale = 0;
        if(StringUtils.isNotBlank(mapScale))
            scale = Double.valueOf(mapScale);
        InputStream inputStream;
        List<BufferedImage> baseImages=new ArrayList<BufferedImage>();
        BufferedImage topImage = null;
        BufferedImage resultImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_BGR);
        Envelope envelope =null;
        Map operaLayer = (Map) exportConfig.get("operaLayer");
        List<Map> ghLayers = (List<Map>) exportConfig.get("ghLayers");
        List<Map> xzLayers = (List<Map>) exportConfig.get("xzLayers");
        String layerName = (String) operaLayer.get("layerName");
        String dataSource = (String) operaLayer.get("dataSource");
        String url = (String) operaLayer.get("url");
        List list = query(layerName,where,null,true,dataSource);
        if(list.size()>0){
            FeatureCollection featureCollection =  geometryService.list2FeatureCollection(list, null, null);
            envelope = featureCollection.getBounds();
            topImage = getExportImage(envelope,url,imageSize,scale);

            if(type.equals(EXPORT_TYPE_GH)){
                for (Map item:ghLayers){
                    BufferedImage tempImage = getExportImage(envelope,(String) item.get("url"),bgWidth+","+bgHeight,scale);
                    baseImages.add(tempImage);
                }
            } else if (type.equals(EXPORT_TYPE_XZ)) {
                for (Map item : xzLayers) {
                    BufferedImage tempImage = getExportImage(envelope, (String) item.get("url"), bgWidth+","+bgHeight, scale);
                    baseImages.add(tempImage);
                }
            } else {
                logger.error("type is not found");
            }
            Graphics2D graphics2D = resultImage.createGraphics();
            resultImage = graphics2D.getDeviceConfiguration().createCompatibleImage(width, height, Transparency.TRANSLUCENT);
            graphics2D.dispose();
            graphics2D = resultImage.createGraphics();
            if (baseImages.size() > 0) {
                for (BufferedImage tempImage : baseImages) {
                    graphics2D.drawImage(tempImage, 0, 0, tempImage.getWidth(), tempImage.getHeight(), null);
                }
            }
            graphics2D.drawImage(topImage, 0, 0, topImage.getWidth(), topImage.getHeight(), null);
            graphics2D.dispose();
        }else
        {
            logger.info("未查询到地块!");
            return null;
//            Graphics2D g2d = resultImage.createGraphics();
//            Font font = new Font("微软雅黑", Font.BOLD, 25);
//            String s = "未查询到地块";
//            g2d.setBackground(Color.white);
//            g2d.clearRect(0, 0, width, height);
//            g2d.setColor(Color.red);
//            FontRenderContext context = g2d.getFontRenderContext();
//            Rectangle2D bounds = font.getStringBounds(s, context);
//            double x = (width - bounds.getWidth()) / 2;
//            double y = (height - bounds.getHeight()) / 2;
//            g2d.drawString(s, (int)x, (int)y);
//            g2d.dispose();
//            resultImage.flush();
        }
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ImageIO.write(resultImage, "png", out);
            inputStream = new ByteArrayInputStream(out.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return inputStream;
    }


    private enum MultipleAnalysis{
        XZ,GH,BP,GD
    }

    private enum MTag {
        GROUP,ID,RESULT
    }

    private enum GhTag{
       JBNT_AREA,YBNT_AREA,SY_AREA,LD_AREA,YD_AREA,YXJSQ_AREA,YTJJSQ_AREA,WLYD_AREA
    }

    private enum BpTag{
        YB_AREA,WB_AREA,TOTAL_AREA
    }

    private enum GdTag{
        YG_AREA,WG_AREA,TOTAL_AREA
    }
    private enum XzTag{
        QSMC,GD_AREA,YD_AREA,LD_AREA,MCD_AREA,QTNYD_AREA,JMD_AREA,JT_AREA,SL_AREA,WLYTD_AREA,QTTD_AREA,NYD_AREA,JSYD_AREA,WLYD_AREA,TOTAL_AREA

    }

    /**
     * 分析的id
     */
    private enum IdTag{
        tdytq,jqdltb,jsydgzq,ghjbnttz,mzzdjsxm,dltb,xzdw,bpdk,gddk
    }
    /**
     * 权属性质
     */
    private enum QSXZ {
        GY("10,20");
        private String label;

        private QSXZ(String value) {
            this.label = value;
        }

        private String getLabel() {
            return label;
        }
    }

    private enum QsTag{
        gy,jt
    }

    /**
     * 公顷 的换算值
     */
    private final double HECTARE_CONV=0.0001;

    private final String QS_GY="国有";

    private final String QS_JT = "集体";


    /**
     *  当涂 多重，复杂综合性的分析
     * @param geometry
     * @param analysisLayers
     * @param dataSource
     * @return
     */
    @Override
    public Map multipleAnalysis(String geometry, List analysisLayers, String dataSource,String tpl) {
        if (isNull(analysisLayers)) throw new RuntimeException("no analysis Layers");
        if (isNull(geometry)) throw new JSONMessageException("geometry is null");
        String xzdwLyr=null;
        String dltbLyr=null;
        String tdghscYear=null;
        Map result = new HashMap();
        List<Map> tdlyxzResult = new ArrayList<Map>();
        try {
            List<Map> analysisList = new ArrayList<Map>();
            for (int i = 0; i < analysisLayers.size(); i++) {
                Map layer = (Map) analysisLayers.get(i);
                String lyrName = (String) layer.get("layerName");
                String outFields = layer.get("fields").toString();
                String[] fields = "*".equals(outFields) ? null : outFields.split(",");

                if(layer.get("id").equals(IdTag.xzdw.name()))
                {
                    xzdwLyr= (String) layer.get("layerName");
                    continue;
                }
                if(layer.get("id").equals(IdTag.dltb.name()))
                {
                    dltbLyr= (String) layer.get("layerName");
                    continue;
                }
                List<Map> list = (List<Map>) intersectByGeoJSON(lyrName,geometry,fields,dataSource);
                Map temp = new HashMap();
                temp.put(MTag.GROUP.name(),layer.get("group"));
                temp.put(MTag.ID.name(),layer.get("id"));
                temp.put(MTag.RESULT.name(),list);
                analysisList.add(temp);
                if(layer.get("group").equals(MultipleAnalysis.GH.name().toLowerCase()))
                {
                    tdghscYear = Arrays.asList(layer.get("layerName").toString().split(LAYER_MIDDLE_FIX_E)).get(1);
                }
            }
            if(StringUtils.isNotBlank(xzdwLyr)&&StringUtils.isNotBlank(dltbLyr))
            {
                Map tdlyxzMap = tdlyxzAnalysis2(dltbLyr,xzdwLyr,geometry,dataSource);
                tdlyxzResult = tdlyxzResult(tdlyxzMap, tpl);
                result.put("tdlyxz",tdlyxzResult);
            }
            if(!isNull(tdghscYear)&&!result.containsKey("tdghsc"))
            {
                Map tdghscMap = tdghscAnalysis2(tdghscYear, geometry, null, dataSource);
                result.put("tdghsc",tdghscResult(tdghscMap));
            }
            Map<String,List> groupMap = groupList(analysisList,MTag.GROUP.name());
            groupMap.put(MultipleAnalysis.XZ.name().toLowerCase(),tdlyxzResult);
            if(!groupMap.isEmpty())
                result.put("result", parseMultiAnalysis(groupMap, geometry));
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }

    /**
     *
     * @param analysisMap
     * @return
     */
    @Override
    public Map multipleAnalysisExcel(Map analysisMap) {
        Map result = new HashMap();
        result.put("gh", getSheetMap("sheet1", analysisMap.get("ghResult"),false));

        LinkedHashMap map =null;
        List<LinkedHashMap> list = null;
        DecimalFormat df = new DecimalFormat("0.00");
        DecimalFormat df_h = new DecimalFormat("0.0000");

        if(analysisMap.containsKey("bpInfo"))
        {
            Map<String,Object> temp = (Map<String, Object>) analysisMap.get("bpInfo");
            list = new ArrayList<LinkedHashMap>();
            for(String key:temp.keySet())
            {
                map = new LinkedHashMap();
                if (key.equals(BpTag.YB_AREA.name()))
                    map.put("lx", "已报");
                else if (key.equals(BpTag.WB_AREA.name()))
                    map.put("lx", "未报");
                else if (key.equals(BpTag.TOTAL_AREA.name()))
                    map.put("lx", "合计");
                map.put(key, df.format(MapUtils.getDouble(temp, key,0.0).doubleValue()));
                map.put("yb_h", df_h.format(MapUtils.getDouble(temp, key,0.0).doubleValue() * HECTARE_CONV));
                map.put("yb_a", df.format(MapUtils.getDouble(temp, key,0.0).doubleValue() * ACRES_CONV));
                list.add(map);
            }
             Map excelMap = getSheetMap("报批汇总",list,false);

            if(analysisMap.containsKey("bpDetail"))
            {
                List<Map<String, ?>> detailList = (List<Map<String, ?>>) analysisMap.get("bpDetail");
                list = new ArrayList<LinkedHashMap>();
                if(detailList.size()>0)
                {
                    Map<String, ?> detailItem = detailList.get(0);
                    List<String> headerList = new ArrayList<String>();
                    List dataList = new ArrayList();
                    List wrapList = new ArrayList();
                    for (String key : detailItem.keySet()) {
                        String keyUpp = key.toUpperCase().trim();
                        if (keyUpp.contains(SE_SHAPE_FIELD) || keyUpp.contains(SE_SHAPE_AREA) || keyUpp.contains(OG_PRO_PERFIX))
                            continue;
                        headerList.add(key);
                    }
                    if(headerList.size()>0)
                    {
                        wrapList.add(headerList);
                        dataList.addAll(wrapList);
                    }
                    for(Map detailMap:detailList)
                    {
                        wrapList = new ArrayList();
                        List tempList = new ArrayList();
                        for(String header:headerList)
                        {
                            tempList.add(detailMap.get(header));
                        }
                        if(tempList.size()>0)
                        {
                            wrapList.add(tempList);
                            dataList.addAll(wrapList);
                        }
                    }
                    if(dataList.size()>0)
                        excelMap.put("报批详情",dataList);
                }
            }
            result.put("bp",excelMap);
        }
        if(analysisMap.containsKey("gdInfo"))
        {
            Map<String,Object> temp = (Map<String, Object>) analysisMap.get("gdInfo");
            list = new ArrayList<LinkedHashMap>();
            for(String key:temp.keySet())
            {
                map = new LinkedHashMap();
                if (key.equals(GdTag.YG_AREA.name()))
                    map.put("lx", "已供");
                else if (key.equals(GdTag.WG_AREA.name()))
                    map.put("lx", "未供");
                else if (key.equals(GdTag.TOTAL_AREA.name()))
                    map.put("lx", "合计");
                map.put(key, df.format(MapUtils.getDouble(temp, key,0.0).doubleValue()));
                map.put("yg_h", df_h.format(MapUtils.getDouble(temp, key,0.0).doubleValue() * HECTARE_CONV));
                map.put("yg_a", df.format(MapUtils.getDouble(temp, key,0.0).doubleValue() * ACRES_CONV));
                list.add(map);
            }
            Map excelMap = getSheetMap("供地汇总",list,false);

            if(analysisMap.containsKey("gdDetail"))
            {
                List<Map<String, ?>> detailList = (List<Map<String, ?>>) analysisMap.get("gdDetail");
                list = new ArrayList<LinkedHashMap>();
                if(detailList.size()>0)
                {
                    Map<String, ?> detailItem = detailList.get(0);
                    List<String> headerList = new ArrayList<String>();
                    List dataList = new ArrayList();
                    List wrapList = new ArrayList();
                    for (String key : detailItem.keySet()) {
                        String keyUpp = key.toUpperCase().trim();
                        if (keyUpp.contains(SE_SHAPE_FIELD) || keyUpp.contains(SE_SHAPE_AREA) || keyUpp.contains(OG_PRO_PERFIX))
                            continue;
                        headerList.add(key);
                    }
                    if(headerList.size()>0)
                    {
                        wrapList.add(headerList);
                        dataList.addAll(wrapList);
                    }
                    for(Map detailMap:detailList)
                    {
                        wrapList = new ArrayList();
                        List tempList = new ArrayList();
                        for(String header:headerList)
                        {
                            tempList.add(detailMap.get(header));
                        }
                        if(tempList.size()>0)
                        {
                            wrapList.add(tempList);
                            dataList.addAll(wrapList);
                        }
                    }
                    if(dataList.size()>0)
                        excelMap.put("供地详情",dataList);
                }
            }
            result.put("gd",excelMap);
        }
        if(analysisMap.containsKey("xzResult"))
        {
            result.put("xz",getSheetMap("sheet1",analysisMap.get("xzResult"),true));
        }
        return result;
    }

    public Map<String,List> getSheetMap(String sheetName,Object data,boolean isXz)
    {
        Map<String,List> sheet=null;
        List<List> sheetList=null;
        DecimalFormat df = new DecimalFormat("0.00");
        try {
            if (data instanceof Map) {
                Map temp = (Map) data;
                sheetList = new ArrayList<List>();
                List tempList = new ArrayList();
                for (Object key : ((Map) data).keySet()) {
                    tempList.add(temp.get(key));
                }
                sheetList.add(tempList);
                sheet = new HashMap<String, List>();
                sheet.put(sheetName, sheetList);
            } else if (data instanceof List)
            {
                List<Map> temp = (List<Map>) data;
                sheetList = new ArrayList<List>();
                for (Map item : temp) {
                    List itemList = new ArrayList();
                    if (isXz&&item.keySet().size()>0) {
                        itemList.add(item.get(XzTag.QSMC.name()));
                        itemList.add(df.format(MapUtils.getDouble(item, XzTag.NYD_AREA.name(), 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, XzTag.GD_AREA.name(), 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "011", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item,"012",0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "013", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "021", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "022", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "023", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "031", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "032", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "033", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "041", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "042", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "122", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "104", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "114", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "117", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "123", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, XzTag.JSYD_AREA.name(), 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "201", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "202", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "203", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "204", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "205", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "101", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "102", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "105", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "106", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "107", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "113", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "118", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, XzTag.WLYD_AREA.name(), 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "043", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "124", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "125", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "126", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "127", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "121", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "111", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "112", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "115", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "116", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, "119", 0.0)));
                        itemList.add(df.format(MapUtils.getDouble(item, XzTag.TOTAL_AREA.name(), 0.0)));
                        itemList.add("");

                    } else {
                        for (Object key : item.keySet()) {
                            itemList.add(item.get(key));
                        }
                    }
                    sheetList.add(itemList);
                }
                sheet = new HashMap<String, List>();
                sheet.put(sheetName,sheetList);
            }
        } catch (Exception e) {
            logger.error("generate sheet error:"+e.getLocalizedMessage());
            throw new RuntimeException("generate sheet error:"+e.getLocalizedMessage());
        }
        return sheet;
    }
    
    
    public Map<String,List> list2Map(List<Map> list){
        assert list!=null;
        Map<String,List> result =new HashMap<String, List>();
        for(Map item:list){
            result.put(item.get(MTag.ID.name()).toString(),(List)item.get(MTag.RESULT.name()));
        }
        return result;
    }
    /**
     *
     * 组织解析 分析所得数据
     * @param map
     * @return
     */
    public Map parseMultiAnalysis(Map<String,List> map,String geometry){
        Map result = new HashMap();
        Object geo = geometryService.readUnTypeGeoJSON(geometry);
        DecimalFormat df_h = new DecimalFormat("0.0000");
        double geoArea = getAreaByGeometry(geo,1);
        logger.info("[geometry area]:"+geoArea);
        try {
            Map<String,List> ghMap = list2Map(map.get(MultipleAnalysis.GH.name().toLowerCase()));
            Map<String,List> bpMap = list2Map(map.get(MultipleAnalysis.BP.name().toLowerCase()));
            Map<String,List> gdMap = list2Map(map.get(MultipleAnalysis.GD.name().toLowerCase()));

            List<Map> xzList = map.get(MultipleAnalysis.XZ.name().toLowerCase());

            LinkedHashMap<String,Object> ghResult=new LinkedHashMap<String, Object>();
            LinkedHashMap<String,Object> bpInfo= new LinkedHashMap<String, Object>();
            LinkedHashMap<String,Object> gdInfo= new LinkedHashMap<String, Object>();
            List bpDetail=new ArrayList();
            List gdDetail=new ArrayList();

            List<LinkedHashMap> xzResult = new ArrayList<LinkedHashMap>();

            List<LinkedHashMap> gyList = new ArrayList<LinkedHashMap>();
            List<LinkedHashMap> jtList = new ArrayList<LinkedHashMap>();

            if(!ghMap.isEmpty())
            {
                List tdytqList = ghMap.get(IdTag.tdytq.name());
                List jqdltbList = ghMap.get(IdTag.jqdltb.name());
                List jsydgzqList = ghMap.get(IdTag.jsydgzq.name());
                ghResult.put(GhTag.JBNT_AREA.name(),df_h.format(getAreaByField(tdytqList, "TDYTQLXDM", "010", HECTARE_CONV)));
                ghResult.put(GhTag.YBNT_AREA.name(), df_h.format(getAreaByField(tdytqList, "TDYTQLXDM", "020", HECTARE_CONV)));
                ghResult.put(GhTag.SY_AREA.name(), df_h.format(getAreaByField(jqdltbList, "DLMC", "农田水利用地", HECTARE_CONV)));
                ghResult.put(GhTag.LD_AREA.name(), df_h.format(getAreaByField(jqdltbList, "DLMC", "林地", HECTARE_CONV)));
                ghResult.put(GhTag.YD_AREA.name(), df_h.format(getAreaByField(jqdltbList, "DLMC", "园地", HECTARE_CONV)));
                ghResult.put(GhTag.YXJSQ_AREA.name(), df_h.format(getAreaByField(jsydgzqList, "GZQLXDM", "010", HECTARE_CONV)));
                ghResult.put(GhTag.YTJJSQ_AREA.name(), df_h.format(getAreaByField(jsydgzqList, "GZQLXDM", "020", HECTARE_CONV)));
                ghResult.put(GhTag.WLYD_AREA.name(), df_h.format(getAreaByField(jqdltbList, "DLMC", "滩涂,河流水面,湖泊水面,自然保留地", HECTARE_CONV)));
            }
            if(!bpMap.isEmpty())
            {
                bpDetail = bpMap.get(IdTag.bpdk.name());
                bpInfo.put(BpTag.YB_AREA.name(), getAreaByList(bpDetail,1));
                bpInfo.put(BpTag.WB_AREA.name(),geoArea-getAreaByList(bpDetail,1)<0?0:(geoArea-getAreaByList(bpDetail,1)));
                bpInfo.put(BpTag.TOTAL_AREA.name(),geoArea);
            }
            if(!gdMap.isEmpty())
            {
                gdDetail = gdMap.get(IdTag.gddk.name());
                gdInfo.put(GdTag.YG_AREA.name(),getAreaByList(gdDetail,1));
                gdInfo.put(GdTag.WG_AREA.name(),geoArea-getAreaByList(gdDetail,1)<0?0:(geoArea-getAreaByList(gdDetail,1)));
                gdInfo.put(GdTag.TOTAL_AREA.name(),geoArea);
            }
            if(!isNull(xzList))
            {
                for(Map xzMap:xzList)
                {
                    LinkedHashMap item = new LinkedHashMap();
                    Map categoryB = (Map) xzMap.get("categoryB");
                    if (xzMap.containsKey("qsxz"))
                    {
                        if (QSXZ.GY.getLabel().indexOf(String.valueOf(xzMap.get("qsxz"))) > -1) {
                            gyList.add(item);
                            item.put(XzTag.QSMC.name(), xzMap.get("xzqmc") + "(" + QS_GY + ")");
                        } else {
                            jtList.add(item);
                            item.put(XzTag.QSMC.name(), xzMap.get("xzqmc") + "(" + QS_JT + ")");
                        }
                    } else
                        item.put(XzTag.QSMC.name(), xzMap.get("xzqmc"));

                    item.put("011",MapUtils.getDouble(categoryB, "011", 0.0));
                    item.put("012",MapUtils.getDouble(categoryB, "012", 0.0));
                    item.put("013",MapUtils.getDouble(categoryB, "013", 0.0));

                    item.put("021",MapUtils.getDouble(categoryB, "021", 0.0));
                    item.put("022",MapUtils.getDouble(categoryB, "022", 0.0));
                    item.put("023",MapUtils.getDouble(categoryB, "023", 0.0));

                    item.put("031",MapUtils.getDouble(categoryB, "031", 0.0));
                    item.put("032",MapUtils.getDouble(categoryB, "032", 0.0));
                    item.put("033",MapUtils.getDouble(categoryB, "033", 0.0));

                    item.put("041",MapUtils.getDouble(categoryB, "041", 0.0));
                    item.put("042",MapUtils.getDouble(categoryB,"042",0.0));

                    item.put("122",MapUtils.getDouble(categoryB, "122", 0.0));
                    item.put("104",MapUtils.getDouble(categoryB, "104", 0.0));
                    item.put("114",MapUtils.getDouble(categoryB, "114", 0.0));
                    item.put("117",MapUtils.getDouble(categoryB, "117", 0.0));
                    item.put("123",MapUtils.getDouble(categoryB, "123", 0.0));

                    item.put("201",MapUtils.getDouble(categoryB, "201", 0.0));
                    item.put("202",MapUtils.getDouble(categoryB, "202", 0.0));
                    item.put("203",MapUtils.getDouble(categoryB, "203", 0.0));
                    item.put("204",MapUtils.getDouble(categoryB, "204", 0.0));
                    item.put("205",MapUtils.getDouble(categoryB, "205", 0.0));

                    item.put("101",MapUtils.getDouble(categoryB, "101", 0.0));
                    item.put("102",MapUtils.getDouble(categoryB, "102", 0.0));
                    item.put("105",MapUtils.getDouble(categoryB, "105", 0.0));
                    item.put("106",MapUtils.getDouble(categoryB, "106", 0.0));
                    item.put("107",MapUtils.getDouble(categoryB, "107", 0.0));

                    item.put("113",MapUtils.getDouble(categoryB, "113", 0.0));
                    item.put("118",MapUtils.getDouble(categoryB, "118", 0.0));

                    item.put("043",MapUtils.getDouble(categoryB,"043",0.0));
                    item.put("124",MapUtils.getDouble(categoryB,"124",0.0));
                    item.put("125",MapUtils.getDouble(categoryB,"125",0.0));
                    item.put("126",MapUtils.getDouble(categoryB,"126",0.0));
                    item.put("127",MapUtils.getDouble(categoryB,"127",0.0));
                    item.put("121",MapUtils.getDouble(categoryB,"121",0.0));

                    item.put("111",MapUtils.getDouble(categoryB, "111", 0.0));
                    item.put("112",MapUtils.getDouble(categoryB, "112", 0.0));
                    item.put("115",MapUtils.getDouble(categoryB, "115", 0.0));
                    item.put("116",MapUtils.getDouble(categoryB, "116", 0.0));
                    item.put("119",MapUtils.getDouble(categoryB, "119", 0.0));


                    item.put(XzTag.GD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("011,012,013".split(",")),1));
                    item.put(XzTag.LD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("021,022,023".split(",")),1));
                    item.put(XzTag.YD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("031,032,033".split(",")),1));
                    item.put(XzTag.MCD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("041,042".split(",")),1));
                    item.put(XzTag.QTNYD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("122,104,114,117,123".split(",")),1));
                    item.put(XzTag.JMD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("201,202,203,204,205".split(",")),1));
                    item.put(XzTag.JT_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("101,102,105,106,107".split(",")),1));
                    item.put(XzTag.SL_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("113,118".split(",")),1));
                    item.put(XzTag.WLYTD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("043,124,125,126,127,121".split(",")),1));
                    item.put(XzTag.QTTD_AREA.name(),getAreaByMapKeys(categoryB,Arrays.asList("111,112,115,116,119".split(",")),1));

                    item.put(XzTag.NYD_AREA.name(), MapUtils.getDouble(item, XzTag.GD_AREA.name()).doubleValue() +
                            MapUtils.getDouble(item, XzTag.YD_AREA.name()).doubleValue() + MapUtils.getDouble(item, XzTag.LD_AREA.name()).doubleValue() +
                            MapUtils.getDouble(item, XzTag.MCD_AREA.name()).doubleValue() + MapUtils.getDouble(item, XzTag.QTNYD_AREA.name()).doubleValue());
                    item.put(XzTag.JSYD_AREA.name(), MapUtils.getDouble(item, XzTag.JMD_AREA.name()).doubleValue() + MapUtils.getDouble(item, XzTag.JT_AREA.name()).doubleValue() +
                            MapUtils.getDouble(item, XzTag.SL_AREA.name()).doubleValue());
                    item.put(XzTag.WLYD_AREA.name(), MapUtils.getDouble(item, XzTag.WLYTD_AREA.name()).doubleValue() + MapUtils.getDouble(item, XzTag.QTTD_AREA.name()).doubleValue());

                    item.put(XzTag.TOTAL_AREA.name(),MapUtils.getDouble(xzMap,"area"));

                    xzResult.add(item);
                }
            }
            LinkedHashMap gyMap =qsList2Map(gyList,QsTag.gy.name());
            LinkedHashMap jtMap = qsList2Map(jtList,QsTag.jt.name());
            if(!gyMap.isEmpty())
                xzResult.add(xzResult.size()-1,gyMap);
            if(!jtMap.isEmpty())
                xzResult.add(xzResult.size()-1,jtMap);

            result.put("ghResult",ghResult);
            result.put("bpInfo",bpInfo);
            result.put("bpDetail",bpDetail);
            result.put("gdInfo",gdInfo);
            result.put("gdDetail",gdDetail);
            result.put("xzResult",xzResult);

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;

    }
    /**
     * 权属性质 list to map
     * @param list
     * @param type
     * @return
     */
    public LinkedHashMap qsList2Map(List<LinkedHashMap> list,String type){
        assert list!=null;
        assert type!=null;
        LinkedHashMap result=new LinkedHashMap();
        try {
            if(list.size()>0)
            {
               if(type.equals(QsTag.gy.name()))
                   result.put(XzTag.QSMC.name(),"合计国有");
                if(type.equals(QsTag.jt.name()))
                    result.put(XzTag.QSMC.name(),"合计集体");
                result.put("011",getDouValueByField(list,"011"));
                result.put("012",getDouValueByField(list,"012"));
                result.put("013",getDouValueByField(list,"013"));
                result.put("021",getDouValueByField(list,"021"));
                result.put("022",getDouValueByField(list,"022"));
                result.put("023",getDouValueByField(list,"023"));
                result.put("031",getDouValueByField(list,"031"));
                result.put("032",getDouValueByField(list,"032"));
                result.put("033",getDouValueByField(list,"033"));

                result.put("041",getDouValueByField(list,"041"));
                result.put("042",getDouValueByField(list,"042"));

                result.put("122",getDouValueByField(list,"122"));
                result.put("104",getDouValueByField(list,"104"));
                result.put("114",getDouValueByField(list,"114"));
                result.put("117",getDouValueByField(list,"117"));
                result.put("123",getDouValueByField(list,"123"));

                result.put("201",getDouValueByField(list,"201"));
                result.put("202",getDouValueByField(list,"202"));
                result.put("203",getDouValueByField(list,"203"));
                result.put("204",getDouValueByField(list,"204"));
                result.put("205",getDouValueByField(list,"205"));

                result.put("101",getDouValueByField(list,"101"));
                result.put("102",getDouValueByField(list,"102"));
                result.put("105",getDouValueByField(list,"105"));
                result.put("106",getDouValueByField(list,"106"));
                result.put("107",getDouValueByField(list,"107"));

                result.put("113",getDouValueByField(list,"113"));
                result.put("118",getDouValueByField(list,"118"));

                result.put("111",getDouValueByField(list,"111"));
                result.put("112",getDouValueByField(list,"112"));
                result.put("115",getDouValueByField(list,"115"));
                result.put("116",getDouValueByField(list,"116"));
                result.put("119",getDouValueByField(list,"119"));

                result.put("043",getDouValueByField(list,"043"));
                result.put("124",getDouValueByField(list,"124"));
                result.put("125",getDouValueByField(list,"125"));
                result.put("126",getDouValueByField(list,"126"));
                result.put("127",getDouValueByField(list,"127"));
                result.put("121",getDouValueByField(list,"121"));

                result.put(XzTag.GD_AREA.name(),getDouValueByField(list,XzTag.GD_AREA.name()));
                result.put(XzTag.LD_AREA.name(),getDouValueByField(list,XzTag.LD_AREA.name()));
                result.put(XzTag.YD_AREA.name(),getDouValueByField(list,XzTag.YD_AREA.name()));
                result.put(XzTag.MCD_AREA.name(),getDouValueByField(list,XzTag.MCD_AREA.name()));
                result.put(XzTag.QTNYD_AREA.name(),getDouValueByField(list,XzTag.QTNYD_AREA.name()));
                result.put(XzTag.JMD_AREA.name(),getDouValueByField(list,XzTag.JMD_AREA.name()));
                result.put(XzTag.JT_AREA.name(),getDouValueByField(list,XzTag.JT_AREA.name()));
                result.put(XzTag.SL_AREA.name(),getDouValueByField(list,XzTag.SL_AREA.name()));
                result.put(XzTag.WLYTD_AREA.name(),getDouValueByField(list,XzTag.WLYTD_AREA.name()));
                result.put(XzTag.QTTD_AREA.name(),getDouValueByField(list,XzTag.QTTD_AREA.name()));

                result.put(XzTag.NYD_AREA.name(),getDouValueByField(list,XzTag.NYD_AREA.name()));
                result.put(XzTag.JSYD_AREA.name(),getDouValueByField(list,XzTag.JSYD_AREA.name()));
                result.put(XzTag.WLYD_AREA.name(),getDouValueByField(list,XzTag.WLYD_AREA.name()));
                result.put(XzTag.TOTAL_AREA.name(),getDouValueByField(list,XzTag.TOTAL_AREA.name()));
            }
            return result;
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw  new RuntimeException(e.getLocalizedMessage());
        }
    }
    /**
     * 按照某个键值对list进行分组
     * @param list
     * @param groupKey
     * @return
     */
    public Map<String,List> groupList(List list,String groupKey){
        Map<String,List> resultMap = new HashMap<String, List>();
        try {
            for(int i=0;i<list.size();i++){
                Map item = (Map) list.get(i);
                if(!item.containsKey(groupKey))continue;
                String groupValue = (String) item.get(groupKey);
                List temp=resultMap.get(groupValue);
                if (isNull(temp))
                {
                    temp = new ArrayList();
                    temp.add(item);
                    resultMap.put(groupValue, temp);
                } else
                    temp.add(item);
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultMap;

    }

    @Override
    public List<Map> jctbAnalysis(String geometry, List analysisLayers, String dataSource) {
        if (isNull(analysisLayers)) throw new RuntimeException("no analysis Layers");
        if (isNull(geometry)) throw new JSONMessageException("geometry is null");
        List results = null;
        try {
            Object geo = geometryService.readUnTypeGeoJSON(geometry);
            if (geo instanceof FeatureCollection) {
                results = new ArrayList();
                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) iterator.next();
                    Map map = singleAnalysisOfJctb(feature, analysisLayers, dataSource);
                    results.add(map);
                }
            }else if(geo instanceof SimpleFeature)
            {
                results = new ArrayList();
                results.add(singleAnalysisOfJctb((SimpleFeature)geo, analysisLayers, dataSource));
            }
        } catch (GeometryServiceException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION,e.getLocalizedMessage());
        } catch (NoSuchElementException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return results;
    }

    /**
     *
     * @param list
     * @return
     */
    @Override
    public List jctbExcelData(List<Map> list) {
        List result = null;
        if(isNull(list))return null;
        result = new ArrayList();
        for(Map item:list){
            List temp = new ArrayList();
            temp.add(list.indexOf(item)+1);
            temp.add(isNull(item.get("OG_PRO_JCBH"))?"":item.get("OG_PRO_JCBH"));
            temp.add(isNull(item.get("OG_PRO_TBLX"))?"":item.get("OG_PRO_TBLX"));
            temp.add(isNull(item.get("OG_PRO_BSM"))?"":item.get("OG_PRO_BSM"));
            temp.add(isNull(item.get("OG_PRO_XZMC"))?"":item.get("OG_PRO_XZMC"));
            temp.add(isNull(item.get("OG_PRO_XMMC"))?"":item.get("OG_PRO_XMMC"));
            temp.add(isNull(item.get("OG_PRO_JSDW"))?"":item.get("OG_PRO_JSDW"));
            temp.add(isNull(item.get("OG_PRO_JCMJ"))?"":item.get("OG_PRO_JCMJ"));
            temp.add(isNull(item.get("JC_GD_AREA"))?"":item.get("JC_GD_AREA"));
            temp.add(item.get(JTag.BP_AREA.name()).equals(0)?"":item.get(JTag.BP_AREA.name()));
            temp.add(item.get(JTag.BP_GD_AREA.name()).equals(0)?"":item.get(JTag.BP_GD_AREA.name()));
            temp.add(item.get(JTag.YG_AREA.name()).equals(0)?"":item.get(JTag.YG_AREA.name()));
            temp.add(item.get(JTag.WG_AREA.name()).equals(0)?"":item.get(JTag.WG_AREA.name()));
            temp.add(item.get(JTag.WPYJ_AREA.name()).equals(0)?"":item.get(JTag.WPYJ_AREA.name()));
            temp.add(item.get(JTag.WPYJ_GD_AREA.name()).equals(0)?"":item.get(JTag.WPYJ_GD_AREA.name()));
            temp.add(item.get(JTag.YXJSQ_AREA.name()).equals(0)?"":item.get(JTag.YXJSQ_AREA.name()));
            temp.add(item.get(JTag.JBNT_AREA.name()).equals(0)?"":item.get(JTag.JBNT_AREA.name()));
            temp.add(item.get(JTag.YBNTQ_AREA.name()).equals(0)?"":item.get(JTag.YBNTQ_AREA.name()));
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            temp.add(isNull(item.get(JTag.PCMC.name()))?"":item.get(JTag.PCMC.name()));
            temp.add(isNull(item.get(JTag.NZYPW.name()))?"":item.get(JTag.NZYPW.name()));
            temp.add(isNull(item.get(JTag.GDBH.name()))?"":item.get(JTag.GDBH.name()));
            temp.add(" ");
            temp.add(" ");
            temp.add(" ");
            result.add(temp);
        }
        return result;
    }

    /**
     *
     * @param dltbLayerName 地类图斑图层名称
     * @param xzdwLayerName 现状地物图层名称
     * @param geometry      geojson格式的图形
     * @param wcfUrl        webservice的地址
     * @param wcfMethod     分析的方法名(wcf中)
     * @return
     */
    @Override
    public Map tdlyxzAnalysisByWcf(String dltbLayerName, String xzdwLayerName, String geometry, String wcfUrl, String wcfMethod) {
        if (StringUtils.isBlank(dltbLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "地类图斑图层名称为空！"));
        if (StringUtils.isBlank(xzdwLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "线状地物图层名称为空！"));
        if (StringUtils.isBlank(geometry))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "分析地块坐标为空！"));
        if(StringUtils.isBlank(wcfUrl))
            throw new RuntimeException(getMessage("tdlyxz.wcf.url.null"));
        if(StringUtils.isBlank(wcfMethod))
            throw new RuntimeException(getMessage("tdlyxz.wcf.method.null"));
        try {
            Client client = new Client(new URL(wcfUrl));
            JSONObject params = new JSONObject();
            params.put(WCFTag.AnalyseGeoJSON.name(),geometry);
            params.put(WCFTag.TBLayerName.name(),dltbLayerName);
            params.put(WCFTag.XWLayerName.name(),xzdwLayerName);
            Object[] objects;
            try {
                objects = client.invoke(wcfMethod, new String[]{JSON.toJSONString(params)});
            } catch (Exception e) {
                logger.error(getMessage("tdlyxz.wcf.error",e.getLocalizedMessage()));
                throw new RuntimeException(getMessage("tdlyxz.wcf.error",e.getLocalizedMessage()));
            }
            Map analysisMap = JSON.parseObject(String.valueOf(objects[0]),Map.class);

            //分别获取图斑、线物的分析详细结果
            List<Map> tbFeatures = (List<Map>) MapUtils.getObject(analysisMap, WCFTag.TBFeatures.name());
            List<Map> xwFeatures = (List<Map>) MapUtils.getObject(analysisMap, WCFTag.XWFeatures.name());

            List<Map> summary = (List<Map>) MapUtils.getObject(analysisMap, WCFTag.Summary.name());
            Map dlbmSummary = (Map) summary.get(0).get("SummaryByDLDM");

            List<Map> detailList = new ArrayList<Map>();
            Map analysisArea = new HashMap();

            Map detailMap=null;
            for(Map tb:tbFeatures)
            {
                detailMap = new HashMap();
                detailMap.put("CCMJ",MapUtils.getDoubleValue(tb,WCFTag.TBDLMJ.name(),0.0));
                detailMap.put("QSDWDM",MapUtils.getString(tb,WCFTag.QSDWDM.name()));
                detailMap.put("QSDWMC",MapUtils.getString(tb,WCFTag.QSDWMC.name()));
                detailMap.put("DLBM",MapUtils.getString(tb,"DLBM"));
                detailMap.put("DLMC",MapUtils.getString(tb,"DLMC"));
                detailMap.put("QSXZ",MapUtils.getString(tb,"QSXZ"));
                detailList.add(detailMap);
            }
            for(Map xw:xwFeatures)
            {
                detailMap = new HashMap();
                detailMap.put("CCMJ",MapUtils.getDoubleValue(xw,WCFTag.XZDWMJ.name(),0.0));
                detailMap.put("QSDWDM",MapUtils.getString(xw,WCFTag.QSDWDM1.name()));
                detailMap.put("QSDWMC",MapUtils.getString(xw,WCFTag.QSDWMC1.name()));
                detailMap.put("DLBM",MapUtils.getString(xw,"DLBM"));
                detailMap.put("DLMC",MapUtils.getString(xw,"DLMC"));
                detailMap.put("QSXZ",MapUtils.getString(xw,"QSXZ"));
                detailList.add(detailMap);
            }
            for(Object key:dlbmSummary.keySet())
            {
                String dlbm = String.valueOf(key).substring(4);
                analysisArea.put(getDlmc(detailList,dlbm),dlbmSummary.get(key));
            }
            Map result = new HashMap();
            result.put("analysisAreaDetail",detailList);
            result.put("analysisArea",analysisArea);
            return result;

        } catch (Exception e) {
            logger.error(getMessage("tdlyxz.wcf.error",e.getLocalizedMessage()));
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     *
     * @param list
     * @param dm
     * @return
     */
    public String getDlmc(List<Map> list,String dm)
    {
        Iterator iterator = list.iterator();
        while (iterator.hasNext())
        {
            Map entry = (Map) iterator.next();
            String dlbm = MapUtils.getString(entry,"DLBM");
            if(dlbm.equals(dm))
                return MapUtils.getString(entry,"DLMC");
        }
        return dm;
    }

    /**
     *
     * @param year
     * @param geometry
     * @param wcfUrl
     * @param wcfMethod
     * @return
     */
    @Override
    public Map tdlyxzAnalysisByWcf(String year, String geometry, String wcfUrl, String wcfMethod) {
        String dltb = TDXZ.DLTB.name().concat(LAYER_MIDDLE_FIX_H).concat(year);
        String xzdw = TDXZ.XZDW.name().concat(LAYER_MIDDLE_FIX_H).concat(year);
        return tdlyxzAnalysisByWcf(dltb,xzdw,geometry,wcfUrl,wcfMethod);
    }

    /**
     *  单个feature的监测图斑分析处理
     * @param feature
     * @param analysisLayers
     * @param dataSource
     * @return
     */
    public Map singleAnalysisOfJctb(SimpleFeature feature,List analysisLayers,String dataSource){
        Map result=null;
        Map<String,List<Map>> analysisMap=new HashMap<String, List<Map>>();
        DecimalFormat df_jctb = new DecimalFormat("0.0");
        try {
            String dltbLyr = null;
            String xzdwLyr = null;
            for (int i = 0; i < analysisLayers.size(); i++) {
                Map layer = (Map) analysisLayers.get(i);
                String lyrId = layer.get("id").toString();
                if(lyrId.equals("dltb"))
                {
                    dltbLyr = String.valueOf(layer.get("layerName"));
                    continue;
                }
                if(lyrId.equals("xzdw"))
                {
                    xzdwLyr = String.valueOf(layer.get("layerName"));
                    continue;
                }
                String lyrName = (String) layer.get("layerName");
                String outFields = layer.get("fields").toString();
                String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                List<Map> analysisList = (List<Map>)intersectBySimpleFeature(lyrName, feature, fields, dataSource);
                analysisMap.put(lyrId.toUpperCase(),analysisList);
                logger.info("["+lyrId.toUpperCase()+"]分析记录数:"+analysisList.size());
            }
            Map tdlyxzMap = tdlyxzAnalysis2(dltbLyr, xzdwLyr, geometryService.toFeatureJSON(feature), dataSource);

            Map tdlyxzAnalysisArea = (Map) tdlyxzMap.get("analysisArea");
            if(!tdlyxzAnalysisArea.isEmpty())
            {
                for(Object key:tdlyxzAnalysisArea.keySet())
                {
                    logger.info("[tdlyxz]分析结果:"+ key + tdlyxzAnalysisArea.get(key));
                }
            }
            List<Map> bpList = analysisMap.get(JCTB.BPDK.name());
            List<Map> gdList = analysisMap.get(JCTB.GDDK.name());
//            List<Map> dltbList = analysisMap.get(JCTB.DLTB.name());
//            List<Map> xzdwList = analysisMap.get(JCTB.XZDW.name());
            List<Map> jsydgzqList = analysisMap.get(JCTB.JSYDGZQ.name());
            List<Map> tdytqList = analysisMap.get(JCTB.TDYTQ.name());

            double geoArea = getAreaByGeometry(feature, ACRES_CONV);

//            double dltb_gd_area =getAreaByField(dltbList,"DLBM","011",1)+getAreaByField(dltbList,"DLBM","012",1)+getAreaByField(dltbList,"DLBM","013",1);
//            double xzdw_gd_area = getXzdwAreaByField(xzdwList,"DLBM","011",1)+getXzdwAreaByField(xzdwList,"DLBM","012",1)+ getXzdwAreaByField(xzdwList,"DLBM","013",1);

            result = new HashMap();
//            Map tdlyxzAnalysisArea = (Map) tdlyxzMap.get("analysisArea");
            result.put(JTag.JC_GD_AREA.name(),df_jctb.format((MapUtils.getDouble(tdlyxzAnalysisArea,"水田",0.0)+MapUtils.getDouble(tdlyxzAnalysisArea,"水浇地",0.0)+MapUtils.getDouble(tdlyxzAnalysisArea,"旱地",0.0))*ACRES_CONV));
            result.put(JTag.BP_AREA.name(), df_jctb.format(getAreaByList(bpList, ACRES_CONV)));
            result.put(JTag.BP_GD_AREA.name(),df_jctb.format(getBpGdArea(bpList,dltbLyr,xzdwLyr,dataSource)*ACRES_CONV));
            result.put(JTag.YG_AREA.name(), df_jctb.format(getAreaByList(gdList, ACRES_CONV)));
            result.put(JTag.WG_AREA.name(),df_jctb.format((geoArea-MapUtils.getDouble(result,JTag.YG_AREA.name(),0.0))<0?0:geoArea-MapUtils.getDouble(result,JTag.YG_AREA.name(),0.0)));
            result.put(JTag.WPYJ_AREA.name(),df_jctb.format((geoArea-MapUtils.getDouble(result,JTag.BP_AREA.name(),0.0))<0?0:geoArea-MapUtils.getDouble(result,JTag.BP_AREA.name(),0.0)));
            result.put(JTag.WPYJ_GD_AREA.name(),df_jctb.format(MapUtils.getDouble(result,JTag.JC_GD_AREA.name(),0.0)-MapUtils.getDouble(result,JTag.BP_GD_AREA.name(),0.0)<0?0:MapUtils.getDouble(result,JTag.JC_GD_AREA.name(),0.0)-MapUtils.getDouble(result,JTag.BP_GD_AREA.name(),0.0)));
            result.put(JTag.YXJSQ_AREA.name(),df_jctb.format(getAreaByField(jsydgzqList,"GZQLXDM","010",ACRES_CONV)));
            result.put(JTag.JBNT_AREA.name(),df_jctb.format(getAreaByField(tdytqList,"TDYTQLXDM","010",ACRES_CONV)));
            result.put(JTag.YBNTQ_AREA.name(),df_jctb.format(getAreaByField(tdytqList,"TDYTQLXDM","020",ACRES_CONV)));

            result.put(JTag.PCMC.name(), getStrValueByField(bpList, "PCJC"));
            result.put(JTag.NZYPW.name(), getStrValueByField(bpList, "PZWH"));
            result.put(JTag.GDBH.name(), getStrValueByField(gdList, "XMBH"));

            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")||p.getName().getLocalPart().equals("crs")) continue;
                result.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }


        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }

        return result;
    }


    /**
     * 获取耕地面积
     * @param list 报批地块集合
     * @param dltb 地类图斑图层等配置
     * @param xzdw 现状地物图层等配置
     * @return
     */
    public double getBpGdArea(List<Map> list,String dltb,String xzdw,String dataSource)
    {
        double area=0;
        try {
            for(Map map:list)
            {
                String wktPolygon = String.valueOf(map.get(SE_SHAPE_FIELD));
                Map tdlyxzMap = tdlyxzAnalysis2(dltb,xzdw,geometryService.toGeoJSON(geometryService.readWKT(wktPolygon)),dataSource);
                Map tdlyxzAnalysisArea = (Map) tdlyxzMap.get("analysisArea");
                area += MapUtils.getDouble(tdlyxzAnalysisArea,"水田",0.0)+MapUtils.getDouble(tdlyxzAnalysisArea,"水浇地",0.0)+MapUtils.getDouble(tdlyxzAnalysisArea,"旱地",0.0);
            }
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area;
    }

    /**
     * sum area of list
     * @param list
     * @return
     */
    public double getAreaByList(List<Map> list, double conv){
        if(isNull(conv))conv=1;
        double area=0;
        for(Map map:list){
            double value =(Double)map.get(SE_SHAPE_AREA);
            area += value;
        }
        return area*conv;
    }

    /**
     * get feature's area
     * @param geo
     * @param conv
     * @return
     */
    public double getAreaByGeometry(Object geo, double conv){
        if(isNull(conv))conv=1;
        double area = 0;
        try {
            SimpleFeature feature=null;
            if(geo instanceof SimpleFeature)
            {
                feature = (SimpleFeature) geo;
                CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
                CoordinateReferenceSystem targetCrs = geometryService.parseUndefineSR("2364");
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (sourceCRS != null && sourceCRS instanceof GeographicCRS) {
                    geometry = geometryService.project((Geometry) feature.getDefaultGeometry(), sourceCRS, targetCrs);
                }
                if (isPolygon(geometry))
                    area = geometry.getArea();
                else
                    area = geometry.getLength();
            }else if(geo instanceof FeatureCollection){

                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    feature = (SimpleFeature) iterator.next();
                    area += getAreaByGeometry(feature, conv);
                }
            }else if(geo instanceof Geometry){
                Geometry temp = (Geometry) geo;
                if(isPolygon(temp))
                    area = temp.getArea();
                else
                    area = temp.getLength();
            }
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area*conv;
    }

    /**
     *
     * @param map
     * @param keys  键值集合
     * @param conv
     * @return
     */
    public double getAreaByMapKeys(Map map,List keys,double conv){

        if(isNull(conv))conv=1;
        assert map !=null;
        assert keys!=null;
        double area=0;
        for(Object key:map.keySet()){
            if(keys.contains(key))
                area += MapUtils.getDouble(map,key,0.0);
        }
        return area*conv;

    }

    /**
     * get area by some field
     * @param list
     * @param field
     * @param value
     * @param conv
     * @return
     */
    public double getAreaByField(List<Map> list,String field,String value,double conv){
        if(isNull(conv))conv=1;
        assert field !=null;
        assert value!=null;
        double area=0;
        List values = Arrays.asList(value.split(","));
        for(Map map:list){
            if(values.contains(map.get(field)))
                area += (Double) map.get(SE_SHAPE_AREA);
        }
        return area*conv;
    }
    /**
     * get string value by some field
     * @param list
     * @param field
     * @return
     */
    public String getStrValueByField(List<Map> list, String field){
        StringBuilder sb=new StringBuilder();
        assert field !=null;
        for(Map map:list){
            if(map.containsKey(field))
            {
                if(StringUtils.isNotBlank(sb.toString()))
                    sb.append(",");
                if(!isNull(map.get(field)))
                    sb.append(map.get(field));
            }
        }
        return sb.toString();
    }
    /**
     * get double value by some field
     * @param list
     * @param field
     * @return
     */
    public double getDouValueByField(List<LinkedHashMap> list, String field){
        double value=0;
        assert field !=null;
        for(Map map:list){
            if(isNull(map.get(field)))continue;
            value += MapUtils.getDouble(map,field,0.0);
        }
        return value;
    }

    /**
     *
     * @param list
     * @param field
     * @return
     */
    public double getDouValueByField2(List<Map> list, String field){
        double value=0;
        assert field !=null;
        for(Map map:list){
            if(isNull(map.get(field)))continue;
            value += Double.valueOf(map.get(field).toString());
        }
        return value;
    }
    /**
     * get xzdw area by field
     * @param list
     * @param field
     * @param value
     * @param conv
     * @return
     */
    public double getXzdwAreaByField(List<Map> list,String field,String value,double conv){
        if(isNull(conv))conv=1;
        double area=0;
        List values = Arrays.asList(value.split(","));
        if(isNull(list))return area*conv;
        for(Map map:list){
            if (values.contains(map.get(field)))
            {
                area += getXzdwArea(map,1);
            }
        }
        return area*conv;
    }

    /**
     * 获取现状地物的面积
     * @param xzdw  单个map或者是list
     * @param conv
     * @return
     */
    public double getXzdwArea(Object xzdw,double conv){
        if(isNull(conv))conv=1;
        double area=0;
        Map map = null;
        try {
            if(xzdw instanceof Map)
            {
                map = (Map) xzdw;
                double length = Double.parseDouble(String.valueOf(map.get(SE_SHAPE_AREA)));
                double kd= Double.parseDouble(String.valueOf(map.get("KD")));
                area = length*kd;
            }else if(xzdw instanceof List)
            {
               List list = (List) xzdw;
                for(int i=0;i<list.size();i++)
                {
                    map = (Map) list.get(i);
                    area +=getXzdwArea(map,1);
                }
            }
        } catch (NumberFormatException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        } catch (Exception e){
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area*conv;
    }

    /**
     * 根据bbox范围和相关参数,获取输出图片
     * @param envelope
     * @param url
     * @param imageSize
     * @param mapScale
     * @return
     */
    private BufferedImage getExportImage(Envelope envelope,String url,String imageSize,double mapScale) {
        BufferedImage image = null;
        String bbox = "bbox="+ envelope.getMinX()+","+envelope.getMinY()+","+envelope.getMaxX()+","+envelope.getMaxY();
        String size = "size="+imageSize;
        StringBuilder exportUrl = new StringBuilder();
        exportUrl.append(url);
        exportUrl.append("/export?");
        exportUrl.append(bbox);
        exportUrl.append("&");
        exportUrl.append(size);
        exportUrl.append("&");
        exportUrl.append("&");
        if (mapScale > 0)
            exportUrl.append("mapScale=" + mapScale);
        exportUrl.append("&");
        exportUrl.append("transparent=true&");
        exportUrl.append("format=png&");
        exportUrl.append("f=image");
        HttpClient httpClient = new HttpClient();
        GetMethod getMethod = new GetMethod(exportUrl.toString());
        try {
            httpClient.executeMethod(getMethod);
            InputStream inputStream = new ByteArrayInputStream(getMethod.getResponseBody());
            if(inputStream.available()>0)
                image = ImageIO.read(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return image;
    }


    /**
     * @param features
     * @return
     */
    private double getTotalArea(JSONArray features) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {
            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
        }
        return area;
    }

    /**
     * @param field
     * @param features
     * @param lxdm
     * @return
     */
    private double getAreaByLxdm(String field, JSONArray features, String lxdm) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {

            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            if (property.get(field) == null)
//                throw new RuntimeException("字段【"+field+"】不存在！");
                return 0;
            else {
                if (property.get(field).equals(lxdm))
                    area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
            }
        }
        return area;
    }

    /**
     * @param fields
     * @param keyword
     * @return
     */
    private String getWhereClause(String[] fields, String keyword) {
        String or = " or ";
        StringBuilder b = new StringBuilder();
        for (String field : fields) {
            b.append(field);
            b.append(" like '");
            b.append(keyword);
            b.append("%'");
            b.append(or);
        }
        b.delete(b.length() - or.length(), b.length());
        return b.toString();
    }


    /**
     * GeoJSON to Map
     *
     * @param geoJSON single geometry
     * @return
     */
    private Map<String, Object> geoJSON2Map(String geoJSON, CoordinateReferenceSystem targetCRS) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        Map<String, Object> columns = null;
        if (geo instanceof Geometry) {
            columns = new HashMap<String, Object>();
            columns.put(SE_SHAPE_FIELD, ((Geometry) geo).toText());
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geoJSON);
            columns = geometryService.simpleFeature2Map(feature);
            try {
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (targetCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, targetCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
            } catch (Exception e) {
                logger.error(e.getLocalizedMessage());
                throw new RuntimeException(e.getLocalizedMessage());
            }
        } else if (geo instanceof GeometryCollection) {
            throw new RuntimeException("current omp version don't support GeometryCollection ");
        } else if (geo instanceof FeatureCollection) {
            throw new RuntimeException("current omp version don't support FeatureCollection ");
        } else {
            throw new RuntimeException(getMessage("geometry.undefined"));
        }
        return columns;
    }

    /**
     * feature to geometry
     *
     * @param feature
     * @param geometry
     * @return
     */
    private Geometry setFeaturePros2Geo(SimpleFeature feature, Geometry geometry) {
        Map<String, Object> map = null;
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                Geometry geo = geometry.getGeometryN(i);
                map = new HashMap<String, Object>();
                for (Property p : feature.getProperties()) {
                    if (p.getName().equals("geometry")) continue;
                    map.put(p.getName().getLocalPart(), p.getValue());
                }
                geo.setUserData(map);
            }
        } else {
            map = new HashMap<String, Object>();
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                map.put(p.getName().getLocalPart(), p.getValue());
            }
            geometry.setUserData(map);
        }
        return geometry;
    }

    /**
     * add pros 2 map  list
     *
     * @param list
     * @param geometry
     * @return
     */
    private List<?> addGeoProperty2List(List<Map<String, Object>> list, Geometry geometry) {
        if (geometry.getUserData() == null) return list;
        for (Map<String, Object> item : list) {
            Map<String, Object> pros = (Map<String, Object>) geometry.getUserData();
            for (Map.Entry entry : pros.entrySet()) {
                item.put("G_".concat((String) entry.getKey()), entry.getValue());
            }
        }
        return list;
    }

    /**
     * @param tpl
     * @return
     */
    private List<Map> getTdlyxzDictList(String tpl) {
        List<Map> groupList = new ArrayList<Map>();
        try {
            Configuration configuration = webMapService.getConfig(tpl);
            List<Dict> dicts = configuration.getDicts();
            for (Dict dict : dicts) {
                if (String.valueOf(dict.getName()).equals("tdlyxz")) {
                    List<Item> dictItems = dict.getDictItems();
                    for (Item item : dictItems) {
                        Map temp = new HashMap();
                        temp.put("dlbm", item.getName());
                        temp.put("dlmc", item.getTitle());
                        temp.put("area", 0);
    //                    temp.put("detail",new ArrayList<Map>());
                        groupList.add(temp);
                    }
                }
            }
        } catch (Exception e) {
            logger.error(getMessage("get.dict.error",tpl));
            throw new RuntimeException(getMessage("get.dict.error",tpl));
        }
        if (groupList.size() == 0)
        {
            logger.error(getMessage("get.dict.error",tpl));
            throw new RuntimeException(getMessage("get.dict.error",tpl));
        }
        return groupList;
    }

    /**
     * check field is in layer
     *
     * @param field
     * @param layerName
     * @return
     */
    private boolean checkFieldInLayer(String field, String layerName, String dataSource) {
        return ArrayUtils.contains(spatialDao.getLayerColumns(layerName, dataSource), field, true);
    }


    private boolean isPolygon(Geometry geometry) {
        String geometryType = geometry.getGeometryType();
        if(geometryType.equals(Geometries.POLYGON.getName())||geometryType.equals(Geometries.MULTIPOLYGON.getName()))
            return true;
        else if(geometryType.equals(Geometries.LINESTRING.getName())||geometryType.equals(Geometries.MULTILINESTRING.getName()))
            return false;
        else if(geometryType.equals(Geometries.GEOMETRYCOLLECTION.getName())){
            return geometry.getArea()>0?true:false;
        }else
            return false;
//        return (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName())) ? true : false;
    }

}
