package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.entity.Document;
import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.service.DocumentService;
import cn.gtmap.onemap.platform.service.GeometryService;
import cn.gtmap.onemap.platform.utils.GeometryUtils;
import cn.gtmap.onemap.platform.utils.SRTransformations;
import cn.gtmap.onemap.platform.utils.UUIDGenerator;
import com.alibaba.fastjson.JSON;
import com.esri.sde.sdk.client.SeLayer;
import com.esri.sde.sdk.pe.PeProjectionException;
import com.vividsolutions.jts.densify.Densifier;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.io.WKTReader;
import com.vividsolutions.jts.operation.valid.IsValidOp;
import com.vividsolutions.jts.operation.valid.TopologyValidationError;
import com.vividsolutions.jts.simplify.DouglasPeuckerSimplifier;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geojson.feature.FeatureJSON;
import org.geotools.geojson.geom.GeometryJSON;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.geotools.referencing.CRS;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.operation.MathTransform;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import java.io.*;
import java.util.*;
import java.util.regex.Pattern;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-17 下午2:35
 */
public class GeometryServiceImpl extends BaseLogger implements GeometryService {

    public static final String TYPE = "type";

    public static final String FEATURE = "Feature";
    public static final String FEATURE_COLLECTION = "FeatureCollection";
    public static final String GEOMETRY_COLLECTION = "GeometryCollection";

    private static final String WKID = "wkid";
    private static final String WKT = "wkt";

    private static final String REGION_FIELD = "regionLayers";
    private static final String REGION_MAP = "regionMap";
    private static final String DEFAULT_CRS = "defaultCrs";
    private static final String COORDINATE_DM = "coordinatesDM";
    private static final String UNDEFINE = "UNDEFINE";

    private static final String EPSG_PERFIX = "EPSG:";

    private static final String BJ_FILE_NAME = "gt.xml";

    private static final String DEFAULT_LAYER_REGION_FIELD = "DEFAULTLAYER";

    private static final String TEMP_PIX = "TMP_";

    private static final String BJ_TITLE = "title";
    private static final String BJ_FEATURE = "feature";

    enum BJ_TAG {
        DATA, DATANAME, ROWDATA, ROW, BL_PLOT, PL_ID, PL_NAME, BL_PNT_COORD, PL_PL_ID, X_COORD, Y_COORD,
        SHAPE_GROUP, ID, NAME, PNT_SERIAL, BL_PROJ_BUILD, PROJ_NAME
    }

    private GeometryFactory geometryFactory = JTSFactoryFinder.getGeometryFactory();
    private WKTReader wktReader;

    //    private String regionField;
    private double simplifyTolerance;
    private Map<String, String> regionLayers;
    private Map regionSet;
    private CoordinateReferenceSystem defaultCrs;
    private Map<String, Object> coordinateDM;

    @Autowired
    private DocumentService documentService;

    /**
     * 读取WKT标准图形
     *
     * @param wkt
     * @return
     */
    @Override
    public Geometry readWKT(String wkt) throws GeometryServiceException {
        try {
            return new WKTReader(geometryFactory).read(wkt);
//            return getWktReader().read(wkt);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.WKT_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析 GeometryJSON  格式数据
     *
     * @param geoJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public Geometry readGeoJSON(String geoJSON) throws GeometryServiceException {
        try {
            GeometryJSON geometryJSON = new GeometryJSON();
            return geometryJSON.read(geoJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析 GeometryCollection
     *
     * @param geoJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public GeometryCollection readGeoCollectionJSON(String geoJSON) throws GeometryServiceException {
        try {
            GeometryJSON geometryJSON = new GeometryJSON();
            return geometryJSON.readGeometryCollection(geoJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }


    /**
     * 解析 FeatureJSON 格式数据
     *
     * @param featureJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public SimpleFeature readFeatureJSON(String featureJSON) throws GeometryServiceException {
        try {
            FeatureJSON fJson = new FeatureJSON();
            return fJson.readFeature(featureJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析 FeatureCollectionJSON
     *
     * @param featureJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public FeatureCollection readFeatureCollectionJSON(String featureJSON) throws GeometryServiceException {
        try {
            FeatureJSON fJson = new FeatureJSON();
            return fJson.readFeatureCollection(featureJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析未指明GeoJSON
     *
     * @param geoJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public Object readUnTypeGeoJSON(String geoJSON) throws GeometryServiceException {
        try {
            return GeometryUtils.parseGeoJSON(geoJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 读取 FeatureJSON 空间参考信息
     *
     * @param featureJSON
     * @return
     */
    @Override
    public CoordinateReferenceSystem readFeatureJSONCRS(String featureJSON) {
        try {
            FeatureJSON fJson = new FeatureJSON();
            return fJson.readCRS(featureJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 投影转换
     *
     * @param geometry
     * @param sourceCRS .
     * @param targetCRS
     * @return
     */
    @Override
    public Geometry project(Geometry geometry, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) throws GeometryServiceException {
       /* try {
            TopologyValidationError error = null;//validGeometry(geometry);
            if (error != null)
                throw new RuntimeException(getMessage("geometry.not.valid", error.toString()));
            MathTransform transform = CRS.findMathTransform(sourceCRS, targetCRS, true);
            return JTS.transform(geometry, transform);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.PROJECT_EXCEPTION, e.getLocalizedMessage());
        }*/
        return projectByAGS(geometry, sourceCRS, targetCRS);
    }

    /**
     * 投影转换 by arcgis sde
     *
     * @param geometry
     * @param sourceCRS
     * @param targetCRS
     * @return
     */
    @Override
    public Geometry projectByAGS(Geometry geometry, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) {
        try {
            return SRTransformations.project(geometry, sourceCRS.toWKT(), targetCRS.toWKT());
        } catch (PeProjectionException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.PROJECT_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 简化图形
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry simplify(Geometry geometry, double tolerance) {
        return geometry.isValid() && geometry.isSimple() ? geometry : DouglasPeuckerSimplifier.simplify(geometry, tolerance);
    }

    /**
     * force simplify
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry forceSimplify(Geometry geometry, double tolerance) {
        return DouglasPeuckerSimplifier.simplify(geometry, tolerance);
    }

    /**
     * densify
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry densify(Geometry geometry, double tolerance) {
        return Densifier.densify(geometry, tolerance);
    }

    /**
     * 获取 wkt 格式空间参考
     *
     * @param wktCRS such as "GEOGCS[" + "\"WGS 84\"," + "  DATUM[" + "    \"WGS_1984\","
     *               + "    SPHEROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\",\"7030\"]],"
     *               + "    TOWGS84[0,0,0,0,0,0,0]," + "    AUTHORITY[\"EPSG\",\"6326\"]],"
     *               + "  PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],"
     *               + "  UNIT[\"DMSH\",0.0174532925199433,AUTHORITY[\"EPSG\",\"9108\"]],"
     *               + "  AXIS[\"Lat\",NORTH]," + "  AXIS[\"Long\",EAST],"
     *               + "  AUTHORITY[\"EPSG\",\"4326\"]]";
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSByWKTString(String wktCRS) {
        try {
            return CRS.parseWKT(SRTransformations.getCoordinateSystem(wktCRS).toString());
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        } catch (PeProjectionException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 获取常用标准格式空间参考
     *
     * @param crs such as "EPSG:4326" , "urn:ogc:def:ellipsoid:EPSG:6.0:7001" , "AUTO2:42001,"+lat+","+lon
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSByCommnonString(String crs) {
        try {
            return CRS.decode(crs, true);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * srid
     *
     * @param srid
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSBySRID(String srid) {
        try {
            return getCRSByWKTString(SRTransformations.getCoordinateSystem(Integer.valueOf(srid)).toString());
        } catch (PeProjectionException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * get SeLayer crs
     *
     * @param layer
     * @return
     */
    @Override
    public CoordinateReferenceSystem getSeLayerCRS(SeLayer layer) {
        return getCRSByWKTString(layer.getCoordRef().getCoordSysDescription());
    }

    /**
     * 获取SimpleFeatureType
     *
     * @param value
     * @return
     */
    @Override
    public SimpleFeatureType getFeatureType(final Map<String, Object> value) {
        SimpleFeatureTypeBuilder typeBuilder = new SimpleFeatureTypeBuilder();
        typeBuilder.setName(FEATURE);
        String[] keys = value.keySet().toArray(new String[0]);
        for (String key : keys) {
            if (SHAPE.equals(key))
                typeBuilder.add(GEOMETRY, Geometry.class);
            else if (FEATURE_CRS.equals(key))
                typeBuilder.add(FEATURE_CRS, String.class, (CoordinateReferenceSystem) value.get(key));
            else
                typeBuilder.add(key, value.get(key) != null ? value.get(key).getClass() : String.class);
        }
        return typeBuilder.buildFeatureType();
    }

    /**
     * Map value to SimpleFeature
     *
     * @param value   {"SHAPE":"WKT FORMATE","PRO1":"VALUE"}
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public SimpleFeature map2SimpleFeature(final Map<String, Object> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        Geometry geometry = value.containsKey(SHAPE) ? readWKT((String) value.get(SHAPE)) : null;
        if (geometry != null && srcCRS != null && destCRS != null)
            geometry = project(geometry, srcCRS, destCRS);
        SimpleFeatureType featureType = getFeatureType(value);
        SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(featureType);
        String[] keys = value.keySet().toArray(new String[0]);
        for (String key : keys) {
            if (SHAPE.equals(key)) featureBuilder.add(geometry);
            else if (FEATURE_CRS.equals(key)) featureBuilder.add(value.get(FEATURE_CRS));//continue;
            else featureBuilder.add(value.get(key));
        }
        return featureBuilder.buildFeature(FEATURE.concat(UUIDGenerator.generate()));
    }

    /**
     * list values to featureCollection
     *
     * @param value
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public FeatureCollection list2FeatureCollection(final List<Map<String, Object>> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        DefaultFeatureCollection collection = new DefaultFeatureCollection(null, null);
        for (Map item : value) {
            collection.add(map2SimpleFeature(item, srcCRS, destCRS));
        }
        return collection;
    }

    /**
     * to featureJSON
     *
     * @param feature
     * @return
     */
    @Override
    public String toFeatureJSON(final Object feature) {
        try {
            FeatureJSON featureJSON = new FeatureJSON(new GeometryJSON(14));
            StringWriter out = new StringWriter();
            if (feature instanceof SimpleFeature) {
                featureJSON.setEncodeFeatureBounds(((SimpleFeature) feature).getBounds() == null ? false : true);
                featureJSON.setEncodeFeatureCRS(((SimpleFeature) feature).getFeatureType().
                        getCoordinateReferenceSystem() == null ? false : true);
                featureJSON.writeFeature((SimpleFeature) feature, out);
            } else if (feature instanceof FeatureCollection) {
                if (((FeatureCollection) feature).size() > 0) {
                    featureJSON.setEncodeFeatureCollectionBounds(((SimpleFeature) ((FeatureCollection) feature).toArray()[0]).getBounds() == null ? false : true);
                    featureJSON.setEncodeFeatureCollectionCRS(((SimpleFeature) ((FeatureCollection) feature).toArray()[0]).getFeatureType().
                            getCoordinateReferenceSystem() == null ? false : true);
                }
                featureJSON.writeFeatureCollection((FeatureCollection) feature, out);
            }
            return out.toString();
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.FEATURE_TO_JSON_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * to geoJSON
     *
     * @param geometry
     * @return
     */
    @Override
    public String toGeoJSON(Geometry geometry) {
        try {
            GeometryJSON geometryJSON = new GeometryJSON(14);
            StringWriter out = new StringWriter();
            geometryJSON.write(geometry, out);
            return out.toString();
        } catch (IOException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOMETRY_TO_JSON_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * @param feature
     * @return
     */
    @Override
    public Map<String, Object> simpleFeature2Map(SimpleFeature feature) {
        Assert.notNull(feature, "feature can't be null");
        Map<String, Object> result = new HashMap<String, Object>();
        for (Property property : feature.getProperties()) {
            if (property.getValue() != null && StringUtils.isNotBlank(String.valueOf(property.getValue())))
                result.put(property.getName().getLocalPart(), property.getValue());
        }
        return result;
    }

    /**
     * @param value wkid or wkt
     * @return
     */
    @Override
    public CoordinateReferenceSystem parseUndefineSR(String value) {
        try {
            int srid = Integer.parseInt(value);
            if (srid != 0) return getCRSBySRID(value);
        } catch (NumberFormatException e) {
            if (value.indexOf(EPSG) == 0)
                return getCRSBySRID(value.substring(EPSG.length() + 1, value.length()));//return getCRSByCommnonString(value);
            else return getCRSByWKTString(value);
        }
        throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, value);
    }

    /**
     * 验证图形拓扑
     *
     * @param geometry
     * @return
     */
    @Override
    public TopologyValidationError validGeometry(Geometry geometry) {
        IsValidOp isValidOp = new IsValidOp(geometry);
        return isValidOp.getValidationError();
    }

    /**
     * get crs by reset regionCode sets
     *
     * @param regionCode
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSByRegionCode(String regionCode) {
        if (StringUtils.isBlank(regionCode)) throw new RuntimeException(getMessage("region.code.not.null"));
        return parseUndefineSR(String.valueOf(regionSet.get(regionCode)));
    }

    /**
     * get region key field
     *
     * @return
     */
    @Override
    public LayerRegion getLayerRegion(String layerName) {
        for (Map.Entry entry : this.regionLayers.entrySet()) {
            if (Pattern.compile((String) entry.getKey()).matcher(layerName.toUpperCase()).matches()) {
                String value = (String) entry.getValue();
                try {
                    return new LayerRegion(layerName, parseUndefineSR(value));
                } catch (Exception e) {
                    return new LayerRegion(layerName, value);
                }
            }
        }
        if (this.regionLayers.containsKey(DEFAULT_LAYER_REGION_FIELD)) {
            String defaultField = this.regionLayers.get(DEFAULT_LAYER_REGION_FIELD);
            logger.info(getMessage("layer.regioncode.default", layerName, defaultField));
            return new LayerRegion(layerName, defaultField);
        }
        throw new RuntimeException(getMessage("layer.regioncode.not.set", layerName));
    }

    /**
     * 行政区字典项配置中是否包含该行政区代码
     *
     * @param regionCode
     * @return
     */
    @Override
    public boolean containsRegionValue(String regionCode) {
        if (regionSet.containsKey(regionCode)) return true;
        return false;
    }

    /**
     * @param geoJSON
     * @return
     */
    @Override
    public double readGeometryAera(String geoJSON) {
        try {
            Object geo = readUnTypeGeoJSON(geoJSON);
            if (geo instanceof Geometry) return ((Geometry) geo).getArea();
            if (geo instanceof SimpleFeature) return ((Geometry) ((SimpleFeature) geo).getDefaultGeometry()).getArea();
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
        }
        return 0;
    }

    /**
     * 获取简化精度
     *
     * @return
     */
    @Override
    public double getSimplifyTolerance() {
        return this.simplifyTolerance;
    }

    /**
     * 获取电子报件地块坐标
     *
     * @param in
     * @return FeatureJSON
     */
    @Override
    public Map getBJCoordinates(InputStream in) throws Exception {
        List<Document> documents = documentService.readZip(in);
        if (documents.size() == 0) {
            File tmp = new File(System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis()));
            try {
                FileOutputStream output = new FileOutputStream(tmp);
                try {
                    IOUtils.copyLarge(in, output, 0, in.available(), new byte[in.available()]);
                    output.close();
                } finally {
                    IOUtils.closeQuietly(output);
                }
                documents = documentService.readZipFile(tmp);
                if (documents.size() == 0)
                    throw new RuntimeException(getMessage("bj.zip.format.error"));
            } finally {
                FileUtils.deleteQuietly(tmp);
            }
        }
        for (Document document : documents) {
            if (document.getFileName().equals(BJ_FILE_NAME)) {
                return parseBjXml(document.getContent());
            }
        }
        throw new RuntimeException(getMessage("bj.gt.xml.not.found"));
    }

    /**
     * parse xml bytes
     *
     * @param bytes
     * @return
     */
    private Map parseBjXml(final byte[] bytes) {
        org.dom4j.Document document = null;
        Map<String, String> result = new HashMap<String, String>();
        SAXReader reader = new SAXReader();
        try {
            document = reader.read(new ByteArrayInputStream(bytes));
        } catch (DocumentException e) {
            throw new RuntimeException(getMessage("doc.read.xml.error", e.getLocalizedMessage()));
        }
        if (document != null) {
            List<Map<String, Object>> features = new ArrayList<Map<String, Object>>();
            Element root = document.getRootElement();
            Node blName = root.selectSingleNode("DATA[@DATANAME='BL_PROJ_BUILD']").selectSingleNode("ROWDATA/ROW");
            if (blName instanceof Element)
                result.put(BJ_TITLE, ((Element) blName).attributeValue(BJ_TAG.PROJ_NAME.name()));
            Node blPlot = root.selectSingleNode("DATA[@DATANAME='BL_PLOT']");
            Node blPntCoords = root.selectSingleNode("DATA[@DATANAME='BL_PNT_COORD']");
            List rows = blPlot.selectNodes("ROWDATA/ROW");
            CoordinateReferenceSystem sourceCrs = null;
            for (Object item : rows) {
                if (item instanceof Element) {
                    Element row = (Element) item;
                    String id = row.attributeValue(BJ_TAG.PL_ID.name());
                    String name = row.attributeValue(BJ_TAG.PL_NAME.name());
                    Map feature = getFeature(id, name, blPntCoords);
                    if (sourceCrs == null) sourceCrs = (CoordinateReferenceSystem) feature.get(FEATURE_CRS);
                    feature.remove(FEATURE_CRS);
                    features.add(feature);
                }
            }
            result.put(BJ_FEATURE, toFeatureJSON(list2FeatureCollection(features, sourceCrs, defaultCrs)));
            return result;
        }
        throw new RuntimeException(getMessage("bj.parse.error"));
    }

    /**
     * get feature
     *
     * @param id
     * @param name
     * @param node
     * @return
     */
    private Map getFeature(String id, String name, Node node) {
        Map<String, Object> feature = new HashMap<String, Object>();
        CoordinateReferenceSystem fearureCrs = null;
        feature.put(BJ_TAG.ID.name(), id);
        feature.put(BJ_TAG.NAME.name(), name);
        Map<String, List<Coordinate>> rings = new LinkedHashMap<String, List<Coordinate>>();
        List rows = node.selectNodes("ROWDATA/ROW[@PL_PL_ID='" + id + "']");
        List<HashMap> sortRows = new ArrayList();
        for (Object item : rows) {
            if (item instanceof Element) {
                Element row = (Element) item;
                HashMap map = new LinkedHashMap();
                map.put(BJ_TAG.SHAPE_GROUP.name(), row.attributeValue(BJ_TAG.SHAPE_GROUP.name()));
                map.put(BJ_TAG.PNT_SERIAL.name(), row.attributeValue(BJ_TAG.PNT_SERIAL.name()));
                map.put(BJ_TAG.Y_COORD.name(), row.attributeValue(BJ_TAG.Y_COORD.name()));
                map.put(BJ_TAG.X_COORD.name(), row.attributeValue(BJ_TAG.X_COORD.name()));
                sortRows.add(map);
            }
        }
        Collections.sort(sortRows, new Comparator<HashMap>() {
            public int compare(HashMap arg0, HashMap arg1) {
                return Integer.valueOf(arg0.get(BJ_TAG.PNT_SERIAL.name()).toString()) - Integer.valueOf(arg1.get(BJ_TAG.PNT_SERIAL.name()).toString());
            }
        });
        for (HashMap row : sortRows) {
            String ringNo = String.valueOf(row.get(BJ_TAG.SHAPE_GROUP.name()));
            double x = Double.valueOf((String) row.get(BJ_TAG.Y_COORD.name()));
            double y = Double.valueOf((String) row.get(BJ_TAG.X_COORD.name()));
            if (rings.containsKey(ringNo))
                rings.get(ringNo).add(new Coordinate(x, y));
            else {
                List<Coordinate> ring = new ArrayList<Coordinate>();
                ring.add(new Coordinate(x, y));
                rings.put(ringNo, ring);
            }
            if (fearureCrs == null) fearureCrs = getCrsByCoordXD(x);
        }

        LinearRing shell = null;
        List<LinearRing> holes = null;
        Collection<List<Coordinate>> coords = rings.values();
        for (List<Coordinate> coordinates : coords) {
            Coordinate first = coordinates.get(0);
            Coordinate last = coordinates.get(coordinates.size() - 1);
            if ((first.x != last.x) || (first.y != last.y))
                coordinates.add(new Coordinate(first.x, first.y));
        }
        Polygon polygon = null;
        try {
            for (Iterator<List<Coordinate>> iterator = coords.iterator(); iterator.hasNext(); ) {
                List<Coordinate> coordinates = iterator.next();
                if (shell == null) {
                    shell = geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0]));
                } else {
                    if (holes == null) holes = new ArrayList<LinearRing>();
                    holes.add(geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0])));
                }
            }
            polygon = geometryFactory.createPolygon(shell, holes != null ? holes.toArray(new LinearRing[0]) : null);
        } catch (Exception e) {
            throw new RuntimeException(getMessage("bj.polygon.coords.error", name, e.getLocalizedMessage()));
        }
        feature.put(SHAPE, polygon.toText());
        feature.put(FEATURE_CRS, fearureCrs);
        return feature;
    }

    /**
     * get crs by coords x
     *
     * @param x
     * @return
     */
    private CoordinateReferenceSystem getCrsByCoordXD(double x) {
        CoordinateReferenceSystem crs = null;
        String s = String.valueOf(Math.round(x));
        if (s.length() == 8) {
            //
            String p = s.substring(0, 2);
            if (coordinateDM.containsKey(p)) return (CoordinateReferenceSystem) coordinateDM.get(p);
            else throw new RuntimeException(getMessage("bj.coords.dm.not.set", p));
        } else if (s.length() == 6) {
            // nanjing
            return (CoordinateReferenceSystem) coordinateDM.get(UNDEFINE);
        } else {
            throw new RuntimeException(getMessage("bj.coords.type.not.support", x));
        }
    }

    /**
     * @return
     */
    private synchronized WKTReader getWktReader() {
        if (wktReader == null) wktReader = new WKTReader(geometryFactory);
        return wktReader;
    }

    /**
     * @param location
     */
    public void setRegionSet(Resource location) {
        try {
            Map values = JSON.parseObject(IOUtils.toString(location.getURI()), Map.class);
            this.simplifyTolerance = Double.valueOf(String.valueOf(values.get("simplifyTolerance")));
            this.regionLayers = (Map) values.get(REGION_FIELD);
            this.regionSet = (Map) values.get(REGION_MAP);
            if (values.containsKey(DEFAULT_CRS)) defaultCrs = parseUndefineSR(String.valueOf(values.get(DEFAULT_CRS)));
            if (values.containsKey(COORDINATE_DM)) {
                coordinateDM = (Map) values.get(COORDINATE_DM);
                for (Map.Entry entry : coordinateDM.entrySet()) {
                    entry.setValue(parseUndefineSR((String) entry.getValue()));
                }
            }
            // validator
        } catch (IOException e) {
            logger.error(" region set file not found ");
        }
    }


}
