package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.dao.SpatialDao;
import cn.gtmap.onemap.platform.dao.impl.ArcSDEDaoImpl;
import cn.gtmap.onemap.platform.dao.impl.OracleSpatialDaoImpl;
import cn.gtmap.onemap.platform.entity.Configuration;
import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.entity.dict.Dict;
import cn.gtmap.onemap.platform.entity.dict.Item;
import cn.gtmap.onemap.platform.event.GISDaoException;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.service.GISService;
import cn.gtmap.onemap.platform.service.GeometryService;
import cn.gtmap.onemap.platform.service.MapService;
import cn.gtmap.onemap.platform.service.WebMapService;
import cn.gtmap.onemap.platform.support.spring.ApplicationContextHelper;
import cn.gtmap.onemap.platform.utils.ArrayUtils;
import cn.gtmap.onemap.platform.utils.EnumUtils;
import cn.gtmap.onemap.platform.utils.Utils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.vividsolutions.jts.geom.*;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.collections.map.MultiKeyMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.geometry.jts.Geometries;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.crs.GeographicCRS;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-20 下午5:25
 */
public class GISServiceImpl extends BaseLogger implements GISService {

    private final String SE_SHAPE_FIELD = Constant.SE_SHAPE_FIELD;

    private final String SE_SHAPE_AREA = Constant.SE_SHAPE_AREA;

    private final String OG_PRO_PERFIX = "OG_PRO_";

    private final String SEARCH_LAYERS = "layers";

    /**
     * 规划图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_E = "_E_";

    /**
     * 现状图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_H = "_H_";

    /**
     * 土地用途总体规划审查
     */
    private enum GHSC {

        TDYTQ("土地用途分区"),
        JSYDGZQ("建设用地管制区"),
        GHJBNTTZ("规划基本农田调整");
//        MZZDJSXM("重点建设项目");

        private String label;

        private GHSC(String value) {
            this.label = value;
        }

        private String getLabel() {
            return label;
        }
    }

    /**
     * 土地利用现状
     */
    private enum TDXZ {
        DLTB, XZDW
    }


    private enum Tag {
        YES, NO;
    }

    private SpatialDao spatialDao;

    @Autowired
    private GeometryService geometryService;

    @Autowired
    private MapService mapService;
    @Autowired
    private WebMapService webMapService;

    private Map searchConfig;

    public void setSearchConfig(Resource path) {
        try {
            searchConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" search config can't found ");
        }
    }


    /**
     * 初始化
     */
    @Override
    public void initialize(Constant.SpatialType type) {
        switch (type) {
            case ARC_SDE:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(ArcSDEDaoImpl.class);
                break;
            case ORACLE_SPATIAL:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(OracleSpatialDaoImpl.class);
                break;
        }
    }

    /**
     * 属性查询
     *
     * @param layerName
     * @param where
     * @param columns
     * @param returnGeometry
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String where, String[] columns, boolean returnGeometry, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(where, getMessage("query.where.notnull"));
        return spatialDao.query(layerName, where, columns, returnGeometry, dataSource);
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param wkt
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String wkt, String[] columns, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(wkt, getMessage("query.geometry.notnull"));
        return spatialDao.query(layerName, wkt, columns, dataSource);
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param geometry
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, Geometry geometry, String[] columns, String dataSource) {
        return query(layerName, geometry.toText(), columns, dataSource);
    }

    /**
     * 空间查询(进行投影转换)
     *
     * @param layerName
     * @param feature
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, SimpleFeature feature, String[] columns, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null && !sourceCRS.equals(layerCRS))
            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        return query(layerName, geometry.toText(), columns, dataSource);
    }


    /**
     * 相交分析
     *
     * @param layerName
     * @param wktPlygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, String wktPlygon, String[] returnFields, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        return spatialDao.intersect(layerName, wktPlygon, returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param polygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Polygon polygon, String[] returnFields, String dataSource) {
        return intersect(layerName, polygon.toText(), returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, CoordinateReferenceSystem sourceCRS,
                             String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null) geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        List<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    result.addAll((Collection<? extends Map<String, Object>>)
                            intersect(layerName, geo.toText(), outFields, dataSource));
                    addGeoProperty2List(result, geo);
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            result.addAll((Collection<? extends Map<String, Object>>)
                    intersect(layerName, geometry.toText(), outFields, dataSource));
            addGeoProperty2List(result, geometry);
        }
        if (sourceCRS != null) {
            for (Map<String, Object> item : result) {
                Geometry geo = geometryService.readWKT((String) item.get(SE_SHAPE_FIELD));
                item.put(Constant.SE_SHAPE_FIELD, geometryService.project(geo, layerCRS, sourceCRS).toText());
            }
        }
        return result;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem sourceCRS = null;
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    List queryResults = query(layerName, geo, outFields, dataSource);

                    if (layerCRS instanceof GeographicCRS) {
                        if (isNull(sourceCRS) && queryResults.size() > 0)
                            sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        geo = geometryService.project(geo, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
                    }
                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            geometryResult = geo1.intersection(geo);
                        } catch (Exception e) {
                            logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                            geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geo);
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, isPolygon(geometryResult) ? geometryResult.getArea() : geometryResult.getLength());
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        result.put(Constant.ORIGINAL_SHAPE_AREA, isPolygon(geo1) ? geo1.getArea() : geo1.getLength());
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            List queryResults = query(layerName, geometry, outFields, dataSource);

            if (layerCRS instanceof GeographicCRS) {
                if (isNull(sourceCRS) && queryResults.size() > 0)
                    sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                geometry = geometryService.project(geometry, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                }
                Geometry geometryResult = null;
                try {
                    geometryResult = geo1.intersection(geometry);
                } catch (Exception e) {
                    logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                    geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geometry);
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, geometryResult.getArea());
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                result.put(Constant.ORIGINAL_SHAPE_AREA, geo1.getArea());
                results.add(result);
            }
        }

        return results;
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersectByGeoJSON(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCRS = null;
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
            results = intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            results = addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            FeatureCollection collection = (FeatureCollection) geo;
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                results.addAll(addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature));
            }
        }
        return results;
    }


    /**
     * @param layerName
     * @param feature
     * @param outField
     * @param dataSource
     * @return
     */
    private List<?> intersectBySimpleFeature(String layerName, SimpleFeature feature, String[] outField, String dataSource) {
        CoordinateReferenceSystem sourceCRS = null;
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        setFeaturePros2Geo(feature, geometry);
        try {
            sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
        } catch (Exception e) {
            logger.error("intersect analysis - feature json crs not define [{}]", e.getLocalizedMessage());
        }
        return intersect(layerName, geometry, sourceCRS, outField, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return GeoJSON 格式要素集 {"type":"FeatureCollection","features":[{"type":"Feature","geometry":{"type":"Polygon","coordinates":[[]]},"properties":{}}}
     */
    @Override
    public List<?> intersect2(String layerName, String geoJson, String[] outField, String dataSource) {
        List results = intersectByGeoJSON(layerName, geoJson, outField, dataSource);
//        FeatureCollection collection = geometryService.list2FeatureCollection(results, null, null);
        return results;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect3(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = intersect(layerName, geometry, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            if (!isNull(sourceCrs) && !isNull(layerCrs))
                geometry = geometryService.project(geometry, sourceCrs, layerCrs);
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                if (!isNull(sourceCrs) && !isNull(layerCrs))
                    geometry = geometryService.project(geometry, sourceCrs, layerCrs);
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature));
//                results.addAll(intersect(layerName, geometry, outField, dataSource));
            }
        }
        return results;
    }

    /**
     * 创建新的有效的geometry
     *
     * @param geometry
     * @return
     */
    private Geometry createValidGeometry(Geometry geometry) {
        try {
            if (geometry instanceof Polygon) {
                Polygon polygon = (Polygon) geometry;
                GeometryFactory factory = geometry.getFactory();
                List<Polygon> polygons = new ArrayList<Polygon>();
                Polygon exteriorPolygon = new Polygon((LinearRing) polygon.getExteriorRing(), null, factory);
                polygons.add(exteriorPolygon);
                for (int i = 0; i < polygon.getNumInteriorRing(); i++) {
                    LinearRing interiorRing = (LinearRing) polygon.getInteriorRingN(i);
                    Polygon interiorPolygon = new Polygon(interiorRing, null, factory);
                    polygons.add(interiorPolygon);
                }
                List<Polygon> newPolygons = new ArrayList<Polygon>();
                List<Polygon> excludePolygons = new ArrayList<Polygon>();
                for (Polygon temp : polygons) {
                    temp = getPolygonWithHoles(polygons, excludePolygons, temp);
                    if (!excludePolygons.contains(temp))
                        newPolygons.add(temp);
                }
                return new MultiPolygon(newPolygons.toArray(new Polygon[0]), factory);
            } else if (geometry instanceof MultiPolygon) {
                MultiPolygon multiPolygon = (MultiPolygon) geometry;
                List<Polygon> polygonList = new ArrayList<Polygon>();
                for (int j = 0; j < multiPolygon.getNumGeometries(); j++) {

                    Polygon polygon = (Polygon) multiPolygon.getGeometryN(j);
                    if (!polygon.isValid()) {
                        MultiPolygon tempMultiPolygon = (MultiPolygon) createValidGeometry(polygon);
                        for (int k = 0; k < tempMultiPolygon.getNumGeometries(); k++) {
                            polygonList.add((Polygon) tempMultiPolygon.getGeometryN(k));
                        }
                    } else {
                        polygonList.add(polygon);

                    }
                }
                return new MultiPolygon(polygonList.toArray(new Polygon[0]), multiPolygon.getFactory());
            } else {
                logger.info("geometryType has not been supported yet");
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 判断一个polygon在polygon组中是否是含有环的polygon
     *
     * @param srcPolygons     原始polygon组
     * @param excludePolygons 内环polygon需排除
     * @param polygon
     * @return
     */
    private Polygon getPolygonWithHoles(List<Polygon> srcPolygons, List<Polygon> excludePolygons, Polygon polygon) {
        List<LinearRing> holes = new ArrayList<LinearRing>();
        for (Polygon item : srcPolygons) {
            if (polygon.equals(item) || excludePolygons.contains(polygon))
                continue;
            if (polygon.contains(item)) {
                holes.add((LinearRing) item.getExteriorRing());
                excludePolygons.add(item);
            }
        }
        if (holes.size() > 0)
            return new Polygon((LinearRing) polygon.getExteriorRing(), holes.toArray(new LinearRing[0]), polygon.getFactory());
        else
            return polygon;
    }


    /**
     * @param value
     * @param feature
     * @return
     */
    private List addFeaturePros2List(List<Map> value, SimpleFeature feature) {
        for (Map item : value) {
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                item.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }
        }
        return value;
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param columns    包含图形、属性数据
     * @param dataSource
     * @return
     */
    @Override
    public String insert(String layerName, Map<String, Object> columns, String dataSource) {
        LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
        if (!isNull(layerRegion.getRegionField())) {
            boolean contains = false;
            for (Map.Entry entry : columns.entrySet()) {
                if (layerRegion.getRegionField().equalsIgnoreCase((String) entry.getKey())
                        && geometryService.containsRegionValue(String.valueOf(entry.getValue()))) {
                    contains = true;
                    break;
                }
            }
            if (!contains)
                throw new RuntimeException(getMessage("insert.region.field.not.null", layerName, layerRegion.getRegionField()));
        }
        return spatialDao.insert(layerName, columns, dataSource);
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param geoJSON    GeoJSON 格式
     * @param dataSource
     * @return primaryKey
     */
    @Override
    public String insert(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (Map.Entry<String, Object> entry : columns.entrySet()) {
                    if (entry.getValue() instanceof Geometry) {
                        Geometry geometry = (Geometry) entry.getValue();
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(entry.getKey());
                    }
                }
                if (featureIterator.hasNext())
                    insert(layerName, columns, dataSource);
                else
                    return insert(layerName, columns, dataSource);
            }
        }
        return insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 插入数据（返回geojson）
     *
     * @param layerName
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
            resultFc.put("type", "FeatureCollection");
            List<LinkedHashMap> resultFeatures = new ArrayList();
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                properties.put("OBJECTID", objectId);
                resultFeatures.add(featureMap);
            }
            resultFc.put("features", resultFeatures);
            return JSON.toJSONString(resultFc);

        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
            JSONObject properties = (JSONObject) featureMap.get("properties");
            String objectId = insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
            properties.put("OBJECTID", objectId);

            return JSON.toJSONString(featureMap);
        }
        return null;

    }

    /**
     * @param layerName
     * @param geoJSON
     * @param check      是否执行拓扑检查
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, boolean check, String dataSource) {

        if (check) {
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
            if (geo instanceof FeatureCollection) {

                LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
                resultFc.put("type", "FeatureCollection");
                List<LinkedHashMap> resultFeatures = new ArrayList();
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    List list = query(layerName, feature, null, dataSource);
//                    List list = query(layerName, feature.getAttribute("geometry").toString(), null, dataSource);
                    if (list.size() > 0) {
                        logger.error("图形已在此图层中存在,插入失败!【{}】", geometryService.toFeatureJSON(feature).toString());
                        if (featureIterator.hasNext())
                            continue;
                        else
                            throw new GISDaoException(GISDaoException.Method.INSERT, "图形已在此图层中存在,插入失败!", GISDaoException.Type.ARC_SDE);
                    }
                    CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    columns = geometryService.simpleFeature2Map(feature);
                    try {
                        for (String key : columns.keySet()) {
                            if (columns.get(key) instanceof Geometry) {
                                Geometry geometry = (Geometry) columns.get(key);
                                if (layerCRS != null && sourceCRS != null)
                                    geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                                columns.put(SE_SHAPE_FIELD, geometry.toText());
                                columns.remove(key);
                                break;
                            }
                        }
                    } catch (Exception e) {
                        throw new RuntimeException(e.getLocalizedMessage());
                    }
                    String objectId = insert(layerName, columns, dataSource);
                    LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                    JSONObject properties = (JSONObject) featureMap.get("properties");
                    properties.put("OBJECTID", objectId);
                    resultFeatures.add(featureMap);
                }
                resultFc.put("features", resultFeatures);
                return JSON.toJSONString(resultFc);

            } else if (geo instanceof SimpleFeature) {

                SimpleFeature feature = (SimpleFeature) geo;
                List list = query(layerName, feature, null, dataSource);
//                List list = query(layerName, feature.getAttribute("geometry").toString(), null, dataSource);
                if (list.size() > 0) {
                    logger.error("图形已在此图层中存在,插入失败!【{}】", geoJSON);
                    throw new GISDaoException(GISDaoException.Method.INSERT, "图形已在此图层中存在,插入失败!", GISDaoException.Type.ARC_SDE);
                }
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                String objectId = insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
                properties.put("OBJECTID", objectId);

                return JSON.toJSONString(featureMap);
            }

        } else {
            return insert2(layerName, geoJSON, dataSource);
        }


        return null;
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, Map<String, Object> columns, String dataSource) {
        return spatialDao.update(layerName, primaryKey, columns, dataSource);
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        return update(layerName, primaryKey, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 删除
     *
     * @param layerName
     * @param primaryKey
     * @param dataSource
     * @return
     */
    @Override
    public boolean delete(String layerName, String primaryKey, String dataSource) {
        return spatialDao.delete(layerName, primaryKey, dataSource);
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry      GeoJSON format
     * @param outFields
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String dltbLayerName, String xzdwLayerName, String geometry, String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        Map<String, Double> analysisResult = new HashMap<String, Double>();
        Map<String, Double> analysisTemp = new HashMap<String, Double>();
        /**
         * 地类图斑相交
         */
        String[] dltbOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "ZLDWDM", "TBBH"});
        List<Map<String, Object>> results = (List<Map<String, Object>>) intersectByGeoJSON(dltbLayerName, geometry, dltbOutFields, dataSource);
        for (Map<String, Object> result : results) {
            Double area = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (area > 0.4) {
                String dlmc = String.valueOf(result.get("DLMC"));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + area);
                analysisTemp.put(String.valueOf(result.get("ZLDWDM")) + String.valueOf(result.get("TBBH")), area);
            }
        }
        /**
         * 现状地物相交并扣除
         */
        String[] xzdwOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "KD", "KCTBDWDM2", "KCTBDWDM1", "KCTBBH1", "KCTBBH2"});
        List<Map<String, Object>> xzResults = (List<Map<String, Object>>) intersectByGeoJSON(xzdwLayerName, geometry, xzdwOutFields, dataSource);
        for (Map<String, Object> result : xzResults) {
            Double length = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (length > 0.08) {
                String dlmc = String.valueOf(result.get("DLMC"));
                Double kd = Double.parseDouble(String.valueOf(result.get("KD")));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + length * kd);
                //
                String where = "ZLDWDM='" + result.get("KCTBDWDM1") + "' and TBBH='" + result.get("KCTBBH1") + "'";
                List<Map<String, Object>> temps = new ArrayList<Map<String, Object>>();
                try {
                    temps = (List<Map<String, Object>>) query(dltbLayerName, where,
                            ArrayUtils.add2Arrays(outFields, new String[]{"DLMC"}), true, dataSource);
                } catch (Exception e) {
                    logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                }
                String kctb1Key = String.valueOf(result.get("KCTBDWDM1")) + String.valueOf(result.get("KCTBBH1"));
                //
                if (StringUtils.isBlank((String) result.get("KCTBDWDM2"))) {
                    for (Map<String, Object> temp : temps) {
                        String tmpDlmc = String.valueOf(temp.get("DLMC"));
                        analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - length * kd);
                        analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - length * kd);
                    }
                } else {
                    //
                    Tag tag = Tag.NO;
                    String tmpDlmc = "";
                    String tmpDlmc1 = "";
                    Double tmpXzdwKd = 0.0;
                    String tmpXzdwKctbdwdm1 = "";
                    String tmpXzdwKctbbh1 = "";
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb1Key)) {
                            Double tmpValue = analysisTemp.get(kctb1Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                tmpXzdwKd = (length / 2.0) * kd;
                                tmpDlmc1 = tmpDlmc;
                                tmpXzdwKctbbh1 = String.valueOf(result.get("KCTBBH1"));
                                tmpXzdwKctbdwdm1 = String.valueOf(result.get("KCTBDWDM1"));
                                analysisTemp.put(tmpXzdwKctbdwdm1 + tmpXzdwKctbbh1, tmpValue - (length / 2.0) * kd);
                            } else
                                tag = Tag.YES;
                        } else
                            tag = Tag.YES;
                    }
                    //
                    where = "ZLDWDM='" + result.get("KCTBDWDM2") + "' and TBBH='" + result.get("KCTBBH2") + "'";
                    try {
                        temps = (List<Map<String, Object>>) query(dltbLayerName, where, new String[]{"DLMC"}, true, dataSource);
                    } catch (Exception e) {
                        logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        temps.clear();
                    }
                    String kctb2Key = String.valueOf(result.get("KCTBDWDM2")) + String.valueOf(result.get("KCTBBH2"));
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb2Key)) {
                            Double tmpValue = analysisTemp.get(kctb2Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                if (!Tag.YES.equals(tag)) {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                } else {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                }
                            } else {
                                tmpDlmc = tmpDlmc1;
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                                analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                            }
                        } else {
                            tmpDlmc = tmpDlmc1;
                            analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                            analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                        }
                    }
                }
            }
        }
        return analysisResult;
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param regionCode
     * @param geometry   GeoJSON format
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String regionCode, String geometry, String dataSource) {
        regionCode = Utils.formatRegionCode(regionCode);
        String dltb = TDXZ.DLTB.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        String xzdw = TDXZ.XZDW.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        return tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
    }

    @Override
    public Map tdlyxzAnalysis2(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        if (StringUtils.isBlank(dltbLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "地类图斑图层名称为空！"));
        if (StringUtils.isBlank(xzdwLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "线状地物图层名称为空！"));
        if (StringUtils.isBlank(geometry))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "分析地块坐标为空！"));
//        Double geometryArea = geometryService.readGeometryAera(geometry);
//        if (geometryArea > 200000 || geometryArea < 100) {
//            throw new RuntimeException("分析地块面积超过20公顷或者小于100平米！");
//        }
        Map<String, Double> dlMap = new HashMap<String, Double>();   //分析地类面积结果
        Map<String, Double> bhMap = new HashMap<String, Double>();
        MultiKeyMap analysisDetailMap = MultiKeyMap.decorate(new HashedMap()); //分析地类面积明细
        String[] columns = {"DLMC"};
        List<Map<String, Object>> xzdwDltbResult = new ArrayList<Map<String, Object>>();
        List<Map<String, Object>> dltbAnalysisResult = new ArrayList<Map<String, Object>>();
        List<Map<String, Object>> xzdwAnalysisResult = new ArrayList<Map<String, Object>>();
        //与地类图斑层进行相交分析
        try {
//            dltbAnalysisResult = (List<Map<String, Object>>) intersectByGeoJSON(dltbLayerName, geometry, "DLMC;ZLDWDM;TBBH;ZLDWMC;QSXZ;DLBM;TBMJ".split(";"), dataSource);
            dltbAnalysisResult = (List<Map<String, Object>>) intersect3(dltbLayerName, geometry, "DLMC;ZLDWDM;TBBH;ZLDWMC;QSXZ;DLBM;TBMJ;QSDWDM;QSDWMC".split(";"), dataSource);
        } catch (Exception e) {
            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
        }

        if (dltbAnalysisResult.size() > 0) {
            for (int i = 0; i < dltbAnalysisResult.size(); i++) {
                Double tmpArea = Double.parseDouble(String.valueOf(dltbAnalysisResult.get(i).get("SHAPE_AREA"))); //相交部分的面积
                if (tmpArea > 0.4) {
                    if (dlMap.containsKey(dltbAnalysisResult.get(i).get("DLMC")))
                        dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), dlMap.get(dltbAnalysisResult.get(i).get("DLMC")) + tmpArea);
                    else
                        dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), tmpArea);
                    bhMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")) + String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), tmpArea);

                    //图斑明细
                    if (analysisDetailMap.containsKey(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")))) {
                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpArea);
                    } else {
                        Map<String, Object> detailValue = new HashMap<String, Object>();
                        detailValue.put("ZLDWDM", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")));
                        detailValue.put("ZLDWMC", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWMC")));
                        detailValue.put("TBBH", String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                        detailValue.put("QSXZ", String.valueOf(dltbAnalysisResult.get(i).get("QSXZ")));
                        detailValue.put("DLMC", String.valueOf(dltbAnalysisResult.get(i).get("DLMC")));
                        detailValue.put("DLBM", String.valueOf(dltbAnalysisResult.get(i).get("DLBM")));
                        detailValue.put("TBMJ", String.valueOf(dltbAnalysisResult.get(i).get("TBMJ")));
                        detailValue.put("QSDWDM", String.valueOf(dltbAnalysisResult.get(i).get("QSDWDM")));
                        detailValue.put("QSDWMC", String.valueOf(dltbAnalysisResult.get(i).get("QSDWMC")));
                        detailValue.put("CCMJ", tmpArea);
                        analysisDetailMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), detailValue);
                    }

                }

            }
        }

        //与线状地物层进行相交分析，并扣除
        try {
            xzdwAnalysisResult = (List<Map<String, Object>>) intersect3(xzdwLayerName, geometry, "DLMC;KD;KCTBDWDM2;KCTBDWDM1;KCTBBH1;KCTBBH2;QSDWDM1;XZDWBH;QSXZ;DLBM;XZDWMJ;QSDWMC1".split(";"), dataSource);
        } catch (Exception e) {
            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
        }

        if (xzdwAnalysisResult.size() > 0) {
            for (int i = 0; i < xzdwAnalysisResult.size(); i++) {
                Double tmpLength = Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("SHAPE_AREA")));
                if (tmpLength > 0.08) {
                    if (dlMap.containsKey(xzdwAnalysisResult.get(i).get("DLMC"))) {
                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) + (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                    } else
                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                    //线物明细
                    if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))) {
                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                    } else {
                        Map<String, Object> detailValue = new HashMap<String, Object>();
                        detailValue.put("ZLDWDM", String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH"))));
                        detailValue.put("ZLDWMC", String.valueOf(xzdwAnalysisResult.get(i).get("")));
                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                        detailValue.put("QSXZ", String.valueOf(xzdwAnalysisResult.get(i).get("QSXZ")));
                        detailValue.put("DLMC", String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")));
                        detailValue.put("DLBM", String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")));
                        detailValue.put("TBMJ", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWMJ")));
                        detailValue.put("QSDWDM1", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")));
                        detailValue.put("QSDWMC1", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWMC1")));
                        detailValue.put("CCMJ", tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                        analysisDetailMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), detailValue);
                    }
                    if (StringUtils.isEmpty(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2"))) || String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")).equalsIgnoreCase("null")) {
                        String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {
                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                dlMap.put(String.valueOf(xzdwDltbResult.get(j).get("DLMC")), dlMap.get(String.valueOf(xzdwDltbResult.get(j).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                String tmpKey = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                bhMap.put(tmpKey, bhMap.get(tmpKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                //扣除线物明细
                                if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                } else {
                                    throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "ERROR", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    throw new RuntimeException("扣除线物明细异常，扣除图斑单位代码1:[ " + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + " ]");
//                                    Map<String, Object> detailValue = new HashMap<String, Object>();
//                                    detailValue.put("ZLDWDM", "error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                    analysisDetailMap.put("error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                }
                            }
                        }
                    } else {
                        //记录扣除状态
                        String kouchuTag = "no";
                        //记录扣除面积
                        double kouchuMj = 0.0;
                        //记录扣除地类
                        String kouchuDlmc = "";
                        //扣除ZLDWDM
                        String kouchuZldwdm = "";
                        //扣除tbbh
                        String kouchuTbbh = "";
                        String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {

                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                    Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                    if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        kouchuMj = (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")));
                                        kouchuDlmc = tmpDlmcKey;
                                        kouchuZldwdm = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1"));
                                        kouchuTbbh = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                        bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        //扣除线物明细
                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);
                                    } else {
                                        kouchuTag = "yes";
//                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积过小", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                        throw new RuntimeException("扣除线物明细异常:[面积过小，扣除图斑单位代码1:[ " + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + " ]]");
                                        Map<String, Object> detailValue = new HashMap<String, Object>();
                                        detailValue.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("TBMJ", tmpValue);
                                        detailValue.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                    }
                                } else {
                                    kouchuTag = "yes";
                                    throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有占用", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    Map<String, Object> detailValue = new HashMap<String, Object>();
//                                    detailValue.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                    detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                    analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                }
                            }
                            if (xzdwDltbResult.size() == 0) {
                                kouchuTag = "yes";
                                throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有找到", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                Map<String, Object> detailValue = new HashMap<String, Object>();
//                                detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                                detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                                analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                            }
                        } else {
                            kouchuTag = "yes";
                            throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有找到", 1, xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                            Map<String, Object> detailValue = new HashMap<String, Object>();
//                            detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
//                            detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
//                            analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                        }

                        tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM2") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH2") + "'";
                        try {
                            xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                        } catch (Exception e) {
                            logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        }
                        if (xzdwDltbResult.size() > 0) {
                            for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")))) {
                                    Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                    if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                        if (!kouchuTag.equals("yes")) {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        } else {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        }
                                    } else {
                                        tmpDlmcKey = kouchuDlmc;
                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                        bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);

                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

//                                        throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "面积过小", 2, xzdwAnalysisResult.get(i).get("KCTBDWDM2")));

                                        Map<String, Object> detailValue1 = new HashMap<String, Object>();
                                        detailValue1.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                        detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                        detailValue1.put("TBMJ", tmpValue);
                                        detailValue1.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);

                                    }
                                } else {
                                    tmpDlmcKey = kouchuDlmc;
                                    dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                    bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);

                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

                                    throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "没有占用", 2, xzdwAnalysisResult.get(i).get("KCTBDWDM2")));

//                                    Map<String, Object> detailValue1 = new HashMap<String, Object>();
//                                    detailValue1.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
//                                    detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
//                                    analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);
                                }
                            }
                        }
                    }

                }
            }
        }
        Map resultMap = new HashMap();
        resultMap.put("analysisArea", dlMap);
        List resultDetailList = new ArrayList();
        Iterator keyIterator = analysisDetailMap.keySet().iterator();
        while (keyIterator.hasNext()) {
            Object tmpKey = keyIterator.next();
            resultDetailList.add(analysisDetailMap.get(tmpKey));
        }
        resultMap.put("analysisAreaDetail", resultDetailList);
//        return JSON.toJSONString(resultMap);
        return resultMap;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> tdghscAnalysis(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        if (StringUtils.isBlank(layerType))
            throw new RuntimeException(getMessage("layer.type.notnull"));
        String layerName = layerType.concat(LAYER_MIDDLE_FIX_E).concat(Utils.formatRegionCode(regionCode));
        return intersect3(layerName, geometry, outFields, dataSource);
//        return intersectByGeoJSON(layerName, geometry, outFields, dataSource);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public String tdghscAnalysis2(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        List result = tdghscAnalysis(layerType, regionCode, geometry, outFields, dataSource);
        FeatureCollection collection = geometryService.list2FeatureCollection(result, null, null);
        return geometryService.toFeatureJSON(collection);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis2(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis2(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 组织土地规划分析的结果（所有图层类型）
     *
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(Map analysisResult) {

        Map resultMap = new HashMap();
        try {
            for (GHSC item : GHSC.values()) {
                List list = new ArrayList();
//                List detailList = new ArrayList();
                Map total = new HashMap();
                Map ghMap = new HashMap();
                double totalArea = 0;
                JSONObject singleObj = JSON.parseObject(analysisResult.get(item.getLabel()).toString());
                JSONArray features = JSON.parseArray(singleObj.get("features").toString());
                totalArea = getTotalArea(features);
                total.put("LXMC", "地块总面积");
                total.put("AREA", totalArea);
                list.add(total);
                switch (item.ordinal()) {
                    case 0:
                        for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name() + area);
                        }
                        break;
                    case 1:
                        for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name() + area);
                        }
                        break;
                    case 2:
                        for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);
                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("TZLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.name());
                        }
                        break;
                    case 3:
                        for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                            List detail = new ArrayList();
                            double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm());
                            double per = 0;
                            if (totalArea > 0)
                                per = area / totalArea * 100;
                            Map temp = new HashMap();
                            temp.put("LXMC", obj.name());
                            temp.put("AREA", area);
                            temp.put("PER", per);

                            for (int i = 0; i < features.size(); i++) {
                                LinkedHashMap detailMap = new LinkedHashMap();
                                JSONObject feature = (JSONObject) features.get(i);
                                Map properties = (Map) feature.get("properties");
                                if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                    detailMap.put("类型名称", obj.name());
//                                    detailMap.put("ghsc", item.getLabel());
                                    detailMap.put("类型代码", properties.get("XMLXDM"));
//                                    detailMap.put("地块名称", properties.get("G_plotName"));
//                                    detailMap.put("地块编号", properties.get("G_plotId"));
                                    detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                    detailMap.put("标识码", properties.get("BSM"));
                                    detailMap.put("面积", properties.get("SHAPE_AREA"));
                                    detail.add(detailMap);
                                }
                            }
                            temp.put("detail", detail);
                            list.add(temp);
                            logger.info(obj.getLxdm());
                        }
                        break;
                }
//                ghMap.put("detail", detailList);
                ghMap.put("info", list);
                resultMap.put(item.getLabel(), ghMap);
            }
        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());

        }

        return resultMap;
    }

    /**
     * 组织土地规划分析的结果（单个图层类型）
     *
     * @param layerType
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(String layerType, String analysisResult) {
        Map resultMap = new HashMap();
        try {

            JSONObject fc = JSON.parseObject(analysisResult);
            JSONArray features = (JSONArray) fc.get("features");

            List list = new ArrayList();
            Map total = new HashMap();
            Map ghMap = new HashMap();
            double totalArea = 0;
            totalArea = getTotalArea(features);
            total.put("LXMC", "地块总面积");
            total.put("AREA", totalArea);
            list.add(total);
            for (GHSC item : GHSC.values()) {

                if (item.name().equals(layerType)) {
                    switch (item.ordinal()) {
                        case 0:
                            for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    Map detailMap = new HashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name() + area);
                            }
                            break;
                        case 1:
                            for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
//                                        detailMap.put("ghsc", item.getLabel());
                                        detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name() + area);
                            }
                            break;
                        case 2:
                            for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TZLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.name());
                            }
                            break;
                        case 3:
                            for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm());
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);

                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("XMLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.info(obj.getLxdm());
                            }
                            break;
                    }
                    ghMap.put("info", list);
                    resultMap.put(item.getLabel(), ghMap);
                }
            }

        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultMap;
    }

    @Override
    public LinkedHashMap tdghscExcelData(Map result) {
        LinkedHashMap map = new LinkedHashMap();
        for (Object key : result.keySet()) {
            List list = new ArrayList();
            List keyList = (List) ((Map) result.get(key)).get("info");
            for (int i = 1; i < keyList.size(); i++) {
                Map temp = (Map) keyList.get(i);
                List tempList = new ArrayList();
                tempList.add(temp.get("LXMC"));
                tempList.add(Math.round(Double.valueOf(temp.get("AREA").toString())));
                list.add(tempList);
            }
            map.put(key, list);
        }
        return map;
    }

    /**
     * 组织土地利用现状分析的结果
     *
     * @param analysisResult
     * @return
     */
    @Override
    public List<Map> tdlyxzResult(Map analysisResult, String tpl) {
        List<Map> resultList = new ArrayList<Map>();
        try {
            List detailList = (List) analysisResult.get("analysisAreaDetail");
            List<Map> dwdmList = new ArrayList<Map>();
            for (int i = 0; i < detailList.size(); i++) {
                Map temp = new HashMap();
                List list = new ArrayList();
                int tag = 0;

                Map map = (Map) detailList.get(i);
                String xzqdm = map.get("ZLDWDM").toString().substring(0, 6);
                list.add(map);

                temp.put("xzqdm", xzqdm);
                temp.put("detail", list);

                if (dwdmList.size() > 0) {
                    for (Map dwdmMap : dwdmList) {
                        if (dwdmMap.get("xzqdm").toString().equals(xzqdm)) {
                            List tempList = (List) dwdmMap.get("detail");
                            tempList.add(map);
                            dwdmMap.put("detail", tempList);
                            tag = 1;
                        }
                    }
                }
                if (tag == 0) {
                    dwdmList.add(temp);
                }
            }

            for (int j = 0; j < dwdmList.size(); j++) {
                List<Map> groupList = getTdlyxzDictList(tpl);
                Map temp = new HashMap();
                Map map = dwdmList.get(j);
                List<Map> list = (List) map.get("detail");
                for (Map detailMap : list) {
                    String dlbm = detailMap.get("DLBM").toString();
                    double area = Double.valueOf(detailMap.get("TBMJ").toString());
                    for (Map groupMap : groupList) {
                        List<Map> childList = new ArrayList<Map>();
                        if (groupMap.containsKey("detail"))
                            childList = (List<Map>) groupMap.get("detail");
                        if (groupMap.get("dlbm").toString().equals(dlbm.substring(0, 2))) {
                            double newArea = Double.valueOf(groupMap.get("area").toString()) + area;
                            groupMap.put("area", newArea);
                            childList = add2List(childList, detailMap);
                            groupMap.put("detail", childList);
                        }
                    }
                }

                temp.put("xzqdm", map.get("xzqdm"));
                temp.put("sumArea", getSumArea(groupList));
                temp.put("categoryA", groupList);
                temp.put("categoryB", getCategory(groupList));

                resultList.add(temp);
            }

        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultList;
    }

    @Override
    public LinkedHashMap tdlyxzExcelData(List<Map> result) {
        LinkedHashMap map = new LinkedHashMap();
        List list1 = new ArrayList();
        List list2 = new ArrayList();
        map.put("一级分类", list1);
        map.put("二级分类", list2);
        for (Map item : result) {
            List temp1 = new ArrayList();
            List temp2 = new ArrayList();
            temp1.add(item.get("xzqdm"));
            temp2.add(item.get("xzqdm"));
            temp2.add(item.get("sumArea"));
            List<Map> categoryA = (List<Map>) item.get("categoryA");
            Map categoryB = (Map) item.get("categoryB");
            for (Map tempMap : categoryA) {
                temp1.add(tempMap.get("area"));
            }
            temp2.add(isNull(categoryB.get("01")) ? 0 : categoryB.get("01"));
            temp2.add(isNull(categoryB.get("011")) ? 0 : categoryB.get("011"));
            temp2.add(isNull(categoryB.get("012")) ? 0 : categoryB.get("012"));
            temp2.add(isNull(categoryB.get("013")) ? 0 : categoryB.get("013"));
            temp2.add(isNull(categoryB.get("02")) ? 0 : categoryB.get("02"));
            temp2.add(isNull(categoryB.get("021")) ? 0 : categoryB.get("021"));
            temp2.add(isNull(categoryB.get("022")) ? 0 : categoryB.get("022"));
            temp2.add(isNull(categoryB.get("023")) ? 0 : categoryB.get("023"));
            temp2.add(isNull(categoryB.get("03")) ? 0 : categoryB.get("03"));
            temp2.add(isNull(categoryB.get("031")) ? 0 : categoryB.get("031"));
            temp2.add(isNull(categoryB.get("032")) ? 0 : categoryB.get("032"));
            temp2.add(isNull(categoryB.get("033")) ? 0 : categoryB.get("033"));
            temp2.add(isNull(categoryB.get("04")) ? 0 : categoryB.get("04"));
            temp2.add(isNull(categoryB.get("041")) ? 0 : categoryB.get("041"));
            temp2.add(isNull(categoryB.get("042")) ? 0 : categoryB.get("042"));
            temp2.add(isNull(categoryB.get("043")) ? 0 : categoryB.get("043"));
            temp2.add(isNull(categoryB.get("20")) ? 0 : categoryB.get("20"));
            temp2.add(isNull(categoryB.get("201")) ? 0 : categoryB.get("201"));
            temp2.add(isNull(categoryB.get("202")) ? 0 : categoryB.get("202"));
            temp2.add(isNull(categoryB.get("203")) ? 0 : categoryB.get("203"));
            temp2.add(isNull(categoryB.get("204")) ? 0 : categoryB.get("204"));
            temp2.add(isNull(categoryB.get("205")) ? 0 : categoryB.get("205"));
            temp2.add(isNull(categoryB.get("10")) ? 0 : categoryB.get("10"));
            temp2.add(isNull(categoryB.get("101")) ? 0 : categoryB.get("101"));
            temp2.add(isNull(categoryB.get("102")) ? 0 : categoryB.get("102"));
            temp2.add(isNull(categoryB.get("104")) ? 0 : categoryB.get("104"));
            temp2.add(isNull(categoryB.get("105")) ? 0 : categoryB.get("105"));
            temp2.add(isNull(categoryB.get("106")) ? 0 : categoryB.get("106"));
            temp2.add(isNull(categoryB.get("107")) ? 0 : categoryB.get("107"));
            temp2.add(isNull(categoryB.get("11")) ? 0 : categoryB.get("11"));
            temp2.add(isNull(categoryB.get("111")) ? 0 : categoryB.get("111"));
            temp2.add(isNull(categoryB.get("112")) ? 0 : categoryB.get("112"));
            temp2.add(isNull(categoryB.get("113")) ? 0 : categoryB.get("113"));
            temp2.add(isNull(categoryB.get("114")) ? 0 : categoryB.get("114"));
            temp2.add(isNull(categoryB.get("115")) ? 0 : categoryB.get("115"));
            temp2.add(isNull(categoryB.get("116")) ? 0 : categoryB.get("116"));
            temp2.add(isNull(categoryB.get("117")) ? 0 : categoryB.get("117"));
            temp2.add(isNull(categoryB.get("118")) ? 0 : categoryB.get("118"));
            temp2.add(isNull(categoryB.get("119")) ? 0 : categoryB.get("119"));
            temp2.add(isNull(categoryB.get("12")) ? 0 : categoryB.get("12"));
            temp2.add(isNull(categoryB.get("122")) ? 0 : categoryB.get("122"));
            temp2.add(isNull(categoryB.get("123")) ? 0 : categoryB.get("123"));
            temp2.add(isNull(categoryB.get("124")) ? 0 : categoryB.get("124"));
            temp2.add(isNull(categoryB.get("125")) ? 0 : categoryB.get("125"));
            temp2.add(isNull(categoryB.get("126")) ? 0 : categoryB.get("126"));
            temp2.add(isNull(categoryB.get("127")) ? 0 : categoryB.get("127"));

            list1.add(temp1);
            list2.add(temp2);

        }
        List temp = new ArrayList();
        temp = (ArrayList) list1.get(0);
        for (int i = 0; i < temp.size(); i++) {
            Object value = temp.get(i);
            if (value instanceof Double) {
                temp.remove(i);
                value = Math.round((Double) value * 100) / 100.00;
                temp.add(i, value);
            }
        }
        temp = (ArrayList) list2.get(0);
        for (int j = 0; j < temp.size(); j++) {
            Object value = temp.get(j);
            if (value instanceof Double) {
                temp.remove(j);
                value = Math.round((Double) value * 100) / 100.00;
                temp.add(j, value);
            }
        }

        return map;
    }

    @Override
    public List<LinkedHashMap> analysisExcelData(List list) {
        List<LinkedHashMap> result = new ArrayList<LinkedHashMap>();
        for (int i = 0; i < list.size(); i++) {
            LinkedHashMap map = new LinkedHashMap();
            List headerList = new ArrayList();
            List dataList = new ArrayList();
            JSONObject obj = (JSONObject) list.get(i);
            JSONArray children = (JSONArray) obj.get("children");
            for (int j = 0; j < children.size(); j++) {
                JSONObject child = (JSONObject) children.get(j);
                List temp = new ArrayList();
                if (j == 0) {
                    for (String key : child.keySet()) {
                        headerList.add(key);
                    }
                }
                if (child.get(headerList.get(0)) instanceof BigDecimal)
                    temp.add(processValue(((BigDecimal) child.get(headerList.get(0))).doubleValue(), 3));
                else if (child.get(headerList.get(1)) instanceof BigDecimal)
                    temp.add(processValue(((BigDecimal) child.get(headerList.get(1))).doubleValue(), 3));
                dataList.add(temp);
            }
            map.put("name", obj.get("label"));
            map.put("header", headerList);
            map.put("data", dataList);
            result.add(map);
        }
        return result;
    }

    /**
     * 处理double类型数据,返回处理后的新数据
     *
     * @param value
     * @param precision
     * @return
     */
    public double processValue(Double value, int precision) {

        switch (precision) {
            case 1:
                return Math.round(value * 100) / 100.0;
            case 2:
                return Math.round(value * 100) / 100.00;
            case 3:
                return Math.round(value * 100) / 100.000;
            default:
                return Math.round(value * 100) / 100;

        }
    }


    public double getSumArea(List<Map> list) {

        double area = 0;
        for (Map map : list) {
            area += Double.valueOf(map.get("area").toString());
        }
        return area;
    }

    public Map getCategory(List<Map> list) {
        Map result = new HashMap();
        try {
            for (Map map : list) {
                result.put(map.get("dlbm"), map.get("area"));
                if (map.containsKey("detail")) {
                    List<Map> detailList = (List<Map>) map.get("detail");
                    for (Map detail : detailList) {
                        result.put(detail.get("dlbm"), detail.get("area"));
                    }
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }

    public List<Map> add2List(List<Map> list, Map map) {
        int tag = 0;
        String dlbm = map.get("DLBM").toString();
        double area = Double.valueOf(map.get("TBMJ").toString());
        Map temp = new HashMap();
        try {
            for (Map item : list) {

                if (item.get("dlbm").equals(dlbm)) {
                    double newArea = Double.valueOf(item.get("area").toString()) + area;
                    item.put("area", newArea);
                    tag = 1;
                }
            }
            if (tag == 0) {
                temp.put("area", area);
                temp.put("dlbm", dlbm);
                list.add(temp);
            }
        } catch (NumberFormatException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return list;
    }


    public List<Map> tdlyxzResult2(Map analysisResult, String tpl) {
        List<Map> resultList = new ArrayList<Map>();
        List<Map> list = (List) analysisResult.get("analysisAreaDetail");
        List<Map> groupList = getTdlyxzDictList(tpl);
        for (Map map : list) {
            String dlbm = map.get("DLBM").toString();
            double area = Double.valueOf(map.get("TBMJ").toString());
            for (Map groupMap : groupList) {
                if (groupMap.get("dlbm").toString().equals(dlbm.substring(0, 2))) {
                    double newArea = Double.valueOf(groupMap.get("area").toString()) + area;
                    ((List) groupMap.get("detail")).add(map);
                    groupMap.put("area", newArea);
                }
            }
        }
        for (int i = 0; i < groupList.size(); i++) {
            Map map = groupList.get(i);
            Map temp = new HashMap();
            double area = Double.valueOf(map.get("area").toString());
            if (area > 0) {
                temp.put("dlmc", map.get("dlmc"));
                temp.put("area", map.get("area"));
                temp.put("detail", map.get("detail"));
                resultList.add(temp);
            }
        }
        return resultList;
    }

    /**
     * 快速检索
     *
     * @param value
     * @return
     */
    @Override
    public List search(String value, boolean returnGeometry) {
        if (searchConfig == null) throw new RuntimeException(getMessage("search.config.not.found"));
        List result = new ArrayList();
        List<Map> layers = (List) searchConfig.get(SEARCH_LAYERS);
        for (Map layer : layers) {
            String name = (String) layer.get("layer");
            String[] fields = ((String) layer.get("fields")).split(",");
            String where = getWhereClause(fields, value);
            result.addAll(spatialDao.query(name, where, fields, returnGeometry, 10, null));
        }
        return result;
    }

    /**
     * @param features
     * @return
     */
    private double getTotalArea(JSONArray features) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {
            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
        }
        return area;
    }

    /**
     * @param field
     * @param features
     * @param lxdm
     * @return
     */
    private double getAreaByLxdm(String field, JSONArray features, String lxdm) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {

            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            if (property.get(field) == null)
//                throw new RuntimeException("字段【"+field+"】不存在！");
                return 0;
            else {
                if (property.get(field).equals(lxdm))
                    area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
            }
        }
        return area;
    }

    /**
     * @param fields
     * @param keyword
     * @return
     */
    private String getWhereClause(String[] fields, String keyword) {
        String or = " or ";
        StringBuilder b = new StringBuilder();
        for (String field : fields) {
            b.append(field);
            b.append(" like '");
            b.append(keyword);
            b.append("%'");
            b.append(or);
        }
        b.delete(b.length() - or.length(), b.length());
        return b.toString();
    }


    /**
     * GeoJSON to Map
     *
     * @param geoJSON single geometry
     * @return
     */
    private Map<String, Object> geoJSON2Map(String geoJSON, CoordinateReferenceSystem targetCRS) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        Map<String, Object> columns = null;
        if (geo instanceof Geometry) {
            columns = new HashMap<String, Object>();
            columns.put(SE_SHAPE_FIELD, ((Geometry) geo).toText());
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geoJSON);
            columns = geometryService.simpleFeature2Map(feature);
            try {
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (targetCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, targetCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
            } catch (Exception e) {
                logger.error(e.getLocalizedMessage());
                throw new RuntimeException(e.getLocalizedMessage());
            }
        } else if (geo instanceof GeometryCollection) {
            throw new RuntimeException("current omp version don't support GeometryCollection ");
        } else if (geo instanceof FeatureCollection) {
            throw new RuntimeException("current omp version don't support FeatureCollection ");
        } else {
            throw new RuntimeException(getMessage("geometry.undefined"));
        }
        return columns;
    }

    /**
     * feature to geometry
     *
     * @param feature
     * @param geometry
     * @return
     */
    private Geometry setFeaturePros2Geo(SimpleFeature feature, Geometry geometry) {
        Map<String, Object> map = null;
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                Geometry geo = geometry.getGeometryN(i);
                map = new HashMap<String, Object>();
                for (Property p : feature.getProperties()) {
                    if (p.getName().equals("geometry")) continue;
                    map.put(p.getName().getLocalPart(), p.getValue());
                }
                geo.setUserData(map);
            }
        } else {
            map = new HashMap<String, Object>();
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                map.put(p.getName().getLocalPart(), p.getValue());
            }
            geometry.setUserData(map);
        }
        return geometry;
    }

    /**
     * add pros 2 map  list
     *
     * @param list
     * @param geometry
     * @return
     */
    private List<?> addGeoProperty2List(List<Map<String, Object>> list, Geometry geometry) {
        if (geometry.getUserData() == null) return list;
        for (Map<String, Object> item : list) {
            Map<String, Object> pros = (Map<String, Object>) geometry.getUserData();
            for (Map.Entry entry : pros.entrySet()) {
                item.put("G_".concat((String) entry.getKey()), entry.getValue());
            }
        }
        return list;
    }

    /**
     * @param tpl
     * @return
     */
    private List<Map> getTdlyxzDictList(String tpl) {
        List<Map> groupList = new ArrayList<Map>();
        Configuration configuration = webMapService.getConfig(tpl);
        List<Dict> dicts = configuration.getDicts();
        for (Dict dict : dicts) {
            if (dict.getName().toString().equals("tdlyxz")) {
                List<Item> dictItems = dict.getDictItems();
                for (Item item : dictItems) {
                    Map temp = new HashMap();
                    temp.put("dlbm", item.getName());
                    temp.put("dlmc", item.getTitle());
                    temp.put("area", 0);
//                    temp.put("detail",new ArrayList<Map>());
                    groupList.add(temp);
                }
            }
        }
        if (groupList.size() == 0)
            throw new RuntimeException(getMessage("get.dict.error"));
        return groupList;
    }

    /**
     * check field is in layer
     *
     * @param field
     * @param layerName
     * @return
     */
    private boolean checkFieldInLayer(String field, String layerName, String dataSource) {
        return ArrayUtils.contains(spatialDao.getLayerColumns(layerName, dataSource), field, true);
    }


    private boolean isPolygon(Geometry geometry) {
        return (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName())) ? true : false;
    }


}
