package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.entity.Document;
import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.utils.*;
import com.alibaba.fastjson.JSON;
import com.esri.sde.sdk.client.SeException;
import com.esri.sde.sdk.client.SeLayer;
import com.esri.sde.sdk.client.SeShape;
import com.esri.sde.sdk.pe.PeProjectionException;
import com.vividsolutions.jts.densify.Densifier;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.io.WKTReader;
import com.vividsolutions.jts.operation.valid.IsValidOp;
import com.vividsolutions.jts.operation.valid.TopologyValidationError;
import com.vividsolutions.jts.simplify.DouglasPeuckerSimplifier;
import freemarker.template.TemplateException;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.apache.poi.ss.usermodel.*;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.geotools.data.FeatureSource;
import org.geotools.data.FeatureWriter;
import org.geotools.data.Transaction;
import org.geotools.data.shapefile.ShapefileDataStore;
import org.geotools.data.shapefile.ShapefileDataStoreFactory;
import org.geotools.data.shapefile.ShpFiles;
import org.geotools.data.shapefile.dbf.DbaseFileHeader;
import org.geotools.data.shapefile.dbf.DbaseFileReader;
import org.geotools.data.shapefile.prj.PrjFileReader;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geojson.feature.FeatureJSON;
import org.geotools.geojson.geom.GeometryJSON;
import org.geotools.geometry.jts.Geometries;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.geotools.referencing.CRS;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.crs.GeographicCRS;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import java.io.*;
import java.net.MalformedURLException;
import java.nio.charset.Charset;
import java.util.*;
import java.util.regex.Pattern;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-17 下午2:35
 */
public class GeometryServiceImpl extends BaseLogger implements GeometryService {

    public static final String TYPE = "type";

    public static final String FEATURE = "Feature";
    public static final String FEATURE_COLLECTION = "FeatureCollection";
    public static final String GEOMETRY_COLLECTION = "GeometryCollection";

    private static final String WKID = "wkid";
    private static final String WKT = "wkt";

    private static final String REGION_FIELD = "regionLayers";
    private static final String REGION_MAP = "regionMap";
    private static final String DEFAULT_CRS = "defaultCrs";
    private static final String COORDINATE_DM = "coordinatesDM";
    private static final String UNDEFINE = "UNDEFINE";

    private static final String EPSG_PERFIX = "EPSG:";

    private static final String BJ_FILE_NAME = "gt.xml";

    private static final String DEFAULT_LAYER_REGION_FIELD = "DEFAULTLAYER";

    private static final String TEMP_PIX = "\\TMP_";

    private static final String BJ_TITLE = "title";
    private static final String BJ_FEATURE = "feature";
    private static final String BJ_AREA = "area";
    private static final String BJ_AREA_NYD = "nydArea";
    private static final String BJ_AREA_GD = "gdArea";
    private static final String BJ_AREA_JSYD = "jsydArea";
    private static final String BJ_AREA_WLYD = "wlydArea";

    private static final String XLS_COORD_TAG="@";

    private static final String PROJCS="PROJCS";

    private static final String GEOGCS="GEOGCS";

    private static final String SHP_FILE_SUFFIX="shp";
    private static final String SHX_FILE_SUFFIX="shx";
    private static final String DBF_FILE_SUFFIX="dbf";
    private static final String PRJ_FILE_SUFFIX="prj";

    private static final String DOTS = ".";

    private static final String TPL_FOLDER= "analysis/";
    private static final String TPL_SUFFIX = ".ftl";

    enum BJ_TAG {
        DATA, DATANAME, ROWDATA, ROW, BL_PLOT, PL_ID, PL_NAME, BL_PNT_COORD, PL_PL_ID, X_COORD, Y_COORD,
        SHAPE_GROUP, ID, NAME, INFO,PNT_SERIAL, BL_PROJ_BUILD, PROJ_NAME,SUM_TOT,UNUSED_TOT,FOREST_TOT,FARM_TOT,BUILD_TOT,TILTH_TOT,
        SUM_GROUP,SUM_STATE,SB_ID,SB_NAME,SB_SB_ID,DLBM,JTMJ,GYMJ,BJTYPE,PROJ_TYPE

    }

    private GeometryFactory geometryFactory = JTSFactoryFinder.getGeometryFactory();
    private WKTReader wktReader;

    //    private String regionField;
    private double simplifyTolerance;
    private Map<String, String> regionLayers;
    private Map regionSet;
    private CoordinateReferenceSystem defaultCrs;
    private Map<String, Object> coordinateDM;

    @Autowired
    private DocumentService documentService;
    @Autowired
    private AgsGeoemtryService agsGeometryService;
    @Autowired
    private TemplateService templateService;
    @Autowired
    private GISService gisService;

    /**
     * 读取WKT标准图形
     *
     * @param wkt
     * @return
     */
    @Override
    public Geometry readWKT(String wkt) throws GeometryServiceException {
        try {
            return new WKTReader(geometryFactory).read(wkt);
//            return getWktReader().read(wkt);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.WKT_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析 GeometryJSON  格式数据
     *
     * @param geoJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public Geometry readGeoJSON(String geoJSON) throws GeometryServiceException {
        try {
            GeometryJSON geometryJSON = new GeometryJSON();
            return geometryJSON.read(geoJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析 GeometryCollection
     *
     * @param geoJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public GeometryCollection readGeoCollectionJSON(String geoJSON) throws GeometryServiceException {
        try {
            GeometryJSON geometryJSON = new GeometryJSON();
            return geometryJSON.readGeometryCollection(geoJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }


    /**
     * 解析 FeatureJSON 格式数据
     *
     * @param featureJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public SimpleFeature readFeatureJSON(String featureJSON) throws GeometryServiceException {
        try {
            FeatureJSON fJson = new FeatureJSON();
            return fJson.readFeature(featureJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析 FeatureCollectionJSON
     *
     * @param featureJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public FeatureCollection readFeatureCollectionJSON(String featureJSON) throws GeometryServiceException {
        try {
            FeatureJSON fJson = new FeatureJSON();
            return fJson.readFeatureCollection(featureJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 解析未指明GeoJSON
     *
     * @param geoJSON
     * @return
     * @throws cn.gtmap.onemap.platform.event.GeometryServiceException
     *
     */
    @Override
    public Object readUnTypeGeoJSON(String geoJSON) throws GeometryServiceException {
        try {
            return GeometryUtils.parseGeoJSON(geoJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 读取 FeatureJSON 空间参考信息
     *
     * @param featureJSON
     * @return
     */
    @Override
    public CoordinateReferenceSystem readFeatureJSONCRS(String featureJSON) {
        try {
            FeatureJSON fJson = new FeatureJSON();
            return fJson.readCRS(featureJSON);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 投影转换
     *
     * @param geometry
     * @param sourceCRS .
     * @param targetCRS
     * @return
     */
    @Override
    public Geometry project(Geometry geometry, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) throws GeometryServiceException {
       /* try {
            TopologyValidationError error = null;//validGeometry(geometry);
            if (error != null)
                throw new RuntimeException(getMessage("geometry.not.valid", error.toString()));
            MathTransform transform = CRS.findMathTransform(sourceCRS, targetCRS, true);
            return JTS.transform(geometry, transform);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.PROJECT_EXCEPTION, e.getLocalizedMessage());
        }*/
        return projectByAGS(geometry, sourceCRS, targetCRS);
    }

    /**
     * project
     * @param feature
     * @param sourceCRS
     * @param targetCRS
     * @return
     * @throws GeometryServiceException
     */
    @Override
    public Object project(final Object feature, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) throws GeometryServiceException {
        if (feature instanceof SimpleFeature) {
            Map<String,Object> map =  simpleFeature2Map((SimpleFeature) feature);
            Geometry geometry = project((Geometry)map.get(GEOMETRY), sourceCRS, targetCRS);
            if(!geometry.isSimple()||(validGeometry(geometry)!=null&&geometry.isValid()))
            {
                geometry = gisService.createValidGeometry(geometry);
            }
            SimpleFeatureType featureType = getFeatureType(map);
            SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(featureType);
            String[] keys = map.keySet().toArray(new String[0]);
            for (String key : keys) {
                if (GEOMETRY.equals(key)) featureBuilder.add(geometry);
                else if (FEATURE_CRS.equals(key)) continue;//featureBuilder.add(map.get(FEATURE_CRS))
                else featureBuilder.add(map.get(key) != null ? map.get(key) : "");
            }
            return featureBuilder.buildFeature(FEATURE.concat(UUIDGenerator.generate()));

        }else if(feature instanceof FeatureCollection){
            DefaultFeatureCollection collection = new DefaultFeatureCollection(null, null);
            FeatureCollection featureCollection = (FeatureCollection) feature;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext())
            {
                SimpleFeature temp = (SimpleFeature) featureIterator.next();
                collection.add((SimpleFeature) project(temp, sourceCRS, targetCRS));
            }
            return collection;
        }
        return null;

    }

    /**
     * 投影转换 by arcgis sde
     *
     * @param geometry
     * @param sourceCRS
     * @param targetCRS
     * @return
     */
    @Override
    public Geometry projectByAGS(Geometry geometry, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) {
        try {
            return SRTransformations.project(geometry, sourceCRS.toWKT(), targetCRS.toWKT());
        } catch (PeProjectionException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.PROJECT_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 简化图形
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry simplify(Geometry geometry, double tolerance) {
        return geometry.isValid() && geometry.isSimple() ? geometry : DouglasPeuckerSimplifier.simplify(geometry, tolerance);
    }

    /**
     * force simplify
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry forceSimplify(Geometry geometry, double tolerance) {
        return DouglasPeuckerSimplifier.simplify(geometry, tolerance);
    }

    /**
     * densify
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry densify(Geometry geometry, double tolerance) {
        return Densifier.densify(geometry, tolerance);
    }

    /**
     * 获取 wkt 格式空间参考
     *
     * @param wktCRS such as "GEOGCS[" + "\"WGS 84\"," + "  DATUM[" + "    \"WGS_1984\","
     *               + "    SPHEROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\",\"7030\"]],"
     *               + "    TOWGS84[0,0,0,0,0,0,0]," + "    AUTHORITY[\"EPSG\",\"6326\"]],"
     *               + "  PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],"
     *               + "  UNIT[\"DMSH\",0.0174532925199433,AUTHORITY[\"EPSG\",\"9108\"]],"
     *               + "  AXIS[\"Lat\",NORTH]," + "  AXIS[\"Long\",EAST],"
     *               + "  AUTHORITY[\"EPSG\",\"4326\"]]";
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSByWKTString(String wktCRS) {
        try {
            return CRS.parseWKT(SRTransformations.getCoordinateSystem(wktCRS).toString());
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        } catch (PeProjectionException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 获取常用标准格式空间参考
     *
     * @param crs such as "EPSG:4326" , "urn:ogc:def:ellipsoid:EPSG:6.0:7001" , "AUTO2:42001,"+lat+","+lon
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSByCommnonString(String crs) {
        try {
            return CRS.decode(crs, true);
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * srid
     *
     * @param srid
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSBySRID(String srid) {
        try {
            return getCRSByWKTString(SRTransformations.getCoordinateSystem(Integer.valueOf(srid)).toString());
        } catch (PeProjectionException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * get SeLayer crs
     *
     * @param layer
     * @return
     */
    @Override
    public CoordinateReferenceSystem getSeLayerCRS(SeLayer layer) {
        return getCRSByWKTString(layer.getCoordRef().getCoordSysDescription());
    }

    /**
     *get layer crs by name and ds
     * @param layerName
     * @param dataSource
     * @return
     */
    @Override
    public CoordinateReferenceSystem getLayerCRS(String layerName, String dataSource) {
        try {
            return gisService.getLayerCRS(layerName, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 获取SimpleFeatureType
     *
     * @param value
     * @return
     */
    @Override
    public SimpleFeatureType getFeatureType(final Map<String, Object> value) {
        SimpleFeatureTypeBuilder typeBuilder = new SimpleFeatureTypeBuilder();
        typeBuilder.setName(FEATURE);
        String[] keys = value.keySet().toArray(new String[0]);
        for (String key : keys) {
            if (SHAPE.equals(key))
                typeBuilder.add(GEOMETRY, Geometry.class);
            else if (FEATURE_CRS.equals(key))
                typeBuilder.add(FEATURE_CRS, String.class, (CoordinateReferenceSystem) value.get(key));
            else
                typeBuilder.add(key, value.get(key) != null ? value.get(key).getClass() : String.class);
        }
        return typeBuilder.buildFeatureType();
    }

    /**
     * Map value to SimpleFeature
     *
     * @param value   {"SHAPE":"WKT FORMATE","PRO1":"VALUE"}
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public SimpleFeature map2SimpleFeature(final Map<String, Object> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        Geometry geometry = value.containsKey(SHAPE) ? readWKT((String) value.get(SHAPE)) : null;
        if (geometry != null && srcCRS != null && destCRS != null && srcCRS != destCRS)
            geometry = project(geometry, srcCRS, destCRS);
        SimpleFeatureType featureType = getFeatureType(value);
        SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(featureType);
        String[] keys = value.keySet().toArray(new String[0]);
        for (String key : keys) {
            if (SHAPE.equals(key)) featureBuilder.add(geometry);
            else if (FEATURE_CRS.equals(key)) continue;//featureBuilder.add(value.get(FEATURE_CRS));
            else featureBuilder.add(value.get(key)!=null?value.get(key):"");
        }
        return featureBuilder.buildFeature(FEATURE.concat(UUIDGenerator.generate()));
    }

    /**
     * list values to featureCollection
     *
     * @param value
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public FeatureCollection list2FeatureCollection(final List<Map<String, Object>> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        DefaultFeatureCollection collection = new DefaultFeatureCollection(null, null);
        if (value != null && value.size() > 0) {
            for (Map item : value) {
                try {
                    collection.add(map2SimpleFeature(item, srcCRS, destCRS));
                } catch (Exception e) {
                    logger.info(getMessage("list.2featureCollection.false", e.getLocalizedMessage()));
                }

            }
        }
        return collection;
    }

    /**
     * to featureJSON
     *
     * @param feature
     * @return
     */
    @Override
    public String toFeatureJSON(final Object feature) {
        try {
            FeatureJSON featureJSON = new FeatureJSON(new GeometryJSON(14));
            StringWriter out = new StringWriter();
            if (feature instanceof SimpleFeature) {
                featureJSON.setEncodeFeatureBounds(((SimpleFeature) feature).getBounds() == null ? false : true);
                featureJSON.setEncodeFeatureCRS(((SimpleFeature) feature).getFeatureType().
                        getCoordinateReferenceSystem() == null ? false : true);
                featureJSON.writeFeature((SimpleFeature) feature, out);
            } else if (feature instanceof FeatureCollection) {
                if (((FeatureCollection) feature).size() > 0) {
                    featureJSON.setEncodeFeatureCollectionBounds(((SimpleFeature) ((FeatureCollection) feature).toArray()[0]).getBounds() == null ? false : true);
                    featureJSON.setEncodeFeatureCollectionCRS(((SimpleFeature) ((FeatureCollection) feature).toArray()[0]).getFeatureType().
                            getCoordinateReferenceSystem() == null ? false : true);
                }
                featureJSON.writeFeatureCollection((FeatureCollection) feature, out);
            }
            return out.toString();
        } catch (Exception e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.FEATURE_TO_JSON_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * to geoJSON
     *
     * @param geometry
     * @return
     */
    @Override
    public String toGeoJSON(Geometry geometry) {
        try {
            GeometryJSON geometryJSON = new GeometryJSON(14);
            StringWriter out = new StringWriter();
            geometryJSON.write(geometry, out);
            return out.toString();
        } catch (IOException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOMETRY_TO_JSON_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * @param feature
     * @return
     */
    @Override
    public Map<String, Object> simpleFeature2Map(SimpleFeature feature) {
        Assert.notNull(feature, "feature can't be null");
        Map<String, Object> result = new HashMap<String, Object>();
        for (Property property : feature.getProperties()) {
            if (property.getValue() != null && StringUtils.isNotBlank(String.valueOf(property.getValue())))
                result.put(property.getName().getLocalPart(), property.getValue());
        }
        return result;
    }

    /**
     * @param value wkid or wkt
     * @return
     */
    @Override
    public CoordinateReferenceSystem parseUndefineSR(String value) {
        try {
            int srid = Integer.parseInt(value);
            if (srid != 0) return getCRSBySRID(value);
        } catch (NumberFormatException e) {
            if (value.indexOf(EPSG) == 0)
                return getCRSBySRID(value.substring(EPSG.length() + 1, value.length()));//return getCRSByCommnonString(value);
            else return getCRSByWKTString(value);
        }
        throw new GeometryServiceException(GeometryServiceException.ExceptionType.CRS_PARSE_EXCEPTION, value);
    }

    /**
     * 验证图形拓扑
     *
     * @param geometry
     * @return
     */
    @Override
    public TopologyValidationError validGeometry(Geometry geometry) {
        IsValidOp isValidOp = new IsValidOp(geometry);
        return isValidOp.getValidationError();
    }

    /**
     * get crs by reset regionCode sets
     *
     * @param regionCode
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCRSByRegionCode(String regionCode) {
        if (StringUtils.isBlank(regionCode)) {
            logger.warn(getMessage("region.code.not.found", regionCode));
            if (regionSet.containsKey("default"))
                return parseUndefineSR(String.valueOf(regionSet.get("default")));
            else
                return parseUndefineSR("2364");
        }
//        throw new RuntimeException(getMessage("region.code.not.null")); //modified by yxf 为null时 不抛出异常 仍然取默认值
        if (!regionSet.containsKey(regionCode)) {
            logger.warn(getMessage("region.code.not.found", regionCode));
            if (regionSet.containsKey("default"))
                return parseUndefineSR(String.valueOf(regionSet.get("default")));
            else
                return parseUndefineSR("2364");
        }
        return parseUndefineSR(String.valueOf(regionSet.get(regionCode)));
    }

    /**
     * get region key field
     *
     * @return
     */
    @Override
    public LayerRegion getLayerRegion(String layerName) {
        for (Map.Entry entry : this.regionLayers.entrySet()) {
            if (Pattern.compile((String) entry.getKey()).matcher(layerName.toUpperCase()).matches()) {
                String value = (String) entry.getValue();
                try {
                    return new LayerRegion(layerName, parseUndefineSR(value));
                } catch (Exception e) {
                    return new LayerRegion(layerName, value);
                }
            }
        }
        if (this.regionLayers.containsKey(DEFAULT_LAYER_REGION_FIELD)) {
            String defaultField = this.regionLayers.get(DEFAULT_LAYER_REGION_FIELD);
            if (defaultField.startsWith("EPSG:")) {
                logger.info(getMessage("layer.regioncode.crs.default", layerName, defaultField));
                return new LayerRegion(layerName, parseUndefineSR(defaultField));
            } else {
                logger.info(getMessage("layer.regioncode.default", layerName, defaultField));
                return new LayerRegion(layerName, defaultField);
            }
        }
        throw new RuntimeException(getMessage("layer.regioncode.not.set", layerName));
    }

    /**
     * 行政区字典项配置中是否包含该行政区代码
     *
     * @param regionCode
     * @return
     */
    @Override
    public boolean containsRegionValue(String regionCode) {
        if (regionSet.containsKey(regionCode)) return true;
        return false;
    }

    /**
     * 获取geojson的面积 仅仅是获取面积 未做投影
     * 对geocrs的图形此方法无效
     * @param geoJSON
     * @return
     */
    @Override
    public double readGeometryAera(String geoJSON) {
        try {
            Object geo = readUnTypeGeoJSON(geoJSON);
            if (geo instanceof Geometry) return ((Geometry) geo).getArea();
            if (geo instanceof SimpleFeature) return ((Geometry) ((SimpleFeature) geo).getDefaultGeometry()).getArea();
            if(geo instanceof FeatureCollection) {
                double area=0;
                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) iterator.next();
                    Geometry geometry = (Geometry) feature.getDefaultGeometry();
                    area +=geometry.getArea();
                }
                return area;
            }
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
        }
        return 0;
    }

    /***
     * get geometry of geo do project first
     * @param geo
     * @return
     */
    @Override
    public double getGeoArea(Object geo,CoordinateReferenceSystem sourceCrs) {
        double area = 0;
        try {
            SimpleFeature feature = null;
            CoordinateReferenceSystem targetCrs = parseUndefineSR("2364");
            if (geo instanceof SimpleFeature) {
                feature = (SimpleFeature) geo;
                if (isNull(sourceCrs))
                    sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (sourceCrs != null && sourceCrs instanceof GeographicCRS) {
                    geometry = project((Geometry) feature.getDefaultGeometry(), sourceCrs, targetCrs);
                }
                if (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName()))
                    area = agsGeometryService.getGeometryArea(geometry.toText());//   area = geometry.getArea();
                else
                    area = geometry.getLength();
            } else if (geo instanceof FeatureCollection) {
                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    feature = (SimpleFeature) iterator.next();
                    area += getGeoArea(feature, sourceCrs);
                }
            } else if (geo instanceof Geometry) {
                Geometry temp = (Geometry) geo;
                Geometry geometry;
                if (isNull(sourceCrs))
                    sourceCrs = getCrsByCoordXD(temp.getCentroid().getX());
                if (sourceCrs != null && sourceCrs instanceof GeographicCRS) {
                    geometry = project(temp, sourceCrs, targetCrs);
                } else
                    geometry = temp;
                if (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName()))
                    area = agsGeometryService.getGeometryArea(geometry.toText());//   area = geometry.getArea();
                else
                    area = geometry.getLength();
            } else {
                if (geo instanceof GeometryCollection) {
                    GeometryCollection geometryCollection = (GeometryCollection) geo;
                    for (int i = 0; i < geometryCollection.getNumGeometries(); i++) {
                        Geometry geometry = geometryCollection.getGeometryN(i);
                        if (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName()))
                            area += getGeoArea(geometry, sourceCrs);
                    }
                }
            }
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area;
    }

    /**
     *获取geometry图形的中心
     * @param geometry
     * @since v2.1.0
     * @return
     */
    @Override
    public Point getGeometryCentre(Geometry geometry) {
        assert geometry!=null;
        if (geometry instanceof Point)
            return (Point) geometry;
        else
            return geometry.getCentroid();
    }

    /**
     * 获取简化精度
     *
     * @return
     */
    @Override
    public double getSimplifyTolerance() {
        return this.simplifyTolerance;
    }

    /***
     * get zip coordinates
     * @param in
     * @return
     * @throws Exception
     */
    @Override
    public Object getZipCoordinates(InputStream in) throws Exception {
        try {
            int tag = 0;
            Map result = new HashMap();
            List<Document> documents = documentService.readZipIn(in);
            if (documents.size() == 0) throw new RuntimeException(getMessage("zip.format.error"));
            for (Document document : documents) {
                if (document.getFileName().equals(BJ_FILE_NAME)) {
                    tag = 1;
                    break;
                }
            }
            switch (tag) {
                case 0:
                    result.put("type","shp");
                    result.put("value",getShpCoordinates(documents));
                    break;
                case 1:
                    result.put("type","bj");
                    result.put("value",getBJCoordinates(documents));
                    break;
            }
            return result;
        } catch (Exception e) {
            throw new RuntimeException(getMessage("zip.coord.get.error", e.getLocalizedMessage()));
        }
    }

    /***
     * 解析shp的zip包
     * @param documents
     * @return
     * @throws Exception
     */
    private String getShpCoordinates(List<Document> documents) throws Exception {
        assert documents != null;
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\" + document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(), output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile = null;
        File dbfFile = null;
        File prjFile = null;
        String result;
        try {
            if (files.length > 0) {
                for (File tempFile : files) {
                    if (tempFile.isFile()) {
                        if (tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                            shpFile = tempFile;
                        if (tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                            dbfFile = tempFile;
                        if (tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                            prjFile = tempFile;
                    }
                }
            }
            result = parseShapefile(shpFile, dbfFile, prjFile, null);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return result;
    }

    /**
     * 解析电子报件包
     * @param documents
     * @return
     * @throws Exception
     */
    private Map getBJCoordinates(List<Document> documents) throws Exception {
        assert documents != null;
        for (Document document : documents) {
            if (document.getFileName().equals(BJ_FILE_NAME)) {
                return parseBjXmlAdv(document.getContent());
            }
        }
        throw new RuntimeException(getMessage("bj.gt.xml.not.found"));
    }

    /**
     * 获取电子报件地块坐标
     *
     * @param in
     * @return FeatureJSON
     */
    @Override
    public Map getBJCoordinates(InputStream in) throws Exception {
        List<Document> documents = documentService.readZip(in);
        if (documents.size() == 0) {
            File tmp = new File(System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis()));
            try {
                FileOutputStream output = new FileOutputStream(tmp);
                try {
                    IOUtils.copyLarge(in, output, 0, in.available(), new byte[in.available()]);
                    output.close();
                } finally {
                    IOUtils.closeQuietly(output);
                }
                documents = documentService.readZipFile(tmp);
                if (documents.size() == 0)
                    throw new RuntimeException(getMessage("bj.zip.format.error"));
            } finally {
                FileUtils.deleteQuietly(tmp);
            }
        }
        for (Document document : documents) {
            if (document.getFileName().equals(BJ_FILE_NAME)) {
                return parseBjXmlAdv(document.getContent());
            }
        }
        throw new RuntimeException(getMessage("bj.gt.xml.not.found"));
    }

    /**
     * 获取excel坐标文件地块坐标(geoJson)
     * @param in
     * @return
     * @throws Exception
     */
    @Override
    public String getExcelCoordinates(InputStream in) throws Exception {
        try {
            Workbook workbook = WorkbookFactory.create(in);
            Sheet sheet = workbook.getSheetAt(0);
            List<Map<String, Object>> features = new ArrayList<Map<String, Object>>();
            CoordinateReferenceSystem sourceCrs = null;
            for (Row row : sheet) {
                String info = row.getCell(0).getStringCellValue();
                logger.debug("excel upload row info:"+info);
                if(row.getCell(7)!=null&&row.getCell(7).getCellType()==Cell.CELL_TYPE_STRING)
                {
                    String crs = row.getCell(7).getStringCellValue();
                    if(crs.startsWith(PROJCS)||crs.startsWith(GEOGCS))
                    {
                        sourceCrs = getCRSByWKTString(crs);
                    }
                }
                if(info.endsWith(XLS_COORD_TAG))
                {
                   features = excel2List(row.getRowNum() + 1, sheet,sourceCrs);
                   break;
                }
            }
            if(features.size()>0)
            {
                logger.debug("excel upload features number:"+features.size());
                for (Map feature:features){
                    if (sourceCrs == null) sourceCrs = (CoordinateReferenceSystem) feature.get(FEATURE_CRS);
                    feature.remove(FEATURE_CRS);
                }
            }else
            {
                logger.error(getMessage("excel.coords.not.found"));
                throw new RuntimeException(getMessage("excel.coords.not.found"));
            }
            return toFeatureJSON(list2FeatureCollection(features, sourceCrs, defaultCrs));
        } catch (Exception e) {
            logger.error(getMessage("excel.parse.error", e.getLocalizedMessage()));
            throw new RuntimeException(getMessage("excel.parse.error", e.getLocalizedMessage()));
        }
    }

    /**
     * 获取shapefile文件坐标
     * @param in
     * @return
     * @throws Exception
     */
    @Override
    public String getShpCoordinates(InputStream in) throws Exception {
        List<Document> documents = documentService.readZip(in);
        if (documents.size() == 0) {
            File tmp = new File(System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis()));
            try {
                FileOutputStream output = new FileOutputStream(tmp);
                try {
                    IOUtils.copyLarge(in, output, 0, in.available(), new byte[in.available()]);
                    output.close();
                } finally {
                    IOUtils.closeQuietly(output);
                }
                documents = documentService.readZipFile(tmp);
                if (documents.size() == 0)
                    throw new RuntimeException(getMessage("bj.zip.format.error"));
            } finally {
                FileUtils.deleteQuietly(tmp);
            }
        }
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\"+document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(),output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile=null;
        File dbfFile=null;
        File prjFile=null;
        String result;
        try {
            if(files.length>0){
                for (File tempFile:files){
                   if(tempFile.isFile()){
                       if(tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                           shpFile = tempFile;
                       if(tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                           dbfFile = tempFile;
                       if(tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                           prjFile = tempFile;
                   }
                }
            }
             result = parseShapefile(shpFile,dbfFile,prjFile,null);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return result;
    }

    /**
     *
     * @param file (zip包)
     * @return
     * @throws Exception
     */
    @Override
    public String getShpCoordinates(File file) throws Exception {
        List<Document> documents = documentService.readZipFile(file);
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\"+document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(),output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile=null;
        File dbfFile=null;
        File prjFile=null;
        String result;
        try {
            if(files.length>0){
                for (File tempFile:files){
                    if(tempFile.isFile()){
                        if(tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                            shpFile = tempFile;
                        if(tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                            dbfFile = tempFile;
                        if(tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                            prjFile = tempFile;
                    }
                }
            }
            result = parseShapefile(shpFile,dbfFile,prjFile,null);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return result;
    }

    /**
     * 获取shapefile文件坐标以及属性(zip包文件)，允许在遇到topo错误后跳过异常继续解析
     * @param file
     * @param tolerateExp
     * true,遇到topo错误后跳过异常继续解析
     * false,遇到topo错误后直接抛出异常，效果与  String getShpCoordinates(File file, String properties) 相同
     * @return
     * @return
     * @throws Exception
     * @since v2.1.5
     */
    @Override
    public String getShpCoordinates(File file, boolean tolerateExp) throws  Exception{
        List<Document> documents = documentService.readZipFile(file);
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\"+document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(),output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile=null;
        File dbfFile=null;
        File prjFile=null;
        String result;
        try {
            if(files.length>0){
                for (File tempFile:files){
                    if(tempFile.isFile()){
                        if(tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                            shpFile = tempFile;
                        if(tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                            dbfFile = tempFile;
                        if(tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                            prjFile = tempFile;
                    }
                }
            }
            result = parseShapefile(shpFile,dbfFile,prjFile,null,tolerateExp);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return result;
    }

    /**
     * 带有自定义属性参数
     * @param file (zip包)
     * @param properties 属性字符串(json格式)
     * @return
     * @throws Exception
     */
    @Override
    public String getShpCoordinates(File file, String properties) throws Exception {
        List<Document> documents = documentService.readZipFile(file);
        Map propertyMap = JSON.parseObject(properties,Map.class);
        if(isNull(propertyMap))
        {
            logger.info("properties is null");
            return getShpCoordinates(file);
        }
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\"+document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(),output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile=null;
        File dbfFile=null;
        File prjFile=null;
        String result;
        try {
            if(files.length>0){
                for (File tempFile:files){
                    if(tempFile.isFile()){
                        if(tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                            shpFile = tempFile;
                        if(tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                            dbfFile = tempFile;
                        if(tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                            prjFile = tempFile;
                    }
                }
            }
            result = parseShapefile(shpFile, dbfFile, prjFile, propertyMap);
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return result;
    }

    /**
     * 带有自定义属性参数，允许在遇到topo错误后跳过异常继续解析
     * @param file
     * @param properties
     * @param tolerateExp
     * true,遇到topo错误后跳过异常继续解析
     * false,遇到topo错误后直接抛出异常，效果与  String getShpCoordinates(File file, String properties) 相同
     * @return
     * @since v2.1.5
     */
    @Override
    public String getShpCoordinates(File file, String properties, boolean tolerateExp) throws Exception {
        List<Document> documents = documentService.readZipFile(file);
        Map propertyMap = JSON.parseObject(properties,Map.class);
        if(isNull(propertyMap))
        {
            logger.info("properties is null");
            return getShpCoordinates(file);
        }
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\"+document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(),output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile=null;
        File dbfFile=null;
        File prjFile=null;
        String result;
        try {
            if(files.length>0){
                for (File tempFile:files){
                    if(tempFile.isFile()){
                        if(tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                            shpFile = tempFile;
                        if(tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                            dbfFile = tempFile;
                        if(tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                            prjFile = tempFile;
                    }
                }
            }
            result = parseShapefile(shpFile, dbfFile, prjFile, propertyMap, tolerateExp);
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return result;
    }

    /**
     *
     * @param in  .shp文件流
     * @return
     * @throws Exception
     */
    @Override
    public String getShpCoordinatesSingle(InputStream in) throws Exception {

        File shpFile = new File(System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis()).concat(DOTS.concat(Document.Type.shp.name())));
        try {
            FileOutputStream output = new FileOutputStream(shpFile);
            try {
                IOUtils.copyLarge(in, output, 0, in.available(), new byte[in.available()]);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
            try {
                if (!isNull(shpFile))
                    return parseShapefile(shpFile, null,null, null);
            } catch (Exception e) {
                throw new RuntimeException(e.getLocalizedMessage());
            }
        } finally {
            FileUtils.deleteQuietly(shpFile);
        }
        return null;
    }

    /**
      
     * @param file
     * @param properties
     * @param layerName
     * @param dataSource
     * @param tolerateExp
     * @return {"success":boolean,result:JSONString}
     */
    @Override
    public Map insertShpCoordinates(File file, String properties, String layerName, String dataSource, boolean tolerateExp) {
        String result = "";
        List rows = null;
        long date = System.currentTimeMillis();
        try {
            rows = getShpContent(file, properties, tolerateExp);
        } catch (Exception e) {
            Map map = new HashMap(){
                {
                    put("success", false);
                }
            };
            map.put("result", e.getLocalizedMessage());
            return map;
        }
        if(!isNull(rows)){
           if(gisService.insertRows(layerName, rows, dataSource)>0){
               Map map = new HashMap(){
                   {
                       put("success", true);
                   }
               };
               for(Object row : rows){
                   Map<String, Object> cols = (Map<String, Object>)row;
                   try {
                       ((Map<String, Object>) row).put("SHAPE_AREA", ((SeShape) cols.get("SHAPE")).getArea());
                   } catch (SeException e) {
                       ((Map<String, Object>) row).put("SHAPE_AREA", 0);
                   }
                   if(((Map<String, Object>) row).containsKey("SHAPE_LENG")){
                       ((Map<String, Object>) row).remove("SHAPE_LENG");
                   }
                   ((Map<String, Object>) row).remove("SHAPE");
               }
               map.put("result", JSON.toJSONString(rows));
               logger.info("插入成功，耗时："+(System.currentTimeMillis()-date)+"ms");
               return map;
           }
        }
        return null;
    }

    public List getShpContent(File file, String properties, boolean tolerateExp) throws Exception {
        List<Document> documents = documentService.readZipFile(file);
        if(isNull(properties))
            properties = "{}";
        Map propertyMap = JSON.parseObject(properties,Map.class);
        String filePath = System.getProperty("java.io.tmpdir").concat(TEMP_PIX + System.currentTimeMillis());
        File folderFile = new File(filePath);
        if (!folderFile.exists()) {
            folderFile.mkdir();
        }
        List rows = null;
        for (Document document : documents) {
            File tmp = new File(filePath.concat("\\"+document.getName().concat(DOTS.concat(document.getType().name()))));
            FileOutputStream output = new FileOutputStream(tmp);
            try {
                IOUtils.write(document.getContent(),output);
                output.close();
            } finally {
                IOUtils.closeQuietly(output);
            }
        }
        File[] files = folderFile.listFiles();
        File shpFile=null;
        File dbfFile=null;
        File prjFile=null;
        String result = "";
        try {
            if(files.length>0){
                for (File tempFile:files){
                    if(tempFile.isFile()){
                        if(tempFile.getName().endsWith(SHP_FILE_SUFFIX))
                            shpFile = tempFile;
                        if(tempFile.getName().endsWith(DBF_FILE_SUFFIX))
                            dbfFile = tempFile;
                        if(tempFile.getName().endsWith(PRJ_FILE_SUFFIX))
                            prjFile = tempFile;
                    }
                }
            }
            rows = parseShapefile2MapList(shpFile, dbfFile, prjFile, propertyMap, tolerateExp);

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        } finally {
            folderFile.delete();
        }
        return rows;
    }

    /**
    *  解析geojson 导出成shp
     * @param geoJson GeoJSON格式  eg.{"type":"Feature","crs":{"type":"name","properties":{"name":"EPSG:4610"}},"geometry":{"type":"Polygon","coordinates":[[[118.65,32.83],[118.65,32.82],[118.64,32.83],[118.65,32.83]]]},"properties":{"PRONAME":"xxx"}}
     * @return
     * @throws Exception
     */
    @Override
    public File exportToShp(String geoJson) throws Exception {
        Assert.notNull(geoJson,getMessage("shp.export.geo.null"));
        List<Map<String,Object>> records=new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem sourceCrs=null;
        Map<String, Object> record;
        Object geo = readUnTypeGeoJSON(geoJson);
        if (geo instanceof FeatureCollection) {
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                record = new HashMap<String, Object>();
                for (Property property : feature.getProperties()) {
                    if (!property.getName().getLocalPart().equalsIgnoreCase(FEATURE_CRS))
                        record.put(property.getName().getLocalPart(), property.getValue());
                }
                records.add(record);
                if(isNull(sourceCrs))
                    sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                if(isNull(sourceCrs))
                    sourceCrs = readFeatureJSONCRS(toFeatureJSON(feature));
            }
        } else if (geo instanceof SimpleFeature) {
            record = new HashMap<String, Object>();
            for (Property property : ((SimpleFeature) geo).getProperties()) {
                if (!property.getName().getLocalPart().equalsIgnoreCase(FEATURE_CRS))
                    record.put(property.getName().getLocalPart(), property.getValue());
            }
            records.add(record);
            if(isNull(sourceCrs))
                sourceCrs = ((SimpleFeature) geo).getFeatureType().getCoordinateReferenceSystem();
        }
        if (records.size() == 0) throw new RuntimeException(getMessage("shp.export.records.null"));
        if (isNull(sourceCrs))
            logger.warn(getMessage("shp.export.crs.null", defaultCrs == null ? null : defaultCrs.toWKT()));
        try {
            String folderPath = System.getProperty("java.io.tmpdir").concat("\\SHP_" + System.currentTimeMillis());
            return createShpZip(folderPath, records, sourceCrs);
        }catch (Exception ex)
        {
            throw new RuntimeException(getMessage("shp.export.error",ex.getLocalizedMessage()));
        }
    }

    /**
     * 解析geojson 导出成shp
     * @param geoJson GeoJSON格式  eg.{"type":"Feature","crs":{"type":"name","properties":{"name":"EPSG:4610"}},"geometry":{"type":"Polygon","coordinates":[[[118.65,32.83],[118.65,32.82],[118.64,32.83],[118.65,32.83]]]},"properties":{"PRONAME":"xxx"}}
     * @param crs 导出shp的空间参考
     * @return
     * @throws Exception
     */
    @Override
    public File exportToShp(String geoJson, CoordinateReferenceSystem crs) throws Exception {
        Assert.notNull(geoJson,getMessage("shp.export.geo.null"));
        List<Map<String,Object>> records=new ArrayList<Map<String, Object>>();
        Map<String, Object> record;
        Object geo = readUnTypeGeoJSON(geoJson);
        if (geo instanceof FeatureCollection) {
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                record = new HashMap<String, Object>();
                for (Property property : feature.getProperties()) {
                    if (!property.getName().getLocalPart().equalsIgnoreCase(FEATURE_CRS))
                        record.put(property.getName().getLocalPart(), property.getValue());
                }
                records.add(record);
            }
        } else if (geo instanceof SimpleFeature) {
            record = new HashMap<String, Object>();
            for (Property property : ((SimpleFeature) geo).getProperties()) {
                if (!property.getName().getLocalPart().equalsIgnoreCase(FEATURE_CRS))
                    record.put(property.getName().getLocalPart(), property.getValue());
            }
            records.add(record);
        }
        if(records.size()==0)throw new RuntimeException(getMessage("shp.export.records.null"));
        try {
            String folderPath = System.getProperty("java.io.tmpdir").concat("\\SHP_" + System.currentTimeMillis());
            return createShpZip(folderPath, records, crs);
        }catch (Exception ex)
        {
            throw new RuntimeException(getMessage("shp.export.error",ex.getLocalizedMessage()));
        }
    }

    /**
     * 根据一定条件 去特定图层中查找符合条件的记录 然后导出这些记录
     * @param layerName   SDE空间图层名
     * @param where       查询条件
     * @param geometry    查询的空间要素，为GeoJSON格式
     * @param outFields   返回字段，默认null
     * @param dataSource  sde数据源
     * @return
     * @throws Exception
     */
    @Override
    public File exportToShp(String layerName, String where, String geometry, String[] outFields, String dataSource) throws Exception {
        Assert.notNull(layerName,getMessage("shp.export.layerName.null"));
        CoordinateReferenceSystem sourceCrs = getLayerCRS(layerName, dataSource);
        List queryResult = new ArrayList();
        if (StringUtils.isNotBlank(where))
            queryResult = gisService.query(layerName, where, outFields, true, dataSource);
        else if (StringUtils.isNotBlank(geometry)) {
            Object geo = readUnTypeGeoJSON(geometry);
            if (geo instanceof FeatureCollection) {
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    queryResult.addAll(gisService.query(layerName, (Geometry) feature.getDefaultGeometry(), outFields, dataSource));
                }
            } else if (geo instanceof SimpleFeature) {
                queryResult.addAll(gisService.query(layerName, (SimpleFeature) geo, outFields, dataSource));
            }
        }
        else
            throw new RuntimeException(getMessage("query.condition.missing"));
        if(queryResult.size()==0)throw new RuntimeException(getMessage("shp.export.query.null"));
        if(isNull(sourceCrs))throw new RuntimeException(getMessage("shp.export.crs.null"));
        try {
            String folderPath=System.getProperty("java.io.tmpdir").concat("\\SHP_" + System.currentTimeMillis());
            return createShpZip(folderPath,queryResult,sourceCrs);
        }catch (Exception ex)
        {
            throw new RuntimeException(getMessage("shp.export.error",ex.getLocalizedMessage()));
        }
    }

    /**
     * 转换shp的zip文件到 dwg文件
     * @param zipFile    zip file 的在线地址
     * @param gpUrl      处理zip的 ArcGIS GP服务地址
     * @since            v2.0.14
     * @return           返回dwg文件的在线下载地址
     */
    @Override
    public String convertShpToDwg(String zipFile, String gpUrl) {
        assert zipFile!=null;
        assert gpUrl!=null;
        logger.debug("[zipfile:]"+zipFile);
        logger.debug("[gpUrl]:"+gpUrl);
        if (!gpUrl.endsWith("/execute"))
            gpUrl = gpUrl.concat("/execute");
        HttpClient httpclient = new DefaultHttpClient();
        HttpPost httpRequest = new HttpPost(gpUrl);
        List<BasicNameValuePair> params = new ArrayList<BasicNameValuePair>();
        params.add(new BasicNameValuePair("input_shape_file", zipFile));
        params.add(new BasicNameValuePair("f", "json"));
        HttpEntity httpentity;
        try {
            httpentity = new UrlEncodedFormEntity(params, "UTF-8");
            httpRequest.setEntity(httpentity);
            HttpResponse httpResponse = httpclient.execute(httpRequest);
            if (httpResponse.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                String responseBody = EntityUtils.toString(httpResponse.getEntity());
                logger.debug("[调用gp服务返回结果:]" + responseBody);
                Map map = JSON.parseObject(responseBody, Map.class);
                List<Map> results = (List<Map>) map.get("results");
                Map outputMap = (Map) results.get(0).get("value");
                return MapUtils.getString(outputMap,"url");
            } else {
                logger.error("调用gp服务异常");
                return null;
            }
        }catch (ClientProtocolException e) {
            throw new RuntimeException(e.getMessage());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e.getMessage());
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e.getMessage());
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * get crs by coords x
     *
     * @param x
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCrsByCoordXD(double x) {
        String s = String.valueOf(Math.round(x));
        switch (s.length()) {
            case 8: {
                String p = s.substring(0, 2);
                if (coordinateDM.containsKey(p)) return (CoordinateReferenceSystem) coordinateDM.get(p);
                else throw new RuntimeException(getMessage("bj.coords.dm.not.set", p));
            }

            case 6: {
                // nanjing
                return (CoordinateReferenceSystem) coordinateDM.get(UNDEFINE);
            }
            case 5: {
                // suzhou wuzhong
                if (coordinateDM.containsKey("SUZHOU_WZ"))
                    return (CoordinateReferenceSystem) coordinateDM.get("SUZHOU_WZ");
                else
                    throw new RuntimeException(getMessage("bj.coords.type.not.support", x));
            }
            default:
                logger.error(getMessage("bj.coords.type.not.support", x));
                throw new RuntimeException(getMessage("bj.coords.type.not.support", x));
        }
    }

    /**
     * render analysis result
     * @param map
     * @param type
     * @return
     */
    @Override
    public String renderAnalysisContent(Map map, String type) {
        try {
            return templateService.getTemplate(map,TPL_FOLDER.concat(type.concat(TPL_SUFFIX)),null);
        } catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException(e.getLocalizedMessage());
        } catch (TemplateException e) {
            e.printStackTrace();
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 生成shapefile的zip包文件
     * @param folderPath
     * @param values
     * @param crs
     * @return zip包路径
     */
    public File createShpZip(String folderPath, List<Map<String, Object>> values, CoordinateReferenceSystem crs) throws IOException {
        try {
            File folderFile = new File(folderPath);
            if (!folderFile.exists())
                folderFile.mkdir();
            String filePath = folderFile.getPath().concat("\\" + folderFile.getName().concat(DOTS.concat(Document.Type.shp.name())));

            File shpFile = new File(filePath);
            if(!shpFile.exists())
                shpFile.createNewFile();
            Map<String, Serializable> params = new HashMap<String, Serializable>();
            params.put(ShapefileDataStoreFactory.URLP.key, shpFile.toURI().toURL());
            ShapefileDataStore ds = (ShapefileDataStore) new ShapefileDataStoreFactory().createNewDataStore(params);

            Map<String, Object> value = values.get(0);
            SimpleFeatureTypeBuilder typeBuilder = new SimpleFeatureTypeBuilder();
            typeBuilder.setName(SHAPEFILE);
            String[] keys = value.keySet().toArray(new String[0]);
            for (String key : keys) {
                if (SHAPE.equals(key) || GEOMETRY.equals(key)) {
                    Object geo = value.get(key);
                    if (geo instanceof String)
                        geo = readWKT(MapUtils.getString(value, key));
                    if (geo instanceof Point)
                        typeBuilder.add("the_geom", Point.class);
                    else if (geo instanceof LineString)
                        typeBuilder.add("the_geom", LineString.class);
                    else if (geo instanceof Polygon)
                        typeBuilder.add("the_geom", Polygon.class);
                    else if (geo instanceof MultiPolygon)
                        typeBuilder.add("the_geom", MultiPolygon.class);
                    else if (geo instanceof MultiLineString)
                        typeBuilder.add("the_geom", MultiLineString.class);
                }
                else {
                    if (key.length() > 10) {
                        String keyN = key.substring(0, 10);
                        logger.warn("字段名 『" + key
                                + "』长度超过10个字符, 将被截取为 『"
                                + keyN + "』");
                        typeBuilder.add(keyN, value.get(key) != null ? value.get(key).getClass() : String.class);
                    } else
                        typeBuilder.add(key, value.get(key) != null ? value.get(key).getClass() : String.class);
                }
            }
            if (!isNull(crs))
                typeBuilder.setCRS(crs);
            else
                typeBuilder.setCRS(defaultCrs==null?DefaultGeographicCRS.WGS84:defaultCrs);

            ds.createSchema(typeBuilder.buildFeatureType());
            ds.forceSchemaCRS(typeBuilder.getCRS());
            ds.setStringCharset(Charset.forName("GBK"));

            logger.debug("[导出shp空间参考]:"+typeBuilder.getCRS().toWKT());
            FeatureWriter<SimpleFeatureType, SimpleFeature> writer = ds.getFeatureWriter(ds.getTypeNames()[0], Transaction.AUTO_COMMIT);
            SimpleFeature feature=null;
            for (Map<String, Object> item : values) {
                feature = writer.next();
                Geometry geometry = item.containsKey(SHAPE) ? readWKT((String) item.get(SHAPE)) : null;
                if (isNull(geometry)) {
                    Object geo = item.get(GEOMETRY);
                    if (geo instanceof Geometry)
                        geometry = (Geometry) geo;
                    else if (geo instanceof String)
                        geometry = item.containsKey(GEOMETRY) ? readWKT((String) item.get(GEOMETRY)) : null;
                }
                if (isNull(geometry)) continue;
                feature.setAttribute("the_geom", geometryFactory.createGeometry(geometry));
                for (Map.Entry entry : item.entrySet()) {
                    String key = String.valueOf(entry.getKey());
                    Object val = entry.getValue();
                    String _k;
                    if (key.equals(SHAPE) || key.equals(FEATURE_CRS) || key.equals(GEOMETRY)) continue;
                    if (key.length() > 10)
                        _k = key.substring(0, 10);
                    else
                        _k = key;
                    if(Utils.isContainChinese(_k))continue;  //中文属性不支持导出
                    if (val instanceof String)
                        feature.setAttribute(_k, String.valueOf(val));
                    else if(val instanceof Double)
                        feature.setAttribute(_k,MapUtils.getDoubleValue(item, key));
                    else
                        feature.setAttribute(_k,val);
                }
            }
            writer.write();
            writer.close();
            ds.dispose();
            File zipFile = ZipUtils.doZip(folderPath, null);
            if (zipFile.exists())
                return zipFile;

        } catch (IOException e) {
            logger.error(getMessage("shp.export.zip.error",e.getLocalizedMessage()));
            throw new RuntimeException(getMessage("shp.export.zip.error",e.getLocalizedMessage()));
        }catch (Exception e){
            logger.error(getMessage("shp.export.error",e.getLocalizedMessage()));
            FilesUtils.delFileOrDirectory(folderPath);
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 解析shapefile成geojson
     * @param shpFile
     * @param dbfFile
     * @return
     */
    private String parseShapefile(File shpFile,File dbfFile,File prjFile,Map properties) throws Exception{

        assert shpFile!=null:getMessage("shp.shapefile.not.found");
//        assert dbfFile!=null:getMessage("shp.dbffile.not.found");

        List<Map<String,Object>> featureList = new ArrayList<Map<String, Object>>();
        ShapefileDataStore shapefileDataStore = null;
        FeatureSource featureSource = null;
        DbaseFileReader reader = null;
        PrjFileReader prjFileReader = null;
        CoordinateReferenceSystem sourceCrs=null;
        if(!isNull(prjFile))
        {
            try {
                prjFileReader = new PrjFileReader(new ShpFiles(prjFile));
                CoordinateReferenceSystem crs = prjFileReader.getCoodinateSystem();
                if(!isNull(crs)) sourceCrs=crs;
            } catch (IOException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            } finally {
                prjFileReader.close();
            }
        }
        try {
            shapefileDataStore = new ShapefileDataStore(shpFile.toURI().toURL());
            shapefileDataStore.setStringCharset(Charset.forName("GBK"));
        } catch (MalformedURLException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        try {
            featureSource = shapefileDataStore.getFeatureSource();
            FeatureCollection featureCollection = featureSource.getFeatures();
            if(featureCollection.size()>0){
                FeatureIterator<SimpleFeature> iterator = featureCollection.features();
                while(iterator.hasNext()){
                    Map<String,Object> map = new HashMap<String, Object>();
                    SimpleFeature feature = iterator.next();
                    Geometry geo = (Geometry) feature.getDefaultGeometry();
                    boolean  valid=true;
                    if(!isNull(sourceCrs)){
                         valid = agsGeometryService.validGeometry(geo.toText(),sourceCrs.toWKT());
//                        if(!valid)
//                            throw new RuntimeException(getMessage("shp.tp.check.error"));
                    }
                    TopologyValidationError validationError = validGeometry(geo);
                     if(!isNull(validationError)&&!valid)
                       throw new RuntimeException(getMessage("shp.tp.check.error",validationError.getMessage()));
                    map.put(SHAPE,geo.toText());
                    if(properties!=null)
                        map.putAll(properties);
                    featureList.add(map);
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        if(!isNull(dbfFile))
        {
            try {
                reader = new DbaseFileReader(new FileInputStream(dbfFile).getChannel(), false, Charset.forName("GBK"));
                DbaseFileHeader header = reader.getHeader();
                int fieldsNum = header.getNumFields();
                int recordIndex=0;
                while (reader.hasNext()) {
                    DbaseFileReader.Row row = reader.readRow();
                    for (int i = 0; i < fieldsNum; i++) {
                        String fieldName = header.getFieldName(i);
                        Object value = row.read(i);
                        Map<String,Object> map = featureList.get(recordIndex);
                        if(map.containsKey(fieldName.toLowerCase())||map.containsKey(fieldName.toUpperCase()))
                            continue;
                        map.put(fieldName,value);
                    }
                    ++recordIndex;
                }
            } catch (IOException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            } finally {
                reader.close();
            }
        }
        return toFeatureJSON(list2FeatureCollection(featureList,sourceCrs,defaultCrs));//modified by yingxiufeng
    }

    /**
     * 解析shapefile成geojson
     * 在遇到topo问题时，是直接抛出异常还是跳过问题继续解析
     * @author shuyuanhao
     * @for 有时候需要容忍部分错误以导入大部分数据
     * @param shpFile
     * @param dbfFile
     * @param prjFile
     * @param properties
     * @param tolerateExp true,跳过异常继续解析;false,直接抛出问题
     * @return
     * @throws Exception
     */
    private String parseShapefile(File shpFile, File dbfFile, File prjFile, Map properties, boolean tolerateExp) throws Exception{
        assert shpFile!=null:getMessage("shp.shapefile.not.found");
//        assert dbfFile!=null:getMessage("shp.dbffile.not.found");

        List<Map<String,Object>> featureList = new ArrayList<Map<String, Object>>();
        ShapefileDataStore shapefileDataStore = null;
        FeatureSource featureSource = null;
        DbaseFileReader reader = null;
        PrjFileReader prjFileReader = null;
        CoordinateReferenceSystem sourceCrs=null;
        if(!isNull(prjFile))
        {
            try {
                prjFileReader = new PrjFileReader(new ShpFiles(prjFile));
                CoordinateReferenceSystem crs = prjFileReader.getCoodinateSystem();
                if(!isNull(crs)) sourceCrs=crs;
            } catch (IOException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            } finally {
                prjFileReader.close();
            }
        }
        try {
            shapefileDataStore = new ShapefileDataStore(shpFile.toURI().toURL());
            shapefileDataStore.setStringCharset(Charset.forName("GBK"));
        } catch (MalformedURLException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        // 被跳过的下标
        List tolerateList = new ArrayList();
        try {
            featureSource = shapefileDataStore.getFeatureSource();
            FeatureCollection featureCollection = featureSource.getFeatures();
            int tolerateIndex = 0;
            if(featureCollection.size()>0){
                FeatureIterator<SimpleFeature> iterator = featureCollection.features();
                while(iterator.hasNext()){
                    tolerateIndex++;
                    Map<String,Object> map = new HashMap<String, Object>();
                    SimpleFeature feature = iterator.next();
                    Geometry geo = (Geometry) feature.getDefaultGeometry();
                    boolean  valid=true;
                    if(!isNull(sourceCrs)){
                        valid = agsGeometryService.validGeometry(geo.toText(),sourceCrs.toWKT());
//                        if(!valid)
//                            throw new RuntimeException(getMessage("shp.tp.check.error"));
                    }
                    TopologyValidationError validationError = validGeometry(geo);
                    if(!isNull(validationError)&&!valid)
                        if(tolerateExp){
                            tolerateList.add(tolerateIndex-1);
                            logger.error(getMessage("shp.tp.check.error","Identifier:" + feature.getID() +"被跳过，因为"+validationError.getMessage()));
                            continue;
                        }else{
                            throw new RuntimeException(getMessage("shp.tp.check.error",validationError.getMessage()));
                        }

                    map.put(SHAPE,geo.toText());
                    if(properties!=null)
                        map.putAll(properties);
                    featureList.add(map);
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        if(!isNull(dbfFile))
        {
            try {
                reader = new DbaseFileReader(new FileInputStream(dbfFile).getChannel(), false, Charset.forName("GBK"));
                DbaseFileHeader header = reader.getHeader();
                int fieldsNum = header.getNumFields();
                int recordIndex=0, tolerateIndex = 0;
                while (reader.hasNext()) {
                    tolerateIndex++;
                    DbaseFileReader.Row row = reader.readRow();
                    if(tolerateList.contains(tolerateIndex-1)){
                        // 该下标的feature已经被跳过
                        continue;
                    }
                    for (int i = 0; i < fieldsNum; i++) {
                        String fieldName = header.getFieldName(i);
                        Object value = row.read(i);
                        Map<String,Object> map = featureList.get(recordIndex);
                        if(map.containsKey(fieldName.toLowerCase())||map.containsKey(fieldName.toUpperCase()))
                            continue;
                        map.put(fieldName,value);
                    }
                    ++recordIndex;
                }
            } catch (IOException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            } finally {
                reader.close();
            }
        }
        return toFeatureJSON(list2FeatureCollection(featureList,sourceCrs,defaultCrs));//modified by yingxiufeng
    }


    /**
     * 将shp文件转换成List<Map<String, Object>>
     * @param shpFile
     * @param dbfFile
     * @param prjFile
     * @param properties
     * @param tolerateExp
     * @return
     * @throws Exception
     */
    private List parseShapefile2MapList(File shpFile, File dbfFile, File prjFile, Map properties, boolean tolerateExp) throws Exception{
        assert shpFile!=null:getMessage("shp.shapefile.not.found");
//        assert dbfFile!=null:getMessage("shp.dbffile.not.found");

        List<Map<String,Object>> featureList = new ArrayList<Map<String, Object>>();
        ShapefileDataStore shapefileDataStore = null;
        FeatureSource featureSource = null;
        DbaseFileReader reader = null;
        PrjFileReader prjFileReader = null;
        CoordinateReferenceSystem sourceCrs=null;
        if(!isNull(prjFile))
        {
            try {
                prjFileReader = new PrjFileReader(new ShpFiles(prjFile));
                CoordinateReferenceSystem crs = prjFileReader.getCoodinateSystem();
                if(!isNull(crs)) sourceCrs=crs;
            } catch (IOException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            } finally {
                prjFileReader.close();
            }
        }
        try {
            shapefileDataStore = new ShapefileDataStore(shpFile.toURI().toURL());
            shapefileDataStore.setStringCharset(Charset.forName("GBK"));
        } catch (MalformedURLException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        // 被跳过的下标
        List tolerateList = new ArrayList();
        try {
            featureSource = shapefileDataStore.getFeatureSource();
            FeatureCollection featureCollection = featureSource.getFeatures();
            int tolerateIndex = 0;
            if(featureCollection.size()>0){
                FeatureIterator<SimpleFeature> iterator = featureCollection.features();
                while(iterator.hasNext()){
                    tolerateIndex++;
                    Map<String,Object> map = new HashMap<String, Object>();
                    SimpleFeature feature = iterator.next();
                    Geometry geo = (Geometry) feature.getDefaultGeometry();
                    boolean  valid=true;
                    if(!isNull(sourceCrs)){
                        valid = agsGeometryService.validGeometry(geo.toText(),sourceCrs.toWKT());
//                        if(!valid)
//                            throw new RuntimeException(getMessage("shp.tp.check.error"));
                    }
                    TopologyValidationError validationError = validGeometry(geo);
                    if(!isNull(validationError)&&!valid)
                        if(tolerateExp){
                            tolerateList.add(tolerateIndex-1);
                            logger.error(getMessage("shp.tp.check.error","Identifier:" + feature.getID() +"被跳过，因为"+validationError.getMessage()));
                            continue;
                        }else{
                            throw new RuntimeException(getMessage("shp.tp.check.error",validationError.getMessage()));
                        }

                    map.put(SHAPE,geo.toText());
                    if(properties!=null)
                        map.putAll(properties);
                    featureList.add(map);
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        if(!isNull(dbfFile))
        {
            try {
                reader = new DbaseFileReader(new FileInputStream(dbfFile).getChannel(), false, Charset.forName("GBK"));
                DbaseFileHeader header = reader.getHeader();
                int fieldsNum = header.getNumFields();
                int recordIndex=0, tolerateIndex = 0;
                while (reader.hasNext()) {
                    tolerateIndex++;
                    DbaseFileReader.Row row = reader.readRow();
                    if(tolerateList.contains(tolerateIndex-1)){
                        // 该下标的feature已经被跳过
                        continue;
                    }
                    for (int i = 0; i < fieldsNum; i++) {
                        String fieldName = header.getFieldName(i);
                        Object value = row.read(i);
                        Map<String,Object> map = featureList.get(recordIndex);
                        if(map.containsKey(fieldName.toLowerCase())||map.containsKey(fieldName.toUpperCase()))
                            continue;
                        map.put(fieldName,value);
                    }
                    ++recordIndex;
                }
            } catch (IOException e) {
                throw new RuntimeException(e.getLocalizedMessage());
            } finally {
                reader.close();
            }
        }
        return featureList;//modified by yingxiufeng
    }

    /**
     *将excel坐标文件解析成list
     * @param startRowNum
     * @param sheet
     * @return
     */
    private List<Map<String, Object>> excel2List(int startRowNum, Sheet sheet,CoordinateReferenceSystem crs){
        List<Map<String, Object>> features = new ArrayList<Map<String, Object>>();
        List<Map<String,Object>> coordList = new ArrayList<Map<String, Object>>();
        for(int i=startRowNum;i<sheet.getLastRowNum();i++){
            Row row = sheet.getRow(i);
            if(isNotNull(row)&&!isNull(row.getCell(0))){
                if(row.getCell(0).getStringCellValue().endsWith(XLS_COORD_TAG)){
                    features.addAll(excel2List(i + 1, sheet,crs));
                    break;
                }else
                {
                    if(!isNull(row.getCell(0).getStringCellValue()))
                    {
                        Map<String,Object> coordMap = new HashMap<String, Object>();
                        coordMap.put(BJ_TAG.PNT_SERIAL.name(),row.getCell(0).getStringCellValue());
                        coordMap.put(BJ_TAG.SHAPE_GROUP.name(),row.getCell(1).getCellType()==0?row.getCell(1).getNumericCellValue():row.getCell(1).getStringCellValue());
                        coordMap.put(BJ_TAG.Y_COORD.name(),row.getCell(2).getCellType()==0?row.getCell(2).getNumericCellValue():row.getCell(2).getStringCellValue());
                        coordMap.put(BJ_TAG.X_COORD.name(),row.getCell(3).getCellType()==0?row.getCell(3).getNumericCellValue():row.getCell(3).getStringCellValue());
                        coordList.add(coordMap);
                        logger.debug(coordMap.toString());
                    }
                }
            }
        }
        Map<String, List<Coordinate>> rings = new LinkedHashMap<String, List<Coordinate>>();
        CoordinateReferenceSystem fearureCrs = crs;
        try {
            for (Map map:coordList){
                String ringNo = String.valueOf(map.get(BJ_TAG.SHAPE_GROUP.name()));
                double x = Double.parseDouble(String.valueOf(map.get(BJ_TAG.X_COORD.name())));
                double y =Double.parseDouble(String.valueOf(map.get(BJ_TAG.Y_COORD.name())));
                if (rings.containsKey(ringNo))
                    rings.get(ringNo).add(new Coordinate(x, y));
                else {
                    logger.debug("set ring of ["+ringNo+"]");
                    try {
                        List<Coordinate> ring = new ArrayList<Coordinate>();
                        ring.add(new Coordinate(x, y));
                        rings.put(ringNo, ring);
                    } catch (Exception e) {
                        logger.error(e.getLocalizedMessage());
                        throw new RuntimeException(e.getLocalizedMessage());
                    }
                }
                if (fearureCrs == null) fearureCrs = getCrsByCoordXD(x);
            }
        } catch (NumberFormatException e) {
            logger.error(getMessage("excel.parse.error", e.getLocalizedMessage()));
            throw new RuntimeException(getMessage("excel.parse.error", e.getLocalizedMessage()));
        }
        LinearRing shell = null;
        List<LinearRing> holes = null;
        Collection<List<Coordinate>> coords = rings.values();
        for (List<Coordinate> coordinates : coords) {
            Coordinate first = coordinates.get(0);
            Coordinate last = coordinates.get(coordinates.size() - 1);
            if ((first.x != last.x) || (first.y != last.y))
                coordinates.add(new Coordinate(first.x, first.y));
        }
        Polygon polygon = null;
        Map<String, Object> feature = new HashMap<String, Object>();
        try {
            for (Iterator<List<Coordinate>> iterator = coords.iterator(); iterator.hasNext(); ) {
                List<Coordinate> coordinates = iterator.next();
                if (shell == null) {
                    shell = geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0]));
                } else {
                    if (holes == null) holes = new ArrayList<LinearRing>();
                    holes.add(geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0])));
                }
            }
            polygon = geometryFactory.createPolygon(shell, holes != null ? holes.toArray(new LinearRing[0]) : null);
        } catch (Exception e) {
            logger.error(getMessage("excel.parse.error", e.getLocalizedMessage()));
            throw new RuntimeException(getMessage("excel.parse.error", e.getLocalizedMessage()));
        }
        Row titleRow = sheet.getRow(startRowNum-1);
        String titleValue = titleRow.getCell(0).getStringCellValue().replaceAll("\\s*", "");
        titleValue = titleValue.replaceAll("，",",");
        List infoList = Arrays.asList(titleValue.split(","));
        String plotName = (String) infoList.get(3);

        feature.put(SHAPE, polygon.toText());
        feature.put(FEATURE_CRS, fearureCrs);
        feature.put(BJ_TAG.PL_NAME.name(),plotName);
        features.add(feature);
        return features;

    }

    /***
     * parse bj xml advanced
     * @param bytes
     * @return
     * @since v2.1.2
     * last modified by yingxiufeng on 2015/02/27
     */
    private Map parseBjXmlAdv(final byte[] bytes) {
        org.dom4j.Document document = null;
        Map<String, Object> result = new HashMap<String, Object>();
        SAXReader reader = new SAXReader();
        try {
            document = reader.read(new ByteArrayInputStream(bytes));
        } catch (DocumentException e) {
            throw new RuntimeException(getMessage("doc.read.xml.error", e.getLocalizedMessage()));
        }
        if (document != null) {
            boolean sFlag=false; //单独选址项目(true)/批次城镇建设用地(false)
            List<Map<String, Object>> features = new ArrayList<Map<String, Object>>();
            Element root = document.getRootElement();
            Node projInfo = root.selectSingleNode("DATA[@DATANAME='BL_PROJ_BUILD']").selectSingleNode("ROWDATA/ROW");
            Node blArea = root.selectSingleNode("DATA[@DATANAME='BL_BUILD_PROJ_LAND']").selectSingleNode("ROWDATA/ROW");
            if (projInfo instanceof Element)
            {
                Element element = (Element) projInfo;
                result.put(BJ_TITLE, element.attributeValue(BJ_TAG.PROJ_NAME.name()));
                /********************************************************
                 * 判断是单独选址还是批次项目
                 * 根据projInfo中的BJTYPE(PROJ_TYPE)来判断
                 * *****************************************************/
                String bjType=Constant.PROJ_PC;
                if (isNotNull(element.attribute(BJ_TAG.BJTYPE.name())))
                    bjType = element.attributeValue(BJ_TAG.BJTYPE.name());
                else if (isNotNull(element.attribute(BJ_TAG.PROJ_TYPE.name())))
                    bjType = element.attributeValue(BJ_TAG.PROJ_TYPE.name());
                if(Constant.PROJ_DDXZ.equals(bjType))
                    sFlag=true;

             }
            if (blArea instanceof Element) {
                result.put(BJ_AREA, ((Element) blArea).attributeValue(BJ_TAG.SUM_TOT.name()));
                result.put(BJ_AREA_NYD, ((Element) blArea).attributeValue(BJ_TAG.FARM_TOT.name()));
                result.put(BJ_AREA_GD, ((Element) blArea).attributeValue(BJ_TAG.TILTH_TOT.name()));
                result.put(BJ_AREA_JSYD, ((Element) blArea).attributeValue(BJ_TAG.BUILD_TOT.name()));
                result.put(BJ_AREA_WLYD, ((Element) blArea).attributeValue(BJ_TAG.UNUSED_TOT.name()));
            }
            /** 批次中的地块 **/
            Node blBound = root.selectSingleNode("DATA[@DATANAME='BL_SURVEY_BOUND']");
            /** 报件中地块的分类面积 **/
            Node blPlotClassArea = root.selectSingleNode("DATA[@DATANAME='BL_LAND_CLASS_AREA_ED']");
            /** 报件中的地块坐标 **/
            Node blPntCoords = root.selectSingleNode("DATA[@DATANAME='BL_PNT_COORD']");

            List plotRows = blBound.selectNodes("ROWDATA/ROW");
            List actualPls = root.selectSingleNode("DATA[@DATANAME='BL_PLOT']").selectNodes("ROWDATA/ROW");
            result.put("isSeparate", sFlag);
            CoordinateReferenceSystem sourceCrs = null;
            //modified by yxf on 2015/08/05
            for (Object item : actualPls) {
                if (item instanceof Element) {
                    Element row = (Element) item;
                    String id = row.attributeValue(BJ_TAG.PL_ID.name());
                    String name = row.attributeValue(BJ_TAG.PL_NAME.name());
                    Map feature = getFeature(id, name, blPntCoords, true, null);
                    if (sourceCrs == null) sourceCrs = (CoordinateReferenceSystem) feature.get(FEATURE_CRS);
                    feature.remove(FEATURE_CRS);
                    features.add(feature);
                }
            }
//            if(!sFlag){
//                for (Object item : plotRows) {
//                    if (item instanceof Element) {
//                        Element row = (Element) item;
//                        String id = row.attributeValue(BJ_TAG.SB_ID.name());
//                        String name = row.attributeValue(BJ_TAG.SB_NAME.name());
//                        Map info = getPlotClassArea(id, blPlotClassArea);
//                        Map feature;
//                        if (info.isEmpty())
//                            feature = getFeature(id, name, blPntCoords,false,null);
//                        else
//                            feature = getFeature(id, name, blPntCoords,false,JSON.toJSONString(info));
//                        if (sourceCrs == null) sourceCrs = (CoordinateReferenceSystem) feature.get(FEATURE_CRS);
//                        feature.remove(FEATURE_CRS);
//                        features.add(feature);
//                    }
//                }
//            }else{
//                for (Object item : actualPls) {
//                    if (item instanceof Element) {
//                        Element row = (Element) item;
//                        String id = row.attributeValue(BJ_TAG.PL_ID.name());
//                        String name = row.attributeValue(BJ_TAG.PL_NAME.name());
//                        Map feature =getFeature(id, name, blPntCoords,true,null);
//                        if (sourceCrs == null) sourceCrs = (CoordinateReferenceSystem) feature.get(FEATURE_CRS);
//                        feature.remove(FEATURE_CRS);
//                        features.add(feature);
//                    }
//                }
//            }

            result.put(BJ_FEATURE, toFeatureJSON(list2FeatureCollection(features, sourceCrs, defaultCrs)));
            return result;
        }
        throw new RuntimeException(getMessage("bj.parse.error"));
    }

    /***
     * get each class area of plots contained in bj(gt.xml)
     * @return
     */
    private Map getPlotClassArea(String id, Node node) {
        Map<String, Object> result = new HashMap<String, Object>();
        if (!isNull(id) && !isNull(node)) {
            /**存储 dlbm和area的map **/
            List<Map> list = new ArrayList<Map>();
            double sumArea = 0;
            double buildArea = 0;
            double tilthArea = 0;
            double unusedArea = 0;
            double farmArea = 0;
            List rows = node.selectNodes("ROWDATA/ROW");
            for (Object item : rows) {
                if (item instanceof Element) {
                    Element row = (Element) item;
                    String sbId = row.attributeValue(BJ_TAG.SB_SB_ID.name());
                    if (id.equals(sbId)) {
                        String dlbm = row.attributeValue(BJ_TAG.DLBM.name());
                        Double area = Double.valueOf(row.attributeValue(BJ_TAG.JTMJ.name())) + Double.valueOf(row.attributeValue(BJ_TAG.GYMJ.name()));
                        Map map = new HashMap();
                        map.put("dlbm", dlbm);
                        map.put("area", area);
                        list.add(map);
                    }
                }
            }
            Map gMap = ArrayUtils.listConvertMap(list, "dlbm");
            if (!isNull(gMap)) {
                for (Object item : gMap.entrySet()) {
                    Map.Entry entry = (Map.Entry) item;
                    List<Map> data = (List<Map>) MapUtils.getObject(gMap, entry.getKey());
                    double classArea = 0;
                    for (Map tmp : data) {
                        classArea += MapUtils.getDouble(tmp, "area", 0.0);
                    }
                    if (ArrayUtils.contains(EnumUtils.TDLYXZ_THREE_TYPE.TILTH.getDlbms(), String.valueOf(entry.getKey()), true))
                        tilthArea += classArea;
                    if (ArrayUtils.contains(EnumUtils.TDLYXZ_THREE_TYPE.FARM.getDlbms(), String.valueOf(entry.getKey()), true))
                        farmArea += classArea;
                    if (ArrayUtils.contains(EnumUtils.TDLYXZ_THREE_TYPE.BUILD.getDlbms(), String.valueOf(entry.getKey()), true))
                        buildArea += classArea;
                    if (ArrayUtils.contains(EnumUtils.TDLYXZ_THREE_TYPE.UNUSED.getDlbms(), String.valueOf(entry.getKey()), true))
                        unusedArea += classArea;
                    sumArea += classArea;
                }
            }

            result.put(BJ_AREA, sumArea * 0.0001);
            result.put(BJ_AREA_NYD, farmArea * 0.0001);
            result.put(BJ_AREA_GD, tilthArea * 0.0001);
            result.put(BJ_AREA_JSYD, buildArea * 0.0001);
            result.put(BJ_AREA_WLYD, unusedArea * 0.0001);
        }
        return result;
    }
    /***
     * get feature from bj advanced
     * @param id
     * @param name
     * @param info
     * @param node
     * @param isSep
     * @return
     * @since v2.1.2
     * last modified by yingxiufeng on 2015/02/27
     */
    private Map getFeature(String id, String name, Node node,boolean isSep,String info) {
        Map<String, Object> feature = new HashMap<String, Object>();
        CoordinateReferenceSystem fearureCrs = null;
        feature.put(BJ_TAG.ID.name(), id);
        feature.put(BJ_TAG.NAME.name(), name);
        if (!isNull(info))
            feature.put(BJ_TAG.INFO.name(), info);
        Map<String, List<Coordinate>> rings = new LinkedHashMap<String, List<Coordinate>>();
        List rows =new ArrayList();
        if (!isSep)
            rows = node.selectNodes("ROWDATA/ROW[@SB_SB_ID='" + id + "']");
        else
            rows = node.selectNodes("ROWDATA/ROW[@PL_PL_ID='" + id + "']");
        List<HashMap> sortRows = new ArrayList();
        for (Object item : rows) {
            if (item instanceof Element) {
                Element row = (Element) item;
                HashMap map = new LinkedHashMap();
                map.put(BJ_TAG.SHAPE_GROUP.name(), row.attributeValue(BJ_TAG.SHAPE_GROUP.name()));
                map.put(BJ_TAG.PNT_SERIAL.name(), row.attributeValue(BJ_TAG.PNT_SERIAL.name()));
                map.put(BJ_TAG.Y_COORD.name(), row.attributeValue(BJ_TAG.Y_COORD.name()));
                map.put(BJ_TAG.X_COORD.name(), row.attributeValue(BJ_TAG.X_COORD.name()));
                sortRows.add(map);
            }
        }
        Collections.sort(sortRows, new Comparator<HashMap>() {
            public int compare(HashMap arg0, HashMap arg1) {
                return Integer.valueOf(arg0.get(BJ_TAG.PNT_SERIAL.name()).toString()) - Integer.valueOf(arg1.get(BJ_TAG.PNT_SERIAL.name()).toString());
            }
        });
        for (HashMap row : sortRows) {
            String ringNo = String.valueOf(row.get(BJ_TAG.SHAPE_GROUP.name()));
            double x = Double.valueOf((String) row.get(BJ_TAG.Y_COORD.name()));
            double y = Double.valueOf((String) row.get(BJ_TAG.X_COORD.name()));
            if (rings.containsKey(ringNo))
                rings.get(ringNo).add(new Coordinate(x, y));
            else {
                List<Coordinate> ring = new ArrayList<Coordinate>();
                ring.add(new Coordinate(x, y));
                rings.put(ringNo, ring);
            }
            if (fearureCrs == null) fearureCrs = getCrsByCoordXD(x);
        }

        LinearRing shell = null;
        List<LinearRing> holes = null;
        Collection<List<Coordinate>> coords = rings.values();
        for (List<Coordinate> coordinates : coords) {
            Coordinate first = coordinates.get(0);
            Coordinate last = coordinates.get(coordinates.size() - 1);
            if ((first.x != last.x) || (first.y != last.y))
                coordinates.add(new Coordinate(first.x, first.y));
        }
        Polygon polygon = null;
        List<Coordinate> coordinates=null;
        try {
            for (Iterator<List<Coordinate>> iterator = coords.iterator(); iterator.hasNext(); ) {
                coordinates = iterator.next();
                if (shell == null) {
                    shell = geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0]));
                } else {
                    if (holes == null) holes = new ArrayList<LinearRing>();
                    holes.add(geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0])));
                }
            }
            polygon = geometryFactory.createPolygon(shell, holes != null ? holes.toArray(new LinearRing[0]) : null);
        } catch (Exception e) {
            throw new RuntimeException(getMessage("bj.polygon.coords.error", name, e.getLocalizedMessage(),JSON.toJSONString(coordinates)));
        }
        feature.put(SHAPE, polygon.toText());
        feature.put(FEATURE_CRS, fearureCrs);
        return feature;
    }

    /**
     * parse xml bytes
     *
     * @param bytes
     * @return
     */
    @Deprecated
    private Map parseBjXml(final byte[] bytes) {
        org.dom4j.Document document = null;
        Map<String, String> result = new HashMap<String, String>();
        SAXReader reader = new SAXReader();
        try {
            document = reader.read(new ByteArrayInputStream(bytes));
        } catch (DocumentException e) {
            throw new RuntimeException(getMessage("doc.read.xml.error", e.getLocalizedMessage()));
        }
        if (document != null) {
            List<Map<String, Object>> features = new ArrayList<Map<String, Object>>();
            Element root = document.getRootElement();
            Node blName = root.selectSingleNode("DATA[@DATANAME='BL_PROJ_BUILD']").selectSingleNode("ROWDATA/ROW");
            Node blArea = root.selectSingleNode("DATA[@DATANAME='BL_BUILD_PROJ_LAND']").selectSingleNode("ROWDATA/ROW");
            if (blName instanceof Element)
                result.put(BJ_TITLE, ((Element) blName).attributeValue(BJ_TAG.PROJ_NAME.name()));
            if(blArea instanceof Element)
            {
                result.put(BJ_AREA,((Element)blArea).attributeValue(BJ_TAG.SUM_TOT.name()));
                result.put(BJ_AREA_NYD,((Element)blArea).attributeValue(BJ_TAG.FARM_TOT.name()));
                result.put(BJ_AREA_GD,((Element)blArea).attributeValue(BJ_TAG.TILTH_TOT.name()));
                result.put(BJ_AREA_JSYD,((Element)blArea).attributeValue(BJ_TAG.BUILD_TOT.name()));
                result.put(BJ_AREA_WLYD,((Element)blArea).attributeValue(BJ_TAG.UNUSED_TOT.name()));
            }
            Node blPlot = root.selectSingleNode("DATA[@DATANAME='BL_PLOT']");
            Node blPntCoords = root.selectSingleNode("DATA[@DATANAME='BL_PNT_COORD']");
            List rows = blPlot.selectNodes("ROWDATA/ROW");
            CoordinateReferenceSystem sourceCrs = null;
            for (Object item : rows) {
                if (item instanceof Element) {
                    Element row = (Element) item;
                    String id = row.attributeValue(BJ_TAG.PL_ID.name());
                    String name = row.attributeValue(BJ_TAG.PL_NAME.name());
                    Map feature = getFeature(id, name, blPntCoords);
                    if (sourceCrs == null) sourceCrs = (CoordinateReferenceSystem) feature.get(FEATURE_CRS);
                    feature.remove(FEATURE_CRS);
                    features.add(feature);
                }
            }
            result.put(BJ_FEATURE, toFeatureJSON(list2FeatureCollection(features, sourceCrs, defaultCrs)));
            return result;
        }
        throw new RuntimeException(getMessage("bj.parse.error"));
    }
    /**
     * get feature
     *
     * @param id
     * @param name
     * @param node
     * @return
     */
    @Deprecated
    private Map getFeature(String id, String name, Node node) {
        Map<String, Object> feature = new HashMap<String, Object>();
        CoordinateReferenceSystem fearureCrs = null;
        feature.put(BJ_TAG.ID.name(), id);
        feature.put(BJ_TAG.NAME.name(), name);
        Map<String, List<Coordinate>> rings = new LinkedHashMap<String, List<Coordinate>>();
        List rows = node.selectNodes("ROWDATA/ROW[@PL_PL_ID='" + id + "']");
        List<HashMap> sortRows = new ArrayList();
        for (Object item : rows) {
            if (item instanceof Element) {
                Element row = (Element) item;
                HashMap map = new LinkedHashMap();
                map.put(BJ_TAG.SHAPE_GROUP.name(), row.attributeValue(BJ_TAG.SHAPE_GROUP.name()));
                map.put(BJ_TAG.PNT_SERIAL.name(), row.attributeValue(BJ_TAG.PNT_SERIAL.name()));
                map.put(BJ_TAG.Y_COORD.name(), row.attributeValue(BJ_TAG.Y_COORD.name()));
                map.put(BJ_TAG.X_COORD.name(), row.attributeValue(BJ_TAG.X_COORD.name()));
                sortRows.add(map);
            }
        }
        Collections.sort(sortRows, new Comparator<HashMap>() {
            public int compare(HashMap arg0, HashMap arg1) {
                return Integer.valueOf(arg0.get(BJ_TAG.PNT_SERIAL.name()).toString()) - Integer.valueOf(arg1.get(BJ_TAG.PNT_SERIAL.name()).toString());
            }
        });
        for (HashMap row : sortRows) {
            String ringNo = String.valueOf(row.get(BJ_TAG.SHAPE_GROUP.name()));
            double x = Double.valueOf((String) row.get(BJ_TAG.Y_COORD.name()));
            double y = Double.valueOf((String) row.get(BJ_TAG.X_COORD.name()));
            if (rings.containsKey(ringNo))
                rings.get(ringNo).add(new Coordinate(x, y));
            else {
                List<Coordinate> ring = new ArrayList<Coordinate>();
                ring.add(new Coordinate(x, y));
                rings.put(ringNo, ring);
            }
            if (fearureCrs == null) fearureCrs = getCrsByCoordXD(x);
        }

        LinearRing shell = null;
        List<LinearRing> holes = null;
        Collection<List<Coordinate>> coords = rings.values();
        for (List<Coordinate> coordinates : coords) {
            Coordinate first = coordinates.get(0);
            Coordinate last = coordinates.get(coordinates.size() - 1);
            if ((first.x != last.x) || (first.y != last.y))
                coordinates.add(new Coordinate(first.x, first.y));
        }
        Polygon polygon = null;
        try {
            for (Iterator<List<Coordinate>> iterator = coords.iterator(); iterator.hasNext(); ) {
                List<Coordinate> coordinates = iterator.next();
                if (shell == null) {
                    shell = geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0]));
                } else {
                    if (holes == null) holes = new ArrayList<LinearRing>();
                    holes.add(geometryFactory.createLinearRing(coordinates.toArray(new Coordinate[0])));
                }
            }
            polygon = geometryFactory.createPolygon(shell, holes != null ? holes.toArray(new LinearRing[0]) : null);
        } catch (Exception e) {
            throw new RuntimeException(getMessage("bj.polygon.coords.error", name, e.getLocalizedMessage()));
        }
        feature.put(SHAPE, polygon.toText());
        feature.put(FEATURE_CRS, fearureCrs);
        return feature;
    }
    /**
     * @return
     */
    private synchronized WKTReader getWktReader() {
        if (wktReader == null) wktReader = new WKTReader(geometryFactory);
        return wktReader;
    }

    /**
     * @param location
     */
    public void setRegionSet(Resource location) {
        try {
            Map values = JSON.parseObject(IOUtils.toString(location.getURI()), Map.class);
            this.simplifyTolerance = Double.valueOf(String.valueOf(values.get("simplifyTolerance")));
            this.regionLayers = (Map) values.get(REGION_FIELD);
            this.regionSet = (Map) values.get(REGION_MAP);
            if (values.containsKey(DEFAULT_CRS)) defaultCrs = parseUndefineSR(String.valueOf(values.get(DEFAULT_CRS)));
            if (values.containsKey(COORDINATE_DM)) {
                coordinateDM = (Map) values.get(COORDINATE_DM);
                for (Map.Entry entry : coordinateDM.entrySet()) {
                    entry.setValue(parseUndefineSR((String) entry.getValue()));
                }
            }
            // validator
        } catch (IOException e) {
            logger.error(" region set file not found ");
        }
    }


}
