package cn.gtmap.hlw.core.util.xml;

import cn.hutool.json.JSONObject;
import cn.hutool.json.XML;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.DomDriver;
import com.thoughtworks.xstream.security.AnyTypePermission;
import groovy.util.Node;
import groovy.util.XmlNodePrinter;
import groovy.util.XmlParser;
import groovy.xml.XmlUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;


/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2023/9/20
 * @description
 */
public class XmlUtils {
    public static Node getXmlNodeByString(String xml) {
        try {
            xml = xml.replaceAll("&(?!amp;|lt;|gt;|quot;|apos;|#\\d+;|#x[0-9A-Fa-f]+;)", "&amp;");
            return new XmlParser().parseText(xml);
        } catch (IOException | SAXException | ParserConfigurationException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * @param clazz 类的字节码文件
     * @param xml   传入的XML字符串
     * @return XML字符串转实体类
     */
    public static <T> T parseFromXml(Class<T> clazz, String xml) {
        XStream xStream = new XStream(new DomDriver());

        // 警告:允许任意类型反序列化,存在安全风险
        XStream.setupDefaultSecurity(xStream);
        xStream.addPermission(AnyTypePermission.ANY);

        xStream.processAnnotations(clazz);
        @SuppressWarnings("unchecked")
        T t = (T) xStream.fromXML(xml);
        return t;
    }


    public static String processCharacterSet(String text) {
        if (StringUtils.isNotBlank(text)) {
            text = text.replace("&lt;", "<").replace("&gt;", ">")
                    .replace("&apos;", "'").replace("&amp;", "&").replace("&quot;", "\"");
        }
        return text;
    }

    public static String replacePageInXml(Node xml, String page) {
        if (xml != null && StringUtils.isNotEmpty(page)) {
            xml.setValue(page);
        }
        return XmlUtil.serialize(xml);
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">chenjia</a>
     * @description 数据转换xml
     */
    public static String processOtherDataSourcePageXml(Node page, Map dataMap, Map detailDataMap) {
        String datas = parsePrintDatasXml(page, dataMap);
        String detail = parseOtherDataSourcePrintDetailXml(page, detailDataMap);
        StringBuilder pageStr = new StringBuilder();
        if (StringUtils.isBlank(datas)) {
            datas = "<datas></datas>";
        }
        pageStr.append(datas);
        pageStr.append(detail);
        return pageStr.toString();
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">chenjia</a>
     * @description 组织主表信息
     */
    static String parsePrintDatasXml(Node page, Map dataMap) {
        StringBuilder builder = new StringBuilder();
        boolean emptyMap = MapUtils.isEmpty(dataMap);
        if (emptyMap) {
            dataMap = Maps.newHashMap();
        }
        List<String> configKeys = Lists.newArrayList();
        page.children().forEach(data -> {
            Node datas = (Node) data;
            if (!StringUtils.equals((CharSequence) datas.name(), "detail")) {
                builder.append(nodeToXmlString((Node) data));
                configKeys.addAll(Arrays.asList(StringUtils.split(((Node) data).text(), '$')));
            }
        });
        /**空字符串替换null,添加未查询出的字段*/
        Map finalDataMap = dataMap;
        configKeys.forEach(key -> {
            finalDataMap.putIfAbsent(key, finalDataMap.containsKey(key) ? finalDataMap.get(key) : (finalDataMap.containsKey(StringUtils.lowerCase(key)) ? finalDataMap.get(StringUtils.lowerCase(key)) : ""));
        });
        String xml = builder.toString();
        Iterator headerIterator = finalDataMap.entrySet().iterator();
        while (headerIterator.hasNext()) {
            Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
            xml = replaceXml(String.valueOf(elem.getKey()), elem.getValue(), xml);
        }
        return xml;
    }

    /**
     * xml to json
     *
     * @param xml
     * @return
     */
    public static String xml2json(String xml) {
        JSONObject jsonObject = XML.toJSONObject(xml.replace("<xml>", "").replace("</xml>", ""));
        return jsonObject.toString();
    }

    /**
     * @param obj 实体类
     * @return 实体类转XML字符串
     */
    public static String toXml(Object obj) {
        XStream xStream = new XStream(new DomDriver());

        // 警告:允许任意类型反序列化,存在安全风险
        XStream.setupDefaultSecurity(xStream);
        xStream.addPermission(AnyTypePermission.ANY);

        xStream.processAnnotations(obj.getClass());
        return xStream.toXML(obj).replaceAll("\\_+", "_");
    }


    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">chenjia</a>
     * @description 节点转换成xml字符串
     */
    public static String nodeToXmlString(Node xml) {
        StringWriter stringWriter = new StringWriter();
        XmlNodePrinter nodePrinter = new XmlNodePrinter(new PrintWriter(stringWriter));
        nodePrinter.setPreserveWhitespace(true);
        nodePrinter.print(xml);
        return stringWriter.toString();
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:chenjia@gtmap.cn">chenjia</a>
     * @description 替换xml参数
     */
    public static String replaceXml(String key, Object value, String xml) {
        xml = xml.replaceAll("\\>\\$" + StringUtils.upperCase(key) + "\\<", "\\>" + value + "\\<");
        xml = xml.replaceAll("\\>\\$" + StringUtils.lowerCase(key) + "\\<", "\\>" + value + "\\<");
        xml = xml.replaceAll("\\>\\$" + key + "\\<", "\\>" + value + "\\<");
        xml = xml.replaceAll("\\$\\{[" + StringUtils.upperCase(key) + "}]+\\}", String.valueOf(value));
        xml = xml.replaceAll("\\$\\{[" + StringUtils.lowerCase(key) + "}]+\\}", String.valueOf(value));
        xml = xml.replaceAll("\\$\\{[" + key + "}]+\\}", String.valueOf(value));
        return xml;
    }

    /**
     * @param page dataMap
     * @return <detail></detail>
     * @author <a href ="mailto:songhaowen@gtmap.cn">songhaowen</a>
     * @description 替换其他数据源 detail 数据
     */
    static String parseOtherDataSourcePrintDetailXml(Node page, Map dataMap) {
        StringBuilder pageXml = new StringBuilder();
        if (dataMap.isEmpty()) {
            return String.valueOf(pageXml);
        }
        //循环处理xml中的detail
        page.children().forEach(data -> {
            Node detailData = (Node) data;
            if (StringUtils.equals("detail", String.valueOf(detailData.name()))) {
                detailData.name();
                //detail xml
                String detailXml = nodeToXmlString(detailData);
                String detailId = String.valueOf(detailData.attribute("ID"));

                //每个detail替换数据后的xml
                StringBuilder dataBulider = new StringBuilder();

                //每个row节点
                StringBuilder rowBulider = new StringBuilder();
                List<String> configKeys = Lists.newArrayList();
                detailData.children().forEach(it -> {
                    Node rowData = (Node) it;
                    rowBulider.append(nodeToXmlString(rowData));
                    configKeys.addAll(Arrays.asList(StringUtils.split(rowData.text(), '$')));
                    String id = String.valueOf(rowData.attribute("ID"));
                    if (StringUtils.isNotBlank(id)) {
                        id = StringUtils.replacePattern(id, "\\$", "");
                        id = StringUtils.replacePattern(id, "\\{", "");
                        id = StringUtils.replacePattern(id, "\\}", "");
                        configKeys.add(id);
                    }
                });

                List<Map> detailList = (List<Map>) dataMap.get(detailId);
                if (CollectionUtils.isNotEmpty(detailList)) {
                    detailList.forEach(rowData -> {
                        //空字符串替换null,添加未查询出的字段
                        configKeys.forEach(key -> {
                            rowData.putIfAbsent(key, rowData.containsKey(key) ? rowData.get(key) : (rowData.containsKey(StringUtils.lowerCase(key)) ? rowData.get(StringUtils.lowerCase(key)) : ""));
                        });
                        String xml = rowBulider.toString();
                        Iterator headerIterator = rowData.entrySet().iterator();
                        while (headerIterator.hasNext()) {
                            Map.Entry<String, String> elem = (Map.Entry<String, String>) headerIterator.next();
                            xml = replaceXml(String.valueOf(elem.getKey()), elem.getValue(), xml);
                        }
                        dataBulider.append(xml);
                    });
                }
                String pageDetail = StringUtils.replace(StringUtils.normalizeSpace(detailXml), StringUtils.normalizeSpace(rowBulider.toString()), StringUtils.normalizeSpace(dataBulider.toString()));
                pageXml.append(pageDetail);
            }
        });
        return pageXml.toString();
    }
}
