package cn.gtmap.hlw.core.util.encryption.ght;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;

/**
 * AES加解密工具类
 *
 * @author <a href="mailto:sunchao@gtmap.cn">sunchao</a
 * @version 1.0
 * @description AES加密解密工具，与前端njxDecode保持一致
 */
public class AesUtil {

    private static final Logger logger = LoggerFactory.getLogger(AesUtil.class);

    private static final String ALGORITHM = "AES";
    private static final String CIPHER_ALGORITHM = "AES/CBC/PKCS5Padding";

    /**
     * AES加密
     *
     * @param content 待加密内容
     * @param key 密钥（16位）
     * @return 加密后的Base64字符串
     */
    public static String encrypt(String content, String key) {
        try {
            // 创建密钥
            SecretKeySpec secretKey = new SecretKeySpec(key.getBytes(StandardCharsets.UTF_8), ALGORITHM);
            // 创建IV（使用key作为IV，与前端保持一致）
            IvParameterSpec iv = new IvParameterSpec(key.getBytes(StandardCharsets.UTF_8));

            // 创建加密器
            Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, secretKey, iv);

            // 执行加密
            byte[] encrypted = cipher.doFinal(content.getBytes(StandardCharsets.UTF_8));

            // 返回Base64编码
            return Base64.encodeBase64String(encrypted);
        } catch (Exception e) {
            logger.error("AES加密失败", e);
            throw new RuntimeException("AES加密失败: " + e.getMessage());
        }
    }

    /**
     * AES解密
     *
     * @param encryptedContent 加密的Base64字符串
     * @param key 密钥（16位）
     * @return 解密后的原文
     */
    public static String decrypt(String encryptedContent, String key) {
        try {
            // 创建密钥
            SecretKeySpec secretKey = new SecretKeySpec(key.getBytes(StandardCharsets.UTF_8), ALGORITHM);
            // 创建IV（使用key作为IV，与前端保持一致）
            IvParameterSpec iv = new IvParameterSpec(key.getBytes(StandardCharsets.UTF_8));

            // 创建解密器
            Cipher cipher = Cipher.getInstance(CIPHER_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, secretKey, iv);

            // Base64解码
            byte[] encryptedBytes = Base64.decodeBase64(encryptedContent);

            // 执行解密
            byte[] decrypted = cipher.doFinal(encryptedBytes);

            // 返回解密后的字符串
            String result = new String(decrypted, StandardCharsets.UTF_8);
            
            // 清理可能的多余字符（与前端保持一致）
            return cleanJsonString(result);
        } catch (Exception e) {
            logger.error("AES解密失败", e);
            throw new RuntimeException("AES解密失败: " + e.getMessage());
        }
    }

    /**
     * 清理JSON字符串（与前端njxDecode逻辑保持一致）
     * 
     * @param str 待清理的字符串
     * @return 清理后的字符串
     */
    private static String cleanJsonString(String str) {
        if (str == null || str.length() <= 1) {
            return str;
        }

        // 去除首尾空白字符和零字符（ZeroPadding可能导致）
        str = str.trim();
        
        // 去除尾部的零字符
        while (str.length() > 0 && str.charAt(str.length() - 1) == '\0') {
            str = str.substring(0, str.length() - 1);
        }
        str = str.trim();

        // 如果是JSON数组
        if (str.length() > 0 && str.charAt(0) == '[') {
            int lastIndex = str.lastIndexOf(']');
            if (lastIndex > 0) {
                return str.substring(0, lastIndex + 1);
            }
        }
        // 如果是JSON对象
        else if (str.length() > 0 && str.charAt(0) == '{') {
            int lastIndex = str.lastIndexOf('}');
            if (lastIndex > 0) {
                return str.substring(0, lastIndex + 1);
            }
        }

        return str;
    }

    /**
     * 判断字符串是否为JSON格式
     *
     * @param str 待判断的字符串
     * @return true-是JSON格式
     */
    public static boolean isJson(String str) {
        if (str == null || str.trim().isEmpty()) {
            return false;
        }
        String trimmed = str.trim();
        return (trimmed.startsWith("{") && trimmed.endsWith("}"))
                || (trimmed.startsWith("[") && trimmed.endsWith("]"));
    }
}

