package cn.gtmap.hlw.core.util.bean;

import cn.gtmap.hlw.core.constant.Constants;
import cn.gtmap.hlw.core.util.string.StringUtil;
import cn.hutool.core.util.ModifierUtil;
import cn.hutool.core.util.ReflectUtil;
import com.alibaba.fastjson.JSON;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 实体转换工具
 *
 * @Author admin
 * @Date 2023/6/17 18:04
 */
public class BeanConvertUtil {

    public static final Logger logger = LoggerFactory.getLogger(BeanConvertUtil.class);

    /**
     * 类型转换：实体Vo <->实体  例如：UserVo <-> User
     * 支持一级复杂对象复制
     */
    public static <T> T copy(Object src, Class<T> targetType) {
        T target = null;
        try {
            //创建一个空目标对象，并获取一个BeanWrapper代理器，用于属性填充，BeanWrapperImpl在内部使用Spring的BeanUtils工具类对Bean进行反射操作，设置属性。
            target = targetType.newInstance();
            BeanWrapper targetBean = new BeanWrapperImpl(target);

            //获取源对象的BeanMap，属性和属性值直接转换为Map的key-value 形式
            BeanMap srcBean = new BeanMap(src);
            for (Object key : srcBean.keySet()) {
                //源对象属性名称
                String srcPropertyName = key + "";
                //源对象属性值
                Object srcPropertyVal = srcBean.get(key);
                //源对象属性类型
                Class srcPropertyType = srcBean.getType(srcPropertyName);
                //目标对象属性类型
                Class targetPropertyType = targetBean.getPropertyType(srcPropertyName);

                //源对象属性值非空判断、目标对象属性类型非空判断，如果为空跳出，继续操作下一个属性
                if ("class".equals(srcPropertyName) || targetPropertyType == null) {
                    continue;
                }

                //类型相等，可直接设置值，比如：String与String 或者 User与User
                if (srcPropertyType == targetPropertyType) {
                    targetBean.setPropertyValue(srcPropertyName, srcPropertyVal);
                }
                //类型不相等，比如：User与UserVo
                else {
                    //如果源复杂对象为null，直接跳过，不需要复制
                    if (srcPropertyVal == null) {
                        continue;
                    }

                    Object targetPropertyVal = targetPropertyType.newInstance();
                    BeanWrapper targetPropertyBean = new BeanWrapperImpl(targetPropertyVal);

                    BeanMap srcPropertyBean = new BeanMap(srcPropertyVal);
                    for (Object srcPropertyBeanKey : srcPropertyBean.keySet()) {
                        String srcPropertyBeanPropertyName = srcPropertyBeanKey + "";
                        Object srcPropertyBeanPropertyVal = srcPropertyBean.get(srcPropertyBeanKey);
                        Class srcPropertyBeanPropertyType = srcPropertyBean.getType(srcPropertyBeanPropertyName);
                        Class targetPropertyBeanPropertyType = targetPropertyBean.getPropertyType(srcPropertyBeanPropertyName);

                        if ("class".equals(srcPropertyBeanPropertyName) || targetPropertyBeanPropertyType == null) {
                            continue;
                        }

                        if (srcPropertyBeanPropertyType == targetPropertyBeanPropertyType) {
                            targetPropertyBean.setPropertyValue(srcPropertyBeanPropertyName, srcPropertyBeanPropertyVal);
                        } else {
                            //复杂对象里面的复杂对象不再进行处理
                        }
                    }
                    //设置目标对象属性值
                    targetBean.setPropertyValue(srcPropertyName, targetPropertyBean.getWrappedInstance());
                }
            }
        } catch (Exception e) {
            logger.error("类型转换异常：", e);
        }
        return target;
    }

    /**
     * 类型转换：实体Vo <->实体  例如：List<UserVo> <-> List<User>
     */
    public static <T> List<T> copyList(List srcList, Class<T> targetType) {
        List<T> newList = new ArrayList<>();
        for (Object entity : srcList) {
            newList.add(copy(entity, targetType));
        }
        return newList;
    }


    /**
     * 功能描述: 将一个 Map 对象转化为一个 JavaBean
     *
     * @param type 要转化的类型
     * @param map  包含属性值的 map
     * @return 转化出来的 JavaBean 对象
     * @throws IntrospectionException    如果分析类属性失败
     * @throws IllegalAccessException    如果实例化 JavaBean 失败
     * @throws InstantiationException    如果实例化 JavaBean 失败
     * @throws InvocationTargetException 如果调用属性的 setter 方法失败
     * @author: <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @version: 2.0, 2018/12/14 9:11
     * throws IntrospectionException, IllegalAccessException,InstantiationException, InvocationTargetException
     */
    public static Object convertMap(Map map, Class type) {
        // 创建 JavaBean 对象
        Object obj = null;
        try {
            // 获取类属性
            BeanInfo beanInfo = Introspector.getBeanInfo(type);
            obj = type.newInstance();

            // 给 JavaBean 对象的属性赋值
            PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
            for (PropertyDescriptor descriptor : propertyDescriptors) {
                String propertyName = descriptor.getName();

                if (map.containsKey(propertyName)) {
                    // 下面一句可以 try 起来，这样当一个属性赋值失败的时候就不会影响其他属性赋值。
                    Object value = null;
                    try {
                        value = map.get(propertyName);
                        if ("".equals(value)) {
                            value = null;
                        }
                        Object[] args = new Object[1];
                        args[0] = value;
                        descriptor.getWriteMethod().invoke(obj, args);
                    } catch (InvocationTargetException e) {
                        logger.error("bean转换工具类，map转bean异常，异常原因：{}", e);
                    }
                }
            }
        } catch (IntrospectionException | InstantiationException | IllegalAccessException e) {
            logger.error("bean转换工具类，map转bean异常，异常原因：{}", e);
        }
        return obj;
    }

    /**
     * 功能描述: 将一个 JavaBean 对象转化为一个  Map
     * 并且将容易引起xss漏洞的半角字符直接替换成全角字符
     *
     * @param bean 要转化的JavaBean 对象
     * @return 转化出来的  Map 对象
     * @throws IntrospectionException    如果分析类属性失败
     * @throws IllegalAccessException    如果实例化 JavaBean 失败
     * @throws InvocationTargetException 如果调用属性的 setter 方法失败
     * @author: <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @version: 2.0, 2018/12/14 9:01
     * throws IntrospectionException, IllegalAccessException, InvocationTargetException
     */
    public static Map convertBean(Object bean) {
        Class type = bean.getClass();
        Map returnMap = new HashMap(Constants.INITIALCAPACITY);
        try {
            BeanInfo beanInfo = Introspector.getBeanInfo(type);
            PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
            for (PropertyDescriptor descriptor : propertyDescriptors) {
                String propertyName = descriptor.getName();
                if (!"class".equals(propertyName)) {
                    Method readMethod = descriptor.getReadMethod();
                    Object result = readMethod.invoke(bean, new Object[0]);
                    if (result != null) {
                        //判断Object对象类型
                        if (result instanceof String) {
                            //将容易引起xss漏洞的半角字符直接替换成全角字符
                            if ("zl".equals(propertyName)) {
                                returnMap.put(propertyName, result.toString());
                            } else {
                                returnMap.put(propertyName, StringUtil.xssEncode(result.toString()));
                            }
                        } else {
                            returnMap.put(propertyName, result);
                        }
                    } else {
                        returnMap.put(propertyName, null);
                    }
                }
            }
        } catch (IntrospectionException | IllegalAccessException | InvocationTargetException e) {
            logger.error("bean转换工具类，bean转map异常，异常原因：{}", e);
        }
        return returnMap;
    }


    /**
     * OBJ copy
     *
     * @param json
     * @param tClass
     * @param <T>
     * @return
     */
    public static <T> T getBeanByJsonObj(Object json, Class<T> tClass) {
        T t = null;
        try {
            if (json instanceof String) {
                //防止String类型被JSON.toJSONString(json)时，外面包裹一层""；无法转成JSON对象
                t = JSON.parseObject(json.toString(), tClass);
            } else {
                t = JSON.parseObject(JSON.toJSONString(json), tClass);
            }
        } catch (Exception e) {
            logger.error("Bean转换工具类getBeanByJsonObj方法转换异常，异常原因：{}", e);
        }
        return t;
    }

    /**
     * List copy
     *
     * @param obj
     * @param tClass
     * @param <T>
     * @return
     */
    public static <T> List<T> getBeanListByJsonArray(Object obj, Class<T> tClass) {
        List t = null;

        try {
            if (obj instanceof String) {
                t = JSON.parseArray(obj.toString(), tClass);
            } else {
                t = JSON.parseArray(JSON.toJSONString(obj), tClass);
            }
        } catch (Exception var4) {
            logger.error("Bean转换工具类getBeanListByJsonArray方法转换异常，异常原因：{}", var4);
        }

        return t;
    }

    /**
     * 使bean中为null的属性转换成空字符串
     *
     * @param bean
     * @author: <a href="mailto:huangyongkai@gtmap.cn">huangyongkai</a>
     * @Date: 2020-9-1
     * @return: void
     */
    public static <T> void nullToEmpty(T bean) {
        Field[] field = bean.getClass().getDeclaredFields();
        //遍历所有属性
        for (Field item : field) {
            //获取属性的名字
            String name = item.getName();
            //将属性的首字符大写，方便构造get，set方法
            name = name.substring(0, 1).toUpperCase() + name.substring(1);
            //获取属性的类型
            String type = item.getGenericType().toString();
            //如果type是类类型，则前面包含"class "，后面跟类名
            if (("class java.lang.String".equals(type))) {
                try {
                    Method mGet = bean.getClass().getMethod("get" + name);
                    //调用getter方法获取属性值
                    String value = (String) mGet.invoke(bean);
                    if (value == null || value.isEmpty()) {
                        Method mSet = bean.getClass().getMethod("set" + name, new Class[]{String.class});
                        mSet.invoke(bean, new Object[]{new String("")});
                    }
                } catch (Exception e) {
                    logger.error("Bean转换工具类nullToEmpty方法转换异常，异常原因：{}", e);
                }
            }
        }
    }

    public static <T> void removeThousandth(T bean) {

        final Field[] fields = ReflectUtil.getFields(bean.getClass());
        for (Field field : fields) {
            if (ModifierUtil.isStatic(field)) {
                continue;
            }
            if (Double.class.equals(field.getType())) {

            }
        }
    }

    /**
     * 判断类中每个属性是否都为空
     * 忽略
     *
     * @param o
     * @return
     */
    public static boolean allFieldIsNull(Object o, String ignoreField) {
        try {
            for (Field field : o.getClass().getDeclaredFields()) {
                field.setAccessible(true);
                if (StringUtils.equals(field.getName(), ignoreField) || StringUtils.equals(field.getName(), "serialVersionUID")) {
                    continue;
                }
                Object object = field.get(o);
                if (object instanceof CharSequence) {
                    if (!org.springframework.util.ObjectUtils.isEmpty(object)) {
                        return true;
                    }
                } /*else {
                    if (null != object) {
                        return true;
                    }
                }*/
            }
        } catch (Exception e) {
            logger.error("判断对象属性为空异常", e);
        }
        return false;
    }


    public static void setProperty(Object targetObject, Object sourceObject, String propertyName) {
        try {
            // 构建 set 和 get 方法名
            String setMethodName = "set" + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
            String getMethodName = "get" + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
            Method setMethod = targetObject.getClass().getMethod(setMethodName, String.class);
            Method getMethod = sourceObject.getClass().getMethod(getMethodName);
            if (setMethod != null && getMethod != null) {
                // 调用 get 方法获取值
                String propertyValue = (String) getMethod.invoke(sourceObject);
                // 设置 paramsModel 对象的属性值
                setMethod.invoke(targetObject, propertyValue);
            }
        } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            logger.error("反射错误", e);
        }
    }
}