package cn.gtmap.hlw.core.util.validator;

import org.apache.commons.lang3.StringUtils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author <a href="mailto:wangxuebiao@gtmap.cn">wangxuebiao</a>
 * @version 2023/6/19 15:36
 * @description 检测密码复杂度
 */
public class ValidationPasswordUtils {

    /**
     * USERPASS_TYPE=0 只需要数字或只需要字母
     */
    private static Pattern PATTERN_0 = Pattern.compile("^[a-z0-9A-Z@$!%*#?&.]{6,}$");

    /**
     * USERPASS_TYPE=1 需要字母和数字
     */
    private static Pattern PATTERN_1 = Pattern.compile("^(?![0-9@$!%*#?&.]+$)(?![a-zA-Z@$!%*#?&.]+$)[0-9A-Za-z@$!%*#?&.]{8,}$");

    /**
     * USERPASS_TYPE=2 需要大写字母且小写字母和数字
     */
    private static Pattern PATTERN_2 = Pattern.compile("^(?=.*[a-z])(?=.*[A-Z])(?=.*\\d)[@$!%*#?&.\\s\\S]{8,}$");

    /**
     * USERPASS_TYPE=3 8`20位，必须包含数字，大小写字母。不能包含其他字符
     */
    private static Pattern PATTERN_3 = Pattern.compile("^(?=.*\\d)(?=.*[a-z])(?=.*[A-Z])[a-zA-Z@$!%*#?&.\\d]{8,20}$");

    /**
     * USERPASS_TYPE=4  大写字母、小写字母、数字和特殊符号(四种中任三种)
     */
    private static Pattern PATTERN_4 = Pattern.compile("^(?![a-zA-Z@$!%*#?&.]+$)(?![A-Z0-9@$!%*#?&.]+$)(?![A-Z\\W_]+$)(?![a-z0-9@$!%*#?&.]+$)(?![a-z\\W_]+$)(?![0-9\\W_]+$)[a-zA-Z0-9@$!%*#?&.\\W_]{8,}$");

    /**
     * USERPASS_TYPE=5 需要大写字母、小写字母、数字和符号
     */
    private static Pattern PATTERN_5 = Pattern.compile("^(?=.*[A-Za-z])(?=.*\\d)(?=.*[@$!%*#?&.])[A-Za-z\\d@$!%*#?&.]{8,}$");

    /**
     * USERPASS_TYPE=6 8`16位长度的密码，必须包含数字、大写字母、小写字母和特殊字符。不能有汉字 沿用checkPasswordMark方法
     */
    private static Pattern PATTERN_6 = Pattern.compile("^(?=.*[A-Za-z])(?=.*\\d)(?=.*[@$!%*#?&.])[A-Za-z\\d@$!%*#?&.]{8,16}$");

    /**
     * USERPASS_TYPE=7 必须包含字母（不区分大小写）、数字和特殊符号  并且大于10位
     */
    private static Pattern PATTERN_7 = Pattern.compile("^(?=.*[A-Za-z])(?=.*\\d)(?=.*[@$!%*#?&.])[A-Za-z\\d@$!%*#?&.]{10,20}$");

    /**
     * 校验密码复杂度
     *
     * @param passwordType
     * @param password
     * @return
     */
    public static boolean checkPasswordComplexity(String passwordType, String password) {
        switch (passwordType) {
            case "0":
                return checkPasswordNum(password);
            case "1":
                return checkPasswordNumAndLetter(password);
            case "2":
                return checkPasswordAa(password);
            case "3":
                return checkPassword(password);
            case "4":
                return checkPasswordAaNumSymbol(password);
            case "5":
                return checkPasswordMark(password);
            case "6":
                return checkPasswordMarkLeng(password);
            case "7":
                return checkPasswordAaNumSymbolTenLeng(password);
            default:
                return false;
        }
    }

    /**
     * 获取密码复杂度错误提示
     * @param passwordType
     * @return
     */
    public static String getEroorTips(String passwordType) {
        switch (passwordType) {
            case "0":
                return "密码只需要数字或只需要字母";
            case "1":
                return "密码需要包含数字和字母，且至少8位";
            case "2":
                return "密码需要同时包含大小写字母，数字，特殊字符（$@!%*#?&），且以字母开头，且至少8位";
            case "3":
                return "密码长度8到20位，且必须包含数字，大小写字母，和特殊符号(@$!%*#?&.)";
            case "4":
                return "密码需要包含大写字母、小写字母、数字和特殊符号其中任意三种，且至少8位";
            case "5":
                return "密码需要包含大写字母、小写字母、数字和特殊符号(@$!%*#?&.)，且至少8位";
            case "6":
                return "密码长度8到16位，且必须包含数字、大写字母、小写字母和特殊字符(@$!%*#?&.)，不能有汉字";
            case "7":
                return "密码必须包含字母（不区分大小写）、数字和特殊符号(@$!%*#?&.),长度10位到20位";
            default:
                return "数据库密码配置项为空";
        }
    }

    /**
     * USERPASS_TYPE=0 只需要数字或只需要字母
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordNum(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_0.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=1 需要字母和数字
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordNumAndLetter(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_1.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=2 需要大写字母且小写字母和数字
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordAa(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_2.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=3 8`20位，必须包含数字，大小写字母。不能包含其他字符
     *
     * @param password
     * @return
     */
    private static boolean checkPassword(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_3.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=4  大写字母、小写字母、数字和特殊符号(四种中任三种)
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordAaNumSymbol(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_4.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=5 需要大写字母、小写字母、数字和符号
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordMark(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_5.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=6 8`16位长度的密码，必须包含数字、大写字母、小写字母和特殊字符。不能有汉字 沿用checkPasswordMark方法
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordMarkLeng(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_6.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }

    /**
     * USERPASS_TYPE=7 必须包含字母（不区分大小写）、数字和特殊符号  并且大于10位
     *
     * @param password
     * @return
     */
    private static boolean checkPasswordAaNumSymbolTenLeng(String password) {
        if (StringUtils.isNotBlank(password)) {
            Matcher matcher = PATTERN_7.matcher(password);
            if (matcher.matches()) {
                return true;
            }
        }
        return false;
    }
}
