package cn.gtmap.hlw.core.util.pdf;

import cn.gtmap.estateplat.utils.CommonUtil;
import cn.gtmap.hlw.core.constant.Constants;
import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.exception.BizException;
import cn.gtmap.hlw.core.util.string.StringUtil;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;

/**
 * @author <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
 * @version 1.0, 2020/10/15
 * @description 数字工具类
 */
public class NumberUtil {

    /**
     * @auto <a href="mailto:wangzhiwen@gtmap.cn">wangzhiwen</a>
     * @description 根据身份证判断是否未成年
     */
    public static Boolean sfzhSfWcn(String idCardNumber) {

        // 提取出生日期（假设身份证号码的第7到14位是出生日期）
        String dateStr = idCardNumber.substring(6, 14);

        // 解析日期字符串并转换为Date类型
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
        Date birthDate = null;
        try {
            birthDate = dateFormat.parse(dateStr);
        } catch (Exception e) {
            throw new BizException(ErrorEnum.SERVICE_ERROR,e);
        }

        // 计算年龄
        int age = 0;
        if (birthDate != null) {
            // 获取当前日期
            Date currentDate = new Date();

            // 计算年龄差值
            long diff = currentDate.getTime() - birthDate.getTime();
            long years = diff / (1000L * 60L * 60L * 24L * 365L);

            age = (int) years;
        }
        // 判断是否未成年人（假设未成年人的年龄小于18岁）
        return age < 18;
    }

    /**
     * 权利比例转化成小数
     *
     * @param qlbl
     * @return
     */
    public static String convertQlbl(String qlbl) {
        if (StringUtils.isBlank(qlbl)) {
            return null;
        }
        if (StringUtil.isChinese(qlbl)) {
            return null;
        }
        if (qlbl.contains(Constants.SLASH)) {
            return qlbl;
        }
        if (qlbl.endsWith(Constants.PERCENT)) {
            qlbl = qlbl.replace("%", "");
        }
        double qlblDouble = Double.parseDouble(qlbl);
        BigDecimal qlblBig = BigDecimal.valueOf(qlblDouble);
        // 检查 qlblBig 是否 ≤ 1（包含负数和 [0,1] 区间）
        if (qlblBig.compareTo(BigDecimal.ONE) <= 0) {
            return qlbl;
        }
        return CommonUtil.formatEmptyValue(qlblBig.divide(new BigDecimal(100L), 2, BigDecimal.ROUND_HALF_UP));
    }

    /**
     * 移除小数点后面多余的0
     *
     * @param number
     * @return
     */
    public static String removeTrailingZeros(String number) {
        if (number == null) {
            return null;
        }
        return number.replaceAll("(\\.\\d*?[1-9])0+$", "$1").replaceAll("\\.0+$", "");
    }

    /**
     * 是否为数字
     *
     * @param str
     * @return
     */
    public static boolean isNumeric(String str) {
        if (StringUtils.isNotBlank(str)) {
            Pattern pattern = Pattern.compile("[0-9]*");
            return pattern.matcher(str).matches();
        }
        return false;
    }
}