package cn.gtmap.hlw.core.validator;

import cn.gtmap.hlw.core.annotation.validator.IdCardCheck;
import cn.gtmap.hlw.core.constant.Constants;
import cn.gtmap.hlw.core.enums.dict.ZjlxEnum;
import cn.gtmap.hlw.core.util.validator.ValidationUtils;
import org.apache.commons.lang3.StringUtils;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.util.Calendar;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2023/10/12
 * @description
 */
public class IdCardCheckValidator implements ConstraintValidator<IdCardCheck, String> {

    private String message;

    @Override
    public void initialize(IdCardCheck constraintAnnotation) {
        this.message = constraintAnnotation.message();
    }

    @Override
    public boolean isValid(String str, ConstraintValidatorContext constraintValidatorContext) {
        //为空不验证
        if (StringUtils.isBlank(str) || StringUtils.contains(Constants.DASH, str)) {
            return true;
        }
        String[] split = str.split(Constants.DASH);
        String zjlx = split[1];
        if (!StringUtils.equals(zjlx, ZjlxEnum.SFZJZL_SFZ.getCode())) {
            return true;
        }
        str = split[0];
        if (StringUtils.isBlank(str)) {
            return true;
        }
        str = str.toUpperCase();
        if (str.length() != 15 && str.length() != 18) {
            ValidationUtils.reset(constraintValidatorContext, str + Constants.DASH + message);
            return false;
        }
        int y = 0, m = 0, d = 0;
        if (str.length() == 15) {
            y = Integer.parseInt("19" + str.substring(6, 8), 10);
            m = Integer.parseInt(str.substring(8, 10), 10);
            d = Integer.parseInt(str.substring(10, 12), 10);
        } else if (str.length() == 18) {
            if (str.indexOf(Constants.X) >= 0 && str.indexOf(Constants.X) != 17) {
                ValidationUtils.reset(constraintValidatorContext, str + Constants.DASH + message);
                return false;
            }
            char verifyBit = getVerifyBit(str);
            if (str.charAt(17) != verifyBit) {
                ValidationUtils.reset(constraintValidatorContext, str + Constants.DASH + message);
                return false;
            }
            y = Integer.parseInt(str.substring(6, 10), 10);
            m = Integer.parseInt(str.substring(10, 12), 10);
            d = Integer.parseInt(str.substring(12, 14), 10);
        }
        int currentY = Calendar.getInstance().get(Calendar.YEAR);
        if (y > currentY || y < 1870) {
            ValidationUtils.reset(constraintValidatorContext, str + Constants.DASH + message);
            return false;
        }
        if (m < 1 || m > 12) {
            ValidationUtils.reset(constraintValidatorContext, str + Constants.DASH + message);
            return false;
        }
        if (d < 1 || d > 31) {
            ValidationUtils.reset(constraintValidatorContext, str + Constants.DASH + message);
            return false;
        }
        return true;
    }

    private static char getVerifyBit(String str) {
        char verifyBit = 0;
        int sum = (str.charAt(0) - '0') * 7 + (str.charAt(1) - '0') * 9 + (str.charAt(2) - '0') * 10
                + (str.charAt(3) - '0') * 5 + (str.charAt(4) - '0') * 8 + (str.charAt(5) - '0') * 4
                + (str.charAt(6) - '0') * 2 + (str.charAt(7) - '0') * 1 + (str.charAt(8) - '0') * 6
                + (str.charAt(9) - '0') * 3 + (str.charAt(10) - '0') * 7 + (str.charAt(11) - '0') * 9
                + (str.charAt(12) - '0') * 10 + (str.charAt(13) - '0') * 5 + (str.charAt(14) - '0') * 8
                + (str.charAt(15) - '0') * 4 + (str.charAt(16) - '0') * 2;
        sum = sum % 11;
        char[] verifyBits = {'1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2'};
        if (sum >= 0 && sum < verifyBits.length) {
            verifyBit = verifyBits[sum];
        }
        return verifyBit;
    }
}
