package cn.gtmap.hlw.core.util.encryption.sm4;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.AlgorithmParameterSpec;

/**
 * @author <a href="mailto:jinfei@gtmap.cn">jinfei</a>
 * @Version 1.0，2023/03/27
 * @description 对称加密AES DES DESede等公用工具类
 */
public abstract class CipherUtil {

    protected static final Logger LOGGER = LoggerFactory.getLogger(CipherUtil.class);

    /**
     * 随机数生成算法  SHA1PRNG和NativePRNG
     */
    protected static final String SHA1_SECURE_RANDOM = "SHA1PRNG";
    protected static final String NATIVE_SECURE_RANDOM = "NativePRNG";

    static {
        // 防止内存中出现多次BouncyCastleProvider的实例
        if (null == Security.getProvider(BouncyCastleProvider.PROVIDER_NAME)) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * @description 加密
     * @author <a href="mailto:jinfei@gtmap.cn">jinfei</a>
     * @date 2023-3-27 18:44
     * @param msgBytes 加密数据
     * @param keyBytes 加密秘钥
     * @param ivBytes 偏移量
     * @param algorithmName 算法名称
     * @param cipherAlgorithm 密码算法
     * @return byte 加密后字节数组
     */
    public static byte[] encrypt(byte[] msgBytes, byte[] keyBytes, byte[] ivBytes, String algorithmName, String cipherAlgorithm) {
        try {
            Cipher cipher = Cipher.getInstance(cipherAlgorithm);
            SecretKeySpec secretKeySpec = new SecretKeySpec(keyBytes, algorithmName);
            if (ivBytes != null) {
                AlgorithmParameterSpec paramSpec = new IvParameterSpec(ivBytes);
                cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, paramSpec);
            } else {
                cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec);
            }
            return cipher.doFinal(msgBytes);
        } catch (Exception e) {
            LOGGER.error("{}加密失败：{}", algorithmName, e);
        }
        return null;
    }

    /**
     * @description 解密
     * @author <a href="mailto:jinfei@gtmap.cn">jinfei</a>
     * @date 2023-3-27 19:07
     * @param msgBytes 解密数据
     * @param keyBytes 解密秘钥
     * @param ivBytes 偏移量
     * @param algorithmName 算法名称
     * @param cipherAlgorithm 密码算法
     * @return byte 解密字节数组
     */
    public static byte[] decrypt(byte[] msgBytes, byte[] keyBytes, byte[] ivBytes, String algorithmName, String cipherAlgorithm) {
        try {
            Cipher cipher = Cipher.getInstance(cipherAlgorithm);
            SecretKeySpec secretKeySpec = new SecretKeySpec(keyBytes, algorithmName);
            if (ivBytes != null) {
                AlgorithmParameterSpec paramSpec = new IvParameterSpec(ivBytes);
                cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, paramSpec);
            } else {
                cipher.init(Cipher.DECRYPT_MODE, secretKeySpec);
            }
            return cipher.doFinal(msgBytes);
        } catch (Exception e) {
            LOGGER.error("{}解密失败：{}", algorithmName, e);
        }
        return null;
    }

    /**
     * @description 生成密钥
     * @author <a href="mailto:jinfei@gtmap.cn">jinfei</a>
     * @date 2023-3-28 15:46
     * @param algorithmName 算法名称
     * @param keySize 算法密钥长度
     * @return byte
     */
    public static byte[] generateKey(String algorithmName, int keySize) {
        return generateKey(algorithmName, keySize, null);
    }

    /**
     * @description 生成密钥
     * @author <a href="mailto:xusong@gtmap.cn">xusong</a>
     * @date 2023-3-28 15:48
     * @param algorithmName 算法名称
     * @param keySize 算法密钥长度
     * @param secureRandom 随机数生成器
     * @return byte
     */
    public static byte[] generateKey(String algorithmName, int keySize, SecureRandom secureRandom) {
        byte[] keyBytes = null;
        try {
            KeyGenerator keyGenerator = KeyGenerator.getInstance(algorithmName);
            if (secureRandom != null) {
                keyGenerator.init(keySize, secureRandom);
            } else {
                keyGenerator.init(keySize);
            }
            SecretKey secretKey = keyGenerator.generateKey();
            keyBytes = secretKey.getEncoded();
        } catch (NoSuchAlgorithmException e) {
            LOGGER.error("生成{}密钥失败：{}", algorithmName, e);
        }
        return keyBytes;
    }
}
