/**













 */
package cn.gtmap.hlw.core.aop;

import cn.gtmap.hlw.core.annotation.PreAuth;
import cn.gtmap.hlw.core.enums.error.ErrorEnum;
import cn.gtmap.hlw.core.exception.BizException;
import cn.gtmap.hlw.core.fun.AuthFun;
import cn.gtmap.hlw.core.util.clazz.ClassUtil;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.expression.BeanFactoryResolver;
import org.springframework.core.MethodParameter;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;

/**
 * @author <a href="mailto:dingweiwei@gtmap.cn">dingweiwei</a>
 * @version 1.0, 2024/8/1
 * @description 用于检查权限 规定访问权限
 */
@Component
@Aspect
public class AuthAspect implements ApplicationContextAware {

	/**
	 * 表达式处理
	 */
	private static final ExpressionParser SPEL_PARSER = new SpelExpressionParser();

	private ApplicationContext applicationContext;

	/**
	 * 获取方法上的参数
	 *
	 * @param method 方法
	 * @param args   变量
	 * @return {SimpleEvaluationContext}
	 */
	private StandardEvaluationContext getEvaluationContext(Method method, Object[] args) {
		// 初始化Sp el表达式上下文，并设置 AuthFun
		final StandardEvaluationContext context = new StandardEvaluationContext(new AuthFun());
		// 设置表达式支持spring bean
		context.setBeanResolver(new BeanFactoryResolver(applicationContext));
		for (int i = 0; i < args.length; i++) {
			// 读取方法参数
			final MethodParameter methodParam = ClassUtil.getMethodParameter(method, i);
			// 设置方法 参数名和值 为sp el变量
			context.setVariable(methodParam.getParameterName(), args[i]);
		}
		return context;
	}

	/**
	 * 处理权限
	 *
	 * @param point 切点
	 */
	private boolean handleAuth(ProceedingJoinPoint point) {
		final MethodSignature ms = (MethodSignature) point.getSignature();
		final Method method = ms.getMethod();
		// 读取权限注解，优先方法上，没有则读取类
		final PreAuth preAuth = ClassUtil.getAnnotation(method, PreAuth.class);
		// 判断表达式
		final String condition = preAuth.value();
		if (StringUtils.isNotBlank(condition)) {
			final Expression expression = SPEL_PARSER.parseExpression(condition);
			// 方法参数值
			final Object[] args = point.getArgs();
			final StandardEvaluationContext context = getEvaluationContext(method, args);
			return Boolean.TRUE.equals(expression.getValue(context, Boolean.class));
		}
		return false;
	}

	/**
	 * 切 方法 和 类上的 @PreAuth 注解
	 *
	 * @param point 切点
	 * @return Object
	 * @throws Throwable 没有权限的异常
	 */
	@Around("@annotation(cn.gtmap.hlw.core.annotation.PreAuth) || "
			+ "@within(cn.gtmap.hlw.core.annotation.PreAuth)")
	public Object preAuth(ProceedingJoinPoint point) throws Throwable {
		if (handleAuth(point)) {
			return point.proceed();
		}
		throw new BizException(ErrorEnum.UN_AUTHORIZED);
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}

}
