package cn.gtmap.gtcc.utils;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

/**
 * @author Fjj
 * @date 2018/1/19
 */
public class FileUtil {

    private FileUtil(){

    }

    static final Logger logger = LoggerFactory.getLogger(FileUtil.class.getName());

    public static final String IMAGE_PNG = "png";
    public static final String IMAGE_JPG = "jpg";
    public static final String IMAGE_BMP = "bmp";
    public static final String IMAGE_GIF = "gif";
    public static final String IMAGE_TIF = "tif";

    static List<String> images = new ArrayList<String>();

    static {
        images.add(IMAGE_PNG);
        images.add(IMAGE_JPG);
        images.add(IMAGE_BMP);
        images.add(IMAGE_GIF);
        images.add(IMAGE_TIF);
    }

    /**
     * 判别文件是否为图片
     *
     * @param file
     * @return
     */
    public static final boolean isImage(File file) {
        Assert.notNull(file, "文件不可为空");
        String suffix = file.getPath().substring(file.getPath().lastIndexOf(".") + 1, file.getPath().length());
        if (StringUtils.isBlank(suffix)) {
            return false;
        }
        if (images.contains(suffix.toLowerCase())) {
            return true;
        }
        return false;
    }

    /**
     * 删除文件或者文件夹(文件夹中存在文件夹时无法处理)
     *
     * @param path
     * @return
     * @throws IOException
     */
    public static final boolean delFileOrDirectory(String path) throws IOException {
        boolean result = false;
        try {
            File file = new File(path);
            if (file.exists()) {
                if (file.isDirectory()) {
                    File[] files = file.listFiles();
                    for (int i = 0; i < files.length; i++) {
                         files[i].delete();
                    }
                    result = file.delete();
                    return result;
                } else {
                    result = file.delete();
                    return result;
                }
            }
            return result;
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 删除指定目录及其下属所有文件
     *
     * @param dir
     * @return
     */
    public static boolean deleteDir(File dir) {
        if (dir.isDirectory()) {
            String[] children = dir.list();
            //递归删除目录中的子目录下
            for (int i = 0; i < children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
        return dir.delete();
    }

    /**
     * 下载文件
     *
     * @param urlString
     * @param saveFolderPath
     * @param saveFileName
     * @return
     * @throws Exception
     */
    public static File downloadFile(String urlString, String saveFolderPath, String saveFileName) {
        InputStream is = null;
        OutputStream os = null;
        String saveFilePath = saveFolderPath + File.separator + saveFileName;
        try {
            URL url = new URL(urlString);
            URLConnection con = url.openConnection();
            con.setConnectTimeout(5 * 1000);
            is = con.getInputStream();

            byte[] bs = new byte[2048];
            // 读取到的数据长度
            int len;
            // 存放上传文件的总的文件夹
            File sf = new File(saveFolderPath);
            if (!sf.exists()) {
                sf.mkdirs();
            }
            os = new FileOutputStream(saveFilePath);
            // 开始读取
            while ((len = is.read(bs)) != -1) {
                os.write(bs, 0, len);
            }
        } catch (Exception e) {
            logger.error(e.toString());
            return null;
        } finally {
            // 上传完毕，关闭所有连接
            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    logger.error(e.toString());
                }
            }
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    logger.error(e.toString());
                }
            }
        }
        return new File(saveFilePath);
    }

    public static byte[] file2byte(String filePath) throws IOException{
        byte[] buffer = null;
        File file = new File(filePath);
        FileInputStream fis = null;
        ByteArrayOutputStream bos = new ByteArrayOutputStream(1024);
        try {
            fis = new FileInputStream(file);
            byte[] b = new byte[1024];
            int n;
            while ((n = fis.read(b)) != -1) {
                bos.write(b, 0, n);
            }
            buffer = bos.toByteArray();
        } catch (FileNotFoundException e) {
            logger.error(e.toString());
        } catch (IOException e) {
            logger.error(e.toString());
        }finally {
            if(fis != null){
                fis.close();
            }
            bos.close();
        }
        return buffer;
    }

    public static void byte2File(byte[] buf, String filePath, String fileName) {
        BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        File file = null;
        try {
            File dir = new File(filePath);
            if (!dir.exists() && dir.isDirectory()) {
                dir.mkdirs();
            }
            file = new File(filePath + File.separator + fileName);
            fos = new FileOutputStream(file);
            bos = new BufferedOutputStream(fos);
            bos.write(buf);
        } catch (Exception e) {
            logger.error(e.toString());
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e) {
                    logger.error(e.toString());
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    logger.error(e.toString());
                }
            }
        }
    }

    /**
     * byte[] to file
     *
     * @param bytes
     * @param filePath
     * @param fileName
     * @return file
     */
    public static void byte2File2(byte[] bytes, String filePath, String fileName) {
        BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        File file;
        try {
            File dir = new File(filePath);
            //判断文件目录是否存在
            if (!dir.exists() && !dir.isDirectory()) {
                dir.mkdirs();
            }
            file = new File(filePath + File.separator + fileName);
            fos = new FileOutputStream(file);
            bos = new BufferedOutputStream(fos);
            bos.write(bytes);
        } catch (Exception e) {
            logger.error(e.toString());
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e1) {
                    logger.error(e1.toString());
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e1) {
                    logger.error(e1.toString());
                }
            }
        }
    }

    /***
     * base64 to file
     * @param base64
     * @param filePath
     * @param fileName
     */
    public static void base64ToFile(String base64, String filePath, String fileName) {
        File dir = new File(filePath);
        //判断文件目录是否存在
        if (!dir.exists() && !dir.isDirectory()) {
            dir.mkdirs();
        }
        try {
            File file = new File(filePath + File.separator + fileName);
            if (!file.exists()) {
                file.createNewFile();
            }
            Files.write(Paths.get(filePath + File.separator + fileName),
                    Base64.getDecoder().decode(base64), StandardOpenOption.CREATE);
        } catch (IOException e) {
            logger.error(e.toString());
        }
    }

    /**
     * 生成临时文件夹
     * @param tempPath
     */
    public static File createTempDir(String tempPath){
        Path Location = Paths.get(tempPath);
        String tempFolderPath = Location.toAbsolutePath().toString().concat(File.separator + "temp_" + System.currentTimeMillis());
        File tempFolder = new File(tempFolderPath);
        if (!tempFolder.exists()) {
            tempFolder.mkdirs();
        }
        return tempFolder;
    }
}
