package cn.gtmap.gtcc.support.jpa.converter;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import javax.persistence.AttributeConverter;
import javax.persistence.Converter;
import java.io.IOException;

/**
 * .JSONAttrConverter
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 2017/11/9 10:43
 */
@Converter(autoApply = true)
public class JSONAttrConverter<T> implements AttributeConverter<T, String> {

    protected final Logger logger = LoggerFactory.getLogger(getClass());

    private Class<T> clazz;

    private final ObjectMapper objectMapper = new ObjectMapper();

    public JSONAttrConverter() {
        this.clazz = (Class<T>) getClass();
    }

    /**
     * Converts the value stored in the entity attribute into the
     * data representation to be stored in the database.
     *
     * @param attribute the entity attribute value to be converted
     * @return the converted data to be stored in the database column
     */
    @Override
    public String convertToDatabaseColumn(T attribute) {
        if (attribute != null) try {
            return objectMapper.writeValueAsString(attribute);
        } catch (JsonProcessingException e) {
            logger.error(e.toString());
        }
        return null;
    }

    /**
     * Converts the data stored in the database column into the
     * value to be stored in the entity attribute.
     * Note that it is the responsibility of the converter writer to
     * specify the correct dbData type for the corresponding column
     * for use by the JDBC driver: i.e., persistence providers are
     * not expected to do such type conversion.
     *
     * @param dbData the data from the database column to be converted
     * @return the converted value to be stored in the entity attribute
     */
    @Override
    public T convertToEntityAttribute(String dbData) {
        if (!StringUtils.isEmpty(dbData)) try {
            return objectMapper.readValue(dbData, clazz);
        } catch (IOException e) {
            logger.error(e.toString());
        }
        return null;
    }
}
