package cn.gtmap.gtcc.support;

import cn.gtmap.gtcc.Constant;
import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.FileCopyUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-3-27 下午1:45
 */
public class BaseController {

    protected static final String RES = "result";

    protected static final String REQUEST_FILE_NOT_FOUND = "request file not found";

    protected static final String LAST_MODIFIED = "Last-Modified";

    protected final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * send file
     *
     * @param file
     * @param response
     */
    protected void sendFile(File file, HttpServletResponse response) throws IOException {
        if (file == null || response == null) {
            return;
        }
        if (file.exists()) {
            response.setDateHeader(LAST_MODIFIED, file.lastModified());
            response.setContentLength((int) file.length());

            try (InputStream stream = new FileInputStream(file);){
                FileCopyUtils.copy(stream, response.getOutputStream());
            } catch (IOException e) {
                logger.error(e.toString());
            }
        } else {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, REQUEST_FILE_NOT_FOUND);
        }

    }

    /**
     * @param inputStream
     * @param response
     * @throws IOException
     */
    protected void sendFile(InputStream inputStream, HttpServletResponse response, String fileName) throws IOException {
        if (inputStream == null || response == null) {
            return;
        }
        if (inputStream.available() > 0) {
            response.addHeader("Content-Disposition", "attachment; filename=" + fileName);
            response.setDateHeader(LAST_MODIFIED, System.currentTimeMillis());
            response.setContentLength(inputStream.available());
            FileCopyUtils.copy(inputStream, response.getOutputStream());
        } else {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, REQUEST_FILE_NOT_FOUND);
        }
    }

    /**
     * send document
     *
     * @param response
     * @param document
     * @throws IOException
     */
    protected void sendDocument(HttpServletResponse response, Document document) throws IOException {
        sendStream(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
    }

    /**
     * send stream
     *
     * @param inputStream
     * @param response
     * @param fileName
     * @throws IOException
     */
    protected void sendStream(InputStream inputStream, HttpServletResponse response, String fileName) throws IOException {
        if (inputStream == null || response == null) {
            return;
        }
        if (inputStream.available() > 0) {
            response.addHeader("Content-Disposition", "attachment; filename=" + URLEncoder.encode(fileName, Constant.CHARSET_UTF8));
            response.setDateHeader(LAST_MODIFIED, System.currentTimeMillis());
            response.setContentLength(inputStream.available());
            FileCopyUtils.copy(inputStream, response.getOutputStream());
        } else {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, REQUEST_FILE_NOT_FOUND);
        }
    }

    /**
     * result map
     *
     * @param value
     * @return
     */
    protected Map<String, Object> result(Object value) {
        Map<String, Object> result = new HashMap<>();
        result.put(RES, value);
        return result;
    }

    /***
     * fail map
     * @param value
     * @return
     */
    protected Map<String, Object> fail(Object value) {
        Map<String, Object> result = new HashMap<>();
        result.put("success", false);
        result.put("msg", value);
        return result;
    }

    /**
     * write error to client
     *
     * @param info
     * @param response
     */
    protected void error(String info, HttpServletResponse response) {
        Map<String, Object> result = new HashMap<>();
        result.put("success", false);
        result.put("msg", info);
        try {
            response.getWriter().write(JSON.toJSONString(result));
        } catch (IOException e) {
            logger.error(e.getLocalizedMessage());
        }
    }

    /**
     * write result to client
     *
     * @param value
     * @param response
     * @return
     */
    protected void result(Object value, HttpServletResponse response) {
        Map<String, Object> result = new HashMap<>();
        result.put(RES, value);
        try {
            response.getWriter().write(JSON.toJSONString(result));
        } catch (IOException e) {
            logger.error(e.getLocalizedMessage());
        }
    }

}
